// -*- mode: java; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*-

package antichess.test;

import antichess.*;
import java.io.*;
import java.util.*;
import junit.framework.*;

/**
 * Tests for the GameReader class
 **/
public class GameReaderTests extends TestCase
{
	public GameReaderTests() {
	}
	
	public void testInvalidGame() {
		try {
			GameReader reader = new GameReader(GameReaderTests.class.getResourceAsStream("invalid_game.xml"));
		} catch(IOException e) {
			fail("IO Exception reading test file: " + e);
		} catch(InvalidGameFileException e) {
			// OK
			return;
		}
		fail("Didn't throw an exception loading invalid_game.xml");
	}

	private Piece[] sampleGamePieces = new Piece[] {
		new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 7, 2),
		new Piece(Player.BLACK, PieceTypeFactory.getKingType(), 7, 4),
		new Piece(Player.BLACK, PieceTypeFactory.getBishopType(), 7, 5),
		new Piece(Player.BLACK, PieceTypeFactory.getKnightType(), 7, 6),
		new Piece(Player.BLACK, PieceTypeFactory.getRookType(), 7, 7),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 0),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 3),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 4),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 5),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 6, 7),
		new Piece(Player.BLACK, PieceTypeFactory.getPawnType(), 5, 6),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 2, 5),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, 0),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, 3),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, 4),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, 6),
		new Piece(Player.WHITE, PieceTypeFactory.getPawnType(), 1, 7),
		new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 2),
		new Piece(Player.WHITE, PieceTypeFactory.getQueenType(), 0, 3),
		new Piece(Player.WHITE, PieceTypeFactory.getKingType(), 0, 4),
		new Piece(Player.WHITE, PieceTypeFactory.getBishopType(), 0, 5),
		new Piece(Player.WHITE, PieceTypeFactory.getKnightType(), 0, 6),
		new Piece(Player.WHITE, PieceTypeFactory.getRookType(), 0, 7),
	};

	private String sampleBoardXml = "<pieces> <square id=\"c8\" side=\"white\" piece=\"rook\" /> <square id=\"e8\" side=\"black\" piece=\"king\" /> <square id=\"f8\" side=\"black\" piece=\"bishop\" /> <square id=\"g8\" side=\"black\" piece=\"knight\" /> <square id=\"h8\" side=\"black\" piece=\"rook\" /> <square id=\"a7\" side=\"black\" piece=\"pawn\" /> <square id=\"d7\" side=\"black\" piece=\"pawn\" /> <square id=\"e7\" side=\"black\" piece=\"pawn\" /> <square id=\"f7\" side=\"black\" piece=\"pawn\" /> <square id=\"h7\" side=\"black\" piece=\"pawn\" /> <square id=\"g6\" side=\"black\" piece=\"pawn\" /> <square id=\"f3\" side=\"white\" piece=\"pawn\" /> <square id=\"a2\" side=\"white\" piece=\"pawn\" /> <square id=\"d2\" side=\"white\" piece=\"pawn\" /> <square id=\"e2\" side=\"white\" piece=\"pawn\" /> <square id=\"g2\" side=\"white\" piece=\"pawn\" /> <square id=\"h2\" side=\"white\" piece=\"pawn\" /> <square id=\"c1\" side=\"white\" piece=\"bishop\" /> <square id=\"d1\" side=\"white\" piece=\"queen\" /> <square id=\"e1\" side=\"white\" piece=\"king\" /> <square id=\"f1\" side=\"white\" piece=\"bishop\" /> <square id=\"g1\" side=\"white\" piece=\"knight\" /> <square id=\"h1\" side=\"white\" piece=\"rook\" /> </pieces>";
	
    public void testSampleGame() {
		GameReader reader;
		AntichessBoard sample = new AntichessBoard();
		try {
			reader = new GameReader(GameReaderTests.class.getResourceAsStream("sample_game.xml"));
		} catch(Exception e) {
			fail("Failure loading sample_game.xml: " + e);
			return;
		}

		try {
			GameReader.loadBoard(sample, sampleBoardXml, Player.BLACK);
		} catch(Exception e) {
			fail("Failure loading sample game <pieces>" + e);
			return;
		}

		Board board = reader.getBoard();
		GameTimer whiteTimer = reader.getWhiteTimer();
		GameTimer blackTimer = reader.getBlackTimer();

		assertTrue("Sample game was timed", reader.isTimed());

		assertEquals("White start time", 300000, whiteTimer.getStartTime());
		assertEquals("White end time", 290000, whiteTimer.getTime());
		assertEquals("Black start time", 300000, blackTimer.getStartTime());
		assertEquals("Black end time", 290000, blackTimer.getTime());

		List<Piece> pieces = new ArrayList<Piece>();
		for(Piece p : sampleGamePieces)
			pieces.add(p);

		ChessBoard expectedBoard = new ChessBoard();
		expectedBoard.loadGame(pieces, Player.BLACK, null);

		assertEquals("It's black's move", board.getPlayer(), Player.BLACK);
		assertEquals("It's black's move (from fragment)", sample.getPlayer(), Player.BLACK);
		assertTrue(board.isGameOver());
		assertEquals(board.getGameOverReason(), ChessBoard.CHECKMATE);

		for(int i=0;i<board.getRows();i++) {
			for(int j=0;j<board.getColumns();j++) {
				assertEquals("Piece at " + i + "," + j,
							 expectedBoard.getPieceAt(i,j),
							 board.getPieceAt(i,j));
				assertEquals("Piece (from fragment) at " + i + "," + j,
							 expectedBoard.getPieceAt(i,j),
							 sample.getPieceAt(i,j));
			}
		}
	}

	public void testStartGame() {
		GameReader reader;
		try {
			reader = new GameReader(GameReaderTests.class.getResourceAsStream("initial_game.xml"));
		} catch(Exception e) {
			fail("Failure loading initial_game.xml: " + e);
			return;
		}
		Board board = reader.getBoard();
		GameTimer whiteTimer = reader.getWhiteTimer();
		GameTimer blackTimer = reader.getBlackTimer();

		assertFalse("Game was not timed", reader.isTimed());
		ChessBoard expectedBoard = new ChessBoard();
		expectedBoard.newGame();

		assertEquals("It's white's move", board.getPlayer(), Player.WHITE);
		assertFalse(board.isGameOver());

		for(int i=0;i<board.getRows();i++) {
			for(int j=0;j<board.getColumns();j++) {
				assertEquals("Piece at " + i + "," + j,
							 expectedBoard.getPieceAt(i,j),
							 board.getPieceAt(i,j));
			}
		}
	}
}
