/**
 ** dl.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 ** dl.c    randy sargent   rsargent@athena.mit.edu
 **
 **         evolved from dl.c by henry minsky  hqm@ai.mit.edu
 **/

#include CONFIG
#include <confexec.h>
#include <util.h>
#include <queue.h>
#include <iob.h>
#include <binrec.h>
#include <s19.h>
#include <board.h>
#include <bs6811.h>
#include <dl6811.h>
#include <parsargs.h>
#include <binrec.h>
#include <radix.h>
#include <filelib.h>

#include "dl.h"

#ifdef MAC
#include <mac-args.h>
#endif

#ifndef IC_LIB_DIRECTORY
#define IC_LIB_DIRECTORY LIB_DIRECTORY "ic/"
#endif
  
IOStream *serial_iostream, *keyboard_iostream;

char lib_directory_config[]=
  MAKE_CONFIGURATION(IC_LIB_DIRECTORY,
		     "Directory where libraries reside");
#define lib_directory GET_CONFIGURATION(lib_directory_config)


char port_config[]=MAKE_CONFIGURATION(DEFAULT_SERIAL_PORT, "Default serial device name (in the form /dev/*)");
#define port GET_CONFIGURATION(port_config)


char config_reg_config[]=MAKE_CONFIGURATION("0C", "Value to initialize config register to (hexadecimal value.  0C turns off eeprom and rom");
#define config_reg GET_CONFIGURATION(config_reg_config)

char bootstrapfile[200];

Int noinit= 0;
Int config= 0;

Int queue_size= 700; /*default queue size*/
Int debug_serial= 0;
Int debug_serial_error= 0;
Int no_init_motors= 0;
Int dl_eeprom= 0;

void dl_rev1_5(void) {
    dl_ignore= 1;
    board_loopback= 1;
    dl_eeprom= 1;
    strcpy(config_reg, "0xff");
}

void dl_ignore_input(void) {
    board_no_return_comm= 1;
    dl_ignore= 1;
}

Arg_descriptor ad[] = {
    {"port", "s", port, "Serial device name (in the form /dev/*)"},
    {"libdir", "s", lib_directory, "Library directory"},
    {"noinit", "b", &noinit, "Do not download bootstrap communications to board"},
    {"bootstrap", "s", bootstrapfile, "Filename (.s19 format) to use instead of standard bootstrap"},
    {"debug_bs", "b", &dl_debug, "Turn on verbose bootstrap downloading debugging messages"},
    {"debug_board", "b", &board_debug, "Turn on verbose second-stage downloading debugging messages"},
    {"config_reg", "s", config_reg, "Set config to N (hexadecimal).  0C turns off eeprom and rom (for IC)"},
    {"config", "b", &config, "Reconfigure executable (executable must be in current directory)"},
    {"no_init_motors", "b", &no_init_motors, "Don't send board-specific code to turn off motors"},
    {"bs_ignore", "b", &dl_ignore, "Ignore serial in during bootstrap download (rev 1.5)"},
    {"loopback", "b", &board_loopback, "Expect board to echo characters (rev 1.5)"},
    {"eeprom", "b", &dl_eeprom, "Program eeprom, not ram"},
    {"1.5", "p", (void*) dl_rev1_5, "Set -bs_ignore, -loopback, -eeprom, -config_reg 0xff"},
    {"ignore_input", "p", (void*) dl_ignore_input, "Don't pay attention to returning serial"},
    {"io_debug", "b", &io_debug, "IO module debugging"},
    {"board_debug", "b", &board_debug, "Board module debugging"},
    LAST_ARG_DESCRIPTOR
};

void dl_help(char **argv)
{
	printf("Arguments to dl are:\n");
	printf("filename.s19                         Download this filename\n"); 
#ifdef MAC
	printf("-port printer    or    -port modem   Set which serial port to use\n");
	printf("-libdir :foldername:                 Set which folder has the .s19 files\n"); 
#endif
#ifdef PC
	printf("-port com1 (or com2-4)               Set which serial port to use\n");
	printf("-libdir directory                    Set which directory has the .s19 files\n");
#endif
#ifdef UNIX
	printf("-port /dev/???                       Set which serial port to use\n");
	printf("-libdir directory                    Set which directory has the .s19 files\n");
#endif
	printf("\n");
	printf("Possibly useful for boards other than the 6.270 board or rug warrior:\n");
	printf("-bootstrap filename.s19     Use filename.s19 as the bootstrap\n");
	printf("-config_reg value           Set config register to value (in hex).\n");
	printf("-bs_ignore                  Ignore serial responses in bootstrap sequence\n");
	printf("-loopback                   Expect hardware serial echoes from board\n");
	printf("-eeprom                     Program EEPROM, not RAM\n");
	printf("-1.5                        Set -bs_ignore, -loopback, -eeprom, config_reg 0xff\n");
	printf("-ignore_input               Completely ignore returning serial\n");
	printf("\n");                            
#ifdef MAC	
	printf("Example: pcoder22.s19 -port modem -libdir :my-favorite-libs:\n");  
#endif
#ifdef PC
	printf("Example: dl pcoder22.s19 -port com2 -libdir /ic/libs\n");
#endif
#ifdef UNIX
	printf("Example: dl pcoder22.s19 -port /dev/ttya -libdir /ic/libs\n");
#endif
	
}

void dl_usage(char **argv)
{    
	dl_help(argv);
    printf("Press any key to continue...\n");    
    io_getchar(keyboard_iostream, -1);
    myexit(1);
}

void myexit(Int n)
{
  io_sigint_interrupt();
  exit(n);
}

/********/
/* MAIN */
/********/

#ifdef MAC
void main()
#else
void main(int argc, char **argv)
#endif
{
    FILE *stream= NULL;
    Stream st;
    char **leftover_args;
    char filename[200];
    Int  config_value, i;

    (void)argc;

#ifdef MAC
    mac_get_args("dl", "pcode_r22.s19 -port modem -libdir :ic-libraries:", dl_help);
#endif

    keyboard_iostream= io_open_stdin();
    io_make_stdin_unbuffered();

    leftover_args= parse_args(ad, argv+1);

    printf("6811 .s19 file downloader.  Version 7.32  %s\n", __DATE__);
    printf("        Copyright Randy Sargent 1994\n");
               
    
    for (i= 0; leftover_args[i]; i++) {  
 	if (leftover_args[i][0]== '-') {
	    dl_usage(argv);
	}
    }
 
    if (config) {
	configure_executable(argv[0], keyboard_iostream, stdout);
	myexit(0);
    }

    config_value= (Int) radix_string_to_integer(config_reg, 'x');

    if (leftover_args[0] == NULL) dl_usage(argv);
    if (leftover_args[1] != NULL) dl_usage(argv);
    
#ifdef MAC
	printf("Using the %s port.\n", port);
#else	    
    printf("Using port %s\n", port);
#endif
    
    serial_iostream= io_open_serial(port);
    if (!serial_iostream) {
	printf("Cannot open serial port %s\n", port);
	dl_usage(argv);
    }
    
    board_set_streams(serial_iostream, keyboard_iostream);

    {
        char *suffix= strrchr(leftover_args[0],'.');
        strcpy(filename, leftover_args[0]);
        if (!suffix) {
            strcat(filename, ".s19");
        } else if (!stricmp(suffix+1, "asm")) {
            printf("Can't download a .asm file, silly!\n");
            dl_usage(argv);
        } else if (stricmp(suffix+1, "s19")) {
            printf("You should be downloading an .s19 file\n");
            dl_usage(argv);
        }
    }

  download_again:
    if (!noinit) {
	Binrec *rec;
	if (bootstrapfile[0]) {
	    FILE *bsfile= fopen_here_or_there_with_format
	      (lib_directory, bootstrapfile, "r", "Using bootstrap file %s\n");
	    if (!bsfile) myexit(1);
	    rec= read_contiguous_s19_file(bsfile);
	}
	else {
	    rec= read_contiguous_s19_string(bootstrapstring);
	}

	if (!rec) {
	    die(("Error in parsing .s19 bootstrap\n"));
	}
	dl_6811_user_friendly(serial_iostream, rec->data, rec->len, keyboard_iostream);
    }

    /* check config register */

    io_serial_init(9600L, serial_iostream);
    msleep(500);
    
    if (!no_init_motors) {
	/* turn off motors for sbot or rev 2 */
	board_write_ram(0x7000, 0x00);  
    }

    if (!board_no_return_comm) {
	Int current_config= board_read_mem(0x103f);
	printf("Config is 0x%02lX\n", current_config);
	if (current_config != config_value) {
		printf("\nBoard's config register is incorrect.\n");
	    printf("Initializing config register to 0x%02lX.\n", config_value);
	    board_erase_config();
	    board_write_eeprom(0x103f, config_value);
	    printf("You will now need to place your board in download mode again.\n");
	    goto download_again;
	}
    } else {
	board_erase_config();
	board_write_eeprom(0x103f, config_value);
    }
    
    /* now download the user prog */
    stream= fopen_here_or_there(lib_directory, filename, "r");
    if (!stream) dl_usage(argv);
    stream_init_from_file(&st, stream);

    printf("\nSending %s\n",filename);

    while (1) {
	Binrec *rec;
	rec= read_s19_record_from_stream(&st);
	if (!rec) break;
	if (dl_eeprom) {
	    long i;
	    for (i= 0; i< rec->len; i++) {
		board_erase_eeprom(rec->addr + i);
		board_write_eeprom(rec->addr + i, rec->data[i]);
	    }
	}
	else {
	    board_download_block(rec->addr, rec->len, rec->data);
	}
	printf(".");
	fflush(stdout);
	free(rec);
    }

    printf("Done.  \nPress the board's RESET button to start %s\n", filename);
    fclose(stream);
#ifdef PC
	printf("Press any key to exit...\n");
	io_getchar(keyboard_iostream, -1);
#endif
    myexit(0);
}

