/* send 24 -- control-x -- will clear input buffer */

/* terminal program */

#include "configx.h"
#include <s19.h>
#include <binrec.h>
#include <binio.h>
#include <common.h>

#include <ctype.h>
#include <signal.h>


#ifdef HAS_TERMIOS_H
#include <termios.h>
#endif
#ifdef HAS_SYS_TERMIOS_H
#include <sys/termios.h>
#endif

#include "ezio.pro"

IOStream *console, *serial;
IOStream *streams[100];

char port_config[]=MAKE_CONFIGURATION(DEFAULT_SERIAL_PORT, "Default serial device name (in the form /dev/?)");
#define port GET_CONFIGURATION(port_config)

long baud= 9600L;
Int showhex= 0;
Int config= 0;
Int s_lines= -1;
Int cr_delay_amount= 50;
Int ezio_debug=1;
Int ezio_timeout=2000;

Int ezio_major_version=0;
Int ezio_minor_version=1;
Int ezio_download_enable= 1;
Int ezio_do_cr= 1;
Table symtab;

Arg_descriptor ad[]=
{
    {"port", "s", port, "Serial device name (in the form /dev/?)"},
    {"baud", "l", &baud, "Baud rate (default 9600)"},
    {"config", "b", &config, "Change default serial port (executable must be in current directory)"},
    LAST_ARG_DESCRIPTOR
};

IOStream *make_window(char *title);

void main(Int argc, char **argv)
{
   char **commands;
   int do_echo=0;
   UNUSED(argc);

   /* parse_args parses all the -flags, and returns an array
      of all the arguments it didn't parse, which in this case are
      commands to send to the board. */
   
   commands =parse_args(ad, argv+1);

   if(commands[0]==NULL) {			/* no extra arguments */
      do_echo=1;
   }

   if(do_echo || config) {
      printf("Ez-io version 0.2\n");
      printf("by Anne Wright (anarch@ai.mit.edu) and Randy Sargent (rsargent@media.mit.edu\n");
      printf("See http://www.ai.mit.edu/people/rsargent/ez-io.html\n");
      
      printf("Port %s, baud %ld\n", port, baud);
   }
   
   if (config) {
       console= io_open_stdin();
       io_make_stdin_unbuffered();
       configure_executable(argv[0], console, stdout);
       exit(0);
   }

   serial= io_open_serial(port);
   if (!serial) {
       printf("Could not open port %s for input\n", port);
       usage(ad, argv, "");
       exit(0);
   }

   io_serial_init(baud, serial);

   if(commands[0]==NULL) {
       printf("No commands;  exiting.\n");
       exit(0);
   }
   else {
       execute_commands(commands);
   }
}

void send_commandf(char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    
    io_discard_input(serial);
    io_putchar(24,serial);		/* send control-x to make
					   board clear input buffer*/
    io_vprintf(serial, fmt, args);
    io_putchar('\r', serial);
    io_flush(serial);
}
    
void send_commands(char **commands)
{
    int i;
    
    io_discard_input(serial);
    io_putchar(24,serial);		/* send control-x to make
					   board clear input buffer*/
    
    for(i=0; commands[i]; i++) {
	io_printf(serial,"%s ", commands[i]);
    }
    io_putchar('\r',serial);
    
    io_flush(serial); /* Buffered output isn't sent until flush */
}

/* timeout is in milliseconds */
void receive_string(char *buf, int timeout)
{
    int c;
    int got_lf= 0;
    int start= mtime();
    
    while( mtime() - start <= timeout) {
	if((c= io_getchar(serial, timeout))==-1) break; /* timed out */

	if (got_lf) {
	    if (c == 'e') {
		/* Seeing the beginning of a prompt. */
		*buf++= 0;
		return;
	    }
	    *buf++= c;
	}
	else if (c == '\n') {
	    /* Got a linefeed.  End of command echo */
	    got_lf= 1;
	}
    }
    /* Timeout */
    *buf++= 0;
    fprintf(stderr, "Timeout from board\n");
    exit(-1);
}

/* TODO: add hexadecimal parsing here */

int parse_number(char *string)
{
    int i;
    if (!string) {
	fprintf(stderr, "Number missing\n");
	exit(-1);
    }
    
    if ((string[0]=='0' && string[1]=='x') ||
	string[0]=='$') {
	/* if start with $ or 0x, do hexadecimal */
	sscanf(string+1+(string[0]=='0'), "%x", &i);
    } else {
	if (string[0]<'0' || string[0]>'9') {
	    fprintf(stderr, "Illegal number\n");
	    exit(-1);
	}
	sscanf(string, "%d", &i);
    }
    return i;
}

void execute_commands(char **commands)
{
    char *command= commands[0];
    
    if (!strcmp(command, "a") ||
	!strcmp(command, "analog")) {
	printf("%d\n", read_analog(parse_number(commands[1])));
    }
    else if (!strcmp(command, "filtered_analog")) {
	printf("%.2f\n", filtered_read_analog(parse_number(commands[1]),5));
    }
    else if (!strcmp(command, "ir")) {
	execute_ir_command(commands+1);
    }
    /* Add your own handlers here */
    /* else if (!strcmp(command, "your_command")) {
          execute_your_command(...);
       } */

    else {
	char buf[1000];
	send_commands(commands);
	receive_string(buf, ezio_timeout);
	printf("%s", buf);
    }
}

void ir_help()
{
}

void execute_ir_command(char **commands)
{
    char *device;
    char *key;

    if (!commands[0] || !commands[1]) {
	ir_help();
	exit(-1);
    }
    device= commands[0];
    key= commands[1];
    
    /* When you add a new device here, please also add it to
       the "ir_help" function */
    
    if (!strcmp(device, "sony_tv")) {
	send_ir_sony_tv(key);
    }
    /* else if (!strcmp(device, "...")) {
          send_...(name);
       } */
    else {
	fprintf(stderr, "Unrecognized remote\n");
	ir_help();
	exit(-1);
    }
}

void send_ir_sony_tv(char *key)
{
    /* Add stuff here! */
}
  
/* Receive a number from the board */
int receive_number(int timeout)
{
    char buf[1000];
    receive_string(buf, timeout);
    return atoi(buf);
}

int read_analog(int ch)
{
    if (ch < 0 || ch > 7) {
	fprintf(stderr, "Analog channel must be 0 to 7\n");
	exit(-1);
    }
    send_commandf("a %d", ch);
    return (receive_number(ezio_timeout));
}

/* Average num_reads sensor readings */
double filtered_read_analog(int ch, int num_reads)
{
    int total= 0, i;
    
    for (i= 0; i< num_reads; i++) {
	total += read_analog(ch);
    }
    return (double) total / (double) num_reads;
}
