/**
 **  cdlib.c   Randy Sargent  rsargent@athena.mit.edu
 **
 **  functions copied from (and modified slightly):
 **	xcd - a motif X client to control a RRD42 cd player.
 **	scott garland (scg@kanaha.idbsu.edu)
 **     who got most of the cd stuff from avolio@decuac.DEC.COM, who
 **	got it from thomas@mipsbx.lkg.dec.com.
 **/

/* 11600-11870 eurythmics wail */

#include CONFIG

#ifdef DECSTATION

#include <sys/cdrom.h>
#include <sys/errno.h>

#include <util.h>
#include <stringlb.h>
#include <fcntl.h>

#include "cdlib.h"

Int cd_fd;

static Int	cd_is_paused = 0, cd_is_ejected = 0;

static struct cd_toc		toc;
static struct cd_toc_entry	*fte, *lte;
static struct cd_playback_status	ps;
static struct cd_playback_control  pc;
static struct cd_playback	pb;
static struct cd_toc_header	th;
static Int    current_volume, current_track, number_of_tracks;

char *cdlib_device_name= "/dev/rrz4c";

void cdlib_open(void)
{
    if (getenv("CDROM")) {
	cdlib_device_name = string_copy(getenv("CDROM"));
    }
    
    if ( !cdlib_device_name ) {
	fprintf(stderr, "Please specify a device name for your CD\n");
	exit(1);
    }

    if ( (cd_fd = open_cd_player(cdlib_device_name)) < 0 ) {
	fprintf(stderr, "Cannot open device %s\n", cdlib_device_name);
	exit(1);
    }
}

void cdlib_close(void)
{
    close(cd_fd);
}

    
void cdlib_eject(void)
{
    ioctl(cd_fd, CDROM_EJECT_CADDY, 0);
    cd_is_ejected= 1;
}

void cdlib_pause(void)
{
    ioctl(cd_fd, CDROM_PAUSE_PLAY, 0);
    cd_is_paused= 1;
}

void cdlib_unpause(void)
{
    ioctl(cd_fd, CDROM_RESUME_PLAY, 0);
    cd_is_paused= 0;
}
    
/*
*	yuk... there must be a better way to stop the cd
*		player, i suppose rebooting is out of the question?
*/
void cdlib_stop(void)  
{
    struct cd_play_audio_msf	msf;

    bzero(&msf, sizeof(msf));

    (void)ioctl(cd_fd, CDROM_PLAY_AUDIO_MSF, &msf);

    cd_is_paused = 0;	/* should probably figure this out as we go */

    return;
}




/**************************************************************************
*
*	functions to access (and otherwise play with) the cd start here
*
*	much of this i got from avolio@decuac.DEC.COM, who i think
*	got it from thomas@mipsbx.lkg.dec.com.
*	
**************************************************************************/

void cdlib_status()
{
    Int	idx = 0;
    Int	dm, tdm, ds, tds;
    
    
    fte = (struct cd_toc_entry *)(toc.toc_buffer + sizeof(th));
    lte = fte + (th.th_ending_track - th.th_starting_track) + 1;
    pb.pb_alloc_length = sizeof(ps);
    pb.pb_buffer = (caddr_t)&ps;
    
    if ( ioctl(cd_fd, CDROM_PLAYBACK_STATUS, &pb) < 0 ) {
	return;
    }
    
    while ( ps.ps_msf.m_units > fte[idx+1].te_msf.m_units ||
	   (ps.ps_msf.m_units == fte[idx+1].te_msf.m_units &&
	    ps.ps_msf.s_units > fte[idx+1].te_msf.s_units) ) {
	
	idx++;
    }
    tds = lte->te_msf.s_units - ps.ps_msf.s_units;
    tdm = lte->te_msf.m_units - ps.ps_msf.m_units;
    if ( tds < 0 )
      tdm -= 1, tds += 60;
    
    ds = fte[idx+1].te_msf.s_units - ps.ps_msf.s_units;
    dm = fte[idx+1].te_msf.m_units - ps.ps_msf.m_units;
    
    if ( ds < 0 )
      dm -= 1, ds += 60;
    
#define cd_percent (100.0 - 100.0 * (tds + tdm * 60) / \
		    ((lte->te_msf.m_units - fte->te_msf.m_units) * 60 + \
		     lte->te_msf.s_units - fte->te_msf.s_units))
  
#define track_percent (100.0 - 100.0 * (ds + dm * 60) / \
		       ((fte[idx+1].te_msf.m_units - fte[idx].te_msf.m_units) * 60 +\
			fte[idx+1].te_msf.s_units - fte[idx].te_msf.s_units))
  
    current_track = fte[idx].te_track_number;
    number_of_tracks = th.th_ending_track;
    
    return;
}



/*
*	this was a pain. hey dec -- "where's the documentation!!"
*
*	level is this case is in the range 0-100. if the 'smart'
*	user overrides the scale resource to change that,
*	then i'll have to fix up the scaling calculation. (which
*	i don't currently do).
*/

void cdlib_set_volume(Int level)
{
    Int			volume = (level * CDROM_MAX_VOLUME) / 100;
    
    pb.pb_alloc_length = sizeof(pc);
    pb.pb_buffer = (caddr_t)&pc;
    bzero(pb.pb_buffer, pb.pb_alloc_length);
    
    pc.pc_chan0_select = CDROM_CHANNEL_0;
    pc.pc_chan1_select = CDROM_CHANNEL_1;
    pc.pc_chan0_volume = pc.pc_chan1_volume = (u_char)volume;

    current_volume= level;
    
    if ( ioctl(cd_fd, CDROM_PLAYBACK_CONTROL, &pb) == -1 ) {
	return;
    }
}


Int open_cd_player(char *cd_device_name)
{
    if ( (cd_fd = open(cd_device_name, O_RDONLY, 0)) < 0 ) {
	return -1;
    }
    
    cdlib_init();
    
    return (cd_fd);
}



Int cdlib_init(void)
{
    Int	mode = CDROM_MSF_FORMAT;

    pb.pb_alloc_length = sizeof(ps);
    pb.pb_buffer = (caddr_t)&ps;
    
    if ( ioctl(cd_fd, CDROM_PLAYBACK_STATUS, &pb) < 0 ) {
	return -1;
    }
    
    if ( ioctl(cd_fd, CDROM_TOC_HEADER, &th) < 0 ) {
	return -1;
    }
    
    toc.toc_address_format = mode;
    toc.toc_starting_track = th.th_starting_track;
    toc.toc_alloc_length = th.th_data_len0 + 256 * th.th_data_len1 + 
      sizeof(th);
    
    if ( toc.toc_buffer == NULL ) 
      toc.toc_buffer = (caddr_t)malloc(toc.toc_alloc_length);
    else 
      toc.toc_buffer = (caddr_t)realloc(toc.toc_buffer, toc.toc_alloc_length);
    
    bzero(toc.toc_buffer, toc.toc_alloc_length);
    
    if ( ioctl(cd_fd, CDROM_SET_ADDRESS_FORMAT, &mode) < 0 ) {
	fprintf(stderr, "can't set address mode\n");
	return -1;
    }

    if ( ioctl(cd_fd, CDROM_TOC_ENTRYS, &toc) < 0 ) {
	fprintf(stderr, "can't do toc entry\n");
	return -1;
    }
    
    fte = (struct cd_toc_entry*)(toc.toc_buffer + sizeof(th));
    lte = fte + (th.th_ending_track - th.th_starting_track) + 1;
    
#ifdef _when_i_fix_this_
    current_track = th.th_starting_track;
    number_of_tracks = th.th_ending_track;
#endif
    cd_is_ejected = 0;

    return 0;
}



void cdlib_play_track(Int start, Int end)
{
    struct cd_play_audio_msf	msf;
    
    
    cd_is_paused = 0;
    if ( cd_is_ejected == 1 )
      cdlib_init();
    
    if ( start < 0 )
      start = th.th_starting_track;
    if ( end < 0 ) 
      end = th.th_ending_track;
    
    fte = (struct cd_toc_entry *)(toc.toc_buffer + sizeof(th));
    lte = fte + (th.th_ending_track - th.th_starting_track) + 1;

    fte += (start - th.th_starting_track);
    lte += (end - th.th_ending_track);
    
    msf.msf_starting_M_unit = fte->te_msf.m_units;
    msf.msf_starting_S_unit = fte->te_msf.s_units;
    msf.msf_starting_F_unit = fte->te_msf.f_units;
    msf.msf_ending_M_unit = lte->te_msf.m_units;
    msf.msf_ending_S_unit = lte->te_msf.s_units;
    msf.msf_ending_F_unit = lte->te_msf.f_units;
    
    if ( msf.msf_ending_F_unit > 0 ) {
	msf.msf_ending_F_unit--;
    } else {
	msf.msf_ending_F_unit = 74;
	if ( msf.msf_ending_S_unit > 0 ) {
	    msf.msf_ending_S_unit--;
	} else {
	    msf.msf_ending_S_unit = 59;
	    msf.msf_ending_M_unit--;
	}
    }
    
    if ( ioctl(cd_fd, CDROM_PLAY_AUDIO_MSF, &msf) < 0 ) {
	return;
    }
    
    cdlib_set_volume(current_volume);
    
    return;
}

void dump_bytes(char *foo, Int n)
{
    Int i;
    for (i= 0; i< n; i++) printf("%c %02X  ", 0xff & foo[i], 0xff & foo[i]);
    printf("\n");
}

void cdlib_track_start(Int track, Int *min, Int *sec, Int *frame)
{
    struct cd_toc_entry *toc_entry;
    
    cd_is_paused = 0;
    if ( cd_is_ejected == 1 )
      cdlib_init();
    
    toc_entry = (struct cd_toc_entry *)(toc.toc_buffer + sizeof(th));
    toc_entry += (track - th.th_starting_track);

    *min= toc_entry->te_msf.m_units;
    *sec= toc_entry->te_msf.s_units;
    *frame= toc_entry->te_msf.f_units;
}

long cdlib_track_start_index(Int track)
{
    Int min, sec, frame;
    cdlib_track_start(track, &min, &sec, &frame);
    return frame + 75 * (sec + (60 * min));
}

void cdlib_play(Int start_m, Int start_s, Int start_f, Int end_m, Int end_s, Int end_f)
{
    struct cd_play_audio_msf	msf;
    
    cd_is_paused = 0;
    if ( cd_is_ejected == 1 )
      cdlib_init();
    
    msf.msf_starting_M_unit = start_m;
    msf.msf_starting_S_unit = start_s;
    msf.msf_starting_F_unit = start_f;
    msf.msf_ending_M_unit = end_m;
    msf.msf_ending_S_unit = end_s;
    msf.msf_ending_F_unit = end_f;
    
    if ( ioctl(cd_fd, CDROM_PLAY_AUDIO_MSF, &msf) < 0 ) {
	return;
    }
    
    cdlib_set_volume(current_volume);
    
    return;
}

void cdlib_play_index(long start, long end)
{
    cdlib_play(start / (60 * 75), (start / 75) % 60, start % 75,
	       end   / (60 * 75), (end   / 75) % 60, end   % 75);
}

void cdlib_position(Int *min, Int *sec, Int *frame)
{
    struct cd_playback_status	ps;
    struct cd_playback	pb;
    
    pb.pb_alloc_length = sizeof(ps);
    pb.pb_buffer = (caddr_t)&ps;
    
    if ( ioctl(cd_fd, CDROM_PLAYBACK_STATUS, &pb) < 0 ) {
	return;
    }

    *min= ps.ps_msf.m_units;
    *sec= ps.ps_msf.s_units;
    *frame= ps.ps_msf.f_units;

    return;
}

long cdlib_position_index(void)
{
    Int min, sec, frame;
    cdlib_position(&min, &sec, &frame);
    return frame + 75 * (sec + (60 * min));
}

#else

void cdlib_open	      (void)  {}
void cdlib_close	      (void)  {}
void cdlib_eject	      (void)  {}
void cdlib_pause	      (void)  {}
void cdlib_unpause     (void)  {}
void cdlib_stop	      (void)    {}
void cdlib_status      ()  {}
void cdlib_set_volume  (Int level)  {}
Int  open_cd_player    (char *cd_device_name)  {return -1;}
Int  cdlib_init	      (void)  {return -1;}
void cdlib_play_track  (Int start, Int end)  {}
void dump_bytes	      (char *foo, Int n)  {}
void cdlib_track_start (Int track, Int *min, Int *sec, Int *frame)  {}
long cdlib_track_start_index(Int track) {return 1;}
void cdlib_play_index  (long start, long end)  {}
void cdlib_position    (Int *min, Int *sec, Int *frame)  {}
long cdlib_position_index (void)  { return 0;}
void cdlib_play	      (Int start_m, Int start_s, Int start_f, Int end_m, Int end_s, Int end_f)  {}

#endif
