/**
 ** cmdline.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* implement tsch/emacs - like command line editing */

/* todo:
   meta-b,f
   parameterize meta-b-d-f, etc
   parameterize paren highlighting
   yank buffer
   incremental search
   */

#include CONFIG
#include <ctype.h>

#include "io.h"

Int cmd_ansi= 1;
#ifdef PC
Int cmd_no_reverse_wrap= 1;
#else
Int cmd_no_reverse_wrap= 0;
#endif

#ifdef DEC
#include "signal.h"
#endif

#define MAXHISTORY 50
#define MAXLEN 1000

Int cmd_string_len(char *s);

IOStream *cmd_in;
FILE *cmd_out;
char *cmd_history[MAXHISTORY]= {
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	};
Int   cmd_current=0;
char cmd_old_line[MAXLEN], *cmd_line=0;
Int   cmd_pos=0, cmd_old_pos=0;
char *cmd_prompt;
Int cmd_char_pos;

#define cmd_char_len(c) (((c) & 127) < 32 ? 2 : 1)

void cmd_ansi_enable(Int enable)
{
    cmd_ansi= enable;
}

void cmd_putc_repeat(Int c, Int n)
{
    while(n-- > 0) {
	if (c == '\b') {
	    if (cmd_no_reverse_wrap && !(cmd_char_pos % 80) && cmd_ansi) {
		Int i;
		/* oh shit */
		/* up line */
		putc(27, cmd_out); putc('[', cmd_out); putc('A', cmd_out);
		/* end of line */
		for (i= 0; i< 79; i++) {
		    putc(27, cmd_out); putc('[', cmd_out); putc('C', cmd_out);
		}
	    } else {
		putc(c, cmd_out);
	    }
	    cmd_char_pos--;
	} else {
	    cmd_char_pos++;
	    putc(c, cmd_out);
	}
    }
}

void cmd_put_char(Int c)
{
    if (cmd_ansi && (c & 128)) fprintf(cmd_out, "\033[4m");
    if ((c & 127) < 32) {
	cmd_char_pos += 1;
	putc('^', cmd_out);
	putc((c & 127) +'@', cmd_out);
    } else putc(c & 127, cmd_out);
    cmd_char_pos += 1;
    if (cmd_ansi && (c & 128)) fprintf(cmd_out, "\033[m");
}

void cmd_set_old(void)
{
    cmd_old_pos= cmd_pos;
    strcpy(cmd_old_line, cmd_line);
}

void cmd_update_line(Int match_parens)
{
    Int  i, x, clear_len, level;
    char *s;

    for (i= 0; cmd_line[i]; i++) cmd_line[i] &= 127;
    if (match_parens) {
	if (cmd_pos && strchr(")]}", cmd_line[cmd_pos-1])) {
	    for (level= 0, i= cmd_pos-1; i >= 0; i--) {
		if (strchr(")]}", cmd_line[i])) level++;
		if (strchr("([{", cmd_line[i])) level--;
		if (!level) {
		    if (cmd_ansi) {
			cmd_line[cmd_pos-1] |= 128;
			cmd_line[i] |= 128;
		    }
		    break;
		}
	    }
	}
    }
    for (i= 0; cmd_old_line[i] == cmd_line[i] && cmd_line[i]; i++);

    if (cmd_old_line[i] != cmd_line[i]) {
	for (x= cmd_old_pos; x< i; x++)
	  cmd_put_char(cmd_line[x]);
	for (x= i; x< cmd_old_pos; x++)
	  cmd_putc_repeat('\b', cmd_char_len(cmd_old_line[x]));
	for (s= &cmd_line[i]; *s; s++) cmd_put_char(*s);
	clear_len= cmd_string_len(&cmd_old_line[i]) - cmd_string_len(&cmd_line[i]);
	cmd_putc_repeat(' ',  clear_len);
	cmd_putc_repeat('\b', clear_len);
	cmd_putc_repeat('\b', cmd_string_len(&cmd_line[cmd_pos]));
    } else {
	for (x= cmd_old_pos; x< cmd_pos; x++)
	  cmd_put_char(cmd_line[x]);
	for (x= cmd_pos; x< cmd_old_pos; x++)
	  cmd_putc_repeat('\b', cmd_char_len(cmd_old_line[x]));
    }
    fflush(cmd_out);
}

Int cmd_string_len(char *s)
{
    Int ret= 0;
    while (*s) ret += cmd_char_len(*s++);
    return ret;
}

void cmd_beep(void)
{
    putc(7, cmd_out);
}

void cmd_get_line(char *prompt, IOStream *in, FILE *out, char *line)
{
    char *l;
    Int c, oldc, newc, i, repeat, cmd_new= cmd_current;

    printf("%s", prompt);
    cmd_prompt= prompt;
    cmd_char_pos= strlen(cmd_prompt);
    
    cmd_in= in;  cmd_out= out;

    cmd_pos= 0;  cmd_line= "";
    cmd_set_old();
    cmd_line= line; cmd_pos= strlen(line);
    cmd_update_line(1);
    
    while(1) {
	repeat= 1;
	while ((c= io_getchar(in, -1)) == 21) repeat *= 4;
	cmd_set_old();
	while (repeat--) {
	    switch(c) {
	      case 27:   /* esc */
		c= io_getchar(in, -1) | 128;
		repeat++;
		break; /* try the modified command */
	      case 'd'+128: case 'D'+128:
		for (i= cmd_pos; cmd_line[i] && !isalnum(cmd_line[i]); i++);
		for (; cmd_line[i] && isalnum(cmd_line[i]); i++);
		strcpy(&cmd_line[cmd_pos], &cmd_line[i]);
		break;
	      case 8  +128: case 127+128:
		for (i= cmd_pos-1; i >=0 && !isalnum(cmd_line[i]); i--);
		for (; i >= 0 && isalnum(cmd_line[i]); i--);
		strcpy(&cmd_line[i+1], &cmd_line[cmd_pos]);
		cmd_pos= i+1;
		break;
	      case '['+128:
		switch(io_getchar(in, -1)) {
		  case 'A': c= 16; break;  /* up arrow */
		  case 'B': c= 14; break;  /* down arrow */
		  case 'C': c=  6; break;  /* right arrow */
		  case 'D': c=  2; break;  /* left arrow */
		  default: goto error;
		}
		repeat++;
		break; /* try the modified command */
	      case '\n': /* return */
	      case '\r': /* return */
		cmd_pos= strlen(cmd_line);
		cmd_update_line(0);
		if (cmd_pos) {
		    if (cmd_history[cmd_current]) free(cmd_history[cmd_current]);
		    cmd_history[cmd_current]= malloc(strlen(cmd_line)+1);
		    strcpy(cmd_history[cmd_current], cmd_line);
		    cmd_current= (cmd_current + 1) % MAXHISTORY;
		}
		return;
	      case '!':
		if (cmd_pos == 1 && cmd_line[0] == '!' && !cmd_line[1]) c= 16;
		else goto normal_char;
	      case 14:   /* ^N */
	      case 16:   /* ^P */
		cmd_new= (cmd_new + MAXHISTORY + 15 - c) % MAXHISTORY;
		if (!cmd_history[cmd_new]) {
		    cmd_new = (cmd_new - 15 + c) % MAXHISTORY;
		    goto error;
		}
	      set_line:
		strcpy(cmd_line, cmd_history[cmd_new]);
		/* FALL THROUGH TO END OF LINE */
	      case 5:    /* ^E */
		cmd_pos= strlen(cmd_line);
		break;
	      case 1:    /* ^A */
		cmd_pos= 0;
		break;
	      case 6:    /* ^F */
		if (cmd_line[cmd_pos]) cmd_pos++;
		else goto error;
		break;
	      case 11:   /* ^K */
		cmd_line[cmd_pos]= 0;
		break;
	      case 2:    /* ^B */
		if (cmd_pos) cmd_pos--; else goto error;
		break;
	      case 8:    /* ^H */
	      case 127:  /* del */
		if (cmd_pos) {
		    cmd_pos--;
		    strcpy(&cmd_line[cmd_pos], &cmd_line[cmd_pos+1]);
		} else goto error;
		break;
	      case 4:    /* ^D */
		if (cmd_line[cmd_pos]) {
		    strcpy(&cmd_line[cmd_pos], &cmd_line[cmd_pos+1]);
		} else goto error;
		break;
	      case 3:    /* ^C */
		exit(0);
		break;
	      case ' ':
		if (cmd_pos &&
		    cmd_line[0]=='!' &&
		    !cmd_line[cmd_pos] &&
		    !strchr(cmd_line, ' ')) {
		    Int i;
		    for (i= (cmd_current-1) % MAXHISTORY;
			 cmd_history[i] && i != cmd_current;
			 i = (i - 1) % MAXHISTORY) {
			if (!strncmp(cmd_history[i], &cmd_line[1], cmd_pos-1)) {
			    cmd_new = i;
			    goto set_line;
			}
		    }
		}
		/* fall through to character */
	      default:
	      normal_char:
		if (c == 17) c= io_getchar(in, -1);
		else if (c < 32 || c > 126) {
                    /*printf("you pressed %x\n", c);*/
                    goto error; 
                }
		for(newc= c, l= &cmd_line[cmd_pos]; oldc= newc; l++) {
		    newc= *l; *l= (char) oldc;
		}
		*l= 0;
		cmd_pos++;
		break;
	      error:
		cmd_beep();
		repeat= 0;
		break;
	    }
	}
	cmd_update_line(1);
    }
}

#ifdef STAND_ALONE
NOPROTO void main(Int argc, char **argv) 
{
    IOStream *in= io_open_stdin();
    char foo[200];
    io_make_stdin_unbuffered();
    strcpy(foo, "hi there");
    while (1) {
	printf("\r\n-> ");
	cmd_get_line(in, stdout, foo);
	foo[0]= 0;
    }
}
#endif
