/**
 ** cmdline2.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* implement tsch/emacs - like command line editing */

/* todo:
   meta-b,f
   parameterize meta-b-d-f, etc
   parameterize paren highlighting
   yank buffer
   incremental search
   */


#include CONFIG
#include <ctype.h>

#include "io.h"

#ifdef UNIX
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <signal.h>
#include <fcntl.h>
#endif

Int cmd_ansi= 1;
Int cmd_no_reverse_wrap= 0;

#define MAXHISTORY 50
#define MAXLEN 1000


IOStream *cmd_in;
FILE *cmd_out;
char *cmd_history[MAXHISTORY]= {
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	0,0,0,0,0, 0,0,0,0,0,
	};
Int   cmd_current=0;
char cmd_old_line[MAXLEN], *cmd_line=0;
Int   cmd_pos=0, cmd_old_pos=0;
char *cmd_prompt;
Int cmd_char_pos;

#define cmd_char_len(c) (((c) & 127) < 32 ? 2 : 1)

Int cmd_string_len(char *s);
Int cmd_get_key(void);

void cmd_ansi_enable(Int enable)
{
    cmd_ansi= enable;
}

#ifdef UNIX
#define ON 1
#define OFF 0

void interrupt(Int s, Int interrupt)
{
    Int  old_fl;

    /* make me the owner of s or the interrupt won't do much good */
    fcntl(s, F_SETOWN, getpid());

    old_fl= fcntl(s, F_GETFL);

    fcntl(s, F_SETFL, interrupt ? old_fl | FASYNC : old_fl & ~FASYNC);
}
#endif

void cmd_sigint_interrupt(void)
{
    printf("Bye!\n");
#ifdef UNIX
    interrupt(0, OFF);
    system("/bin/stty cooked echo -nl");
#endif
#ifdef DEC
    exit(0);
#endif    
}

void cmd_make_stdin_unbuffered(void)
{
    
#ifdef UNIX    
    system("stty cbreak -echo");
    
#ifdef DEC
    signal(SIGINT, (void*) cmd_sigint_interrupt);
#else
    on_exit(cmd_sigint_interrupt);
#endif

#endif
}

void cmd_putc_repeat(Int c, Int n)
{
    while(n-- > 0) {
	if (c == '\b') {
	    if (cmd_no_reverse_wrap && !(cmd_char_pos % 80) && cmd_ansi) {
		Int i;
		/* oh shit */
		/* up line */
		putc(27, cmd_out); putc('[', cmd_out); putc('A', cmd_out);
		/* end of line */
		for (i= 0; i< 79; i++) {
		    putc(27, cmd_out); putc('[', cmd_out); putc('C', cmd_out);
		}
	    } else {
		putc(c, cmd_out);
	    }
	    cmd_char_pos--;
	} else {
	    cmd_char_pos++;
	    putc(c, cmd_out);
	}
    }
}

void cmd_put_char(Int c)
{
    if (cmd_ansi && (c & 128)) fprintf(cmd_out, "\033[4m");
    if ((c & 127) < 32) {
	cmd_char_pos += 1;
	putc('^', cmd_out);
	putc((c & 127) +'@', cmd_out);
    } else putc(c & 127, cmd_out);
    cmd_char_pos += 1;
    if (cmd_ansi && (c & 128)) fprintf(cmd_out, "\033[m");
}

void cmd_set_old(void)
{
    cmd_old_pos= cmd_pos;
    strcpy(cmd_old_line, cmd_line);
}

void cmd_update_line(Int match_parens)
{
    Int  i, x, clear_len, level;
    char *s;

    for (i= 0; cmd_line[i]; i++) cmd_line[i] &= 127;
    if (match_parens) {
	if (cmd_pos && strchr(")]}", cmd_line[cmd_pos-1])) {
	    for (level= 0, i= cmd_pos-1; i >= 0; i--) {
		if (strchr(")]}", cmd_line[i])) level++;
		if (strchr("([{", cmd_line[i])) level--;
		if (!level) {
		    if (cmd_ansi) {
			cmd_line[cmd_pos-1] |= 128;
			cmd_line[i] |= 128;
		    }
		    break;
		}
	    }
	}
    }
    for (i= 0; cmd_old_line[i] == cmd_line[i] && cmd_line[i]; i++);

    if (cmd_old_line[i] != cmd_line[i]) {
	for (x= cmd_old_pos; x< i; x++)
	  cmd_put_char(cmd_line[x]);
	for (x= i; x< cmd_old_pos; x++)
	  cmd_putc_repeat('\b', cmd_char_len(cmd_old_line[x]));
	for (s= &cmd_line[i]; *s; s++) cmd_put_char(*s);
	clear_len= cmd_string_len(&cmd_old_line[i]) - cmd_string_len(&cmd_line[i]);
	cmd_putc_repeat(' ',  clear_len);
	cmd_putc_repeat('\b', clear_len);
	cmd_putc_repeat('\b', cmd_string_len(&cmd_line[cmd_pos]));
    } else {
	for (x= cmd_old_pos; x< cmd_pos; x++)
	  cmd_put_char(cmd_line[x]);
	for (x= cmd_pos; x< cmd_old_pos; x++)
	  cmd_putc_repeat('\b', cmd_char_len(cmd_old_line[x]));
    }
    fflush(cmd_out);
}

Int cmd_string_len(char *s)
{
    Int ret= 0;
    while (*s) ret += cmd_char_len(*s++);
    return ret;
}

void cmd_beep(void)
{
    putc(7, cmd_out);
}

Int cmd__handler(Int init, char *prompt_in,
		 IOStream *in_arg, FILE *out_arg, char *line_arg);

void cmd_start_get_line(char *prompt, IOStream *in, FILE *out, char *line)
{
    cmd__handler(1, prompt, in, out, line);
}
    
void cmd_get_line(char *prompt, IOStream *in, FILE *out, char *line)
{
    cmd__handler(1, prompt, in, out, line);
    while (cmd_get_key() != 1);
}

Int cmd_get_key(void)
{
    return cmd__handler(0,0,0,0,0);
}

Int cmd__handler(Int init, char *prompt_in,
		 IOStream *in_arg, FILE *out_arg, char *line_arg)
{
    static Int did_command;
    static char *prompt;
    static char *l;
    static Int c, oldc, newc, i, repeat, cmd_new;
    static IOStream *in;
    static FILE *out;
    static char *line;
    static Int meta, cursor, quote;
    Int iter;

    if (init) {
	prompt= prompt_in;
	in= in_arg;
	out= out_arg;
	line= line_arg;
	cmd_new= cmd_current;

	printf("%s", prompt);
	cmd_prompt= prompt;
	cmd_char_pos= strlen(cmd_prompt);
	
	cmd_in= in;  cmd_out= out;
	
	cmd_pos= 0;  cmd_line= "";
	cmd_set_old();
	cmd_line= line; cmd_pos= strlen(line);
	cmd_update_line(1);
	repeat= 1;
	meta= 0;
	cursor= 0;
	quote= 0;
	return 0;
    }

    c= io_getchar(in, 0);
    if (c == EOF) return -1;
    if (meta) {
	c |= 128;
	meta= 0;
    }

    if (cursor) {
	switch(c) {
	  case 'A': c= 16; break;  /* up arrow */
	  case 'B': c= 14; break;  /* down arrow */
	  case 'C': c=  6; break;  /* right arrow */
	  case 'D': c=  2; break;  /* left arrow */
	}
	cursor= 0;
    }
    
    cmd_set_old();
    did_command= 1;
    for (iter= 0; iter < repeat; iter ++) {
	if (quote) goto normal_char;
	switch(c) {
	  case 21:
	    repeat *= 4;
	    did_command= 0;
	    break;
	  case 27:   /* esc */
	    meta= 1;
	    did_command= 0;
	    break;
	  case 'Q'-64:
	    quote= 1;
	    did_command= 0;
	    break;
	  case '['+128:
	    cursor= 1;
	    did_command= 0;
	    break;
	  case 'd'+128: case 'D'+128:
	    for (i= cmd_pos; cmd_line[i] && !isalnum(cmd_line[i]); i++);
	    for (; cmd_line[i] && isalnum(cmd_line[i]); i++);
	    strcpy(&cmd_line[cmd_pos], &cmd_line[i]);
	    break;
	  case 8  +128: case 127+128:
	    for (i= cmd_pos-1; i >=0 && !isalnum(cmd_line[i]); i--);
	    for (; i >= 0 && isalnum(cmd_line[i]); i--);
	    strcpy(&cmd_line[i+1], &cmd_line[cmd_pos]);
	    cmd_pos= i+1;
	    break;
	  case '\n': /* return */
	  case '\r': /* return */
	    cmd_pos= strlen(cmd_line);
	    cmd_update_line(0);
	    if (cmd_pos) {
		if (cmd_history[cmd_current]) free(cmd_history[cmd_current]);
		cmd_history[cmd_current]= malloc(strlen(cmd_line)+1);
		strcpy(cmd_history[cmd_current], cmd_line);
		cmd_current= (cmd_current + 1) % MAXHISTORY;
	    }
	    return 1;
	  case '!':
	    if (cmd_pos == 1 && cmd_line[0] == '!' && !cmd_line[1]) c= 16;
	    else goto normal_char;
	  case 14:   /* ^N */
	  case 16:   /* ^P */
	    cmd_new= (cmd_new + MAXHISTORY + 15 - c) % MAXHISTORY;
	    if (!cmd_history[cmd_new]) {
		cmd_new = (cmd_new - 15 + c) % MAXHISTORY;
		goto error;
	    }
	  set_line:
	    strcpy(cmd_line, cmd_history[cmd_new]);
	    /* FALL THROUGH TO END OF LINE */
	  case 5:    /* ^E */
	    cmd_pos= strlen(cmd_line);
	    break;
	  case 1:    /* ^A */
	    cmd_pos= 0;
	    break;
	  case 6:    /* ^F */
	    if (cmd_line[cmd_pos]) cmd_pos++;
	    else goto error;
	    break;
	  case 11:   /* ^K */
	    cmd_line[cmd_pos]= 0;
	    break;
	  case 2:    /* ^B */
	    if (cmd_pos) cmd_pos--; else goto error;
	    break;
	  case 8:    /* ^H */
	  case 127:  /* del */
	    if (cmd_pos) {
		cmd_pos--;
		strcpy(&cmd_line[cmd_pos], &cmd_line[cmd_pos+1]);
	    } else goto error;
	    break;
	  case 4:    /* ^D */
	    if (cmd_line[cmd_pos]) {
		strcpy(&cmd_line[cmd_pos], &cmd_line[cmd_pos+1]);
	    } else goto error;
	    break;
	  case 3:    /* ^C */
	    exit(0);
	    break;
	  case ' ':
	    if (cmd_pos &&
		cmd_line[0]=='!' &&
		!cmd_line[cmd_pos] &&
		!strchr(cmd_line, ' ')) {
		Int i;
		for (i= (cmd_current-1) % MAXHISTORY;
		     cmd_history[i] && i != cmd_current;
		     i = (i - 1) % MAXHISTORY) {
		    if (!strncmp(cmd_history[i], &cmd_line[1], cmd_pos-1)) {
			cmd_new = i;
			goto set_line;
		    }
		}
	    }
	    /* fall through to character */
	  default:
	  normal_char:
	    if (c < 32 || c > 127) goto error; 
	    for(newc= c, l= &cmd_line[cmd_pos]; oldc= newc; l++) {
		newc= *l; *l= (char) oldc;
	    }
	    *l= 0;
	    cmd_pos++;
	    break;
	  error:
	    cmd_beep();
	    iter= repeat;
	    break;
	}
    }
    if (did_command) {
	repeat= 1;
	meta= 0;
	cursor= 0;
	quote= 0;
    }
    cmd_update_line(1);
    return 0;
}

#ifdef STAND_ALONE
NOPROTO void main(Int argc, char **argv)
{
    IOStream *in= io_open_stdin();
    char foo[200];
    cmd_make_stdin_unbuffered();
    strcpy(foo, "hi there");
    while (1) {
	printf("\r\n-> ");
	cmd_get_line(in, stdout, foo);
	foo[0]= 0;
    }
}
#endif
