/* stolen from david michael (djsm@ai.mit.edu),
   Mike Bolotski (misha@ai.mit.edu) */

#include CONFIG

#include <util.h>

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/extensions/XShm.h>
#include <sys/types.h>
#ifdef HAS_SHARED_MEMORY
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#include "sbm_unix.h"

/* global variables */

Int sbm_try_shared_memory= 0;

Int sbm_init_default(Sbm *sbm, Int width, Int height, Int scale)
{
    return sbm_init(sbm,width,height,scale,0,8);
}

Int sbm_init(Sbm *sbm, Int width, Int height, Int scale, char *displayname, Int depth)
{
  XSetWindowAttributes xswa;
  Int black, white, screen;
  Display *display;
  GC gc;
  Window window;
  XEvent event;
  XImage *image;
  /*Colormap cmap;*/

  display= XOpenDisplay(displayname);
  if (!display) {
      fprintf(stderr, "Can't open X display\n");
      exit(-1);
  }
  screen=  DefaultScreen(display);

  sbm->scale= scale;
  width *= scale;
  height *= scale;
  
  sbm->width= width;
  sbm->height= height;
  sbm->display= display;
  sbm->screen= screen;
  
  black = BlackPixel(display,screen);
  white = WhitePixel(display,screen);

  xswa.event_mask=ExposureMask|StructureNotifyMask|
    ButtonPressMask|KeyPressMask;
  xswa.background_pixel= white;
  xswa.border_pixel= black;
  xswa.backing_store=Always;
  xswa.backing_planes=AllPlanes;
  xswa.bit_gravity=NorthWestGravity;

  window= XCreateWindow
    (display,DefaultRootWindow(display),
     0,0,width,height,0,
     DefaultDepth(display,DefaultScreen(display)),
     InputOutput,
     DefaultVisual(display,DefaultScreen(display)),
     CWEventMask|CWBackPixel|CWBorderPixel|
     CWBackingStore|CWBackingPlanes|CWBitGravity,
     &xswa);

  sbm->window= window;

  /*
  cmap = XCopyColormapAndFree(display,DefaultColormap(display,screen));
  XSetWindowColormap(display,win,cmap);
  */
  
  XMapWindow(display,window);
  XNextEvent (display, &event);
  while (event.type!=Expose)
    XNextEvent(display,&event);

  gc = DefaultGC(display,screen);
  sbm->gc= gc;
  
  /*image = sbm_create_ximage(display,width,height,8,1);*/

  image= 0;
#ifdef HAS_SHARED_MEMORY
  if (sbm_try_shared_memory) {
      image = sbm_create_shmximage(display,width,height,depth, &sbm->shminfo);
  }
#endif

  if (image) {
      sbm->shm= 1;
  }
  else {
      image= sbm_create_ximage(display,width,height,depth);
      sbm->shm= 0;
  }
  sbm->image= image;
  return 0;
}

void sbm_refresh(Sbm *sbm)
{
    if (sbm->shm) {
#ifdef HAS_SHARED_MEMORY	
	XShmPutImage(sbm->display, sbm->window, sbm->gc, sbm->image,
		     0,0,0,0,
		     sbm->image->width, sbm->image->height, 1);
#endif	
    }
    else {
	XPutImage(sbm->display, sbm->window, sbm->gc, sbm->image,
		  0,0,0,0,
		  sbm->image->width, sbm->image->height);
    }
    XSync(sbm->display, 0);
    /*XFlush(sbm->display);*/
}

void sbm_destroy(Sbm *sbm)
{
    UNUSED(sbm);
}



/*
 * Create an X11 image given a desired
 * width, height and depth.
 */
XImage *
sbm_create_ximage(display, w, h, depth)
Display	*display;
Int	w, h, depth;
{
	register long howmuch;
	Visual *visual;
	XImage *r_ximage;

	visual = DefaultVisual(display, DefaultScreen(display));
	if(depth == 1)
		r_ximage = XCreateImage(display, visual, 1, XYBitmap,
			0, 0, w, h, 16, 0);
	else	
		r_ximage = XCreateImage(display, visual, depth, ZPixmap,
			0, 0, w, h, 16, 0);
	if(r_ximage == NULL) {
		fprintf(stderr, "XCreateImage failed\n");
		return(NULL);
	}

	howmuch = r_ximage->height*r_ximage->bytes_per_line;
	r_ximage->data = (char *)malloc(howmuch);
	if(r_ximage->data == NULL) {
		perror("malloc");
		return(NULL);
	}
	
	/* Clear the memory out */
	memset(r_ximage->data, 0, howmuch);
	return(r_ximage);
}

#ifdef HAS_SHARED_MEMORY

/*
 * Create a shared memory X11 image
 */
XImage *
sbm_create_shmximage(display, w, h, depth, shminfo)
Display		*display;
Int		w, h, depth;
XShmSegmentInfo *shminfo;
{
	XImage		*ximage;
	Visual		*visual;
	Int		majorv, minorv, howmuch;
	Bool		sharedPixmaps;

	if(!XShmQueryVersion(display,&majorv,&minorv,&sharedPixmaps)) {
		fprintf(stderr,"X11 Shared memory images not available");
		return(NULL);
	}

	visual = DefaultVisual(display, DefaultScreen(display));
	if(depth == 1)
		ximage = XShmCreateImage(display, visual, 1, 
			XYBitmap, 0, shminfo, w, h);
	else	
		ximage = XShmCreateImage(display, visual, depth, 
			ZPixmap, 0, shminfo, w, h);

	howmuch = h*ximage->bytes_per_line;

	shminfo->shmid = shmget(IPC_PRIVATE, howmuch, IPC_CREAT|0777);

	if(shminfo->shmid < 0) {
		perror("shmget");
		return(NULL);
	}
	shminfo->shmaddr = (char *)shmat(shminfo->shmid, 0, 0);

	if(shminfo->shmaddr == ((char *)-1)) {
		perror("shmat");
		return(NULL);
	}
	ximage->data = shminfo->shmaddr;

	/* Clear the memory out */
	memset(ximage->data, 0, howmuch);

	shminfo->readOnly = False;
	XShmAttach(display, shminfo);
	return(ximage);
}

#endif

#ifdef HAS_SHARED_MEMORY

/*
 * Destroy the X11 shared memory image specified
 */
void
sbm_destroy_shmximage(display, ximage, shminfo)
Display		*display;
XImage		*ximage;
XShmSegmentInfo *shminfo;
{
	Int rc;

	if(ximage) {
		XShmDetach(display, shminfo);
		rc = shmdt(shminfo->shmaddr);
		if(rc < 0)
			perror("shmdt");
		rc = shmctl(shminfo->shmid, IPC_RMID, 0);
		if(rc < 0)
			perror("shmctl");
	}
	return;
}

/*
 * Use the private color map and install it in
 * the canvas
 */
void
sbm_load_private_cmap(display, cmap, r, g, b, map_length, offset)
Display		*display;
Colormap	cmap;
u_char		*r, *g, *b;
Int		map_length, offset;
{
	register Int	i;
	u_long		pixels[256], plane_masks[256];
	XColor		colors[256];
	
	/* Allocate 256 cells again */
	XAllocColorCells(display, 
		cmap, True, plane_masks, 0, pixels, 256);

	/* Now set the values of the colour array */
	for(i=0; i<map_length; i++) {
		colors[i].pixel = i + offset;
		colors[i].red = r[i]<<8;
		colors[i].green = g[i]<<8;
		colors[i].blue = b[i]<<8;
		colors[i].flags = DoRed | DoGreen | DoBlue;
	}
	
	/* Free the ones we don't need */
	if(offset) {
		/* Free window manager colors */
		for(i=0; i<offset; i++)
			pixels[i] = i;
		XFreeColors(display, cmap, pixels, offset, 0);

		/* Free colors at the top of the map */
		for(i=(offset+map_length); i<256; i++)
			pixels[i-(offset+map_length)] = i;
		XFreeColors(display, cmap, pixels,
				 	256-(offset+map_length), 0);
	}

	/* Store it in the colour map */
	XStoreColors(display, cmap, colors, map_length);
	
	/* Now install it in the h/ware */
	XInstallColormap(display, cmap);
	return;
}

#endif

/*
 * Try to allocate read/write cells from the default color map.
 * This is the preferred choice as the window manager will install
 * the map for us. Need to return the value of the offset so
 * that we can compensate for it elsewhere.
 */
Int
sbm_load_default_cmap(display, cmap, r, g, b, map_length, offset)
Display		*display;
Colormap	cmap;
u_char		*r, *g, *b;
Int		map_length, *offset;
{
	Status		rc;
	register Int	i;
	u_long		pixels[256], plane_masks[256];
	XColor		colors[256];

	rc = XAllocColorCells(display, 
		cmap, True, plane_masks, 0, pixels, map_length);

	if(!rc) {
		/* XAllocColorCells failed, return a failure */
		return(-1);
	}

	if((pixels[0]+map_length-1) != pixels[map_length-1]) {
		/*
		 * The map returned wasn't contigous.
		 * Free the colors returned in pixels.
		 * Return a failure.
		 */
		XFreeColors(display, cmap, pixels, map_length, 0);
		return(-1);
	}

	/* Now set the values of the colour array */
	for(i=0; i<map_length; i++) {
		colors[i].pixel = i + pixels[0];
		colors[i].red = r[i]<<8;
		colors[i].green = g[i]<<8;
		colors[i].blue = b[i]<<8;
		colors[i].flags = DoRed | DoGreen | DoBlue;
	}

	/* Return the value of offset to the user */
	*offset = pixels[0];

	/* Store it in the colour map */
	XStoreColors(display, cmap, colors, map_length);
	return(0);
}

long            colortab4[256];
short           colortab2[256];
unsigned char   colortab1[256];

void sbm_set_colortab_entry(Int color, long xcolor)
{
    if (color < 0 || color > 255) errorf("Illegal color %d\n", color);
    
    colortab4[color]= xcolor | (xcolor << 8) | (xcolor << 16) | (xcolor << 24);
    colortab2[color]= xcolor | (xcolor << 8);
    colortab1[color]= xcolor;
}

void sbm_set_color(Sbm *sbm, double r, double g, double b, Int color)
{
    XColor c;
    c.red= (short) (r*65535);
    c.green= (short) (g*65535);
    c.blue= (short) (b*65535);
    XAllocColor(sbm->display, DefaultColormap(sbm->display, sbm->screen), &c);
    sbm_set_colortab_entry(color, c.pixel);
}

void sbm_color_all(Sbm *sbm, long *color)
{
    if (sbm->width/sbm->scale != 128 ||
	sbm->height/sbm->scale != 128) {
	errorf("sbm_plot_fixnum_vectors not implemented for given dimensions");
    }

    switch (sbm->scale) {
      case 1: sbm_color_all_1_128_128(sbm, color); break;
      case 2: sbm_color_all_2_128_128(sbm, color); break;
      case 4: sbm_color_all_4_128_128(sbm, color); break;
      default: errorf("sbm_plot_fixnum_vectors not implemented at scale %n",
		      sbm->scale);
    }
}

void sbm_color_all_1_128_128(Sbm *sbm, long *color)
{
    Int i;

    for (i= 0; i< 128; i++) {
	char *current= sbm->image->data + i*128;
	char *dest=    current + 128;
	while (current < dest) {
	    *current= colortab1[(*color)>>16];
	    current++;
	    color++;
	}
    }
}

void sbm_color_all_2_128_128(Sbm *sbm, long *color)
{
    Int i;

    for (i= 0; i< 128; i++) {
	long *current= (long*) (sbm->image->data + i*512);
	long *dest=    current + 64;
	while (current < dest) {
	    long c1= colortab2[(*color++)>>16];
	    long c2= colortab2[(*color++)>>16];
#ifdef MSB_FIRST
	    long c3= c2 | (c1 << 16);
#else
	    long c3= c1 | (c2 << 16);
#endif	    
	    *current=      c3;
	    *(current+64)= c3;
	    current++;
	}
    }
}

void sbm_color_all_4_128_128(Sbm *sbm, long *color)
{
    Int i;

    for (i= 0; i< 128; i++) {
	long *current= (long*) (sbm->image->data + i*2048);
	long *dest=    current + 128;
	while (current < dest) {
	    long c= colortab4[(*color++)>>16];
	    *current=       c;
	    *(current+128)= c;
	    *(current+256)= c;
	    *(current+384)= c;
	    current++;
	}
    }
}

void sbm_plot_fixnum_vectors(Sbm *sbm, long *x, long *y, long *color, long n)
{
    if (sbm->width/sbm->scale != 128 ||
	sbm->height/sbm->scale != 128) {
	errorf("sbm_plot_fixnum_vectors not implemented for given dimensions");
    }
    
    switch (sbm->scale) {
      case 1: sbm_plot_fixnum_vectors_1_128_128(sbm, x, y, color, n); break;
      case 2: sbm_plot_fixnum_vectors_2_128_128(sbm, x, y, color, n); break;
      case 4: sbm_plot_fixnum_vectors_4_128_128(sbm, x, y, color, n); break;
      default: errorf("sbm_plot_fixnum_vectors not implemented at scale %n",
		      sbm->scale);
    }
	
}

void sbm_plot_fixnum_vectors_1_128_128(Sbm *sbm, long *x, long *y, long *color, long n)
{
    unsigned char *data= (unsigned char *) sbm->image->data;
    unsigned char *ptr;
    unsigned char c;
    long xc, yc;
    
    while (n--) {
	xc= ((64<<16) + *x++) >> 16;
	yc= ((64<<16) - 1 - *y++) >> 16;
	if (!((xc | yc) & ~127)) {
	    ptr= data + xc + yc *128;
	    c= colortab1[(*color)>>16];
	    *ptr= c;
	}
	color++;
    }
}
      
void sbm_plot_fixnum_vectors_2_128_128(Sbm *sbm, long *x, long *y, long *color, long n)
{
    unsigned char *data= (unsigned char *) sbm->image->data;
    unsigned char *ptr;
    short c;
    long xc, yc;
    
    while (n--) {
	xc= ((64<<16) + *x++) >> 16;
	yc= ((64<<16) - 1 - *y++) >> 16;
	if (!((xc | yc) & ~127)) {
	    ptr= data + 2 * (xc + yc * 256);
	    c= colortab2[(*color)>>16];
	    *(short*)(ptr+  0)= c;
	    *(short*)(ptr+256)= c;
	}
	color++;
    }
}
      
void sbm_plot_fixnum_vectors_4_128_128(Sbm *sbm, long *x, long *y, long *color, long n)
{
    unsigned char *data= (unsigned char *) sbm->image->data;
    unsigned char *ptr;
    long c;
    long xc, yc;
    
    while (n--) {
	xc= ((64<<16) + *x++) >> 16;
	yc= ((64<<16) - 1 - *y++) >> 16;
	if (!((xc | yc) & ~127)) {
	    ptr= data + 4 * (xc + yc * 512);
	    c= colortab4[(*color)>>16];
	    *(long*)(ptr+   0)= c;
	    *(long*)(ptr+ 512)= c;
	    *(long*)(ptr+1024)= c;
	    *(long*)(ptr+1536)= c;
	}
	color++;
    }
}
      

      
