/**
 ** stringlb.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 ** stringlb.c
 **
 ** Manages strings on the heap
 **
 ** v1.0  Thu May  9 09:53:31 1991  Randy Sargent  created
 **/

#include CONFIG

#include <util.h>
#include <filelib.h>

#include <ctype.h>

#include "stringlb.h"
#include "growbuf.h"

char *string_copy(char *src)
{
    char *ret= malloc(strlen(src) + 1);
    if (ret) strcpy(ret, src);
    return ret;
}

void *memory_copy(void *src, size_t len)
{
   char *ret= malloc(len);
   memcpy(ret, src, len);
   return ret;
}

char *string_left(char *src, size_t n)
{
    char *ret= malloc(n + 1);
    if (ret) {
	strncpy(ret, src, n);
	ret[n]= 0;
    }
    return ret;
}

char *string_right(char *src, size_t n)
{
    char *ret= malloc(n + 1);
    if (ret) {
	strncpy(ret, src + strlen(src) - n, n+1);
    }
    return ret;
}

Int string_lastchar(char *str)
{
    if (str[0] == 0) return -1;
    return str[strlen(str) - 1];
}

char *string_mid(char *src, size_t start, size_t len)
{
    char *ret;

    if (len == (size_t) -1) len= strlen(src+start);
    ret= malloc(len + 1);
    if (ret) {
	strncpy(ret, src+start, len);
	ret[len]= 0;
    }
    return ret;
}
    
char *string_substr(char *src, size_t start, size_t end_plus_one)
{
    return string_mid(src, start, end_plus_one - start);
}

char *string_delete(char *src, size_t start, size_t len)
{
    char *ret;

    ret= malloc(strlen(src) - len + 1);
    if (ret) {
	strncpy(ret, src, start);
	strcpy(ret + start, src + start + len);
    }
    return ret;
}    
    
char *string_append(char *a, char *b)
{
    size_t len_a= strlen(a), len_b= strlen(b);
    char *ret= malloc(len_a + len_b + 1);

    if (ret) {
	strcpy(ret, a);
	strcpy(ret + len_a, b);
    }
    return ret;
}

char *string_appendn(char *first, ...)
{
    size_t len;
    char *ret;
    char *s;
    va_list args;

    len= strlen(first);
    
    va_start(args, first);
    while (1) {
	s= va_arg(args, char*);
	if (!s) break;
	len += strlen(s);
    }
    va_end(args);
    
    ret= malloc(len + 1);
    if (!ret) return 0;
    strcpy(ret, first);
    
    va_start(args, first);
    while (1) {
	s= va_arg(args, char*);
	if (!s) break;
	strcat(ret, s);
    }
    va_end(args);

    return ret;
}

void string_free(char *s)
{
    if (s) free(s);
}

char *string_free_soon(char *s)
{
    static char *to_free[20]={0,0,0,0,0,0,0,0,0,0,
		              0,0,0,0,0,0,0,0,0,0};		
    static Int in= 0;

    if (to_free[in]) free(to_free[in]);
    to_free[in]= s;
    in=(in+1)%20;
    return s;
}

/* replaced by filenm_remove_suffix */
NOPROTO char *string_remove_suffix(char *filename)
{
   return filenm_remove_suffix(filename);
}

/* replaced by filenm_get_suffix */
NOPROTO char *string_get_suffix(char *filename)
{
   return filenm_get_suffix(filename);
}

/* replaced by filenm_remove_directory */
NOPROTO char *string_remove_pathname(char *filename)
{
   return filenm_remove_directory(filename);
}

char *string_find_substring(char *str, char *sub)
{
    size_t sub_len= strlen(sub);
    while (*str) {
	if (!strncmp(str, sub, sub_len)) return str;
	str++;
    }
    return 0;
}

char *string_find_substring_i(char *str, char *sub)
{
    size_t sub_len= strlen(sub);
    while (*str) {
	if (!strnicmp(str, sub, sub_len)) return str;
	str++;
    }
    return 0;
}

char *string_skip_spaces(char *str)
{
    while (isspace(*str)) str++;
    return str;
}

char *string_skip_chars(char *str, char *chars_to_skip)
{
    while (*str) {
	if (!strchr(chars_to_skip, *str)) break;
	str++;
    }
    return str;
}

char *string_find_chars(char *str, char *chars_to_skip)
{
    while (*str) {
	if (strchr(chars_to_skip, *str)) break;
	str++;
    }
    return str;
}

char *string_get_token(char **str, char *delims)
{
   char *beginning= string_skip_chars(*str, delims);
   char *end= string_find_chars(beginning, delims);
   char *ret= string_left(beginning, end-beginning);
   *str= string_skip_chars(end, delims);
   return ret;
}
    
char **string_get_tokens(char *in, char *delims)
{
   Growbuf g;
   growbuf_init(&g);
   in= string_skip_chars(in, delims);
   while (in[0]) {
      growbuf_add_ptr(&g, string_get_token(&in, delims));
   }
   growbuf_add_ptr(&g, NULL);
   return (char**) GROWBUF_DATA(&g);
}

void string_make_lower(char *str)
{
    while (*str) {
	*str= (char) tolower(*str);
	str++;
    }
}

char *string_vprintf(char *format, va_list args)
{
    char buf[5000];
    
    vsprintf(buf, format, args);

    return string_copy(buf);
}

char *string_printf(char *format, ...)
{
    char *ret;
    va_list args;
    
    va_start(args, format);
    ret= string_vprintf(format, args);
    va_end(args);

    return ret;
}

char *string_shorten(char *s, size_t len)
{
    size_t num_periods;
    size_t left, right, slen;
    char periods[4];
    
    slen= strlen(s);
    if (slen <= len) {
	return string_copy(s);
    }

    strcpy(periods, "...");
    
    num_periods= 3;
    if (len < 6) num_periods= 2;
    if (len < 4) num_periods= 1;
    if (len < 2) num_periods= 0;
    periods[num_periods]= 0;

    right= (len - num_periods) / 2;
    left= (len - num_periods - right);
    
    return string_appendn(free_soon(string_left(s, left)),
			  periods,
			  free_soon(string_right(s, right)),
			  NULL);
}

char *string_replace_char(char *s, Int from, Int to)
{
    char *ret= string_copy(s);
    Int i;
    for (i= 0; ret[i]; i++) {
        if (ret[i] == from) ret[i]= to;
    }
    return ret;
}

/*
void stable_sort(void *data, size_t n_elems, size_t elemsize, int (*compare)(void *, void *))
{
   void *to= malloc(n_elems * elemsize);
   void *from= data;
   size_t size;

   for (size= 1; size < n_elems; size *= 2) {
      for (base= 0; base < n_elems-size;  base += size*2) {
	 int f1= base;
	 int f2= base+size;
	 int t= base;
*/	 
	 
