/******************************************************************
 * AUTO Paranoia IV
 * CopyPolicy: GNU Public License 2 applies
 * (C) 1999 Monty xiphmont@mit.edu
 * 
 * Sample callback building block and message registration
 *
 ******************************************************************/

#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>

#include "paranoia_types.h"
#include "trans_paranoia.h"
#include "auto_paranoia.h"
#include "paranoia_errors.h" /* error numbers for all paranoia */
#include "report_paranoia.h"

/* platform includes */
#include "linux_callback.h"

static char *auto_report_lookup(report_base *b,int ERR,int WARN,int INFO,
				int LONG,int err,char *procdev,
				int devstate,char *devicename){
  char *errS;
  char *warnS;
  char *infoS;
  char *longS;
  char errnoS[80];
  char devstateS[80];
  char *procdevname;
  char userS[80];
  struct passwd *user=getpwuid(geteuid());
  
  if(user==NULL)
    sprintf(userS,"%d",geteuid());
  else
    sprintf(userS,"%s",user->pw_name);;

  errS=(ERR?"1":"0");
  warnS=(WARN?"1":"0");
  infoS=(INFO?"1":"0");
  longS=(LONG?"1":"0");
  sprintf(errnoS,"%d",err);
  if(procdev==NULL)procdev="unknown";
  sprintf(iftypeS,"%d",iftype);
  sprintf(hardtypeS,"%d",hardtype);
  sprintf(devstateS,"%d",devstate);

  {
    char *search=__pi_procdevs;

    while(*search){
      if(strncmp(*search,procdev,strlen(*search))){
	procdevname=*(search+1);
	break;
      }
    }
    if(procdevname==NULL)procdevname="unknown";
  }

  return(report_lookup(b,errS,warnS,infoS,longS,errnoS,procdev,
		       devstateS,userS,devicename,procdevname));
}

static void auto_report_add(report_base *b,char *message,char *errS,
			     char *warnS,char *infoS, char *longS,
			     int err,char *procdev,int devstate,
			     char *user){
  char errnoS[80];
  char devstateS[80];

  if(err==P_ERROR_WILDCARD)
    sprintf(errnoS,"*");
  else
    sprintf(errnoS,"%d",err);

  if(devstate==A_STATE_WILDCARD)
    sprintf(devstateS,"*");
  else
    sprintf(devstateS,"%d",devstate);

  report_add(b,message,errS,warnS,infoS,longS,errnoS,procdev,devstateS,user);
}

/* axes: 0:ERROR,1:WARN,2:INFO,3:LONG,4:err,
         5:iftype,6:procdev,7:user,         
   +pr:  8:devicename,9:procdevname */

report_base *auto_callback_reportdb(void){
  report_base *r=report_create(8,10);

  /* Wildcard entry that reads AUTO Paranoia; some apps might want to
     replace/customize this message */

  auto_report_add(r,"AUTO Paranoia: Callback made for an unregistered "
		  "condition...  Please email the contents of this "
		  "message to xiphmont@mit.edu:\n"
		  "  err:%0 warn:%1 info:%2 long:%3 errno:%4 procdev:%5 "
		  "devstate:%6 user:%7",
		  "*","*","*","*",P_ERROR_WILDCARD,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"The device file %8 does not exist, is a dangling link"
		  ", or directory permissions prevent finding it.  To use %8,"
		  " make certain that the device entry exists and the "
		  "directories leading to it are accessible by %7.",
		  "1","*","*","1",P_ERROR_EBADNAME,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device file %8 does not exit, or is inaccessible.",
		  "1","*","*","0",P_ERROR_EBADNAME,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device file %8 exists, and the kernel has support"
		  "for %9 devices, however the system reports that there is"
		  "no actual hardware to match the device entry %8.",
		  "1","*","*","1", P_ERROR_ENXIO,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"No hardware detected matching device file %8.",
		  "1","*","*","0", P_ERROR_ENXIO,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Found device %8, but permissions do not allow access"
		  ".  To use %8, make certain that permissions allow access by"
		  " %7.",
		  "1","*","*","1", P_ERROR_EACCES,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device %8 permissions do not allow access by %7.",
		  "1","*","*","0", P_ERROR_EACCES,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Kernel reports: out of memory.",
		  "1","*","*","*", P_ERROR_ENOMEM,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Kernel reports: too many files open.",
		  "1","*","*","*", P_ERROR_EMFILE,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Unable to open device file %8; this kernel has no "
		  "%9 interface support and kerneld was not able to "
		  "find or load a %9 kernel module.  To use "
		  "%9 devices be certain that the kernel includes "
		  "%9 device support, either compiled in or as a "
		  "loadable module, and that the module is loaded or "
		  "configure for automatic loading.",
		  "1","*","*","1", P_ERROR_ENODEV,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Unable to open device file %8; the device file exists,"
		 " but there does not appear to be kernel support for the "
		 "device, and kerneld did not find a relevant module.  If "
		 "this is the proper device to load (Paranoia was not able "
		 "to identify the interface type), be certain that kernel "
		 "support is present and any relevant modules are loaded "
		 "and configured.",
		  "1","*","*","1", P_ERROR_ENODEV,"unknown",
		  A_STATE_WILDCARD,"*");

  auto_report_add(r,"Unable to open device file %8; no kernel or module "
		  "support.",
		  "1","*","*","0", P_ERROR_ENODEV,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device file %8 is a %9 device; Paranoia doesn't "
		  "currently support using %9 devices.",
		  "1","*","*","*", P_ERROR_ENOSYS,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device file %8 unexpectedly not a %9 device.",
		  "1","*","*","*", P_ERROR_ENOSYS,"*",A_STATE_MATCHTEMP,"*");

  auto_report_add(r,"File %8 is not a device file.",
		  "1","*","*","*", P_ERROR_ENOTDEVICE,"*",
		  A_STATE_WILDCARD,"*");

  /* this message will often be replaced by a specific platform include */
  auto_report_add(r,"Paranoia was not able to determine the bus:target:lun "
		  "numbers for device %8. Without this support, "
		  "Paranoia cannot automap %8 to a matching generic "
		  "device for use.  Please specify the desired generic "
		  "device explicitly.",
		  "1","*","*","1", P_ERROR_BTL,"*",A_STATE_WILDCARD,"*");
  auto_report_add(r,"Paranoia could not get bus:target:lun of device %8",
		  "1","*","*","0", P_ERROR_BTL,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Paranoia could not find any device files that appeared "
		  "to be %9 devices.  If Paranoia is being used with %9 "
		  "support, be certain that %9 device entries exist in /dev.",
		  "1","*","*","1", P_ERROR_NOMAPFILES,"*",
		  A_STATE_WILDCARD,"*");
		  
  auto_report_add(r,"Could not find any %9 device files.",
		  "1","*","*","0", P_ERROR_NOMAPFILES,"*",
		  A_STATE_WILDCARD,"*");

  auto_report_add(r,"Paranoia found %9 device files while automapping, "
		  "but none allowed read/write permission for user %7.  "
		  "Please check the device permissions.",
		  "1","*","*","1", P_ERROR_NOMAPACCESS,"*",
		  A_STATE_WILDCARD,"*");
  auto_report_add(r,"All %9 device files do not allow r/w access.",
		  "1","*","*","0", P_ERROR_NOMAPACCESS,"*",
		  A_STATE_WILDCARD,"*");

  auto_report_add(r,"Detected %9 kernel support and %9 device "
		  "files, but unable to find the %9 device file to match "
		  "device %8.  Please specify the %9 device name "
		  "explicitly.",
		  "1","*","*","1", P_ERROR_NOMAPFULL,"*",A_STATE_WILDCARD,"*");
  auto_report_add(r,"Cannot find a %9 device matching %8",
		  "1","*","*","0", P_ERROR_NOMAPFULL,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Detected %9 kernel support and %9 device "
		  "files, but unable to find the %9 device file to match "
		  "device %8.  Some %9 device entries did not allow "
		  "read/write permission; please check that permissions "
		  "for the the %9 device matching %8 are correct.",
		  "1","*","*","1", P_ERROR_NOMAPSOME,"*",A_STATE_WILDCARD,"*");
  auto_report_add(r,"Cannot find/access a %9 device matching %8",
		  "1","*","*","0", P_ERROR_NOMAPSOME,"*",A_STATE_WILDCARD,"*");

  /* Warning entries */

  auto_report_add(r,"This system doesn't seem to have /proc filesystem"
		  " support; /proc makes device autodetection and error"
		  " reporting easier.  Making do without...",
		  "*","1","*","*",P_WARN_NOPROC,"*",A_STATE_WILDCARD,"*");

  /* Informational (progress type) entries */

  auto_report_add(r,"Device entry %8 doesn't seem to to have any support "
		  "in the kernel at the moment; attempting to force the "
		  "appropriate kernel module to load...",
		  "*","*","1","*", P_INFO_LOADMOD,"*",A_STATE_WILDCARD,"*");
  auto_report_add(r,"Successfully loaded module for %8.",
		  "*","*","1","*", P_INFO_LOADEDMOD,"*",A_STATE_WILDCARD,"*");
  auto_report_add(r,"Hmm... appear to have failed loading module.",
		  "*","*","1","*", P_INFO_XLOADEDMOD,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Device entry %8 is a %9 device; attempting to map this "
		  "device to a matching generic device that supports raw "
		  "packet command interaction...",
		  "*","*","1","*", P_INFO_DEVTRYMAP,"*",A_STATE_WILDCARD,"*");

  auto_report_add(r,"Found matching %9 device: %8",
		  "*","*","1","*", P_INFO_DEVGOTMAP,"*",A_STATE_WILDCARD,"*");

  __pi_auto_callback_platform(r);

  return(r);
}

void auto_callback_setup(auto_device *a){
  if(!a->reportdb)a->reportdb=auto_callback_reportdb();
}

char *auto_callback(int err,int argtype,void *arg,int verbosity){
  auto_device *a=NULL;

  switch(argtype){
  case CALLBACK_ARG_AUTO:
    a=(auto_device *)arg;
    break;
  default:
    return(trans_callback(err,argtype,arg,verbosity));
  }

  /* if we don't have an associated report database, construct it now! */
  if(!a->reportdb)auto_callback_setup(a);

  return(auto_report_lookup(a->reportdb,verbosity&CV_ERROR,verbosity&CV_WARN,
			    verbosity&CV_INFO,verbosity&CV_LONG,err,
			    a->procdev,a->devstate,a->devicename));
}

