/******************************************************************
 * AUTO Paranoia IV
 * CopyPolicy: GNU Public License 2 applies
 * (C) 1999 Monty xiphmont@mit.edu
 * 
 * Linux toplevel autosense layer
 *
 ******************************************************************/

#ifdef LINUX
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#ifdef SBPCD_H
#include <linux/sbpcd.h>
#endif

#ifdef UCDROM_H
#include <linux/ucdrom.h>
#endif

#ifndef CDROMAUDIOBUFSIZ      
#define CDROMAUDIOBUFSIZ        0x5382 /* set the audio buffer size */
#endif

#include <linux/../scsi/sg.h>
#include <linux/../scsi/scsi.h>

#include <linux/cdrom.h>
#include <linux/major.h>

#include "trans_paranoia.h"  /* linux specific transport interface */
#include "auto_paranoia.h"   /* includes for this layer */
#include "paranoia_errors.h" /* error numbers for all paranoia */

#ifndef PG_MAJOR
#define PG_MAJOR 97          /* only became official recently */
#endif

#ifndef PCD_MAJOR
#define PCD_MAJOR 46          /* only became official recently */
#endif


/* With the advent of movable majors, this should be the primary way
   to ID a device.  This does not try to suck in a module for us.
   Unless opening the file sucks in the module, we can't force a load;
   I'm worried about adding root privilege features to this lib. */

static int linux_major_from_proc(int major,int cb_p){
  int proc=open("/proc/devices",O_RDONLY),done=0,devtype=DEVTYPE_UNKNOWN;
  int proccbp=-1;

  if(proc==-1)return(DEVTYPE_ERROR);

  while(!done){
    char buffer[80],procdev[80];
    int i,newline=0,procmajor;
    

    /* read until 80 chars, or newline */
    for(i=0;i<79 && !done && !newline;i++){
      switch(read(proc,buffer+i,1)){
      case 0:
	done=1;
	break;
      case -1:
	switch(errno){
	case EINTR:
	  i--;
	  continue;
	default:
	  close(proc);
	  return(DEVTYPE_ERROR);
	}
      default:
	break;
      }
      if(buffer[i]=='\n')newline=1;
    }
    buffer[i]='\0';

    if(!strncasecmp(buffer,"Character",9)){
      proccbp=S_IFCHR;
    }else if(!strncasecmp(buffer,"Block",5)){
      proccbp=S_IFBLK;
    }else{

      /* we're interested in sr, sg, pg, pcd; note that pcd cannot be
	 automatically matched to pg; alert that */
      
      if(sscanf(buffer,"%d %s",&procmajor,procdev)==2)
	if(procmajor==major && proccbp==cb_p){
	  done=1;
	  if(strcmp(procdev,"sg")==0)
	    devtype=DEVTYPE_LINUX_SG;
	  else if(strcmp(procdev,"sr")==0)
	    devtype=DEVTYPE_LINUX_SR;
	  else if(strcmp(procdev,"pg")==0)
	    devtype=DEVTYPE_LINUX_PG;
	  else if(strcmp(procdev,"pcd")==0)
	    devtype=DEVTYPE_LINUX_PCD;
	  else if(strncmp(procdev,"ide0",4)==0)
	    devtype=DEVTYPE_LINUX_IDE;
	  else if(strncmp(procdev,"ide",3)==0)
	    devtype=DEVTYPE_LINUX_IDE_SEC;
	  else
	    devtype=DEVTYPE_OTHER;
	}
    }
  }
  return(devtype);
}

struct  sg_id {
  long    l1; /* target | lun << 8 | channel << 16 | low_ino << 24 */
  long    l2; /* Unique id */
} sg_id;

typedef struct scsiid{
  int bus;
  int id;
  int lun;
} scsiid;

#ifndef SCSI_IOCTL_GET_BUS_NUMBER
#define SCSI_IOCTL_GET_BUS_NUMBER 0x5386
#endif

/* Even *this* isn't as simple as it bloody well should be :-P */
static int get_scsi_id(int fd, scsiid *id){
  struct sg_id argid;
  int busarg;

  /* get the host/id/lun */

  if(ioctl(fd,SCSI_IOCTL_GET_IDLUN,&argid))return(-1);
  id->bus=argid.l2; /* for now */
  id->id=argid.l1&0xff;
  id->lun=(argid.l1>>8)&0xff;

  if(ioctl(fd,SCSI_IOCTL_GET_BUS_NUMBER,&busarg)==0)
    id->bus=busarg;
  
  return(0);
}


/* device is the path of the dev enrty to use (or some identifier on
   systems without dev entries, ie bus:taget:lun on nonunix)

   commandset is the commandset we want to use on the device; COMMAND_CDDA 
   is the only command set implemented so far. 

   callback is our reporting callback hook 
*/

auto_device *auto_open(char *device,void (*callback)(int,int,void *)){
  struct stat st;
  int fd=-1,openerr=0;
  int type;
  int major,cb_p;
  auto_device state;
  char mapname[80];
  int moduleloadflag=0;

  mapname[0]='\0';

  memset(&state,0,sizeof(state));
  state.devicename=device;
  state.devstate=A_STATE_OPENING;
  state.devtype=DEVTYPE_UNKNOWN;

  if(stat(device,&st)){
    errno=errno_translate(errno);
    if(callback)callback(errno,CALLBACK_ARG_AUTO,&state);
    return(NULL);
  }
  major=(int)(st.st_rdev>>8);
  cb_p=st.st_mode & (S_IFBLK|S_IFCHR);
    
  if(!S_ISCHR(st.st_mode) &&
     !S_ISBLK(st.st_mode)){
    errno=P_ERROR_ENOTDEVICE;
    if(callback)callback(errno,CALLBACK_ARG_AUTO,&state);
    return(NULL);
  }
  
  /* Before opening, see if we have proc support and a device entry
     for this major.  This allows us to possibly ID the device for a
     better error message should the open fail due to not being able
     to load the proper module. */
  
  type=linux_major_from_proc(major,cb_p); /* this is not for device
					type id; we're checking to see
					if proc support exists and if
					our device is a module that
					must be loaded */

  if(type==DEVTYPE_ERROR){
    
    /* Uh oh, no proc support; warn the application as our
       autodetect/error reporting will be compromised */
    
    if(callback)callback(P_WARN_NOPROC,CALLBACK_ARG_AUTO,&state);
    
  }else if(type==DEVTYPE_UNKNOWN){
    
    /* OK, there's currently no device entry.  We'll have to force a
       module load */
    
    if(callback)callback(P_INFO_LOADMOD,CALLBACK_ARG_AUTO,&state);
    moduleloadflag=1;
  }
  
  /* Open the device file entry.  If we fail, don't bail yet.  We
     might be able to learn more about why first. */
  
  errno=0;
  fd=open(device,O_RDWR);
  if(fd==-1 && errno==EROFS){
    /* stupid linux bug; some nongeneric devices cannot be opened RW
       regardless of permissions */
    errno=0;
    fd=open(device,O_RDONLY);
  }
  openerr=errno;

  state.devtype=linux_major_from_proc(major,cb_p); /* For real now;
						      the module is
						      loaded (if it
						      can be) */

  if(state.devtype==DEVTYPE_UNKNOWN || state.devtype==DEVTYPE_ERROR){ 
    /* Still no dev major entry... */

    if(moduleloadflag){
      if(callback)callback(P_INFO_XLOADEDMOD,CALLBACK_ARG_AUTO,&state);
      errno=P_ERROR_ENODEV;
      if(callback)callback(errno,CALLBACK_ARG_AUTO,&state);
      return(NULL);
    }
    /* guess by 'typical' major */

    switch(major){
    case SCSI_CDROM_MAJOR:
      state.devtype=DEVTYPE_LINUX_SR; 
      break;

    case SCSI_GENERIC_MAJOR:
      state.devtype=DEVTYPE_LINUX_SG; 
      break;

    case PG_MAJOR: /* typically 97 right now */
      state.devtype=DEVTYPE_LINUX_PG;
      break;

    case PCD_MAJOR: 
      state.devtype=DEVTYPE_LINUX_PCD;
      break;

    case IDE0_MAJOR: 
      state.devtype=DEVTYPE_LINUX_IDE; /* cdromness is checked by the
					  cooked OS layer later */
      break;
    case IDE1_MAJOR: 
    case IDE2_MAJOR: 
    case IDE3_MAJOR: 
      state.devtype=DEVTYPE_LINUX_IDE_SEC; /* cdromness is checked by the
					  cooked OS layer later */
      break;

    default:
      /* OK, nothing obvious.  Either unknown, or paride on a
	 nonstandard major. We only get here when /proc is also
	 missing */
      {
	/* paranoid to the very end */
	char buffer[MAXPATHLEN],*real;
	real=realpath(device,buffer);
	
	if(real && strncmp("/dev/pg",real,7)==0){

	  /* OK, almost certainly generic paride */
	  state.devtype=DEVTYPE_LINUX_PG;
	}
      }
    }
  }else{
    if(moduleloadflag)
      if(callback)callback(P_INFO_LOADEDMOD,CALLBACK_ARG_AUTO,&state);
  }

  if(fd==-1){
    /* OK.  At this point, we were unable to open the device, but we
       probably know more about why */
    errno=errno_translate(openerr);
    callback(errno,CALLBACK_ARG_AUTO,&state);
    return(NULL);
  }

  /* If this is a SCSI cdrom device, we need to find the matching generic
     device */
  if(state.devtype==DEVTYPE_LINUX_SR){
    scsiid sr,sg;
    auto_device match;
    int done=0,devopened=0,accessdenied=0,i,j;

    memset(&match,0,sizeof(match));
    match.devtype=DEVTYPE_LINUX_SG;
    match.devstate=A_STATE_MATCHTEMP;
    
    if(callback)(*callback)(P_INFO_DEVTRYMAP,CALLBACK_ARG_AUTO,&state);
    
    /* what device are we? */
    if(get_scsi_id(fd,&sr)){
      /* D'oh! */
      errno=P_ERROR_BTL;
      if(callback)(*callback)(errno,CALLBACK_ARG_AUTO,&state);
      close(fd);
      return(NULL);
    }
    close(fd);

    match.devicename=mapname;

    /* Get the generic match */
    /* go through most likely /dev nodes for a match */
    for(i=0;i<26 && !done;i++){
      for(j=0;j<2 && !done;j++){
	
	switch(j){
	case 0:
	  /* number */
	  sprintf(mapname,"/dev/sg%d",i);
	  break;
	case 1:
	  /* number */
	  sprintf(mapname,"/dev/sg%c",i+'a');
	  break;
	}
        
	errno=0;
	fd=open(mapname,O_RDWR);
	if(fd!=-1){
	  devopened=1;
	  if(get_scsi_id(fd,&sg)==0){
	    if(sr.bus==sg.bus && sr.id==sg.id && sr.lun==sr.lun){
	      /* Match.  Everything is OK */
	      
	      if(callback)(*callback)(P_INFO_DEVGOTMAP,CALLBACK_ARG_AUTO,
				      &match);
	      done=1;
	    }else
	      close(fd);
	  }else{
	    /* As the above one went through, this will only happen if
	       we're not (for some reason) a SCSI device.  A regular
	       file with an unfortunate name perhaps? */

	    errno=P_ERROR_ENOSYS;
	    if(callback)(*callback)(errno,CALLBACK_ARG_AUTO,&match);
	    close(fd);

	  }
	}else{
	  /* Couldn't open.  Why couldn't we open? */
	  errno=errno_translate(errno);
	  
	  /* We're searching and expecting some of the device entries
             not to exist.  Other open errors (such as ENODEV) we want
             to report immediately */

	  switch(errno){
	  case P_ERROR_EACCES:
	    /* Perhaps only the device file we want is accessible.
               It's OK, keep looking, but remember */
	    accessdenied=1;

	    break;
	  case P_ERROR_EBADNAME:
	  case P_ERROR_ENXIO:
	    /* Don't expect to find them all. */

	    break;
	  case P_ERROR_ENODEV:
	    /* Ah ha.  No generic support.  Fall through. */
	  default:
	    callback(errno,CALLBACK_ARG_AUTO,&match);
	    return(NULL);
	  }
	}
      }
    }

    /* Are we 'done'?  If not, why did we not find anything? */

    if(done){
      /* fd is set to what we expect, just need to modify the device
	 name to the map. */
      device=mapname;
      state.devtype=DEVTYPE_LINUX_SG;
    }else{

      /* Could we access any of the scsi generic files? We know that
	 if we got here, either the appropriate device is missing, or
	 permission to access is denied.  Lack of generic SCSI support
	 (assuming accessible devices) would have been reported
	 earlier although it's possible that this case is being masked
	 by inaccessible files. */

      /* Grid (devopened,accessdenied):
	 0,0 ==    No sg device files (possibly no sg support)
	 0,1 ==    Files exist, none accessible (possibly no sg support)
         1,0 ==    Not enough dev sg files? (no match otherwise OK setup)
         1,1 ==    Proper file does not exist/not accessible (OK setup) */

      match.devicename=device;
      if(!devopened)
	if(!accessdenied)
	  errno=P_ERROR_NOMAPFILES;
	else
	  errno=P_ERROR_NOMAPACCESS;
      else
	if(!accessdenied)
	  errno=P_ERROR_NOMAPFULL;
	else
	  errno=P_ERROR_NOMAPSOME;
      
      if(callback)(*callback)(errno,CALLBACK_ARG_AUTO,&match);
	
      return(NULL);
    }
  }

  /* Dispatch according to device */

  {
    auto_device *ret;
    transport_device *(*dev_open)(int,void (*)(int,int,void *))=NULL;

    switch(state.devtype){
    case DEVTYPE_LINUX_SG:
      dev_open=&linuxsg_open;
      break;
    case DEVTYPE_LINUX_PG:
      /*      dev_open=&linuxpg_open;*/
      break;
    case DEVTYPE_LINUX_IDE:
    case DEVTYPE_LINUX_IDE_SEC:
    case DEVTYPE_LINUX_PCD: /* No ioctls yet actually */
      dev_open=&linuxioctl_open;
      break;
    default:
      /* Not a useable device type :-( */
      errno=P_ERROR_ENOSYS;
      if(callback)(*callback)(errno,CALLBACK_ARG_AUTO,&state);
      return(NULL);
    }
    
    ret=calloc(1,sizeof(auto_device));
    ret->devicename=calloc(strlen(device)+1,sizeof(char));
    strcat(ret->devicename,device);
    ret->devstate=A_STATE_OPENING;

    /* Get a transport device */
    if((ret->device=(*dev_open)(fd,callback))==NULL){
      close(fd);
      auto_close(ret,NULL);
      return(NULL);
    }
    
    ret->devstate=A_STATE_READY;
    return(ret);
  }
}

void auto_close(auto_device *d,void (*callback)(int,int,void *)){
  if(d){
    if(d->device)
      if(d->device->close)(*d->device->close)(d->device,callback);
    free(d);
  }
}

#endif /*LINUX*/







