/******************************************************************
 * Paranoia IV
 * CopyPolicy: GNU Public License 2 applies
 * Copyright (C) 1999 Monty xiphmont@mit.edu
 ******************************************************************/

#ifndef types_para_h
#define types_para_h

#include "paranoia_report.h"

/******************************************************************
 * Beginning with defines for setting up hardware device types
 * (HARDTYPE_xxxxxx), OS device interface types (IFFAMILY_xxxxx),
 * and packet command types (PACKET_xxxxxx).  Declare structure
 * types for the various flavors of packet commands, eg, SCSI, USB, 
 * etc (xxxx_packet_command structures unioned into packet_command).
 * Lastly, define the transport_device structure to glue it all 
 * together.
 ******************************************************************/

#define MAX_SENSE_DATA 96

#define CALLBACK_ARG_SCSI_PACKET 1

typedef struct scsi_packet_command {
  unsigned char *command;
  int cmdsize;

  unsigned char *inbuff;
  int bytesin;

  unsigned char *outbuff;
  int bytesout;   /* expected bytes */

  int bytesread;  /* bytes actually read */
  int completion_sec;
  int completion_usec;

  /* Sense data (on error). I don't use bitfields like Joerg because
     they are way too unpredictable with anything but GCC, and there's
     no workaround to get the packing right on several compilers.*/

  unsigned char sensedata[MAX_SENSE_DATA]; 
  int sensebytes;

  struct transport_device *t;
} scsi_packet_command;

/* hooks for other generic interfaces to be added later */
typedef union {
  struct scsi_packet_command scsi;
} packet_command;

/* high-level hardware types ********************************************/

#define HARDTYPE_WILDCARD           1
#define HARDTYPE_UNKNOWN            0
#define HARDTYPE_OTHER              2

#define HARDTYPE_DISK               3
#define HARDTYPE_CDROM              4
#define HARDTYPE_TAPE               5
#define HARDTYPE_FLOPPY             6
#define HARDTYPE_SCANNER            7

/* /dev interfaces recognized by PARANOIA *******************************/

#define IFFAMILY_WILDCARD           1

#define IFFAMILY_ERROR              2 /* Means for determining type
					 (usually /proc) unavailable */ 
#define IFFAMILY_UNKNOWN            0 /* type determination infrastructure
					 exists, but the type was not
					 registered */
#define IFFAMILY_OTHER              3 /* type was registered, but not a type
					 for which we autosense */

/* useful interface types.  Some are generic, some are 'platform
   specific'.  Use the general interface for a device (ie, use
   'SCSIGENERIC' for parallel port SCSI devices, not PARSCSIGENERIC)
   unless the OS differentiates between them (like Linux). The
   IFFAMILY plus device type should be sufficient to determine the
   low-level transport driver to use. */

/* Linux examples.  ATAPI CDROMS                -> IDESPECIAL
                    ATAPI CDROMS w/ IDESCSI emu -> SCSIGENERIC
		    SCSI CDROMS                 -> SCSIGENERIC
		    SBPRO IDE-like CDROM        -> PROPSPECIAL*/

#define IFFAMILY_PROPSPECIAL        4

#define IFFAMILY_SCSISPECIAL        5
#define IFFAMILY_SCSIGENERIC        6

#define IFFAMILY_IDESPECIAL         7
/* Only if the OS does not include ATAPI packet devs with SCSI devices */
#define IFFAMILY_IDEGENERIC         8

/* Only if the OS does not include PARIDE support in normal IDE */
#define IFFAMILY_PARIDESPECIAL      9  
#define IFFAMILY_PARIDEGENERIC      10
       
/* Only if the OS does not include PARSCSI support in normal SCSI */
#define IFFAMILY_PARSCSISPECIAL    11   
#define IFFAMILY_PARSCSIGENERIC     12

#ifdef IFFNAMES
static char *iff_names[]={"unknown",
			  "wildcard",
			  "error",
			  "unusable",
			  "proprietary ioctl", /* "device" ==> /dev */

			  "specialized SCSI",
			  "generic (raw) SCSI",

			  "specialized IDE",
			  "generic (raw) ATAPI",

			  "specialized parallel port IDE",
			  "generic (raw) parallel port ATAPI",

			  "specialized parallel port SCSI",
			  "generic (raw) parallel port SCSI",

			  NULL};
#endif
       
#ifdef HARDNAMES
static char *hard_names[]={"unknown",
			   "wildcard",
			   "unknown",
			   "unusable",
			   
			   "disk",
			   "cdrom",
			   "tape",
			   "floppy",
			   "scanner",
			   NULL};
#endif
       

/* packet command set types ********************************************/

#define PACKET_WILDCARD 1
#define PACKET_UNKNOWN  0
#define PACKET_SCSI     2  /*  "SCSI-like"; use scsi_packet_command  */
#define PACKET_COOKED   3  /*  specialized ioctl interface */

#define CALLBACK_ARG_TRANS  128

#define SCSIFLAG_SCSI   1
#define SCSIFLAG_ATAPI  2
#define SCSIFLAG_MMC    4

typedef struct transport_device{
  int packettype; /* packet_set: scsi-like, usb, special, etc */
  int packet_flags;
  int iffamily;   /* device interface family used by device */
  int hardtype;   /* worm, cdrom, etc */

  int fd;

  int max_request_bytes;
  int max_response_bytes;
  
  unsigned char *inq;
  int            inqbytes;
  char *devmodel;
  int prot_revision_flags;

  int (*close)(struct transport_device *,void (*callback)(int,int,void *));
  int (*reset)(struct transport_device *,void (*callback)(int,int,void *));
  int (*command)(struct transport_device *,packet_command *,
		 void (*callback)(int,int,void *));
  int (*special)(struct transport_device *,int commandset,
		 void (*callback)(int,int,void *));

  void *platform_dev;
  report_base *reportdb;
} transport_device;

/* built-in callback verbosity argument flags **************************/

#define CV_NONE                   0  /* verbosity: No message */ 
#define CV_ERROR                  1  /* verbosity: report errors */
#define CV_WARN                   2  /* verbosity: report warnings */
#define CV_INFO                   4  /* verbosity: report progress */
#define CV_LONG                   8  /* verbosity: long style reports */

/******************************************************************
 * AUTO library setup; this adds device scanning utilities on top 
 * of transport.  Returns "auto devices", just a wrapper around a 
 * transport device that adds some addiitonal sense info
 ******************************************************************/

/******************************************************************
 * Commandset probing and setup; defines the various command sets
 * Paranoia can be used to autoconfigure
 ******************************************************************/

#define CALLBACK_ARG_AUTO        256

#define A_STATE_WILDCARD          0 /* used for matching */
#define A_STATE_OPENING           1 /* device still in auto_open */
#define A_STATE_MATCHTEMP         2 /* temporary device created in auto_open
				       for matching purposes */
#define A_STATE_READY             3 /* device ready for use */

typedef struct auto_device{

  char *procdev; /* platform-specific device type strings used
		    internally for autosearch/match.  On systems with
		    /proc, this is a /proc/devices entry.  Can be used
		    to similar end on other systems. */
  int devstate;   

  char *devicename;
  transport_device *device;

  report_base *reportdb;
} auto_device;

#endif
