/******************************************************************
 * TRANSPORT Paranoia IV
 * CopyPolicy: GNU Public License 2 applies
 * Copyright (C) 1998 Monty xiphmont@mit.edu
 * 
 * Linux Generic PARIDE transport layer
 *
 ******************************************************************/

#include "trans_paranoia.h"
#include "paranoia_errors.h"
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>

#if defined(LINUX) && defined(PG)
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <string.h>
#include <linux/pg.h>

/* platform specific information to bury in transport_device */
typedef struct linuxpg_device{
  int fd;
  long maxreqbytes;
  long status;
} linuxpg_device;

static int linuxpg_reset(transport_device *d,void (*callback)(int,int,void *)){
  linuxpg_device *pg=(linuxpg_device *)(d->platform_dev);
  struct pg_write_hdr hdr={PG_MAGIC,PG_RESET,0};
  
  while(1){
    if(write(pg->fd,(char *)&hdr,sizeof(hdr))==-1){
      switch(errno){
      case EINTR:
	continue;
      default:
	errno=errno_translate(errno);
	if(callback)callback(errno,CALLBACK_ARG_TRANS,d);
	return(-1);
      }
    }else{
      return(0);
    }
  }
}

int linuxpg_close(transport_device *d,void (*callback)(int,int,void *)){
  if(d){
    linuxpg_device *pg=(linuxpg_device *)(d->platform_dev);
    if(pg){
      if(pg->fd>=0)close(pg->fd);
      free(pg);
    }
    free(d);
  }
  return(0);
}

/* The externally visible command request needs to fetch sense.  This
   one does not... so that sense can use it without possibly recursing */

static int linuxpg_icommand(transport_device *d,packet_command *p){ 
  struct pg_write_hdr writestruct;
  struct pg_read_hdr  readstruct;
  linuxpg_device *pg=(linuxpg_device *)(d->platform_dev);
  int ret;

  pg->status=0;

  if(p->bytesin>d->max_request_bytes)
    return(P_ERROR_EPACKETBIG);

  if(p->bytesout>d->max_response_bytes)
    return(P_ERROR_EPACKETBIG);

  /* I like my way better.  The raw packet commands are easier to read */

  writestruct.magic=PG_MAGIC;
  writestruct.func=PG_COMMAND;
  writestruct.dlen=p->bytesin;
  writestruct.timeout=10; /* arbitrary right now */
  memcpy(&(writestruct.packet),p->inbuff,p->cmdsize);

  p->bytesread=0;

  {
    struct iovec vector[2];
    vector[0].iov_base=&writestruct;
    vector[0].iov_len=sizeof(writestruct);
    vector[1].iov_base=p->inbuff+p->cmdsize;
    vector[1].iov_len=p->bytesin-p->cmdsize;

    while(1){
      /*      clear_garbage(d);*/
      ret = writev(pg->fd,vector,2);
      if(ret==-1){
	switch(errno){
	case EINTR:
	  continue;
	default:
	  errno=errno_translate(errno);
	  return(-1);
	}
      }else
	break;
    }
  }

  /* Read it back */

  {
    struct iovec vector[2];
    vector[0].iov_base=&readstruct;
    vector[0].iov_len=sizeof(readstruct);
    vector[1].iov_base=p->outbuff;
    vector[1].iov_len=p->bytesout;
    
    while(1){
      ret=readv(pg->fd,vector,2);
      if(ret==-1){
	switch(errno){
	case EINTR:
	  continue;
	default:
	  errno=errno_translate(errno);
	  return(-1);

	  /* we don't check sense here as this command may have been
             called to do the sense.  The sense is done by our wrapper */
	  return(readstruct.scsi);
	}
      }else{
	p->bytesread=ret-sizeof(readstruct);
	pg->status=readstruct.scsi;
	break;
      }
    }
  }
  return(0);
}

/* We don't automatically get sense bytes on error from pg like we do sg */
static int linuxpg_sense(transport_device *d,packet_command *p){
  int ret;

  /* request sense command: */
  unsigned char command[6]={0x03,0,0,0,MAX_SENSE_DATA,0};

  packet_command packet={command,6,NULL,0,p->sensedata,MAX_SENSE_DATA,0,
			 {},0};

  ret=linuxpg_icommand(d,&packet);

  p->sensebytes=packet.bytesread;
  return(ret);
}

static int linuxpg_command(transport_device *d,packet_command *p,
			    void (*callback)(int,int,void *)){ 

  memset(p->sensedata,0,MAX_SENSE_DATA);
  p->sensebytes=0;

  {
    linuxpg_device *pg=(linuxpg_device *)(d->platform_dev);
    int ret=linuxpg_icommand(d,p);
    
    if(ret==-1){
      
      /* error sending command or reading back result */
      if(callback)callback(errno,CALLBACK_ARG_TRANS,d);
      return(-1);
      
    }else{
      if(pg->status){
	
	/* Linux PARIDE went OK, but the device had some error
	   processing the request */
	
	linuxpg_sense(d,p);
	errno=P_ERROR_CHECKSENSE;
	return(-1);
	
      }else{
	/* No worries. */      
	return(0);
      }
    }
  }
}

transport_device *linuxpg_open(int fd,void (*callback)(int,int,void *)){
  linuxpg_device *pg=calloc(1,sizeof(linuxpg_device));
  transport_device *ret=calloc(1,sizeof(transport_device));

  /* need to ping to see if this is really a PG device and if so, what type */


  ret->platform_dev=pg;
  ret->devtype=TRANSSET_ATAPI;
  ret->iftype=TRANSIF_PG;
  ret->close=&linuxpg_close;
  ret->reset=&linuxpg_reset;
  ret->command=&linuxpg_command;
  
  pg->maxreqbytes=32768; /* I don't suck it out of the defines on purpose */
  pg->fd=fd;

  ret->max_request_bytes=pg->maxreqbytes; /* Grant: Is this right? */
  ret->max_response_bytes=pg->maxreqbytes;

  return(ret); /* yay! */
}

#else

/* Only newer kernels have PARIDE support */

transport_device *linuxpg_open(int fd,void (*callback)(int,int,void *)){
  transport_device temp;
  temp.iffamily=IFFAMILY_PARSCSIGENERIC;
  temp.packettype=PACKET_SCSI;
  temp.packet_flags=SCSIFLAG_ATAPI;
  temp.hardtype=HARDTYPE_UNKNOWN;

  errno=P_ERROR_ENOTCONFIG;
  if(callback)callback(errno,CALLBACK_ARG_TRANS,&temp);
  return(NULL);
}

#endif



