
#include <stdio.h>

#define EV_CODE_C
#include <X10/Xlib.h>
#include "ev_tokens.h"

extern char *calloc();

extern ENV_PTR	curenv;
extern OBJ_PTR	curobj;
extern CODE_PTR curcode;

char	theProgName[64];

/* utilities for: creating (new), finding (find), and adding to lists (add) */

/* this function should be called before anything gets parsed */

int
ev_MakeCBinding(prog)
	char *prog;	/* should be argv[0] */
{
HashTablePtr cprog_link();

	C_ht = cprog_link( prog );
	strcpy( theProgName, prog );
	return 0;
}

/* first, the environment functions */

/* ---------------------------------------- */

ENV_PTR
newenv(ename)
	char *ename;
{
ENV_PTR	ep;

	if( findenv(ename) != NULL )
		ev_Error( "duplicate environment already exists!" );
	ep = (ENV_PTR) calloc(1,sizeof(ENVIRONMENT));
	if( ep==NULL )
		ev_Error( "out of memory in newenv()" );
	if( ename && strlen(ename) )
	{
		ep->name = calloc(1,strlen(ename)+1);
		strcpy( ep->name, ename );
	}
	return ep;
}

ENV_PTR
findenv( ename )
	char *ename;
{
ENV_PTR	ep;

	for( ep=envlist; ep; ep=ep->next )
		if( !strcmp(ep->name,ename) )
			return ep;
	return NULL;
}

int
addenv(elist,ep)	

	ENV_PTR *	elist;	/* the env list */
	ENV_PTR		ep;	/* the one to add */
{	
ENV_PTR	tep, ttep;

	if( !elist || !ep )
		ev_Error( "internal error; NULL ptr in addenv()" );

	if( findenv(ep->name) )
	{
		fprintf( stderr, "duplicate environment '%s' already exists; new env NOT added\n",
			ep->name );
		return -1;
	}

	if( *elist == NULL )	/* envlist is NULL */
	{
		*elist = ep;
		ep->next = NULL;
	}
	else
	{
		/* append at tail */
		for( tep = *elist; tep->next; tep = tep->next );
		tep->next = ep;
		ep->next = NULL;
	}
	return 0;
}

/* ---------------------------------------- */
/* the window binding functions for objects */

WBIND_PTR
newbinding( op )
	OBJ_PTR	op;
{
WBIND_PTR	wbp;

	wbp = (WBIND_PTR) calloc(1,sizeof(WBINDING));
	if( !wbp )
		ev_Error( "out of memory in newbinding()" );
	wbp->magic = WBIND_MAGIC;
	wbp->wname = op->name;
	wbp->obj = op;
	wbp->w = (Window) NULL;
	return wbp;
}

/* ---------------------------------------- */
/* the object functions */

OBJ_PTR
newobj(oname)
	char *oname;
{
OBJ_PTR	op;

	if( !oname || !strlen(oname) )
		return NULL;

	op = (OBJ_PTR) calloc(1,sizeof(OBJ));
	op->name = calloc(1,strlen(oname)+1);
	strcpy(op->name,oname);
	op->binding = newbinding(op);
	return op;
}

	
OBJ_PTR
findobj( envp, oname )
	ENV_PTR	envp;
	char *oname;
{
HashNodePtr	hnp;

	if( !envp || !envp->obj_ht || !oname || !strlen(oname) )
		return NULL;
	hnp = hash_lookup( envp->obj_ht, oname );
	if( hnp )
		return ((OBJ_PTR) hnp->ptr);
	else
		return NULL;
}

int
addobj( envp, op )
	ENV_PTR	envp;
	OBJ_PTR	op;
{
OBJ_PTR	top;
HashNodePtr	hnp;

	/* must add to obj list and env hash table */

	if( !op || !envp ) /* OLD: ignored dups!! || findobj(envp,op->name) ) */
		return -1;

	if( envp->obj_ht == NULL )
		envp->obj_ht = hash_create( "Objects", 256 );

	/* NEW: if obj already exists, replaces it by insertion in hash table and objlist */
	hnp = hash_lookup( envp->obj_ht, op->name );
	if( hnp )
	{
		printf("addobj(): adding duplicate object '%s'\n",op->name);
		if( (OBJ_PTR)hnp->ptr == envp->objlist )
		{
			op->next = envp->objlist->next;
			envp->objlist = op;
		}
		else
		{
			for( top = envp->objlist; 
				top && top->next != (OBJ_PTR)hnp->ptr; top = top->next);
			if( top )
				top->next = op;		/* set prev */
			op->next = ((OBJ_PTR)(hnp->ptr))->next;
		}
		/* note that this version DOES NOT free the over-written obj!! */
		return 0;
	}

	hnp = hash_install( envp->obj_ht, op->name, NULL, 0 );
	hnp->ptr = (char *) op;

	if( envp->objlist == NULL )
	{
		envp->objlist = op;
		op->next = NULL;
	}
	else
	{
		/* append at tail */
		for( top = envp->objlist; top && top->next; top = top->next );
		top->next = op;
		op->next = NULL;
	}
	return 0;
}

/* --------------------------------- */
/* the c-dcl functions */

C_DCL_PTR
newCdcl(cname,ctype)
	char *cname;
	int	ctype;
{
C_DCL_PTR	cdp;

	if( !cname || !strlen(cname) )
		return NULL;
	cdp = (C_DCL_PTR) calloc(1,sizeof(C_DCL));
	cdp->name = calloc(1,strlen(cname)+1);
	strcpy(cdp->name,cname);
	cdp->type = ctype;
	return cdp;	
}

C_DCL_PTR
findCdcl(cname)
	char *cname;
{
C_DCL_PTR	tcdp;

	for( tcdp = C_declist; tcdp; tcdp=tcdp->next )
		if( !strcmp(cname,tcdp->name) )
			return tcdp;
	return NULL;
}

int
addCdcl( cdp )
	C_DCL_PTR cdp;
{
C_DCL_PTR	tcdp;


	/* if duplicate name, does nothing; old remains valid */
	if( findCdcl(cdp->name) )
		return -1;
	
	if( C_declist == NULL )
	{
		C_declist = cdp;
		cdp->next = NULL;
	}
	else
	{
		for( tcdp = C_declist; tcdp && tcdp->next; tcdp=tcdp->next );
		tcdp->next = cdp;
		cdp->next = NULL;
	}

	return 0;
}

/* --------------------------------- */
/* the code functions */

CODE_PTR
newcode(cname,ctype)
	char *	cname;
	int	ctype;
{
CODE_PTR	cp;

	if( curobj && findcode(curobj->codelist,cname) )
		ev_Error( "object code segment already exists!" );
	else if( curenv && findcode(curenv->global_code,cname) )
		ev_Error( "global code segment already exists!" );

	cp = (CODE_PTR) calloc(1,sizeof(CODE));
	if( cp==NULL )
		ev_Error( "out of memory in newcode()" );
	if( cname && strlen(cname) )
	{
		cp->name = calloc(1,strlen(cname)+1);
		strcpy( cp->name, cname );
	}
	cp->type = ctype;
	if( cp->type != CFUNC_CODESEG )
	{
		cp->code = (unsigned long *) 
			calloc( MAXSTR, sizeof(unsigned long) );
	}
	cp->pc = 0;
	cp->ninstr = 0;
	cp->varlist = NULL;
	return cp;
}

CODE_PTR
findcode( clist, cname )
	CODE_PTR clist;
	char *	cname;
{
CODE_PTR	cp;

	if( !clist || !cname || !strlen(cname) )
		return NULL;
	for( cp=clist; cp; cp=cp->next )
		if( !strcmp(cp->name,cname) )
			return cp;
	return NULL;
}

int
addcode(clist,cp)

	CODE_PTR *	clist;	/* the code list */
	CODE_PTR	cp;	/* the one to add */
{	
CODE_PTR	tcp;

	if( !clist || !cp )
		ev_Error( "internal error; NULL ptr in addcode()" );

	if( findcode( *clist, cp->name ) )
	{
		fprintf(stderr,"duplicate codeseg '%s' found in code-list; new codeseg NOT added\n",
			cp->name );
		return -1;
	}
			
	if( *clist == NULL )
	{
		*clist = cp;
		cp->next = NULL;
	}
	else
	{
		/* append at tail */
		for( tcp = *clist; tcp && tcp->next; tcp = tcp->next );
		tcp->next = cp;
		cp->next = NULL;
	}
	return 0;
}

/* ------------------------------- */

/* the var functions */

VAR_PTR
newvar(vname)
	char *vname;
{
VAR_PTR		vp;
char *calloc();

	vp = (VAR_PTR) calloc(1,sizeof(VAR));
	if( vp==NULL )
		ev_Error( "out of memory in newvar()" );
	vp->magic = VAR_MAGIC;
	if( vname && strlen(vname) )
	{
		vp->name = calloc(1,strlen(vname)+1);
		strcpy(vp->name,vname);
	}
	vp->type = T_int;	/* default */
	vp->val.ival = 0;
	vp->val.sval = calloc(1,MAXSTR);
	return vp;
}

int
initvar( vp )			/* for temporary auto vars */
	VAR_PTR vp;
{
	vp->name = NULL;
	vp->magic = VAR_MAGIC;
	vp->type = T_int;
	vp->val.ival = 0;
	vp->val.sval = calloc(1,MAXSTR);	/* trashes what was there,
						if anything */
}

VAR_PTR
findvar( vlist, str )

	VAR_PTR	vlist;
	char *str;
{
VAR_PTR	vp;

	if( !vlist || !str || !strlen(str) )
		/* ev_Error( "internal error; bad ptr given to findvar()" ); */
		return NULL;
	for( vp = vlist; vp; vp = vp->next )
		if( strlen(vp->name) && !strcmp( vp->name, str ) )
			return vp;
	return NULL;
}

int
addvar( vlist, vp )

	VAR_PTR *	vlist;
	VAR_PTR		vp;
{
VAR_PTR	tvp;

	if( !vlist || !vp )
		ev_Error( "internal error; NULL ptr in addvar()" );
	if( vp->magic != VAR_MAGIC )
		ev_Error( "internal error; bad ptr in addvar()" );

	if( *vlist == NULL )
	{
		*vlist = vp;
		vp->next = NULL;
	}
	else	/* append at tail */
	{
		for( tvp = *vlist; tvp->next; tvp = tvp->next );
		tvp->next = vp;
		vp->next = NULL;
	}
	return 0;
}

/* -------------------------- */
/* the C var functions */

int
IsTrue( vp )
	VAR_PTR	vp;
{
	ev_GetCval( vp );	/* does nothing if non-C */

	switch( vp->type )
	{
	case T_int:
	case T_Cint:
		if( vp->val.ival == 0 )
			return 0;
		else
			return 1;
		break;
	case T_float:
	case T_Cflt:
		if( vp->val.fval == 0.0 )
			return 0;
		else
			return 1;
		break;
	case T_string:
	case T_Cstr:
		if( !strlen(vp->val.sval) )
			return 0;
		else
			return 1;
		break;
	default:
		fprintf(stderr, 
			"EventScript Exec err: bad var->type in IsTrue()\n");
		return 0;	/* error; should never happen */
		break;
	}
	return 0;
}

int
toInt( vp )
	VAR_PTR vp;
{
	/* assumes the C-things have already been 'fetched' from C */

	switch( vp->type )
	{
	case T_int:
	case T_Cint:
		break;
	case T_float:
	case T_Cflt:
		vp->val.ival = (int) vp->val.fval;
		break;
	case T_string:
	case T_Cstr:
		vp->val.ival = atoi(vp->val.sval);
		break;
	default:
		break;
	}
	vp->type = T_int;
}

int
toFlt( vp )
	VAR_PTR vp;
{
double atof();

	switch( vp->type )
	{
	case T_int:
	case T_Cint:
		vp->val.fval = (float) vp->val.ival;
		break;
	case T_float:
	case T_Cflt:
		break;
	case T_string:
	case T_Cstr:
		vp->val.fval = (float) atof(vp->val.sval);
		break;
	default:
		break;	
	}
	vp->type = T_float;
}

int
toNum( vp )
	VAR_PTR	vp;
{
	if( vp->type == T_string )
	{
		if( strpos(vp->val.sval,'.') >= 0 )
			toFlt( vp );
		else
			toInt( vp );
	}
}

int
toStr( vp )
	VAR_PTR vp;
{

	if( vp->type != T_string )
	{
		if( vp->type == T_int )
			sprintf(vp->val.sval,"%d", vp->val.ival);
		else if( vp->type == T_float )
			sprintf(vp->val.sval,"%f", vp->val.fval);
		vp->type = T_string;
	}
}

int
ev_GetCval( vp )	/* pulls in the value from C-land */

	VAR_PTR	vp;
{
	if( !vp )
		return -1;
	switch( vp->type )
	{
	case T_int:
	case T_float:
	case T_string:
		return -1;
		break;
	case T_Cint:
		vp->val.ival = *((int *) vp->Cptr);
		break;
	case T_Cflt:
		vp->val.fval = *((float *) vp->Cptr);
		break;
	case T_Cstr:
		strcpy( vp->val.sval, vp->Cptr );
		break;
	default:
		fprintf(stderr,"ev_GetCval(): unknown var type\n");
		return -1;
		break;
	}
	return 0;
}

int
ev_SetCval( varp, valp )	/* sends the value out to C-land */
	VAR_PTR varp, valp;
{
VAR	v;
VAR_PTR	vp = &v;

	if( !varp || !valp || !IsCvar(varp) )
		return -1;

	bcopy( valp, vp, sizeof(VAR) );

	switch( varp->type )
	{
	case T_int:
	case T_float:
	case T_string:
		return -1;
		break;
	case T_Cint:
		toInt( vp );
		varp->val.ival = vp->val.ival;
		*((int *) varp->Cptr) = varp->val.ival;
		break;
	case T_Cflt:
		toFlt( vp );
		varp->val.fval = vp->val.fval;
		*((float *) varp->Cptr) = vp->val.fval;
		break;
	case T_Cstr:
		toStr( vp );
		strcpy( varp->val.sval, vp->val.sval );
		strcpy( varp->Cptr, vp->val.sval );
		break;
	default:
		fprintf(stderr,"ev_SetCval(): unknown var type\n");
		return -1;
		break;
	}

	return 0;
}

/* ---------------------------- */

/* the constant functions */

VAR_PTR
newint(i)
	int i;
{
VAR_PTR	vp;

	if( (vp = findint(i)) != NULL )
		return vp;
	vp = newvar(NULL);
	vp->type = T_int;
	vp->val.ival = i;
	if( !curenv )
		ev_Error( "no current env in newint()" );
	addvar( &(curenv->constants), vp );
	return vp;
}

VAR_PTR
ev_newstr(s)
	char *s;
{
VAR_PTR vp;

	if( (vp=findstr(s)) != NULL )
		return vp;
	vp = newvar(NULL);
	vp->type = T_string;
	strcpy(vp->val.sval,s);
	if( !curenv )
		ev_Error( "no current env in ev_newstr()" );
	addvar( &(curenv->constants), vp );
	return vp;
}

VAR_PTR
newflt(f)
	float f;
{
VAR_PTR vp;

	if( (vp=findflt(f)) != NULL )
		return vp;
	vp = newvar(NULL);
	vp->type = T_float;
	vp->val.fval = f;
	if( !curenv )
		ev_Error( "no current env in newflt()" );
	addvar( &(curenv->constants), vp );
	return vp;
}

VAR_PTR
findint( i )		/* look in global constants list */
	int	i;
{
VAR_PTR vp;
VAR_PTR	vlist = NULL;

	if( !curenv )
		ev_Error( "internal error; no curenv in findint()" );

	vlist = curenv->constants;
	if( !vlist )
		return NULL;
	if( vlist->magic != VAR_MAGIC )
		ev_Error( "internal error; bad env constants in findint()" );
	
	for( vp=vlist; vp; vp = vp->next )
		if( !strlen(vp->name) && vp->type == T_int && vp->val.ival==i )
			return vp;
	return NULL;
}

VAR_PTR
findflt( f )		/* look in global constants list */

	float f;
{
VAR_PTR vp;
VAR_PTR	vlist = NULL;

	if( !curenv )
		ev_Error( "internal error; no curenv in findflt()" );

	vlist = curenv->constants;
	if( !vlist )
		return NULL;
	if( vlist->magic != VAR_MAGIC )
		ev_Error( "internal error; bad env constants in findflt()" );
		
	for( vp=vlist; vp; vp = vp->next )
		if( !strlen(vp->name) && vp->type == T_float && vp->val.fval==f )
			return vp;
	return NULL;
}

VAR_PTR
findstr( s )		/* look in global constants list */

	char *s;
{
VAR_PTR vp;
VAR_PTR	vlist = NULL;

	if( !curenv )
		ev_Error( "internal error; no curenv in findstr()" );

	vlist = curenv->constants;
	if( !vlist )
		return NULL;
	if( vlist->magic != VAR_MAGIC )
		ev_Error( "internal error; bad env constants in findstr()" );
		
	for( vp=vlist; vp; vp = vp->next )
		if( !strlen(vp->name) && vp->type == T_string 
			&& !strcmp(vp->val.sval,s) )
			return vp;
	return NULL;
}

/* easy one-step function for the parser when it encounters a variable
	or constant */

VAR_PTR
makevar()
{
VAR_PTR	vp = NULL;
int	atoi();
double	atof();

	switch( theToken.type )
	{
	case T_int:
		vp = newint(atoi(theToken.value));
		break;
	case T_float:
		vp = newflt((float) atof(theToken.value));
		break;
	case T_string:
		vp = ev_newstr(theToken.value);
		break;
	case T_word:
		/* see if it's a local, common, or global var;
		   know it's not a function name, so
		   if no match, make it a var, and add to
		   the correct list */

		if( curcode && (vp = findvar(curcode->varlist,theToken.value)) )
		{
			;
		}
		else if( curobj && (vp = findvar(curobj->common_vars,theToken.value)) )
		{
			;
		}
		else if( curenv && (vp = findvar(curenv->global_vars,theToken.value)) )
		{
			;
		}
		else	/* not found in any list; must create and add new var */
		{
			vp = newvar(theToken.value);
			if( curcode )
				addvar( &(curcode->varlist), vp );
			else if( curobj )
				addvar( &(curobj->common_vars), vp );
			else if( curenv )
				addvar( &(curenv->global_vars), vp );
		}
		break;
	default:
		ev_Error( "unknown var type: %d, in makevar()" );
		break;
	}
	return vp;
}

/* allocate a new expression */

EXPR_PTR
newexpr()
{
EXPR_PTR	ep;

	ep = (EXPR_PTR) calloc(1,sizeof(EXPR));
	if( !ep )
		ev_Error( "out of memory in newexpr()" );
	ep->magic = EXPR_MAGIC;
	return ep;
}

/* make a new function call */

FCALL_PTR
newfcall( codep )
	CODE_PTR	codep;
{
FCALL_PTR	fcp = NULL;

	if( !codep )
		return NULL;
	fcp = (FCALL_PTR) calloc(1,sizeof(FCALL));
	if( !fcp )
		ev_Error( "out of memory in newfcall()" );
	fcp->magic = FCALL_MAGIC;
	fcp->func = codep;
	return fcp;
}

int
addparm( fcp, pp )

	FCALL_PTR	fcp;
	char *		pp;	/* pointer to parm: var, expr, or fcall */
{

	if( !fcp || !pp )
		return -1;

	if( fcp->nparms < MAXPARMS )
		fcp->parms[fcp->nparms++] = pp;
	else
		return -1;
	return 0;
}

/* add the current instruction (in a long int) to the current code stream */

int
append_instr( codeseg, val )
	CODE_PTR	codeseg;
	long		val;
{
char	msg[MAXSTR];

	if( !codeseg )
		ev_Error( "bad codeseg in append_instr() ");
		
	if( codeseg->pc < MAXSTR )
		codeseg->code[codeseg->pc++] = val;
	else
	{
		sprintf(msg,"instruction overflow, codeseg <%s>",codeseg->name);
		ev_Error( msg );
	}
}

VAR_PTR
save_vars( cp )
	CODE_PTR cp;
{
VAR_PTR	vp, tvp;
VAR_PTR vlist = NULL;
char *calloc();

	for( vp=cp->varlist; vp; vp=vp->next )
	{
		tvp = (VAR_PTR) calloc(1,sizeof(VAR));
		bcopy( vp, tvp, sizeof(VAR) );
		addvar( &vlist, tvp );
	}
	return vlist;	
}

int
restore_vars( cp, vlist )

	CODE_PTR 	cp;
	VAR_PTR 	vlist;
{
VAR_PTR	vp, tvp, temp, theNext;

	for( vp=cp->varlist, tvp=vlist; vp; vp=vp->next,tvp=theNext )
	{
		temp = vp->next;
		if( !vp->IsStatic && !vp->IsParm )	/* don't restore statics or parms  */
			bcopy( tvp, vp, sizeof(VAR) );
		vp->next = temp;
		theNext = tvp->next;
		free( (char *) tvp );
	}
	return 0;
}

/* ----------------------------- */

/* following are utils for printing out things */

int
dump_code( cp )
	CODE_PTR	cp;
{
VAR_PTR		vp;

	printf("\nDUMP of codeseg <%s>\n",cp->name );
	printf("\nVariable List:\n\n");
	for( vp=cp->varlist; vp; vp=vp->next )
		print_var( vp );
	printf("\nCODE: (%d instructions)\n\n",cp->pc);
	print_code( cp );
}

int
print_var( vp )
	VAR_PTR	vp;
{
	if( strlen(vp->name) )
		printf("VARIABLE <%s>; ",vp->name);
	else
		printf("CONSTANT ");
		
	switch( vp->type )
	{
	case T_int:
		printf("integer; value: %d\n",vp->val.ival);
		break;
	case T_float:
		printf("float; value: %f\n",vp->val.fval);
		break;
	case T_string:
		printf("string; value <%s>\n",vp->val.sval);
		break;
	case T_word:
		printf("word; (illegal!!); <%s>\n",vp->val.sval);
		break;
	default:
		printf("illegal type code %d !!!\n",vp->type);
		break;
	}		
	
}

int
print_code( cp )
	CODE_PTR cp;
{
int		i;
int level=0;
int *ip;
VAR_PTR		vp;

	if( !cp )
		return -1;
	switch( cp->type )
	{
	case CFUNC_CODESEG:
		printf("-- << Cfunc '%s' >>\n",cp->name);
		return 0;
		break;
	case MSG_CODESEG:
		printf("\n");
		printf("on %s",cp->name);
		if( cp->nparms > 0 )
			printf("\t");
		for( i=0; i<cp->nparms; i++ )
		{
			printf("%s",cp->parms[i]->name);
			if( (i+1) < cp->nparms )
				printf(", ");
		}
		break;
	case FUNC_CODESEG:
		printf("\n");
		printf("%s( ",cp->name);
		for( i=0; i<cp->nparms; i++ )
		{
			printf("%s",cp->parms[i]->name);
			if( (i+1) < cp->nparms )
				printf(", ");
		}
		printf(" )");
		break;
	default:
		printf("print_code(): ERROR: unknown type of code seg\n");
		return -1;
		break;
	}

	printf("\n{\n");

	/* print the local vars */
	
	for( vp = cp->varlist; vp; vp = vp->next )
	{
		if( !vp->IsParm )
		{
			if( vp->IsStatic )
				printf("static %s;\n",vp->name);
			else
				printf("var %s;\n",vp->name);
		}
	}
	printf("\n");
	++level;
	indent(level);
		
	for( i=1; i<cp->pc; i++ )	/* starts past the initial '{' */
	{
		switch( cp->code[i] )
		{
		case T_cmdend:
			printf(";\n");
			if( cp->code[i+1] != T_endblock )
				indent(level);
			else
				indent(level-1);
			break;
		case T_if:
			printf("if ");
			break;
		case T_else:
			printf("else ");
			break;
		case T_begblock:
			printf("\n");
			indent(level);
			printf("{\n");
			++level;
			indent(level);
			break;
		case T_endblock:
			--level;
			printf("} ");
			if( cp->code[i+1] == T_endblock )
			{
				printf("\n");
				indent(level-1);
			}
			else if( cp->code[i+1] != T_while && cp->code[i+1] != T_until )
			{
				printf("\n");
				indent(level);
			}
			break;
		case T_repeat:
			printf("repeat ");
			break;
		case T_while:
			printf("while ");
			break;
		case T_until:
			printf("until ");
			break;
		case T_next:
			printf("next ");
			break;
		case T_exit:
			printf("exit ");
			break;
		case T_put:
			printf("put ");
			break;
		case T_into:
			printf("into ");
			break;
		case T_send:
			printf("send ");
			break;
		case T_to:
			printf("to ");
			break;
		default:
			if( cp->code[i] > 500 )
			{
				ip = (int *) cp->code[i];
				if( ip && *ip == VAR_MAGIC )
					show_var( (VAR_PTR) cp->code[i] );
				else if( ip && *ip == EXPR_MAGIC )
					print_expr( (EXPR_PTR) cp->code[i] );
				else if( ip && *ip == FCALL_MAGIC )
					print_fcall( (FCALL_PTR) cp->code[i] );
			}
			else
				printf("<ERROR: unknown token: #%d>\n",cp->code[i]);
			break;
		}
	}
	return 0;	
}

static int
indent(lev)
	int lev;
{
int i;

	for(i=0;i<lev;i++)
		printf("  ");
}

int
print_expr(ep)
	EXPR_PTR ep;
{
int *		ip;
int *		ip2;

	if( ep )
	{
		printf("( ");
		ip = (int *) ep->left;
		ip2 = (int *) ep->right;
		if( ip && *ip == EXPR_MAGIC )
			print_expr( (EXPR_PTR) ep->left );
		else if( ip && *ip == VAR_MAGIC )
			show_var( (VAR_PTR) ep->left );
		else if( ip && *ip == FCALL_MAGIC )
			print_fcall( (FCALL_PTR) ep->left );
		printf("%s ",OpStr(ep->op));
		if( ip2 && *ip2 == EXPR_MAGIC )
			print_expr( (EXPR_PTR) ep->right );
		else if( ip2 && *ip2 == VAR_MAGIC )
			show_var( (VAR_PTR) ep->right );
		else if( ip2 && *ip2 == FCALL_MAGIC )
			print_fcall( (FCALL_PTR) ep->right );
		printf(") ");
	}
	return 0;
}

int
show_var( vp )
	VAR_PTR vp;
{
	if( !vp || vp->magic != VAR_MAGIC )
	{
		printf("[bad var ptr: %ld] ",(long) vp);
		return 0;
	}
		
	if( strlen(vp->name) )
		printf("%s ",vp->name);
	else
	{
		switch( vp->type )
		{
		case T_int:
			printf("%d ",vp->val.ival);
			break;
		case T_float:
			printf("%f ",vp->val.fval);
			break;
		case T_string:
			printf("\"%s\" ",vp->val.sval);
			break;
		case T_word:
			printf("[illegal word: %s] ",vp->name);
			break;
		default:
			printf("[illegal var type: %d] ",vp->type);
			break;
		}
	}
	return 0;
}
