package rocs.geom;

import Jama.*;

/** Utilities to convert to and from [DX DY DTHETA] and proper
 * transformation matrices 
 **/
public class GeomUtil
{
    public static Matrix transformationMatrix33(Matrix T)
    {
	if (T.getColumnDimension()==3 && T.getRowDimension()==3)
	    return T.copy();

	assert(T.getRowDimension()==3 && T.getColumnDimension()==1);

	return transformationMatrix33(T.get(0,0), T.get(1,0), T.get(2,0));
    }

    public static Matrix transformationMatrix33(double dx, double dy, double dt)
    {
	Matrix P=new Matrix(3,3);

	P.set(0,0,Math.cos(dt));
	P.set(0,1,-Math.sin(dt));
	P.set(0,2,dx);
	P.set(1,0,Math.sin(dt));
	P.set(1,1,Math.cos(dt));
	P.set(1,2,dy);
	P.set(2,2,1);

	return P;
    }

    public static Matrix transformationMatrix31(Matrix P)
    {
	if (P.getColumnDimension()==1 && P.getRowDimension()==3)
	    return P.copy();

	assert(P.getRowDimension()==3 && P.getColumnDimension()==3);

	Matrix S=new Matrix(3,1);

	double dx=P.get(0,2);
	double dy=P.get(1,2);

	S.set(0,0,P.get(0,2)); // dx
	S.set(1,0,P.get(1,2)); // dy
	S.set(2,0,Math.atan2(P.get(1,0), P.get(0,0))); // dt

	return S;
    }

    public static Matrix transformationMultiply31(Matrix a, Matrix b)
    {
	Matrix Ta=transformationMatrix33(a);
	Matrix Tb=transformationMatrix33(b);

	Matrix T=Ta.times(Tb);

	return transformationMatrix31(T);
    }

    public static Matrix transformationMultiply33(Matrix a, Matrix b)
    {
	Matrix Ta=transformationMatrix33(a);
	Matrix Tb=transformationMatrix33(b);

	Matrix T=Ta.times(Tb);

	return transformationMatrix33(T);
    }


    public static Matrix transformationInverse31(Matrix a)
    {
	Matrix Ta=transformationMatrix33(a);

	return transformationMatrix31(a.inverse());
    }
}
