package maslab.orc;
 
import java.net.*;
import java.io.*;
import java.awt.image.*;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.util.*;
import java.util.concurrent.*;

import maslab.util.*;

/** Low-level access to the OrcBoard; when possible, use a wrapper
 * class. 
 **/
public class Orc
{
    public enum DrawMode {
	ERASE(0),
	DRAW(1),
	XOR(2);
	
	int val;

	DrawMode(int val)
	{
	    this.val=val;
	}

	public int getCode()
	{
	    return val;
	}
    }

    public enum LcdFont {
	TINY('A'),
	TINY_INV('a'),
	SMALL('B'),
	SMALL_INV('b'),
	MEDIUM('C'),
	MEDIUM_INV('c'),
	LARGE('D'),
	LARGE_INV('d');
	
	int val;

	LcdFont(int val)
	{
	    this.val=val;
	}

	public int getCode()
	{
	    return val;
	}
    }

    /** Pin modes, for use with pinModeSet **/
    public enum PinMode { 
	DIGITAL_IN("Dig In",0),
	DIGITAL_IN_PULLUP("Dig In (Pull Up)",1),
	DIGITAL_IN_PULLDOWN("Dig In (Pull Dn)",2),
	DIGITAL_OUT("Dig Out",3),
	DIGITAL_OUT_SLOW("Dig Out (Slow)",4),
	SERVO("Servo",5),
	PWM("PWM",10),
	SONAR_PING("Sonar Ping",6),
	SONAR_ECHO("Sonar Echo",7),
	ANALOG_IN("Analog In",8), 
	ANALOG_OUT("Analog Out",9),
	QUADPHASE("Quad Phase",11),
	/** Requires U2 firmware upgrade. **/
	QUADPHASEFAST("Quad Phase Fast",14),
	MONOPHASE("Mono Phase",12);
	
	private int val;
	private String s;

	PinMode(String s, int val)
	{
	    this.val=val;
	    this.s=s;
	}

	public int getCode()
	{
	    return val;
	}

	public String toString()
	{
	    return s;
	}
    };

    /** Motor Direction constants, for use with motorSet. **/
    public enum MotorDirection {
	FORW(1),
	BACK(2),
	DISABLED(0);
	
	private int code;

	MotorDirection(int code)
	{
	    this.code=code;
	}
	
	public int getCode()
	{
	    return code;
	}

	public int getSign()
	{
	    if (this==FORW)
		return 1;
	    else if (this==BACK)
		return -1;
	    return 0;
	}
    }

    protected Socket sock;
    protected InputStream ins;
    protected OutputStream outs;

    protected Logger log=new Logger(this);

    protected static final byte MASTER = (byte) 0x00;    
    protected static final byte SLAVE  = (byte) 0x40;    
    protected static final byte PAD    = (byte) 0x80;

    protected BlockingQueue<OrcCommandConnection> commandConnections;
    protected OrcAsyncConnection asyncConnection;

    static final int NUM_CMD_HANDLERS = 3;

    /** Bit field mask for the Pad's Stick button **/
    public static final int BUTTON_STICK=1;

    /** Bit field mask for the Pad's Menu button **/
    public static final int BUTTON_MENU=2;

    /** Bit field mask for the Pad's Stop button **/
    public static final int BUTTON_STOP=4;

    public static final int BUTTON_UP=8;
    public static final int BUTTON_DOWN=16;
    public static final int BUTTON_LEFT=32;
    public static final int BUTTON_RIGHT=64;

    String host;
    int    cmdport, asyncport;

    ArrayList<OrcUpdateListener> updateListeners=new ArrayList<OrcUpdateListener>();

    int padOldUpDown=-1, padOldLeftRight=-1;

    static class OrcPacketHolder
    {
	long      time;
	OrcPacket packet;
	long      maximumLatency=50; // in ms

	OrcPacketHolder()
	{
	    time=-1;
	    packet=null;
	}
    }

    OrcPacketHolder masterHolder = new OrcPacketHolder();
    OrcPacketHolder slaveHolder  = new OrcPacketHolder();
    OrcPacketHolder padHolder    = new OrcPacketHolder();

    NotifyThread notifyThread    = new NotifyThread();

    /** Create a new Orc object, connecting to localhost on the
     * default ports (7320, 7321) 
     **/
    public Orc() throws IOException
    {
	this("localhost", 7320, 7321);
    }

    public Orc(String host) throws IOException
    {
	this(host, 7320, 7321);
    }

    /** Create a new Orc object.
     *
     * @param host The host to connect to
     * @param cmdport The TCP port for command connections
     * @param asyncport The TCP port for async event connections
     **/
    public Orc(String host, int cmdport, int asyncport) throws IOException
    {
	this.host=host;
	this.cmdport=cmdport;
	this.asyncport=asyncport;

	commandConnections = new LinkedBlockingQueue<OrcCommandConnection>();

	refillHandlers();

	asyncConnection=new OrcAsyncConnection(this, host, asyncport);

	padHolder.maximumLatency=1000;

	notifyThread.start();
    }

    /** Reliably perform a transaction. This doesn't mean the
     * transaction was successful, only that a response was
     * successfully received.
     *
     * @param request The request packet
     * @return The response packet.
     **/
    protected OrcPacket doTransaction(OrcPacket request)
    {
	int naks=0;

	while(true)
	    {
		try {
		    OrcCommandConnection handler=commandConnections.take();
		    OrcPacket response=handler.doTransaction(request);

		    if (response==null)
			refillHandlers();
		    else
			commandConnections.add(handler);
		
		    if (response==null || response.buf.length<4 || response.buf[3]!=0)
			{
			    naks++;
			    if (naks>=10 && naks%100==0)
				{
				    log.warn("Repeated failures "+naks+" code="+response.buf[3]);
				    log.warn(" SENT: "+request.toString());
				    log.warn(" RECV: "+response.toString());
					     
				}
			    continue;
			}

		    if (false)
			{
			    System.out.print("SEND: ");
			    request.print();
			    System.out.print("RECV: ");
			    response.print();
			}

		    return response;
		} catch (InterruptedException ex) {
		    log.warn("Ack!",ex);
		} catch (IOException ex) {
		    log.warn("Disconnected");
		}
	    }
    }

    protected void refillHandlers()
    {
	while (commandConnections.size()<NUM_CMD_HANDLERS)
	    {
		try {
		    commandConnections.add(new OrcCommandConnection(this, host, cmdport));
		} catch (IOException ex) {
		    log.error("Reconnect failed");
		    try {
			Thread.sleep(1000);
		    } catch (InterruptedException iex) {
		    }
		}
	    }
    }

    protected MotorDirection codeToMotorDirection(int code)
    {
	switch (code)
	    {
	    case 0:
		return MotorDirection.DISABLED;
	    case 1:
		return MotorDirection.FORW;
	    case 2:
		return MotorDirection.BACK;
	    }
	log.error("Bogus motor direction received");
	return MotorDirection.DISABLED;
    }

    /** Set the maximum number of milliseconds that the cache will be
     * used for. Set to 0 to disable the cache. **/
    public void setCacheLifetime(int ms)
    {
	masterHolder.maximumLatency=ms;
	slaveHolder.maximumLatency=ms;
	padHolder.maximumLatency=ms;
    }

    /** Returns the motor's goal direction.
     * @param port [0,3]
     **/
    public MotorDirection motorGoalDirectionRead(int port)
    {
	assert(port>=0 && port<=3);

	int v=read8(slaveHolder, OrcPacket.SLAVE_MOTORDIR+port/2);
	if ((port&1)==0)
	    v=v&0x0f;
	else
	    v=(v>>4)&0x0f;

	return codeToMotorDirection(v&3);
    }

    /** Returns the motor's actual direction.
     * @param port [0,3]
     **/
    public MotorDirection motorActualDirectionRead(int port)
    {
	assert(port>=0 && port<=3);

	int v=read8(slaveHolder, OrcPacket.SLAVE_MOTORDIR+port/2);
	if ((port&1)==0)
	    v=v&0x0f;
	else
	    v=(v>>4)&0x0f;

	return codeToMotorDirection((v>>2)&3);
    }

    /** Read the motor's current PWM. **/
    public int motorActualRead(int port)
    {
	assert(port>=0 && port<=3);
	
	return (read8(slaveHolder, OrcPacket.SLAVE_MOTOR+1+port*3))
	    *motorActualDirectionRead(port).getSign();
    }

    /** Read the motor's current goal PWM **/
    public int motorGoalRead(int port)
    {
	assert(port>=0 && port<=3);
	
	return (read8(slaveHolder, OrcPacket.SLAVE_MOTOR+port*3))
	    *motorGoalDirectionRead(port).getSign();
    }

    protected double voltageToCurrent(double senseVoltage)
    {
	// V=IR, R=0.18 ohm
	return (senseVoltage/0.18);
    }

    /** Returns the current consumption for the motor.
     * @param port [0,3]
     * @return The current, in amps.
     **/
    public double motorCurrentRead(int port)
    {
	assert(port>=0 && port<=3);

	return voltageToCurrent(analogRead(20+port));
    }

    /** Returns the current consumption for the servo.
     * @param port [0,3]
     * @return The current, in amps.
     **/
    public double servoCurrentRead(int port)
    {
	// current supported only on ports 0 & 1.
	assert (port<2);

	return voltageToCurrent(analogRead(24+port));
    }

    // maps the user-numbered analog ports to the numbering used in the firmware
    // if returned value >=256, then the port is on the slave, realport-=256
    protected int analogPortToRealPort(int port)
    {
	assert(port>=0 && port<26);

	int realport=0;

	if (port<16)
	    {
		assert(port>=8 || port==0 || port==3);
		
		if (port>=8)
		    realport=port-8;
		else if (port==0)
		    realport=8;
		else if (port==3)
		    realport=9;

		return realport;
	    }
	else
	    {
		assert(port==16 || port==19 || port>=20);

		if (port==16)
		    realport=6;
		else if (port==19)
		    realport=7;
		else
		    realport=port-20;

		return realport+256;
	    }
    }

    /** ports 0-19 correspond to numbered ports. (Some of these can't
     * do analog in.)  20-25 correspond to curent-sense inputs on
     * motors0-3 and servos0-1
    **/
    public double analogRead(int port)
    {
	int realport=analogPortToRealPort(port);

	if (realport<256)
	    {
		double v=read16(masterHolder, 16+realport*2);
		return v*5.0/65535.0;
	    }
	else
	    {
		realport-=256;

		double v=read16(slaveHolder, 27+realport*2);
		return v*5.0/65535.0;
	    }
    }

    /** Configure the analog low-pass filter for a port.
     * @param port [0,19] that supports analog in.
     * @param val  [0,10]. The parameter of the low-pass filter, higher numbers
     * representing much stronger filtering.
     **/
    public void analogLPFWrite(int port, int val)
    {
	int realport=analogPortToRealPort(port);

	if (port<256)
	    {
		doTransaction(new OrcPacket(MASTER, 'F', realport, val));
	    }
	else
	    {
		realport-=256;

		doTransaction(new OrcPacket(SLAVE, 'F', realport, val));
	    }
    }

    /** Read the current low-pass filter value.
     * @param port [0,19] that supports analog in.
     **/
    public int analogLPFRead(int port)
    {
	int realport=analogPortToRealPort(port);

	if (port<256)
	    {
		int v=read8(masterHolder, 36+realport/2);
		if ((realport&1)==0)
		    return v&0x0f;
		return (v>>4)&0x0f;
	    }
	else
	    {
		realport-=256;

		int v=read8(slaveHolder, 43+realport/2);
		if ((realport&1)==0)
		    return v&0x0f;
		return (v>>4)&0x0f;
	    }

    }

    /** Read the DAC (digital-to-analog) NOT IMPLEMENTED.
     * @param port [12]
     * @return v [0,255]
     **/
    public int dacRead(int port)
    {
	assert (port==12);
	return read8(masterHolder, OrcPacket.MASTER_DAC);
    }

    /** Write to the DAC (digital-to-analog).
     * @param port [12]
     * @param v [0,255]
     **/
    public void dacWrite(int port, int v)
    {
	assert (port==12);
	doTransaction(new OrcPacket(MASTER, 'V', v));
    }

    /** Read the digital port.
     * @param port [0,19]
     **/
    public boolean digitalRead(int port)
    {
	assert(port>=0 && port<=19);

	if (port<16)
	    {
		return (read16(masterHolder, 6)&(1<<port))>0;
	    }
	else
	    {
		return (read8(slaveHolder, 6)&(1<<(port-16)))>0;
	    }
    }


    /** Write the digital port.
     * @param port [0,19]
     * @param v The value
     **/
    public void digitalWrite(int port, boolean v)
    {
	assert(port>=0 && port<=20);

	if (port<16)
	    doTransaction(new OrcPacket(MASTER, 'D', port, v ? 1 : 0));
	else
	    doTransaction(new OrcPacket(SLAVE, 'D', port-16, v ? 1 : 0));	    
    }

    /** Read the servo's current position.
     * @param port [0,3]
     * @return The PWM value, [0,65535]
     **/
    public int servoRead(int port)
    {
	assert(port>=0 && port<=3);

	return read16(masterHolder, 8+port*2);
    }

    /** Write the servo's position.
     * @param port [0,3]
     * @param v [0,65535] The PWM value.
     **/
    public void servoWrite(int port, int v)
    {
	assert(port>=0 && port<=3);
	assert(v>=0 && v<=65535);

	doTransaction(new OrcPacket(MASTER, 'S', port, v>>8, v&0x00ff));
    }

    /** Initiate a new ultrasound ping.
     * @param port The port, 4 or 6.
     **/
    public void ultrasoundPing(int port)
    {
	assert(port==4 || port==6);

	doTransaction(new OrcPacket(MASTER, 'R', port));
    }

    /** Determine the range for an ultrasound ping.
     * @param port The port, 5 or 7.
     * @return The range, assuming speed of sound of 33146 cm/s. If
     * the measurement is still in progress, returns zero.
     **/
    public double ultrasoundRange(int port)
    {
	assert(port==5 || port==7);
	
	int i=read16(masterHolder, 49+port-5);

	if (i==0xffff)
	    return 0;

	double seconds=i/1000000.0;
	double range=33146*seconds/2.0;

	return range;
    }

    /** Returns a continuously running 4bit counter of the number of
     *  pings on this port 
     **/
    public int ultrasoundPingCount(int port)
    {
	assert(port==4 || port==6);
	
	int i=read8(masterHolder, 53);
	if (port==4)
	    return (i&0x0f);
	else
	    return ((i>>4)&0x0f);

    }

    /** Configure a port. Check which modes are possible before
     * calling.
     * @param port The port, [0,19].
     * @param m The mode.
     **/
    public void pinModeWrite(int port, PinMode m)
    {
	assert(port>=0 && port<=19);

	if (port<16)
	    {
		doTransaction(new OrcPacket(MASTER, 'C', port, m.getCode()));
	    }
	else
	    {
		doTransaction(new OrcPacket(SLAVE, 'C', port-16, m.getCode()));
	    }

	log.output("Set port "+port+" to mode "+m);
    }

    /** Determine the new pin mode.
     * @param port The port, [0,19].
     **/
    public PinMode pinModeRead(int port)
    {
	assert(port>=0 && port<=19);
	int v;

	if (port<16)
	    {
		v=read8(masterHolder, 41+port/2);
		if (port%2==0)
		    v=v&0x0f;
		else
		    v=(v>>4)&0x0f;

	    }
	else
	    {
		v=read8(slaveHolder, 4+(port-16)/2);
		if (port%2==0)
		    v=v&0x0f;
		else
		    v=(v>>4)&0x0f;
	    }

	for (PinMode m : PinMode.values())
	    if (m.getCode()==v)
		return m;
	
	log.error("Unknown mode type "+v+"!");
	return null;

    }

    // source: 4=48MHz, 5=vc1, 6=vc2, 7=32k, 1=vc3
    /** NOT IMPLEMENTED. **/
    public void pwmWrite(int port, int source, int period, int dutycycle)
    {
	doTransaction(new OrcPacket(MASTER, 'K', 4, period, dutycycle));
    }

    /** NOT IMPLEMENTED. **/
    public int pwmSourceRead(int port)
    {
	return read8(masterHolder, OrcPacket.MASTER_CLOCK);
    }

    /** NOT IMPLEMENTED. **/
    public int pwmPeriodRead(int port)
    {
	return read8(masterHolder, OrcPacket.MASTER_CLOCK+1);
    }

    /** NOT IMPLEMENTED. **/
    public int pwmDutyCycleRead(int port)
    {
	return read8(masterHolder, OrcPacket.MASTER_CLOCK+2);
    }

    /** Returns the number of errors that have occured on the
     * quadphase port. A small number of errors can be expected, but a
     * large number of errors means the sensor is not working
     * properly or is counting too quickly. This value is meaningless
     * for monophase ports.
     * @param port The port, [16,19].
     **/
    public int quadphaseErrorsRead(int port)
    {
	assert(port>=16 && port<=19);

	int i;

	if (port==16 || port==17)
	    i=read8(slaveHolder, 9);
	else
	    i=read8(slaveHolder, 12);

	return i;

    }
    
    /** Read the quadphase/monophase counter.
     * @param port The port, [16,19].
     **/
    public int quadphaseRead(int port)
    {
	assert(port>=16 && port<=19);

	int i;

	if (port==16 || port==17)
	    i=read16(slaveHolder, 7);
	else
	    i=read16(slaveHolder, 10);

	return i;
    }

    /** Perform an emergency allStop. The Orc board will become
     * unresponsive after calling this function. 
     **/
    public void allStop()
    {
	doTransaction(new OrcPacket(MASTER, 'X'));
    }

    /** Set a motor's goal speed and direction.
     *
     * @param port [0,3]
     * @param pwm [0,255]. For directions other than disabled, low
     * pwms imply braking.
     * @param dir  The direction
     **/
    public void motorWrite(int port, int pwm, MotorDirection dir)
    {
	assert(port>=0 && port<=3);
	assert(pwm>=0 && pwm<=255);

	doTransaction(new OrcPacket(SLAVE, 'M', port, dir.getCode(), pwm));
    }

    /** Set a motor's goal speed and direction.
     *
     * @param port [0,3]
     * @param pwm [-255,255]. Negative PWMs correspond to backwards,
     * positive PWM to forwards. The motor is never disabled with this
     * API.
     **/
    public void motorWrite(int port, int pwm)
    {
	assert(port>=0 && port<=3);
	assert(pwm>=-255 && pwm<=255);

	if (pwm>=0)
	    motorWrite(port, pwm, MotorDirection.FORW);
	else
	    motorWrite(port, -pwm, MotorDirection.BACK);
	    
    }

    /** Set the rate at which motors can change the PWM. When motors
     * are written, it only changes the goal speed and direction. The
     * actual speed and direction change more gradually, according to
     * the slew rate. Larger slew rates mean that the PWM changes faster.
     *
     * @param port [0,3]
     * @param slew [1,255]. Motor speeds are updated at 64Hz, so the
     * worst case slewing interval is seconds=512/(64*slew).
     **/
    public void motorSlewWrite(int port, int slew)
    {
	assert(port>=0 && port<=3);
	assert(slew>=1 && slew<=255);

	doTransaction(new OrcPacket(SLAVE, 'W', port, slew));
    }

    /** Returns the current slew rate.
     * @param port [0,3]
     **/
    public int motorSlewRead(int port)
    {
	assert(port>=0 && port<=3);

	return(read8(slaveHolder, 15+port*3));
    }

    /** Wait for a button to be pressed and released, and return it.
     * @return the logical OR of all simultaneously pressed buttons.
     **/
    public int padButtonsGet()
    {
	int v=0;

	synchronized (padHolder)
	    {
		while (v==0)
		    {
			
			v=padButtonsPoll();
			if (v>0)
			    break;
			
			try {
			    padHolder.wait();
			} catch (InterruptedException ex) {
			}
		    }

		while (padButtonsPoll()!=0)
		    {
			try {
			    padHolder.wait();
			} catch (InterruptedException ex) {
			}
		    }			
	    }

	return v;
    }

    /** Test the current status of the buttons and return immediately.
     * @return the logical OR of all the buttons currently
     * depressed. The corresponding bit is 1 if the button is
     * depressed.
     **/
    public int padButtonsPoll()
    {
	int newUpDown=read8(padHolder, 7);
	int newLeftRight=read8(padHolder, 8);

	int v=read8(padHolder, 6);

	if (padOldUpDown!=-1)
	    {
		if ((newUpDown&0xf0)!=(padOldUpDown&0xf0))
		    v|=BUTTON_UP;
		if ((newUpDown&0x0f)!=(padOldUpDown&0x0f))
		    v|=BUTTON_DOWN;
		if ((newLeftRight&0xf0)!=(padOldLeftRight&0xf0))
		    v|=BUTTON_LEFT;
		if ((newLeftRight&0x0f)!=(padOldLeftRight&0x0f))
		    v|=BUTTON_RIGHT;
	    }

	padOldUpDown=newUpDown;
	padOldLeftRight=newLeftRight;

	return v;
    }

    /** Returns the Pad's current joystick position (X axis) **/
    public int joyX()
    {
	return(read8(padHolder, 4));
    }

    /** Returns the Pad's current joystick position (Y axis) **/
    public int joyY()
    {
	return(read8(padHolder, 5));
    }

    /** Clear the LCD **/
    public void lcdClear()
    {
	doTransaction(new OrcPacket(PAD, 'C'));
    }

    /** Move the console cursor position. **/
    public void lcdConsoleGoto(int x, int y)
    {
	doTransaction(new OrcPacket(PAD, 'G', x, y));
    }

    /** Write text to the console at the current cursor position. **/
    public void lcdConsoleWrite(String s)
    {
	int len=s.length();
	int pos=0;
	int maxlen=32;

	while (pos<len)
	    {
		int thislen=maxlen;
		if (thislen+pos>len)
		    thislen=len-pos;

		byte[] data=s.substring(pos, pos+thislen).getBytes();

		doTransaction(new OrcPacket(data, data.length, PAD, 'W'));

		pos+=thislen;
	    }
    }

    /** Fill the entire LCD with a pattern. See the orcmanual for
     * documentation on this function. **/
    public void lcdFill(int even, int odd)
    {
	doTransaction(new OrcPacket(PAD, 'F', even, odd));
    }

    /** Draw a line. **/
    public void lcdDrawLine(int x0, int y0, int x1, int y1)
    {
	doTransaction(new OrcPacket(PAD, 'L', x0, y0, x1, y1));
    }

    /** Clear the portion of the lcd being used by the console, and
	move the cursor to the top left position. **/
    public void lcdConsoleHome()
    {
	doTransaction(new OrcPacket(PAD, 'H'));
    }

    /** Set the console size and location. Note that the units for the
    * parameters are in characters, columns, and rows-- not
    * pixles. There are a total of eight rows, and approximately 32
    * columns usable for the console. In general, the top status bar consumes
    * the top line.
    *
    * @param top The top line of the console, [0,7]
    * @param height The number of lines in the console, [0,7].
    * @param left The leftmost column of the console [0,31]
    * @param width The number of columns in the console.
    **/
    public void lcdConsoleSettings(int top, int height, int left, int width)
    {
	doTransaction(new OrcPacket(PAD, 'S', top, height, left, width));
    }

    /** Set the draw mode. This command affects line and pixel drawing
     * commands, and is reset to DrawMode.DRAW by a screen clear.
     **/
    public void lcdDrawMode(DrawMode dm)
    {
	doTransaction(new OrcPacket(PAD, 'M', dm.getCode()));
    }

    /** Draw text at an arbitrary location.
     * @param font The font to draw in
     * @param x The x location (in pixels), [0, 127]
     * @param y The y location (in pixels), [0, 63]
     * @param s The string to write, no more than 32 characters long.
     **/
    public void lcdDrawText(LcdFont font, int x, int y, String s)
    {
	byte[] data=s.getBytes();
	int len=data.length;
	if (len>32)
	    len=32;
	doTransaction(new OrcPacket(data, len, PAD, 'T', x, y, font.getCode()));
    }

    /** Read the LCD's frame buffer. Reads a horizontal swath of
     * pixels, 8 pixels high.
     *
     * @param x The x offset to begin reading from.
     * @param y The y offset to begin reading from. Must be a multiple of 8.
     * @param length The number of bytes to read.
     * @param buf The buffer to deposit the data into
     * @param offset The offset in the buffer
     **/
    public void lcdRead(int x, int y, int length, byte[] buf, int offset)
    {
	OrcPacket resp=doTransaction(new OrcPacket(PAD, 'r', x, y, length));

	for (int i=0;i<length;i++)
	    buf[offset+i]=resp.buf[4+i];
    }

    /** Write to the LCD's frame buffer. Writes a horizontal swath of
     * pixels, 8 pixels high.
     *
     * @param x The x offset to begin writing from.
     * @param y The y offset to begin writing from. Must be a multiple of 8.
     * @param buf The buffer to get the data from
     * @param blocksize The number of bytes to use from the buffer
     **/
    public void lcdWrite(int x, int y, byte[] buf, int blocksize)
    {
	OrcPacket resp=doTransaction(new OrcPacket(buf, blocksize, PAD, 'R', x, y));
    }

    /** Read the entire LCD frame buffer. This function is fairly slow.
     * @return A 128x64 BufferedImage in TYPE_BYTE_GRAY format.
     **/
    public BufferedImage lcdRead()
    {
	BufferedImage bi=new BufferedImage(128,64,BufferedImage.TYPE_BYTE_GRAY);

	int xstride=32;
	int ystride=8;   // must be 8!

	byte[] buf=new byte[xstride];

	for (int y=0;y<64;y+=ystride)
	    {
		for (int x=0;x<128;x+=xstride)
		    {
			lcdRead(x,y,xstride, buf, 0);

			for (int xi=0;xi<xstride;xi++)
			    {
				int b=buf[xi];

				for (int yi=0;yi<8;yi++)
				    {
					bi.setRGB(x+xi, y+yi, (b&1)>0 ? 0x00000000 : 0xffffffff);
					b=b>>1;
				    }
			    }
		    }
	    }

	return bi;
    }

   /** Draw an image on the lcd.
     *
     * @param xi The top left x coordinate, [0,127]
     *
     * @param yi The top left y coordinate, [0,63] and a multiple of 8.
     *
     * @param i The image. For good results, the image should be
     * reduced to dithered black and white. This procedure uses only
     * the green channel to determine whether a pixel should be on or
     * off.
     **/
    public void lcdWrite(int xi, int yi, BufferedImage i)
    {
        assert(0<=xi && xi<=127);
        assert(0<=yi && yi<=63);

        int w=i.getWidth();
        int h=i.getHeight();

        int maxblocksize=32;
        byte[] data=new byte[maxblocksize];

        for (int y0=0;y0<h;y0+=8)
            {
                if ((yi+y0)>(64-8))
                    continue;

                for (int x0=0;x0<w;x0+=maxblocksize)
                    {
                        if ((xi+x0)>(128-maxblocksize))
                            continue;

                        int thisblocksize=maxblocksize;
			if (x0+thisblocksize>=w)
			    {
                                thisblocksize=w-x0;
                            }
			
                        // create a data block, 'thisblocksize' columns wide
                        for (int xt=0;xt<thisblocksize;xt++)
                            {
                                int acc=0;

                                // each column 8 pixels tall.
                                for (int yt=7;yt>=0;yt--)
                                    {
                                        int p=0;

                                        if ((x0+xt)<w && ((y0+yt)<h))
                                            p=i.getRGB(x0+xt,y0+yt);

                                        if ((p&0x00ff00)!=0)
                                            p=0;
                                        else
                                            p=1;

                                        acc=(acc<<1) | p;
                                    }

                                data[xt]=(byte) acc;
                            }

                        // send the data block
                        lcdWrite(xi+x0, yi+y0, data, thisblocksize);
                    }
	    }
    }

    /** Read the Orc board's internal clock, a 16 bit counter updating
     * at 512 Hz. 
     **/
    public long clockRead()
    {
	return(read16(masterHolder, OrcPacket.MASTER_TIMESTAMP));
    }

    /** 
     * Read the raw gyro integrator value. This is not calibrated, and not intended
     * for use by mere mortals.
     **/
    public long gyroReadRaw()
    {
	return(read32(masterHolder, OrcPacket.MASTER_GYRO));
    }


    /** Perform an I2C Read transaction REQUIRES FIRMWARE UPGRADE.
     * @param addr I2C Address, [0,127]
     * @param buf Buffer to deposit data into
     * @param len Length of read (<64)
     * @return true if read succeeds.
     **/
    public boolean i2cRead(int addr, byte[] buf, int len)
    {
	OrcPacket resp=doTransaction(new OrcPacket(MASTER, 'i', addr, len));

	if (resp.buf[3]!=0) // check result code
	    return false;

	// data begins at offset 4
	for (int i=0;i<len;i++)
	    buf[i]=resp.buf[4+i];
	
	return true;
    }

    /** Perform an I2C Write transaction REQUIRES FIRMWARE UPGRADE.
     * @param addr I2C Address, [0,127]
     * @param buf Buffer to data to write
     * @param len Length of write (<64)
     * @return true if write succeeds.
     **/
    public boolean i2cWrite(int addr, byte[] buf, int len)
    {
	OrcPacket req=new OrcPacket(buf, len, MASTER, 'I', addr);
	OrcPacket resp=doTransaction(req);

	return resp.buf[3]==0; // check result code
    }

    /** Perform a combined write/read I2C transaction UNTESTED. REQUIRES FIRMWARE UPGRADE.
     * @param addr I2C Address, [0,127]
     * @param writebuf The buffer which will be written first
     * @param writelen The length of the buffer (<64)
     * @param readbuf  The buffer in which to deposit read data
     * @param readlen  The number of bytes to read (<64)
     * @return true if the transaction succeeds.
     **/
    public boolean i2cWriteRead(int addr, byte[] writebuf, int writelen, 
				byte[] readbuf, int readlen)
    {
	OrcPacket req=new OrcPacket(writebuf, writelen, MASTER, 'J', addr, readlen);
	OrcPacket resp=doTransaction(req);

	if (resp.buf[3]!=0) // check result code
	    return false;

	// data begins at offset 4
	for (int i=0;i<readlen;i++)
	    readbuf[i]=resp.buf[4+i];
	
	return true;
    }

    /** Compute a-b, accounting for wraparound with 16 bit numbers. */
    public static int diff16(long a, long b)
    {
	long SIZE=0x10000;
	long MAX=SIZE-1;

	a&=MAX;
	b&=MAX;

	long d=(a-b);

	if (d<-SIZE/2)
	    return ((int) (SIZE+d));
	if (d>SIZE/2)
	    return -((int) (SIZE-d));
	return ((int) d);
    }

    /** Compute a-b, accounting for wraparound with 32 bit numbers. */
    public static long diff32(long a, long b)
    {
	long SIZE=0x100000000L;
	long MAX=SIZE-1;

	a&=MAX;
	b&=MAX;

	long d=(a-b);

	if (d<-SIZE/2)
	    return SIZE+d;
	if (d>SIZE/2)
	    return -(SIZE-d);
	return d;
    }

    //////////// INTERNAL FUNCTIONS ////////////////
    /** Wait until we've received a recent packet. **/
    protected void uptodate(OrcPacketHolder h)
    {
	synchronized(h)
	    {
		long time=System.currentTimeMillis();
		
		while(time-h.time>h.maximumLatency || h.packet==null || h.packet.buf==null)
		    {
			try {
			    h.wait();
			} catch (InterruptedException ex) {
			}
		    }
	    }
    }

    /** Read an 8bit field from a packet. **/
    protected int read8(OrcPacketHolder ph, int offset)
    {
	uptodate(ph);

	return ph.packet.buf[offset]&0x00ff;
    }

    /** Read a 16 bit field from a packet. **/
    protected int read16(OrcPacketHolder ph, int offset)
    {
	uptodate(ph);
	OrcPacket p=ph.packet;

	return ((p.buf[offset]&0x00ff)<<8)+(p.buf[offset+1]&0x00ff);
    }

    /** Read a 32 bit field from a packet. **/
    protected int read32(OrcPacketHolder ph, int offset)
    {
	uptodate(ph);
	OrcPacket p=ph.packet;

	return ((p.buf[offset]&0x00ff)<<24) +
	    ((p.buf[offset+1]&0x00ff)<<16) +
	    ((p.buf[offset+2]&0x00ff)<<8) +
	    ((p.buf[offset+3]&0x00ff));
    }

    /** Add an OrcUpdateListener. It will be notified whenever the Orc
     * publishes new data on a different thread.
     **/
    public void addUpdateListener(OrcUpdateListener ul)
    {
	synchronized (updateListeners)
	    {
		if (!updateListeners.contains(ul))
		    updateListeners.add(ul);
	    }
    }

    /** Remove an OrcUpdateListener.
     **/
    public void removeUpdateListener(OrcUpdateListener ul)
    {
	synchronized (updateListeners)
	    {
		if (updateListeners.contains(ul))
		    updateListeners.remove(ul);
	    }
    }


    /** Signal all the listeners. **/
    protected void notifyUpdateListeners()
    {
	notifyThread.update();
    }

    class NotifyThread extends Thread
    {
	public NotifyThread()
	{
	    setDaemon(true);
	}

	/** Call update() to cause all listeners to be reinvoked **/
	public synchronized void update()
	{
	    this.notify();
	}

	/** This is the thread that does all the notifying. **/
	public void run()
	{
	    while(true)
		{
		    synchronized(this)
			{
			    try {
				this.wait();
			    } catch (InterruptedException ex) {
			    }
			}

		    synchronized (updateListeners)
			{
			    for (OrcUpdateListener ul: updateListeners)
				ul.orcUpdated(Orc.this);
			}
		}
	}
    }

}
