package maslab.orc;

import maslab.util.*;

import java.io.*;

/** Orc wrapper for a servo. **/
public class Servo
{
    Orc orc;
    int port;

    public int minpwm, maxpwm;
    public double minangle, maxangle;

    public Servo(Orc orc, int port)
    {
	this.orc=orc;
	this.port=port;

	orc.pinModeWrite(port, Orc.PinMode.SERVO);
    }

    /** Create a servo using default parameters of an MPI-MX400,
     * seeking from 0 to 180 degrees. **/
    public static Servo makeMpiMX400(Orc orc, int port)
    {
	Servo s=new Servo(orc, port);
	s.minpwm=5000; s.maxpwm=18000;
	s.minangle=0; s.maxangle=180;

	return s;
    }

    /** Create a servo using default parameters of an Hitec HS300
     * seeking from 0 to 180 degrees. **/
    public static Servo makeHitecHS300(Orc orc, int port)
    {
	Servo s=new Servo(orc, port);
	s.minpwm=7000; s.maxpwm=20000;
	s.minangle=0; s.maxangle=180;

	return s;
    }

    /** Seek the servo.
     * @param angle The angle to seek to. 
     **/
    public void seek(double angle)
    {
	int pwm=(int) (minpwm+(maxpwm-minpwm)*(angle-minangle)/maxangle);
	
	orc.servoWrite(port, pwm);
    }

    /** Determine the servo's current command.
     * @return The angle the servo is attempting to seek to.
     **/
    public double tell()
    {
	int pwm=orc.servoRead(port);

	return minangle+(maxangle-minangle)*(pwm-minpwm)/(maxpwm-minpwm);
    }

    /** Measure the servo's current consumption. 
     **/
    public double current()
    {
	return orc.servoCurrentRead(port);
    }

    /** A very simple little servo calibration program. **/
    public static void main(String[] args)
    {
	Orc orc;
	
	try {
	    orc=new Orc();
	} catch (Exception ex) {
	    System.out.println("Couldn't connect to orc.");
	    return;
	}

	Servo s=new Servo(orc, 0);
	
	System.out.println("Assuming there's a servo on port 0...");

	BufferedReader ins=new BufferedReader(new InputStreamReader(System.in));

	while(true)
	    {
		String line="";

		try {
		    System.out.print("Set pwm to: ");
		    line=ins.readLine();

		    int pwm=Integer.parseInt(line);
		    orc.servoWrite(0, pwm);
		} catch (Exception ex) {
		    System.out.println("Huh?");
		}


		System.out.println("Seeked to: "+orc.servoRead(0));
		System.out.println("Current: "+orc.servoCurrentRead(0));
	    }
    }
}
