package maslab.telemetry;

import java.io.*;
import java.util.*;
import java.net.*;

import maslab.telemetry.*;
import maslab.util.*;

/** Manages sending and receiving packets via a hub for
 * inter-process/inter-robot communication.
 **/
public class JugClient
{
    // for folks who are interested in when we connect/disconnect
    // from a hub.
    ArrayList<StatusListener> statusListeners=new ArrayList<StatusListener>();
    boolean connectionState = false;

    MTQueue<JugPacket> outqueue=new MTQueue<JugPacket>();

    // String -> Channel
    // we use this to keep track of where to send messages and also
    // as our list of channels that exist.
    HashMap<String,Channel> channels=new HashMap<String,Channel>();

    // (String channelname)
    LinkedList<String> advertisements=new LinkedList<String>();

    protected Logger log=new Logger("JugClient");

    MonitorThread monitorthread;

    class Channel
    {
	// the name of this channel
	String name; 
	
	// the list of local listeners (JugMessageListener)
	LinkedList<JugMessageListener> listeners; 

	// the number of JugClients subscribed to this channel,
	// not to be confused with listeners.size();
	int numsubscribers; 

	// how many bytes are queued to go up?
	int queueBytes;
	
	// how many queued up bytes before we stop enqueing new messages?
	// -1 for unlimited
	int queueLimit=-1;

	// does the hub tell us that this channel is currently advertised?
	boolean advertised=false;

	public Channel(String name)
	{
	    this.name=name;
	    listeners=new LinkedList<JugMessageListener>();
	    numsubscribers=0;
	}
    }

    /** Create a new client and connect to the hub on localhost using
     *  the default TCP port 
     **/
    public JugClient() 
    {
	try {
	    // we use getByName(127.0.0.1) instead of getLocalHost
	    // for people who have bogus hostnames configured.
	    monitorthread=new MonitorThread(InetAddress.getByName("127.0.0.1"),
					    JugHub.DEFAULTPORT);
	    monitorthread.setDaemon(true);
	    monitorthread.start();
	} catch (UnknownHostException ex) {
	    log.error("Can't connect to localhost: "+ex);
	}

    }

    /** Create a new client and connect to the hub on the specified
     * host and port.
     **/
    public JugClient(InetAddress addr, int port) throws IOException
    {
	monitorthread=new MonitorThread(addr, port);
	monitorthread.setDaemon(true);
	monitorthread.start();
    }

    public synchronized void reconnect(InetAddress addr, int port) throws IOException
    {
	monitorthread.threadquit=true;
	if (monitorthread.sock != null)
	    monitorthread.sock.close();
	try {
	    monitorthread.join();
	} catch (InterruptedException ex) {
	}

	monitorthread=new MonitorThread(addr, port);
	monitorthread.setDaemon(true);
	monitorthread.start();
    }


    /** Add a StatusListener who will be notified of
     * connection/disconnection events and new channel events.
     **/    
    public void addStatusListener(StatusListener listener)
    {
	synchronized(statusListeners)
	    {
		statusListeners.add(listener);
	    }

	// Send a message to the new listener with our current connection state
	StatusEvent e = new StatusEvent(this);
	e.name = "connected";
	e.booleanValue = connectionState;
	listener.statusChanged(e);
    }

    /** Remove a StatusListener
     **/
    public void removeStatusListener(StatusListener listener) {
	synchronized(statusListeners)
	    {
		statusListeners.remove(listener); 
	    }
    }


    /** Get a list of all channels which are currently advertised.
     * @return ArrayList of String
     **/
    public ArrayList<String> getChannels()
    {
	ArrayList<String> al=new ArrayList<String>();

	synchronized(channels)
	    {
		Iterator i=channels.keySet().iterator();
		
		int idx=0;
		while (i.hasNext())
		    {
			Channel c=getChannel((String) i.next());
			if (c.advertised)
			    al.add(c.name);
		    }
	    }
	return al;
    }

    Channel getChannel(String cname)
    {
	Channel c=(Channel) channels.get(cname);
	if (c==null)
	    {
		c=new Channel(cname);
		channels.put(cname, c);
	    }
	return c;
    }

    /** Subscribe to a channel.
     * @param cname The name of the channel.
     * @param listener The listener to be notified of received data.
     **/
    public synchronized void subscribe(String cname, JugMessageListener listener)
    {
	Channel c=getChannel(cname);
	synchronized(c.listeners)
	    {
		c.listeners.add(listener);

		// send the subscribe message if this is our first listener
		// on this channel
		if (c.listeners.size()==1)
		    {
			JugPacket p=new JugPacket(JugPacket.COMMAND_SUB, cname);
			outqueue.put(p);
		    }
	    }
    }

    /** Unsubscribe to a channel.
     * @param cname The name of the channel.
     * @param listener The listener that was subscribed
     **/
    public synchronized void unsubscribe(String cname, JugMessageListener listener)
    {
	Channel c=getChannel(cname);
	synchronized(c.listeners)
	    {
		c.listeners.remove(listener);

		// if there are no more listeners on this channel, unsubscribe.
		if (c.listeners.size()==0)
		    {
			JugPacket p=new JugPacket(JugPacket.COMMAND_UNSUB, cname);
			outqueue.put(p);
		    }
	    }
    }

    /** Advertise the existence of a new channel.
     * @param channel The channel name that you will be publishing
     * on. 
     **/
    public synchronized void advertise(String channel)
    {
	JugPacket p=new JugPacket(JugPacket.COMMAND_ADV,channel);

	synchronized(advertisements)
	    {
		if (!advertisements.contains(channel))
		    {
			advertisements.add(channel);
			outqueue.put(p);
		    }
	    }
    }

    /** Unadvertise a channel.
     * @param channel The channel name that you will no longer be
     * publishing on
     **/
    public synchronized void unadvertise(String channel)
    {
	JugPacket p=new JugPacket(JugPacket.COMMAND_UNADV, channel);

	synchronized(advertisements)
	    {
		if (advertisements.contains(channel))
		    {
			advertisements.remove(channel);
			outqueue.put(p);
		    }
	    }
    }

    /** Publish data on a previously advertised channel.
     * @param channel The channel name.
     * @param data The data (opaque) to be sent. The data is NOT
     * copied and thus should not be modified.
     **/
    public void publish(String channel, byte[] data)
    {
	if (!monitorthread.isConnected)
	    return;

	Channel c=getChannel(channel);

	synchronized(c)
	    {
		if (c.numsubscribers>0)
		    {
			//			if (c.queueLimit>=0 && c.queueBytes>c.queueLimit)
			//			    return;
			
			JugPacket p=new JugPacket(JugPacket.COMMAND_MSG, channel, data);
			
			if (JugHub.localJugHub!=null) 
			    {
				JugHub.localJugHub.directReceive(p);
				return;
			    }
			else
			    {
				// send via TCP
				//				c.queueBytes+=p.size();
				outqueue.put(p);
			    }
		    }
	    }
    }

    /** Limit the number of messages that can be queued up on a particular channel.
     * @param channel The channel name
     * @param size The maximum number of bytes before new published
     * messages are dropped. If the size is less than zero, no limit
     * is enforced.
     **/
    public void setMaximumQueueSize(String channel, int size)
    {
	Channel c=getChannel(channel);

	c.queueLimit=size;
    }

    /** As a debugging aid, dump the sizes of each queue to stdout. 
     **/
    public void displayQueueSizes()
    {
	synchronized(channels)
	    {
		Iterator i=channels.values().iterator();
		
		while (i.hasNext())
		    {
			Channel c=(Channel) i.next();
			
			System.out.println(c.name+"  "+c.queueBytes+"  "+c.queueLimit);
		    }
	    }
    }
    
    /** Determine if a message would actually be sent on a channel. This function
     * can be used as an optimization by calling it before actually constructing
     * an expensive packet and calling publish.
     * @param channel The channel name.
     * @return true if the message would be sent.
     **/
    public boolean publishing(String channel)
    {
	if (!monitorthread.isConnected)
	    return false;

	Channel c=getChannel(channel);
	return c.numsubscribers>0;
    }

    /** Called internally whenever we receive a packet from the hub.
     **/
    protected void receivedPacket(JugPacket p)
    {
	DataInputStream dins=null;
	Channel c=null;
			    
	switch(p.command)
	    {
	    case JugPacket.NOTICE_ADV:
		// this will create the channel
		c=getChannel(p.channelName);
		c.advertised=true;
		notifyListeners("newchannel",true);
		break;

	    case JugPacket.NOTICE_UNADV:
		c=getChannel(p.channelName);
		c.advertised=false;
		notifyListeners("newchannel",true);
		break;

	    case JugPacket.NOTICE_MSG:
		receivedMessage(p.channelName, p.data);
		break;

	    case JugPacket.NOTICE_SUBCOUNT:
		try {
		    dins=p.getDataInputStream();
		    c=getChannel(p.channelName);
		    c.numsubscribers=dins.readInt();
		} catch (Exception ex) {
		    log.warn("unexpected exception",ex);
		}
		break;

	    default:
		log.warn("unknown packet type "+p.command);
		break;
	    }

    }

    /** Called internally whenever a message is received from the hub. This function
     * calls the listeners.
     **/
    protected void receivedMessage(String channel, byte[] data)
    {
	Channel c=getChannel(channel);
	synchronized(c.listeners)
	    {
		if (c.listeners.size()==0)
		    {
			log.warn("got a message for channel "+channel+" but there are no subscribers.");
			return;
		    }

		Iterator i=c.listeners.iterator();
		while (i.hasNext())
		    {
			JugMessageListener ml=(JugMessageListener) i.next();
			ml.messageReceived(channel, data);
		    }
	    }
    }
 
    /** Notify all StatusListeners that our connection state has
     * changed
     **/
    protected void notifyListeners(String name, boolean value)
    {
	StatusEvent e = new StatusEvent(this);
	e.name = name;
	e.booleanValue = value;

	synchronized(statusListeners)
	    {
		Iterator l = statusListeners.iterator();
		while (l.hasNext())
		    {
			StatusListener sl = (StatusListener)l.next();
			sl.statusChanged(e);
		    }
	    }

    }


    /** The monitor thread is responsible for maintaining a connection to the Hub,
     * including reconnecting when connectivity is lost.
     **/
    protected class MonitorThread extends Thread
    {
	InetAddress hubaddr;
	int hubport;
	Socket sock;
	boolean threadquit=false;

	JugPacket EXIT=new JugPacket(0);

	boolean isConnected=false;

	public MonitorThread(InetAddress hubaddr, int hubport)
	{
	    this.hubaddr=hubaddr;
	    this.hubport=hubport;

	    setDaemon(true);
	}
	
	public void run()
	{
	    while(!threadquit)
		{
		    ReaderThread reader=null;
		    WriterThread writer=null;
		    

		    try {
			sock=new Socket(hubaddr, hubport);
			writer=new WriterThread();
			writer.setDaemon(true);
			writer.start();
			reader=new ReaderThread();
			reader.setDaemon(true);
			reader.start();

			// if we got this far w/o an exception, we're connected.
			isConnected=true;

			synchronized(channels)
			    {
				// resubscribe to all of our channels
				Iterator i=channels.values().iterator();
				while (i.hasNext())
				    {
					Channel c=(Channel) i.next();
					synchronized(c.listeners)
					    {
						if (c.listeners.size()>0)
						    {
							JugPacket p=new JugPacket(JugPacket.COMMAND_SUB, c.name);
							outqueue.put(p);
						    }
					    }
				    }
			    }

			// readvertise all of our channels
			synchronized(advertisements)
			    {
				Iterator i=advertisements.iterator();
				while (i.hasNext())
				    {
					String cname=(String) i.next();
					JugPacket p=new JugPacket(JugPacket.COMMAND_ADV, cname);
					outqueue.put(p);
				    }
			    }
			
			// Send a StatusEvent to indicate reconnection
			notifyListeners("connected",true);
			connectionState = true;

		    } catch (IOException ex) {
		    }
		    
		    try {
			if (writer!=null)
			    writer.join();
			if (reader!=null)
			    reader.join();
		    } catch (InterruptedException ex) {
			log.warn("unable to join threads",ex);
		    }

		    isConnected=false;

		    // Only send connection events if state has actually changed
		    if (connectionState)
			{
			    notifyListeners("connected",false);
			    log.verbose("lost connection to hub");
			    connectionState = false;
			}

		    //Mark all channels as unadvertised
		    Iterator i=channels.values().iterator();
		    while (i.hasNext())
			{
			    Channel c=(Channel) i.next();
			    c.advertised = false;
			}
		    
		    try {
			Thread.sleep(1000);
		    } catch (InterruptedException ex) {
		    }
		}
	}
	    
	/** Writes objects in the outqueue to the socket. **/
	protected class WriterThread extends Thread
	{
	    DataOutputStream outs;
	    
	    public WriterThread() throws IOException
	    {
		setDaemon(true);
		outs=new DataOutputStream(new BufferedOutputStream(sock.getOutputStream()));
	    }
	
	    public void run() 
	    {
		while(true)
		    {
			Object o=outqueue.getBlock();
		    
			if (o==EXIT)
			    break;
		    
			try {
			    if (o instanceof byte[])
				outs.write((byte[]) o);
			    else if (o instanceof String)
				outs.writeBytes((String) o);
			    else if (o instanceof JugPacket)
				{
				    JugPacket p=(JugPacket) o;
				    p.write(outs);			
				    
				    if (p.command==JugPacket.COMMAND_MSG)
					{
					    Channel c=getChannel(p.channelName);
					    //					    c.queueBytes-=p.size();
					}
				    
				}
			    else
				log.error("Unknown type "+o.getClass().getName());

			    log.vverbose("wrote packet");
			} catch (Exception ex) {
			    // assume connection closed.
			    break;
			}
		    }
	    }

	}

	/** Processes data from the hub. **/
	protected class ReaderThread extends Thread
	{
	    DataInputStream ins;

	    public ReaderThread() throws IOException
	    {
		setDaemon(true);
		ins=new DataInputStream(new BufferedInputStream(sock.getInputStream()));
	    }
	
	    public void run()
	    {
		while(true)
		    {
			try {
			    JugPacket p=new JugPacket(ins);

			    receivedPacket(p);
			} catch (IOException ex) {
			    log.log(Logger.VERBOSE,"lost connection",ex);
			    break;
			} catch (Exception ex) {
			    log.log(Logger.WARN,"Unexpected exception " + ex,ex);
			    ex.printStackTrace(System.out);
			}		    
		    }
		
		// this will kill the writer thread.
		outqueue.put(EXIT);
	    }
	}
	
    }

}
