package maslab.telemetry.botclient;

import maslab.telemetry.*;
import maslab.util.*;

import javax.swing.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.awt.*;
import java.util.*;
import java.net.*;
import java.io.*;

// A note on semantics: Plugin describes a class for handling data.  A
// handler is an instance of a plugin.


/** The main class for the BotClient application, used for viewing
 * robot data remotely. **/
public class BotClient extends JFrame implements ActionListener
{
    public static final long serialVersionUID=1001;

    public static String version = "4.0";

    static Logger log = new Logger("BotClient");

    public static JDesktopPane aDesktop;

    JugClient jc;
    PluginController pc;

    JDesktopPane desktop;
    JLabel statusBar;


    public static void main(String args[])
    {
	Logger.consolePublisher.setGlobalLevel(Logger.OUTPUT);
	new BotClient();
    }


    public BotClient()
    {
	setSize(800,600);
	setTitle("BotClient");
	getContentPane().setLayout(new BorderLayout());
	createMenus();
	desktop = new JDesktopPane();
	desktop.setBackground(Color.blue);
	aDesktop = desktop;	// Make this desktop available statically
	getContentPane().add(desktop,BorderLayout.CENTER);
	statusBar = new JLabel("BotClient version " + version);
	getContentPane().add(statusBar,BorderLayout.SOUTH);
	setVisible(true);
 	addWindowListener(new WindowAdapter(){public void windowClosing(WindowEvent e){System.exit(0);}});

	jc = new JugClient();
	jc.advertise("USERCMD$hidden");

	pc = new PluginController();

	ControlPanel cp = new ControlPanel(this);
	jc.addStatusListener(cp);
	desktop.add(cp);
    }

    public void actionPerformed(ActionEvent e)
    {
	String s = e.getActionCommand();

	if (s == "Exit")
	    System.exit(0);

	if (s == "New Channel View")
	    new ManualAdd(this);

    }


    void createMenus()
    {
	JMenuBar mb = new JMenuBar();
	setJMenuBar(mb);

	//----------File menu
	JMenu m = new JMenu("File");
	m.setMnemonic(KeyEvent.VK_F);
	mb.add(m);
	

	//Exit
	JMenuItem i = new JMenuItem("Exit",KeyEvent.VK_X);
 	m.add(i);
	i.addActionListener(this);

	//--------- View menu
	m = new JMenu("View");
	m.setMnemonic(KeyEvent.VK_V);
	mb.add(m);

	//New Channel View
	i = new JMenuItem("New Channel View",KeyEvent.VK_N);
	m.add(i);
	i.addActionListener(this);
	
    }


    /** Set the address of the robot.  **/
    public void setIP(InetAddress ip)
    {
	try{
	    jc.reconnect(ip,JugHub.DEFAULTPORT);
	} catch(IOException ex){
	    log.log(Logger.ERROR,"Error reconnecting to " + ip,ex);
	}
	    
    }

    /** Send a command to the robot.  
	@param cmd contains both the command and any arguments **/
    public void userCommand(String cmd)
    {
	byte[] data = cmd.getBytes();
	jc.publish("USERCMD$hidden",data);
    }

    /** Get a list of available channels. 
     @return An ArrayList of String objects containing the channel
     names, in name$type format. **/
    public ArrayList getChannels()
    {
	return jc.getChannels();
    }

    /** Get a list of plugins that can handle a given type. 
	@param type The name of a channel type, such as "text" or
	"jpegimage". 
	@return Array of fully qualified class names for the
	plugins. **/
    String[] getPlugins(String type) { return
    pc.getPlugins(type);

    }

    /** Get a list of all the channel types for which we have plugins. 
     @return Array of type names. **/
    String[] getTypes()
    {
	return pc.getTypes();
    }

    /** Create a new default handler.  Try to instantiate a new plugin
     * for the given channel, using the default plugin for the given
     * type.
     @param channel The channel name (without the "$type" suffix).
     @param type The channel type. **/
    public void newDefaultHandler(String channel, String type)
    {
	String pluginName = pc.getDefaultPlugin(type);
	if (pluginName == null)
	    return;
	newHandler(channel,type,pluginName);
    }

    /** Create a new handler.  Tries to instantiate a new plugin of
	type pluginName to handle the given channel.
	@param channel The channel name (without the "$type" suffix).
	@param type The channel type. 
	@param pluginName The name of the plugin to use.  Can be a
	fully qualified class name, or if the class is within
	maslab.telemetry.botclient, it may be the short name (for
	example, "TextPlugin"). **/
    public void newHandler(String channel, String type, String pluginName)
    {
	JInternalFrame frame = new JInternalFrame();
	Plugin p = pc.newHandler(pluginName,frame);

	if (p == null)
	    return;

	Watchdog w = new Watchdog(p);
	frame.addInternalFrameListener(w);
	desktop.add(frame);

	frame.setLocation(desktop.getWidth()/2 - frame.getWidth()/2,desktop.getHeight()/2 - frame.getHeight()/2);

	if (!p.addChannel(channel,type))
	    {
		JOptionPane.showMessageDialog(desktop,pluginName + " is unable to handle channel" + channel + " of type " + type);
		return;
	    }

	w.channels.add(channel+"$"+type);
	jc.subscribe(channel+"$"+type,p);

	try{
	    StatusListener listen = (StatusListener)p;
	    jc.addStatusListener(listen);
	} catch(ClassCastException cce){}

    }
    
    /** Listens for when a plugin closes, and handles unsubscriptions
     * for the plugin's channels.**/
    class Watchdog extends InternalFrameAdapter
    {
	Plugin plugin;
	public HashSet<String> channels; // of the form Name$Type

	public Watchdog(Plugin plugin)
	{
	    this.plugin = plugin;
	    channels = new HashSet<String>();
	}

	public void internalFrameClosing(InternalFrameEvent e)
	{
	    Iterator i = channels.iterator();
	    while (i.hasNext())
		jc.unsubscribe((String)i.next(),plugin);

	    try{
		StatusListener listen = (StatusListener)plugin;
		jc.removeStatusListener(listen);
	    } catch(ClassCastException cce){}
	}

    }
    

}

