package maslab.util;

import java.awt.image.*;
import java.awt.*;
import java.awt.color.ColorSpace;

/** Useful image utilities. **/
public class ImageUtil
{
    static Logger log=new Logger("ImageUtil");

    /**
     * Convert an image to a specified type.
     * @param in The input image, in any format.
     * @param type The desired type, e.g. BufferedImage.TYPE_3BYTE_BGR
     * @return An image with analagous content as in, but of the
     * requested type. Or, if the input image was already in the
     * requested format, the input image is returned.
     **/
    public static BufferedImage convertImage(BufferedImage in, int type)
    {
        if (in.getType()==type)
            return in;

        log.warn("Performing slow image type conversion");

        int w = in.getWidth();
        int h = in.getHeight();

        BufferedImage out=new BufferedImage(w,h,type);

        for (int y=0; y<h; y++)
            for (int x=0; x<w; x++)
                out.setRGB(x,y,in.getRGB(x,y));

        return out;
    }


    /** Convert an arbitrary input image into a black and white
     * dithered image (using the Floyd-Steinberg dithering algorithm
     * with auto thresholding) suitable for display on the lcd
     * panel. You should perform scaling operations before calling
     * this function.
     *
     * @param in The image to convert.
     * @return The dithered black and white image.
     **/
    public static BufferedImage ditherImage(BufferedImage in)
    {

        BufferedImage out=new BufferedImage(in.getWidth(),
                                            in.getHeight(),
                                            BufferedImage.TYPE_INT_RGB);

        int width=in.getWidth();
        int height=in.getHeight();

        int[] d=in.getRGB(0,0,width,height,null,0, width);
        int[] dout=new int[width*height];

        int p=0;
        int sum=0;

        // convert the picture to gray.
        for (int y=0;y<height;y++)
            {
                for (int x=0;x<width;x++,p++)
                    {
                        int c=d[p];

                        int gray=(int) (.299*((c&0x00ff0000)>>16) +
					.587*((c&0x0000ff00)>>8) +
                                        .114*((c&0x000000ff)));

                        sum+=gray;
                        d[p]=gray;
                    }
            }

        // estimate a good threshold.
        int thresh=sum/(width*height)/2;
        //      thresh=128;

        p=0;
        for (int y=0;y<height;y++)
            {
                for (int x=0;x<width;x++,p++)
                    {
                        int gray=d[p];
                        int bw=0;
                        if (gray>=thresh)
                            bw=255;

                        dout[p]=bw | (bw<<8) | (bw<<16) | 0xff000000;

                        int err=gray-bw;
                        if (x<(width-1))
                            d[p+1]+=err*7/16;

                        if (x>0 && y<(height-1))
                            d[p-1+width]+=err*3/16;

                        if (y<(height-1))
                            d[p+width]+=err*5/16;
			if (x<(width-1) && y<(height-1))
                            d[p+width+1]+=err*1/16;
                    }
            }

        out.setRGB(0,0,width,height,dout,0,width);

        return out;
    }

   /**
       Downsample an image (obsolete; do not use)
       @param in The image to downsample.  It will not be modified.
       @param interval The sampling interval.  If internval=n,The
       output image will contain every nth pixel from the original
       image, both vertically and horizontally.  So the number of
       pixels is reduced by n^2.
       @return The downsampled image.
    */
    public static BufferedImage downSample(BufferedImage in, int interval)
    {
        int w = in.getWidth();
        int h = in.getHeight();
        int sw = w/interval;
        int sh = h/interval;

	in = convertImage(in, BufferedImage.TYPE_INT_RGB);

        BufferedImage out = new BufferedImage(sw,sh,BufferedImage.TYPE_INT_RGB);

	int[] inbuf = ((DataBufferInt) in.getRaster().getDataBuffer()).getData();
	int[] outbuf = ((DataBufferInt) out.getRaster().getDataBuffer()).getData();

	int pos=0; // offset in inbuf
	int spos=0; // offset in outbuf

	for (int y=0, sy=0; sy<sh; sy++, y+=interval, pos+=w)
	    {
		for (int x=0, sx=0; sx<sw; sx++, x+=interval, spos++)
		    {
			outbuf[spos]=inbuf[pos+x];
		    }
	    }

        return out;
    }


    /**
     * Clone an image. Not particularly fast.
     * @param in The image to be cloned.
     * @return An identical image.
     **/
    public static BufferedImage cloneImage(BufferedImage in)
    {
        int type=in.getType();
        int w = in.getWidth();
        int h = in.getHeight();

        BufferedImage out=new BufferedImage(w,h,type);

        for (int y=0; y<h; y++)
            for (int x=0; x<w; x++)
                out.setRGB(x,y,in.getRGB(x,y));

	return out;
    }

}
