package maslab.util;

import java.util.*;

/** The base class of all LoggerPublishers. **/
public class LoggerPublisher
{
    int globalLevelThreshold=Logger.ERROR;
    HashMap<String, Integer> thresholds=new HashMap<String, Integer>();

    boolean reentry=false;

    /** Called when new data is published by any channel.
     * @param name The channel name
     * @param level The level
     * @param message The message to be displayed.
     **/
    public synchronized void publish(String name, int level, String message)
    {
	// Avoid infinite loops resulting by error conditions by disallowing
	// error messages that WE might generate.
	if (reentry)
	    return;

	reentry=true;

	if (acceptingMessage(name, level))
	    {
		publishUnconditional(name, level, message);
	    }

	reentry=false;
    }

    
    public void publishUnconditional(String name, int level, String message)
    {
	long nowTime=System.currentTimeMillis();
	double elapsedTime=(nowTime-Logger.startTime)/1000.0;
	String levelName=Logger.levelToString(level);
	
	System.out.println(StringUtil.padRight(name,20,' ') + 
			   StringUtil.padLeft(StringUtil.formatDouble(elapsedTime,3),10,' ') + " " +
			   StringUtil.padRight(levelName,10,' ') +
			   message);
    }

    /** Set a new default threshold for publishing. This level
     * will be applied to any channel which hasn't been given
     * a level specifically.
     *
     * @param threshold One of the static levels provided by the
     * Logger class.
     **/
    public void setGlobalLevel(int threshold)
    {
	this.globalLevelThreshold=threshold;
    }

    /** Set a per-channel threshold for publishing. This overrides
     * the global threshold.
     *
     * @param channel The channel name
     * @param threshold One of the static levels provided by the
     * Logger class.
     **/
    public void setLevel(String channel, int threshold)
    {
	thresholds.put(channel, new Integer(threshold));
    }

    /** Tests our internal log message acceptance criteria, returning
     * true if the message should be actually published.
     **/
    protected boolean acceptingMessage(String name, int level)
    {
	boolean accepted=false;

	Integer i=thresholds.get(name);
	if (i!=null)
	    {
		if (level<=i.intValue())
		    accepted=true;
	    }
	else if (level<=globalLevelThreshold)
	    accepted=true;

	return accepted;
    }
}
