ROUTINE(interp,		bivariate interpolation of irregulrly spaced x-y data)
subroutine  idbvip(md,ncp,ndp,xd,yd,zd,nip,xi,yi,zi,iwk,wk)
dimension   xd(100),yd(100),zd(100),xi(1000),yi(1000),zi(1000),iwk(4100),wk(800)
# this subroutine performs bivariate interpolation when the pro-
# jections of the data points in the x-y plane are irregularly
# distributed in the plane.

# the input parameters are
#     md  = mode of computation (must be 1, 2, or 3),
#         = 1 for new ncp and/or new xd-yd,
#         = 2 for old ncp, old xd-yd, new xi-yi,
#         = 3 for old ncp, old xd-yd, old xi-yi,
#     ncp = number of additional data points used for esti-
#           mating partial derivatives at each data point
#           (must be 2 or greater, but smaller than ndp),
#     ndp = number of data points (must be 4 or greater),
#     xd  = array of dimension ndp containing the x
#           coordinates of the data points,
#     yd  = array of dimension ndp containing the y
#           coordinates of the data points,
#     zd  = array of dimension ndp containing the z
#           coordinates of the data points,
#     nip = number of output points at which interpolation
#           is to be performed (must be 1 or greater),
#     xi  = array of dimension nip containing the x
#           coordinates of the output points,
#     yi  = array of dimension nip containing the y
#           coordinates of the output points.
# the output parameter is
#     zi  = array of dimension nip where interpolated z
#           values are to be stored.
# the other parameters are
#     iwk = integer array of dimension
#              max0(31,27+ncp)*ndp+nip
#           used internally as a work area,
#     wk  = array of dimension 8*ndp used internally as a
#           work area.
# the very first call to this subroutine and the call with a new
# ncp value, a new ndp value, and/or new contents of the xd and
# yd arrays must be made with md=1.  the call with md=2 must be
# preceded by another call with the same ncp and ndp values and
# with the same contents of the xd and yd arrays.  the call with
# md=3 must be preceded by another call with the same ncp, ndp,
# and nip values and with the same contents of the xd, yd, xi,
# and yi arrays.  between the call with md=2 or md=3 and its
# preceding call, the iwk and wk arrays must not be disturbed.
# use of a value between 3 and 5 (inclusive) for ncp is recom-
# mended unless there are evidences that dictate otherwise.

# this subroutine calls the idcldp, idlctn, idpdrv, idptip, and
# idtang subroutines.
# declaration statements
common/idlc/nit
common/idpi/itpv
# setting of some input parameters to local variables (for md=1,2,3)
md0 = md
ncp0 = ncp
ndp0 = ndp
nip0 = nip
# error check.  (for md=1,2,3)
if(md0<1 | md0>3 | ncp0<2 | ncp0>=ndp0 | ndp0<4 | nip0<1) ERROR(Bad input values)
if (md0<2) {
	iwk(1) = ncp0
	iwk(2) = ndp0
	}
else {
	ncppv = iwk(1)
	ndppv = iwk(2)
	if (ncp0!=ncppv | ndp0!=ndppv) ERROR(Bad input values)
	}
if (md0<3)
	iwk(3) = nip
else {
	nippv = iwk(3)
	if (nip0!=nippv) ERROR(Bad input values)
	}
# allocation of storage areas in the iwk array.  (for md=1,2,3)
jwipt = 16
jwiwl = 6*ndp0+1
jwiwk = jwiwl
jwipl = 24*ndp0+1
jwiwp = 30*ndp0+1
jwipc = 27*ndp0+1
jwit0 = max0(31,27+ncp0)*ndp0
if (md0<=1) {	# triangulates the x-y plane.  (for md=1)
	call idtang(ndp0,xd,yd,nt,iwk(jwipt),nl,iwk(jwipl),iwk(jwiwl),iwk(jwiwp),wk)
	iwk(5) = nt
	iwk(6) = nl
	if (nt==0) return
	}
if (md0<=1) {	# determines ncp points closest to each data point.  (for md=1)
	call idcldp(ndp0,xd,yd,ncp0,iwk(jwipc))
	if (iwk(jwipc)==0) return
	}
if (md0!=3) {	# locates all points at which interpolation is to be performed (for md=1,2)
	nit = 0
	jwit = jwit0
	do iip = 1,nip0 {
		jwit = jwit+1
		call idlctn(ndp0,xd,yd,nt,iwk(jwipt),nl,iwk(jwipl),xi(iip),yi(iip),iwk(jwit),iwk(jwiwk),wk)
		}
	}
# estimates partial derivatives at all data points (for md=1,2,3)
call idpdrv(ndp0,xd,yd,zd,ncp0,iwk(jwipc),wk)
# interpolates the zi values.  (for md=1,2,3)
itpv = 0
jwit = jwit0
do iip = 1,nip0 {
	jwit = jwit+1
	call idptip(xd,yd,zd,nt,iwk(jwipt),nl,iwk(jwipl),wk,iwk(jwit),xi(iip),yi(iip),zi(iip))
	}
return
end



subroutine  idcldp(ndp,xd,yd,ncp,ipc)
# this subroutine selects several data points that are closest
# to each of the data point.
# the input parameters are
#     ndp = number of data points,
#     xd,yd = arrays of dimension ndp containing the x and y
#           coordinates of the data points,
#     ncp = number of data points closest to each data
#           points.
# the output parameter is
#     ipc = integer array of dimension ncp*ndp, where the
#           point numbers of ncp data points closest to
#           each of the ndp data points are to be stored.
# this subroutine arbitrarily sets a restriction that ncp must
# not exceed 25.
# declaration statements
dimension   xd(100),yd(100),ipc(400)
dimension   dsq0(25),ipc0(25)
data  ncpmx/25/
# statement function
dsqf(u1,v1,u2,v2) = (u2-u1)**2+(v2-v1)**2
# preliminary processing
ndp0 = ndp
ncp0 = ncp
if (ndp0<2 | ncp0<1 | ncp0>ncpmx | ncp0>=ndp0) ERROR(Bad input values)
do ip1 = 1,ndp0 {	# - selects ncp points.
	x1 = xd(ip1)
	y1 = yd(ip1)
	j1 = 0
	dsqmx = 0.0
	do ip2 = 1,ndp0
		if (ip2!=ip1) {
			dsqi = dsqf(x1,y1,xd(ip2),yd(ip2))
			j1 = j1+1
			dsq0(j1) = dsqi
			ipc0(j1) = ip2
			if (dsqi>dsqmx) {
				dsqmx = dsqi
				jmx = j1
				}
			if (j1>=ncp0) break
			}
	ip2mn = ip2+1
	if (ip2mn<=ndp0)
		do ip2 = ip2mn,ndp0
			if (ip2!=ip1) {
				dsqi = dsqf(x1,y1,xd(ip2),yd(ip2))
				if (dsqi<dsqmx) {
					dsq0(jmx) = dsqi
					ipc0(jmx) = ip2
					dsqmx = 0.0
					do j1 = 1,ncp0
						if (dsq0(j1)>dsqmx) {
							dsqmx = dsq0(j1)
							jmx = j1
							}
					}
				}
# - checks if all the ncp+1 points are collinear.
	ip2 = ipc0(1)
	dx12 = xd(ip2)-x1
	dy12 = yd(ip2)-y1
	do j3 = 2,ncp0 {
		ip3 = ipc0(j3)
		dx13 = xd(ip3)-x1
		dy13 = yd(ip3)-y1
		if (dy13*dx12-dx13*dy12!=0.0) go to 30
		}
# - searches for the closest noncollinear point.
	nclpt = 0
	do ip3 = 1,ndp0
		if (ip3!=ip1) {
			do j4 = 1,ncp0
				if (ip3==ipc0(j4)) next 2
			dx13 = xd(ip3)-x1
			dy13 = yd(ip3)-y1
			if (dy13*dx12-dx13*dy12!=0.0) {
				dsqi = dsqf(x1,y1,xd(ip3),yd(ip3))
				if (nclpt!=0 & dsqi>=dsqmn) next
				nclpt = 1
				dsqmn = dsqi
				ip3mn = ip3
				}
			}
	if (nclpt==0) ERROR(All colinear data points)
	dsqmx = dsqmn
	ipc0(jmx) = ip3mn
# - replaces the local array for the output array.
	30  j1 = (ip1-1)*ncp0
	do j2 = 1,ncp0 {
		j1 = j1+1
		ipc(j1) = ipc0(j2)
		}
	}
return
end



subroutine idgrid(xd,yd,nt,ipt,nl,ipl,nxi,nyi,xi,yi,ngp,igp)
# this subroutine organizes grid points for surface fitting by
# sorting them in ascending order of triangle numbers and of the
# border line segment number.
# the input parameters are
#     xd,yd = arrays of dimension ndp containing the x and y
#           coordinates of the data points, where ndp is the
#           number of the data points,
#     nt  = number of triangles,
#     ipt = integer array of dimension 3*nt containing the
#           point numbers of the vertexes of the triangles,
#     nl  = number of border line segments,
#     ipl = integer array of dimension 3*nl containing the
#           point numbers of the end points of the border
#           line segments and their respective triangle
#           numbers,
#     nxi = number of grid points in the x coordinate,
#     nyi = number of grid points in the y coordinate,
#     xi,yi = arrays of dimension nxi and nyi containing
#           the x and y coordinates of the grid points,
#           respectively.
# the output parameters are
#     ngp = integer array of dimension 2*(nt+2*nl) where the
#           number of grid points that belong to each of the
#           triangles or of the border line segments are to
#           be stored,
#     igp = integer array of dimension nxi*nyi where the
#           grid point numbers are to be stored in ascending
#           order of the triangle number and the border line
#           segment number.
# declaration statements
dimension xd(100),yd(100),ipt(585),ipl(300),xi(101),yi(101),ngp(800),igp(10201)
# statement functions
side(u1,v1,u2,v2,u3,v3) = (u1-u3)*(v2-v3)-(v1-v3)*(u2-u3)
spdt(u1,v1,u2,v2,u3,v3) = (u1-u2)*(u3-u2)+(v1-v2)*(v3-v2)
# preliminary processing
nt0 = nt
nl0 = nl
nxi0 = nxi
nyi0 = nyi
nxinyi = nxi0*nyi0
ximn = amin1(xi(1),xi(nxi0))
ximx = amax1(xi(1),xi(nxi0))
yimn = amin1(yi(1),yi(nyi0))
yimx = amax1(yi(1),yi(nyi0))
# determines grid points inside the data area.
jngp0 = 0
jngp1 = 2*(nt0+2*nl0)+1
jigp0 = 0
jigp1 = nxinyi+1
do it0 = 1,nt0 {
	ngp0 = 0
	ngp1 = 0
	it0t3 = it0*3
	ip1 = ipt(it0t3-2)
	ip2 = ipt(it0t3-1)
	ip3 = ipt(it0t3)
	x1 = xd(ip1)
	y1 = yd(ip1)
	x2 = xd(ip2)
	y2 = yd(ip2)
	x3 = xd(ip3)
	y3 = yd(ip3)
	xmn = amin1(x1,x2,x3)
	xmx = amax1(x1,x2,x3)
	ymn = amin1(y1,y2,y3)
	ymx = amax1(y1,y2,y3)
	insd = 0
	do ixi = 1,nxi0
		if (xi(ixi)>=xmn&&xi(ixi)<=xmx) {
			if (insd!=1) {
				insd = 1
				iximn = ixi
				}
			}
		else if (insd!=0) {
			iximx = ixi-1
			go to 80
			}
	if (insd==0)
		go to 90
	iximx = nxi0
	80  do iyi = 1,nyi0 {
		yii = yi(iyi)
		if (yii>=ymn&&yii<=ymx)
			do ixi = iximn,iximx {
				xii = xi(ixi)
				l = 0
				if (side(x1,y1,x2,y2,xii,yii)>=0) {
					if (side(x1,y1,x2,y2,xii,yii)==0)
						l = 1
					if (side(x2,y2,x3,y3,xii,yii)>=0) {
						if (side(x2,y2,x3,y3,xii,yii)==0)
							l = 1
						if (side(x3,y3,x1,y1,xii,yii)>=0) {
							if (side(x3,y3,x1,y1,xii,yii)==0)
								l = 1
							izi = nxi0*(iyi-1)+ixi
							if (l!=1) {
								ngp0 = ngp0+1
								jigp0 = jigp0+1
								igp(jigp0) = izi
								}
							else {
								if (jigp1<=nxinyi)
									do jigp1i = jigp1,nxinyi
										if (izi==igp(jigp1i))
											next 2
								ngp1 = ngp1+1
								jigp1 = jigp1-1
								igp(jigp1) = izi
								}
							}
						}
					}
				}
		}
	90  jngp0 = jngp0+1
	ngp(jngp0) = ngp0
	jngp1 = jngp1-1
	ngp(jngp1) = ngp1
	}
# determines grid points outside the data area.
# - in semi-infinite rectangular area.
do il0 = 1,nl0 {
	ngp0 = 0
	ngp1 = 0
	il0t3 = il0*3
	ip1 = ipl(il0t3-2)
	ip2 = ipl(il0t3-1)
	x1 = xd(ip1)
	y1 = yd(ip1)
	x2 = xd(ip2)
	y2 = yd(ip2)
	xmn = ximn
	xmx = ximx
	ymn = yimn
	ymx = yimx
	if (y2>=y1)
		xmn = amin1(x1,x2)
	if (y2<=y1)
		xmx = amax1(x1,x2)
	if (x2<=x1)
		ymn = amin1(y1,y2)
	if (x2>=x1)
		ymx = amax1(y1,y2)
	insd = 0
	do ixi = 1,nxi0
		if (xi(ixi)>=xmn&&xi(ixi)<=xmx) {
			if (insd!=1) {
				insd = 1
				iximn = ixi
				}
			}
		else if (insd!=0) {
			iximx = ixi-1
			go to 100
			}
	if (insd==0)
		go to 110
	iximx = nxi0
	100  do iyi = 1,nyi0 {
		yii = yi(iyi)
		if (yii>=ymn&&yii<=ymx)
			do ixi = iximn,iximx {
				xii = xi(ixi)
				l = 0
				if (side(x1,y1,x2,y2,xii,yii)>=0) {
					if (side(x1,y1,x2,y2,xii,yii)!=0)
						next 1
					l = 1
					}
				if (spdt(x2,y2,x1,y1,xii,yii)>=0) {
					if (spdt(x2,y2,x1,y1,xii,yii)==0)
						l = 1
					if (spdt(x1,y1,x2,y2,xii,yii)>=0) {
						if (spdt(x1,y1,x2,y2,xii,yii)==0)
							l = 1
						izi = nxi0*(iyi-1)+ixi
						if (l!=1) {
							ngp0 = ngp0+1
							jigp0 = jigp0+1
							igp(jigp0) = izi
							}
						else {
							if (jigp1<=nxinyi)
								do jigp1i = jigp1,nxinyi
									if (izi==igp(jigp1i))
										next 2
							ngp1 = ngp1+1
							jigp1 = jigp1-1
							igp(jigp1) = izi
							}
						}
					}
				}
		}
	110  jngp0 = jngp0+1
	ngp(jngp0) = ngp0
	jngp1 = jngp1-1
	ngp(jngp1) = ngp1
# - in semi-infinite triangular area.
	ngp0 = 0
	ngp1 = 0
	ilp1 = mod(il0,nl0)+1
	ilp1t3 = ilp1*3
	ip3 = ipl(ilp1t3-1)
	x3 = xd(ip3)
	y3 = yd(ip3)
	xmn = ximn
	xmx = ximx
	ymn = yimn
	ymx = yimx
	if (y3>=y2&&y2>=y1)
		xmn = x2
	if (y3<=y2&&y2<=y1)
		xmx = x2
	if (x3<=x2&&x2<=x1)
		ymn = y2
	if (x3>=x2&&x2>=x1)
		ymx = y2
	insd = 0
	do ixi = 1,nxi0
		if (xi(ixi)>=xmn&&xi(ixi)<=xmx) {
			if (insd!=1) {
				insd = 1
				iximn = ixi
				}
			}
		else if (insd!=0) {
			iximx = ixi-1
			go to 120
			}
	if (insd==0)
		go to 130
	iximx = nxi0
	120  do iyi = 1,nyi0 {
		yii = yi(iyi)
		if (yii>=ymn&&yii<=ymx)
			do ixi = iximn,iximx {
				xii = xi(ixi)
				l = 0
				if (spdt(x1,y1,x2,y2,xii,yii)>=0) {
					if (spdt(x1,y1,x2,y2,xii,yii)!=0)
						next 1
					l = 1
					}
				if (spdt(x3,y3,x2,y2,xii,yii)>=0) {
					if (spdt(x3,y3,x2,y2,xii,yii)!=0)
						next 1
					l = 1
					}
				izi = nxi0*(iyi-1)+ixi
				if (l!=1) {
					ngp0 = ngp0+1
					jigp0 = jigp0+1
					igp(jigp0) = izi
					}
				else {
					if (jigp1<=nxinyi)
						do jigp1i = jigp1,nxinyi
							if (izi==igp(jigp1i))
								next 2
					ngp1 = ngp1+1
					jigp1 = jigp1-1
					igp(jigp1) = izi
					}
				}
		}
	130  jngp0 = jngp0+1
	ngp(jngp0) = ngp0
	jngp1 = jngp1-1
	ngp(jngp1) = ngp1
	}
return
end



subroutine idlctn(ndp,xd,yd,nt,ipt,nl,ipl,xii,yii,iti,iwk,wk)
# this subroutine locates a point, i.e., determines to what tri-
# angle a given point (xii,yii) belongs.  when the given point
# does not lie inside the data area, this subroutine determines
# the border line segment when the point lies in an outside
# rectangular area, and two border line segments when the point
# lies in an outside triangular area.
# the input parameters are
#     ndp = number of data points,
#     xd,yd = arrays of dimension ndp containing the x and y
#           coordinates of the data points,
#     nt  = number of triangles,
#     ipt = integer array of dimension 3*nt containing the
#           point numbers of the vertexes of the triangles,
#     nl  = number of border line segments,
#     ipl = integer array of dimension 3*nl containing the
#           point numbers of the end points of the border
#           line segments and their respective triangle
#           numbers,
#     xii,yii = x and y coordinates of the point to be
#           located.
# the output parameter is
#     iti = triangle number, when the point is inside the
#           data area, or
#           two border line segment numbers, il1 and il2,
#           coded to il1*(nt+nl)+il2, when the point is
#           outside the data area.
# the other parameters are
#     iwk = integer array of dimension 18*ndp used inter-
#           nally as a work area,
#     wk  = array of dimension 8*ndp used internally as a
#           work area.
# declaration statements
dimension xd(100),yd(100),ipt(585),ipl(300),iwk(1800),wk(800)
dimension ntsc(9),idsc(9)
common/idlc/nit
# statement functions
side(u1,v1,u2,v2,u3,v3) = (u1-u3)*(v2-v3)-(v1-v3)*(u2-u3)
spdt(u1,v1,u2,v2,u3,v3) = (u1-u2)*(u3-u2)+(v1-v2)*(v3-v2)
# preliminary processing
ndp0 = ndp
nt0 = nt
nl0 = nl
ntl = nt0+nl0
x0 = xii
y0 = yii
# processing for a new set of data points
if (nit!=0) {
# checks if in the same triangle as previous.
	it0 = itipv
	if (it0<=nt0) {
		it0t3 = it0*3
		ip1 = ipt(it0t3-2)
		x1 = xd(ip1)
		y1 = yd(ip1)
		ip2 = ipt(it0t3-1)
		x2 = xd(ip2)
		y2 = yd(ip2)
		if (side(x1,y1,x2,y2,x0,y0)>=0.0) {
			ip3 = ipt(it0t3)
			x3 = xd(ip3)
			y3 = yd(ip3)
			if (side(x2,y2,x3,y3,x0,y0)>=0.0)
				if (side(x3,y3,x1,y1,x0,y0)>=0.0)
					go to 150
			}
		}
	else {
# checks if on the same border line segment.
		il1 = it0/ntl
		il2 = it0-il1*ntl
		il1t3 = il1*3
		ip1 = ipl(il1t3-2)
		x1 = xd(ip1)
		y1 = yd(ip1)
		ip2 = ipl(il1t3-1)
		x2 = xd(ip2)
		y2 = yd(ip2)
		if (il2!=il1) {
# checks if between the same two border line segments.
			if (spdt(x1,y1,x2,y2,x0,y0)<=0.0) {
				ip3 = ipl(3*il2-1)
				x3 = xd(ip3)
				y3 = yd(ip3)
				if (spdt(x3,y3,x2,y2,x0,y0)<=0.0)
					go to 150
				}
			}
		else if (spdt(x1,y1,x2,y2,x0,y0)>=0.0)
			if (spdt(x2,y2,x1,y1,x0,y0)>=0.0)
				if (side(x1,y1,x2,y2,x0,y0)<=0.0)
					go to 150
		}
	}
else {
	nit = 1
# - divides the x-y plane into nine rectangular sections.
	xmn = xd(1)
	xmx = xmn
	ymn = yd(1)
	ymx = ymn
	do idp = 2,ndp0 {
		xi = xd(idp)
		yi = yd(idp)
		xmn = amin1(xi,xmn)
		xmx = amax1(xi,xmx)
		ymn = amin1(yi,ymn)
		ymx = amax1(yi,ymx)
		}
	xs1 = (xmn+xmn+xmx)/3.0
	xs2 = (xmn+xmx+xmx)/3.0
	ys1 = (ymn+ymn+ymx)/3.0
	ys2 = (ymn+ymx+ymx)/3.0
# - determines and stores in the iwk array triangle numbers of
# - the triangles associated with each of the nine sections.
	do isc = 1,9 {
		ntsc(isc) = 0
		idsc(isc) = 0
		}
	it0t3 = 0
	jwk = 0
	do it0 = 1,nt0 {
		it0t3 = it0t3+3
		i1 = ipt(it0t3-2)
		i2 = ipt(it0t3-1)
		i3 = ipt(it0t3)
		xmn = amin1(xd(i1),xd(i2),xd(i3))
		xmx = amax1(xd(i1),xd(i2),xd(i3))
		ymn = amin1(yd(i1),yd(i2),yd(i3))
		ymx = amax1(yd(i1),yd(i2),yd(i3))
		if (ymn<=ys1) {
			if (xmn<=xs1)
				idsc(1) = 1
			if (xmx>=xs1&&xmn<=xs2)
				idsc(2) = 1
			if (xmx>=xs2)
				idsc(3) = 1
			}
		if (ymx>=ys1&&ymn<=ys2) {
			if (xmn<=xs1)
				idsc(4) = 1
			if (xmx>=xs1&&xmn<=xs2)
				idsc(5) = 1
			if (xmx>=xs2)
				idsc(6) = 1
			}
		if (ymx>=ys2) {
			if (xmn<=xs1)
				idsc(7) = 1
			if (xmx>=xs1&&xmn<=xs2)
				idsc(8) = 1
			if (xmx>=xs2)
				idsc(9) = 1
			}
		do isc = 1,9
			if (idsc(isc)!=0) {
				jiwk = 9*ntsc(isc)+isc
				iwk(jiwk) = it0
				ntsc(isc) = ntsc(isc)+1
				idsc(isc) = 0
				}
# - stores in the wk array the minimum and maximum of the x and
# - y coordinate values for each of the triangle.
		jwk = jwk+4
		wk(jwk-3) = xmn
		wk(jwk-2) = xmx
		wk(jwk-1) = ymn
		wk(jwk) = ymx
		}
	}
# locates inside the data area.
# - determines the section in which the point in question lies.
isc = 1
if (x0>=xs1)
	isc = isc+1
if (x0>=xs2)
	isc = isc+1
if (y0>=ys1)
	isc = isc+3
if (y0>=ys2)
	isc = isc+3
# - searches through the triangles associated with the section.
ntsci = ntsc(isc)
if (ntsci>0) {
	jiwk = -9+isc
	do itsc = 1,ntsci {
		jiwk = jiwk+9
		it0 = iwk(jiwk)
		jwk = it0*4
		if (x0>=wk(jwk-3))
			if (x0<=wk(jwk-2))
				if (y0>=wk(jwk-1))
					if (y0<=wk(jwk)) {
						it0t3 = it0*3
						ip1 = ipt(it0t3-2)
						x1 = xd(ip1)
						y1 = yd(ip1)
						ip2 = ipt(it0t3-1)
						x2 = xd(ip2)
						y2 = yd(ip2)
						if (side(x1,y1,x2,y2,x0,y0)>=0.0) {
							ip3 = ipt(it0t3)
							x3 = xd(ip3)
							y3 = yd(ip3)
							if (side(x2,y2,x3,y3,x0,y0)>=0.0)
								if (side(x3,y3,x1,y1,x0,y0)>=0.0)
									go to 150
							}
						}
		}
	}
# locates outside the data area.
do il1 = 1,nl0 {
	il1t3 = il1*3
	ip1 = ipl(il1t3-2)
	x1 = xd(ip1)
	y1 = yd(ip1)
	ip2 = ipl(il1t3-1)
	x2 = xd(ip2)
	y2 = yd(ip2)
	if (spdt(x2,y2,x1,y1,x0,y0)>=0.0)
		if (spdt(x1,y1,x2,y2,x0,y0)<0.0) {
			il2 = mod(il1,nl0)+1
			ip3 = ipl(3*il2-1)
			x3 = xd(ip3)
			y3 = yd(ip3)
			if (spdt(x3,y3,x2,y2,x0,y0)<=0.0)
				go to 140
			}
		else if (side(x1,y1,x2,y2,x0,y0)<=0.0) {
			il2 = il1
			go to 140
			}
	}
it0 = 1
go to 150
140  it0 = il1*ntl+il2
# normal exit
150  iti = it0
itipv = it0
return
end



subroutine  idpdrv(ndp,xd,yd,zd,ncp,ipc,pd)
# this subroutine estimates partial derivatives of the first and
# second order at the data points.
# the input parameters are
#     ndp = number of data points,
#     xd,yd,zd = arrays of dimension ndp containing the x,
#           y, and z coordinates of the data points,
#     ncp = number of additional data points used for esti-
#           mating partial derivatives at each data point,
#     ipc = integer array of dimension ncp*ndp containing
#           the point numbers of ncp data points closest to
#           each of the ndp data points.
# the output parameter is
#     pd  = array of dimension 5*ndp, where the estimated
#           zx, zy, zxx, zxy, and zyy values at the data
#           points are to be stored.
# declaration statements
dimension   xd(100),yd(100),zd(100),ipc(400),pd(500)
real        nmx,nmy,nmz,nmxx,nmxy,nmyx,nmyy
# preliminary processing
ndp0 = ndp
ncp0 = ncp
ncpm1 = ncp0-1
# estimation of zx and zy
do ip0 = 1,ndp0 {
	x0 = xd(ip0)
	y0 = yd(ip0)
	z0 = zd(ip0)
	nmx = 0.0
	nmy = 0.0
	nmz = 0.0
	jipc0 = ncp0*(ip0-1)
	do ic1 = 1,ncpm1 {
		jipc = jipc0+ic1
		ipi = ipc(jipc)
		dx1 = xd(ipi)-x0
		dy1 = yd(ipi)-y0
		dz1 = zd(ipi)-z0
		ic2mn = ic1+1
		do ic2 = ic2mn,ncp0 {
			jipc = jipc0+ic2
			ipi = ipc(jipc)
			dx2 = xd(ipi)-x0
			dy2 = yd(ipi)-y0
			dnmz = dx1*dy2-dy1*dx2
			if (dnmz!=0.0) {
				dz2 = zd(ipi)-z0
				dnmx = dy1*dz2-dz1*dy2
				dnmy = dz1*dx2-dx1*dz2
				if (dnmz<0.0) {
					dnmx = -dnmx
					dnmy = -dnmy
					dnmz = -dnmz
					}
				nmx = nmx+dnmx
				nmy = nmy+dnmy
				nmz = nmz+dnmz
				}
			}
		}
	jpd0 = 5*ip0
	pd(jpd0-4) = -nmx/nmz
	pd(jpd0-3) = -nmy/nmz
	}
# estimation of zxx, zxy, and zyy
do ip0 = 1,ndp0 {
	jpd0 = jpd0+5
	x0 = xd(ip0)
	jpd0 = 5*ip0
	y0 = yd(ip0)
	zx0 = pd(jpd0-4)
	zy0 = pd(jpd0-3)
	nmxx = 0.0
	nmxy = 0.0
	nmyx = 0.0
	nmyy = 0.0
	nmz = 0.0
	jipc0 = ncp0*(ip0-1)
	do ic1 = 1,ncpm1 {
		jipc = jipc0+ic1
		ipi = ipc(jipc)
		dx1 = xd(ipi)-x0
		dy1 = yd(ipi)-y0
		jpd = 5*ipi
		dzx1 = pd(jpd-4)-zx0
		dzy1 = pd(jpd-3)-zy0
		ic2mn = ic1+1
		do ic2 = ic2mn,ncp0 {
			jipc = jipc0+ic2
			ipi = ipc(jipc)
			dx2 = xd(ipi)-x0
			dy2 = yd(ipi)-y0
			dnmz = dx1*dy2-dy1*dx2
			if (dnmz!=0.0) {
				jpd = 5*ipi
				dzx2 = pd(jpd-4)-zx0
				dzy2 = pd(jpd-3)-zy0
				dnmxx = dy1*dzx2-dzx1*dy2
				dnmxy = dzx1*dx2-dx1*dzx2
				dnmyx = dy1*dzy2-dzy1*dy2
				dnmyy = dzy1*dx2-dx1*dzy2
				if (dnmz<0.0) {
					dnmxx = -dnmxx
					dnmxy = -dnmxy
					dnmyx = -dnmyx
					dnmyy = -dnmyy
					dnmz = -dnmz
					}
				nmxx = nmxx+dnmxx
				nmxy = nmxy+dnmxy
				nmyx = nmyx+dnmyx
				nmyy = nmyy+dnmyy
				nmz = nmz+dnmz
				}
			}
		}
	pd(jpd0-2) = -nmxx/nmz
	pd(jpd0-1) = -(nmxy+nmyx)/(2.0*nmz)
	pd(jpd0) = -nmyy/nmz
	}
return
end



subroutine  idtang(ndp,xd,yd,nt,ipt,nl,ipl,iwl,iwp,wk)
# this subroutine performs triangulation.  it divides the x-y
# plane into a number of triangles according to given data
# points in the plane, determines line segments that form the
# border of data area, and determines the triangle numbers
# corresponding to the border line segments.
# at completion, point numbers of the vertexes of each triangle
# are listed counter-clockwise.  point numbers of the end points
# of each border line segment are listed counter-clockwise,
# listing order of the line segments being counter-clockwise.
# the lun constant in the data initialization statement is the
# logical unit number of the standard output unit and is,
# therefore, system dependent.
# this subroutine calls the idxchg function.
# the input parameters are
#     ndp = number of data points,
#     xd  = array of dimension ndp containing the
#           x coordinates of the data points,
#     yd  = array of dimension ndp containing the
#           y coordinates of the data points.
# the output parameters are
#     nt  = number of triangles,
#     ipt = integer array of dimension 6*ndp-15, where the
#           point numbers of the vertexes of the (it)th
#           triangle are to be stored as the (3*it-2)nd,
#           (3*it-1)st, and (3*it)th elements,
#           it=1,2,...,nt,
#     nl  = number of border line segments,
#     ipl = integer array of dimension 6*ndp, where the
#           point numbers of the end points of the (il)th
#           border line segment and its respective triangle
#           number are to be stored as the (3*il-2)nd,
#           (3*il-1)st, and (3*il)th elements,
#           il=1,2,..., nl.
# the other parameters are
#     iwl = integer array of dimension 18*ndp used
#           internally as a work area,
#     iwp = integer array of dimension ndp used
#           internally as a work area,
#     wk  = array of dimension ndp used internally as a
#           work area.
# declaration statements
dimension   xd(100),yd(100),ipt(585),ipl(600),iwl(1800),iwp(100),wk(100)
dimension   itf(2)
data  ratio/1.0e-6/,nrep/100/,lun/6/
# statement functions
dsqf(u1,v1,u2,v2) = (u2-u1)**2+(v2-v1)**2
side(u1,v1,u2,v2,u3,v3) = (v3-v1)*(u2-u1)-(u3-u1)*(v2-v1)
# preliminary processing
ndp0 = ndp
ndpm1 = ndp0-1
if (ndp0<4) ERROR(Fewer than 4 data points)
# determines the closest pair of data points and their midpoint.
dsqmn = dsqf(xd(1),yd(1),xd(2),yd(2))
ipmn1 = 1
ipmn2 = 2
do ip1 = 1,ndpm1 {
	x1 = xd(ip1)
	y1 = yd(ip1)
	ip1p1 = ip1+1
	do ip2 = ip1p1,ndp0 {
		dsqi = dsqf(x1,y1,xd(ip2),yd(ip2))
		if (dsqi==0.0) ERROR(Identical data points)
		if (dsqi<dsqmn) {
			dsqmn = dsqi
			ipmn1 = ip1
			ipmn2 = ip2
			}
		}
	}
dsq12 = dsqmn
xdmp = (xd(ipmn1)+xd(ipmn2))/2.0
ydmp = (yd(ipmn1)+yd(ipmn2))/2.0
# sorts the other (ndp-2) data points in ascending order of
# distance from the midpoint and stores the sorted data point
# numbers in the iwp array.
jp1 = 2
do ip1 = 1,ndp0
	if (ip1!=ipmn1&&ip1!=ipmn2) {
		jp1 = jp1+1
		iwp(jp1) = ip1
		wk(jp1) = dsqf(xdmp,ydmp,xd(ip1),yd(ip1))
		}
do jp1 = 3,ndpm1 {
	dsqmn = wk(jp1)
	jpmn = jp1
	do jp2 = jp1,ndp0
		if (wk(jp2)<dsqmn) {
			dsqmn = wk(jp2)
			jpmn = jp2
			}
	its = iwp(jp1)
	iwp(jp1) = iwp(jpmn)
	iwp(jpmn) = its
	wk(jpmn) = wk(jp1)
	}
# if necessary, modifies the ordering in such a way that the
# first three data points are not collinear.
ar = dsq12*ratio
x1 = xd(ipmn1)
y1 = yd(ipmn1)
dx21 = xd(ipmn2)-x1
dy21 = yd(ipmn2)-y1
for(jp=3; jp<=ndp0; jp=jp+1){
	ip = iwp(jp)
	if (abs((yd(ip)-y1)*dx21-(xd(ip)-x1)*dy21)>ar) break
	}
if(jp>ndp0) ERROR(All data points collinear)
if (jp!=3) {
	jpmx = jp
	jp = jpmx+1
	do jpc = 4,jpmx {
		jp = jp-1
		iwp(jp) = iwp(jp-1)
		}
	iwp(3) = ip
	}
# forms the first triangle.  stores point numbers of the ver-
# texes of the triangle in the ipt array, and stores point num-
# bers of the border line segments and the triangle number in
# the ipl array.
ip1 = ipmn1
ip2 = ipmn2
ip3 = iwp(3)
if (side(xd(ip1),yd(ip1),xd(ip2),yd(ip2),xd(ip3),yd(ip3))<0.0) {
	ip1 = ipmn2
	ip2 = ipmn1
	}
nt0 = 1
ntt3 = 3
ipt(1) = ip1
ipt(2) = ip2
ipt(3) = ip3
nl0 = 3
nlt3 = 9
ipl(1) = ip1
ipl(2) = ip2
ipl(3) = 1
ipl(4) = ip2
ipl(5) = ip3
ipl(6) = 1
ipl(7) = ip3
ipl(8) = ip1
ipl(9) = 1
# adds the remaining (ndp-3) data points, one by one.
do jp1 = 4,ndp0 {
	ip1 = iwp(jp1)
	x1 = xd(ip1)
	y1 = yd(ip1)
# - determines the visible border line segments.
	ip2 = ipl(1)
	jpmn = 1
	dxmn = xd(ip2)-x1
	dymn = yd(ip2)-y1
	dsqmn = dxmn**2+dymn**2
	armn = dsqmn*ratio
	jpmx = 1
	dxmx = dxmn
	dymx = dymn
	dsqmx = dsqmn
	armx = armn
	do jp2 = 2,nl0 {
		ip2 = ipl(3*jp2-2)
		dx = xd(ip2)-x1
		dy = yd(ip2)-y1
		ar = dy*dxmn-dx*dymn
		if (ar<=armn) {
			dsqi = dx**2+dy**2
			if (ar<(-armn)||dsqi<dsqmn) {
				jpmn = jp2
				dxmn = dx
				dymn = dy
				dsqmn = dsqi
				armn = dsqmn*ratio
				}
			}
		ar = dy*dxmx-dx*dymx
		if (ar>=(-armx)) {
			dsqi = dx**2+dy**2
			if (ar>armx||dsqi<dsqmx) {
				jpmx = jp2
				dxmx = dx
				dymx = dy
				dsqmx = dsqi
				armx = dsqmx*ratio
				}
			}
		}
	if (jpmx<jpmn)
		jpmx = jpmx+nl0
	nsh = jpmn-1
	if (nsh>0) {
# - shifts (rotates) the ipl array to have the invisible border
# - line segments contained in the first part of the ipl array.
		nsht3 = nsh*3
		do jp2t3 = 3,nsht3,3 {
			jp3t3 = jp2t3+nlt3
			ipl(jp3t3-2) = ipl(jp2t3-2)
			ipl(jp3t3-1) = ipl(jp2t3-1)
			ipl(jp3t3) = ipl(jp2t3)
			}
		do jp2t3 = 3,nlt3,3 {
			jp3t3 = jp2t3+nsht3
			ipl(jp2t3-2) = ipl(jp3t3-2)
			ipl(jp2t3-1) = ipl(jp3t3-1)
			ipl(jp2t3) = ipl(jp3t3)
			}
		jpmx = jpmx-nsh
		}
# - adds triangles to the ipt array, updates border line
# - segments in the ipl array, and sets flags for the border
# - line segments to be reexamined in the iwl array.
	jwl = 0
	do jp2 = jpmx,nl0 {
		jp2t3 = jp2*3
		ipl1 = ipl(jp2t3-2)
		ipl2 = ipl(jp2t3-1)
		it = ipl(jp2t3)
# - - adds a triangle to the ipt array.
		nt0 = nt0+1
		ntt3 = ntt3+3
		ipt(ntt3-2) = ipl2
		ipt(ntt3-1) = ipl1
		ipt(ntt3) = ip1
# - - updates border line segments in the ipl array.
		if (jp2==jpmx) {
			ipl(jp2t3-1) = ip1
			ipl(jp2t3) = nt0
			}
		if (jp2==nl0) {
			nln = jpmx+1
			nlnt3 = nln*3
			ipl(nlnt3-2) = ip1
			ipl(nlnt3-1) = ipl(1)
			ipl(nlnt3) = nt0
			}
# - - determines the vertex that does not lie on the border
# - - line segments.
		itt3 = it*3
		ipti = ipt(itt3-2)
		if (ipti==ipl1||ipti==ipl2) {
			ipti = ipt(itt3-1)
			if (ipti==ipl1||ipti==ipl2)
				ipti = ipt(itt3)
			}
# - - checks if the exchange is necessary.
		if (idxchg(xd,yd,ip1,ipti,ipl1,ipl2)!=0) {
# - - modifies the ipt array when necessary.
			ipt(itt3-2) = ipti
			ipt(itt3-1) = ipl1
			ipt(itt3) = ip1
			ipt(ntt3-1) = ipti
			if (jp2==jpmx)
				ipl(jp2t3) = it
			if (jp2==nl0&&ipl(3)==it)
				ipl(3) = nt0
# - - sets flags in the iwl array.
			jwl = jwl+4
			iwl(jwl-3) = ipl1
			iwl(jwl-2) = ipti
			iwl(jwl-1) = ipti
			iwl(jwl) = ipl2
			}
		}
	nl0 = nln
	nlt3 = nlnt3
	nlf = jwl/2
	if (nlf!=0) {
# - improves triangulation.
		ntt3p3 = ntt3+3
		do irep = 1,nrep {
			do ilf = 1,nlf {
				ilft2 = ilf*2
				ipl1 = iwl(ilft2-1)
				ipl2 = iwl(ilft2)
# - - locates in the ipt array two triangles on both sides of
# - - the flagged line segment.
				ntf = 0
				do itt3r = 3,ntt3,3 {
					itt3 = ntt3p3-itt3r
					ipt1 = ipt(itt3-2)
					ipt2 = ipt(itt3-1)
					ipt3 = ipt(itt3)
					if (ipl1==ipt1||ipl1==ipt2||ipl1==ipt3)
						if (ipl2==ipt1||ipl2==ipt2||ipl2==ipt3) {
							ntf = ntf+1
							itf(ntf) = itt3/3
							if (ntf==2)
								go to 170
							}
					}
				if (ntf<2)
					next 1
# - - determines the vertexes of the triangles that do not lie
# - - on the line segment.
				170  it1t3 = itf(1)*3
				ipti1 = ipt(it1t3-2)
				if (ipti1==ipl1||ipti1==ipl2) {
					ipti1 = ipt(it1t3-1)
					if (ipti1==ipl1||ipti1==ipl2)
						ipti1 = ipt(it1t3)
					}
				it2t3 = itf(2)*3
				ipti2 = ipt(it2t3-2)
				if (ipti2==ipl1||ipti2==ipl2) {
					ipti2 = ipt(it2t3-1)
					if (ipti2==ipl1||ipti2==ipl2)
						ipti2 = ipt(it2t3)
					}
# - - checks if the exchange is necessary.
				if (idxchg(xd,yd,ipti1,ipti2,ipl1,ipl2)!=0) {
# - - modifies the ipt array when necessary.
					ipt(it1t3-2) = ipti1
					ipt(it1t3-1) = ipti2
					ipt(it1t3) = ipl1
					ipt(it2t3-2) = ipti2
					ipt(it2t3-1) = ipti1
					ipt(it2t3) = ipl2
# - - sets new flags.
					jwl = jwl+8
					iwl(jwl-7) = ipl1
					iwl(jwl-6) = ipti1
					iwl(jwl-5) = ipti1
					iwl(jwl-4) = ipl2
					iwl(jwl-3) = ipl2
					iwl(jwl-2) = ipti2
					iwl(jwl-1) = ipti2
					iwl(jwl) = ipl1
					do jlt3 = 3,nlt3,3 {
						iplj1 = ipl(jlt3-2)
						iplj2 = ipl(jlt3-1)
						if (iplj1==ipl1&&iplj2==ipti2||iplj2==ipl1&&iplj1==ipti2)
							ipl(jlt3) = itf(1)
						if (iplj1==ipl2&&iplj2==ipti1||iplj2==ipl2&&iplj1==ipti1)
							ipl(jlt3) = itf(2)
						}
					}
				}
			nlfc = nlf
			nlf = jwl/2
			if (nlf==nlfc)
				break 1
# - - resets the iwl array for the next round.
			jwl = 0
			jwl1mn = (nlfc+1)*2
			nlft2 = nlf*2
			do jwl1 = jwl1mn,nlft2,2 {
				jwl = jwl+2
				iwl(jwl-1) = iwl(jwl1-1)
				iwl(jwl) = iwl(jwl1)
				}
			nlf = jwl/2
			}
		}
	}
# rearranges the ipt array so that the vertexes of each triangle
# are listed counter-clockwise.
do itt3 = 3,ntt3,3 {
	ip1 = ipt(itt3-2)
	ip2 = ipt(itt3-1)
	ip3 = ipt(itt3)
	if (side(xd(ip1),yd(ip1),xd(ip2),yd(ip2),xd(ip3),yd(ip3))<0.0) {
		ipt(itt3-2) = ip2
		ipt(itt3-1) = ip1
		}
	}
nt = nt0
nl = nl0
return
end



function  idxchg(x,y,i1,i2,i3,i4)
# this function determines whether or not the exchange of two
# triangles is necessary on the basis of max-min-angle criterion
# by c. l. lawson.
# the input parameters are
#     x,y = arrays containing the coordinates of the data
#           points,
#     i1,i2,i3,i4 = point numbers of four points p1, p2,
#           p3, and p4 that form a quadrilateral with p3
#           and p4 connected diagonally.
# this function returns an integer value 1 (one) when an ex-
# change is necessary, and 0 (zero) otherwise.
# declaration statements
dimension   x(100),y(100)
equivalence(c2sq,c1sq),(a3sq,b2sq),(b3sq,a1sq),(a4sq,b1sq),(b4sq,a2sq),(c4sq,c3sq)
# preliminary processing
x1 = x(i1)
y1 = y(i1)
x2 = x(i2)
y2 = y(i2)
x3 = x(i3)
y3 = y(i3)
x4 = x(i4)
y4 = y(i4)
# calculation
idx = 0
u3 = (y2-y3)*(x1-x3)-(x2-x3)*(y1-y3)
u4 = (y1-y4)*(x2-x4)-(x1-x4)*(y2-y4)
if (u3*u4>0.0) {
	u1 = (y3-y1)*(x4-x1)-(x3-x1)*(y4-y1)
	u2 = (y4-y2)*(x3-x2)-(x4-x2)*(y3-y2)
	a1sq = (x1-x3)**2+(y1-y3)**2
	b1sq = (x4-x1)**2+(y4-y1)**2
	c1sq = (x3-x4)**2+(y3-y4)**2
	a2sq = (x2-x4)**2+(y2-y4)**2
	b2sq = (x3-x2)**2+(y3-y2)**2
	c3sq = (x2-x1)**2+(y2-y1)**2
	s1sq = u1*u1/(c1sq*amax1(a1sq,b1sq))
	s2sq = u2*u2/(c2sq*amax1(a2sq,b2sq))
	s3sq = u3*u3/(c3sq*amax1(a3sq,b3sq))
	s4sq = u4*u4/(c4sq*amax1(a4sq,b4sq))
	if (amin1(s1sq,s2sq)<amin1(s3sq,s4sq))
		idx = 1
	}
idxchg = idx
return
end



subroutine  idplin(xd,yd,zd,nt,ipt,nl,ipl,iti,xii,yii,zii)
# this subroutine performs punctual interpolation (linear)
# determines the z value at a point.
# the input parameters are
#     xd,yd,zd = arrays of dimension ndp containing the x,
#           y, and z coordinates of the data points, where
#           ndp is the number of the data points,
#     nt  = number of triangles,
#     ipt = integer array of dimension 3*nt containing the
#           point numbers of the vertexes of the triangles,
#     nl  = number of border line segments,
#     ipl = integer array of dimension 3*nl containing the
#           point numbers of the end points of the border
#           line segments and their respective triangle
#           numbers,
#     iti = triangle number of the triangle in which lies
#           the point for which interpolation is to be
#           performed,
#     xii,yii = x and y coordinates of the point for which
#           interpolation is to be performed.
# the output parameter is
#     zii = interpolated z value.
# declaration statements
dimension   xd(100),yd(100),zd(100),ipt(585),ipl(300)
common/idpi/itpv
dimension   x(3),y(3),z(3)
# preliminary processing
it0 = iti
ntl = nt+nl
if (it0>ntl)	# outside data - set to na
	call setna(zii)
else {
# calculation of zii by interpolation.
# checks if the necessary coefficients have been calculated.
	if (it0!=itpv) {
# loads coordinate and partial derivative values at the
# vertexes.
		jipt = 3*(it0-1)
		do i = 1,3 {
			jipt = jipt+1
			idp = ipt(jipt)
			x(i) = xd(idp)
			y(i) = yd(idp)
			z(i) = zd(idp)
			}
# determines the coefficients for the coordinate system
# transformation from the x-y system to the u-v system
# and vice versa.
		x0 = x(1)
		y0 = y(1)
		z0 = z(1)
		a = x(2)-x0
		b = x(3)-x0
		c = y(2)-y0
		d = y(3)-y0
		dlt = a*d-c*b
		}
	dx = xii-x0
	dy = yii-y0
	bcoef = (a*dy-c*dx)/dlt
	acoef = (d*dx-b*dy)/dlt
	zii = z0+acoef*(z(2)-z0)+bcoef*(z(3)-z0)
	}
return
end



subroutine  idptip(xd,yd,zd,nt,ipt,nl,ipl,pdd,iti,xii,yii,zii,miss)
# this subroutine performs punctual interpolation or extrapola-
# tion, i.e., determines the z value at a point.
# the input parameters are
#     xd,yd,zd = arrays of dimension ndp containing the x,
#           y, and z coordinates of the data points, where
#           ndp is the number of the data points,
#     nt  = number of triangles,
#     ipt = integer array of dimension 3*nt containing the
#           point numbers of the vertexes of the triangles,
#     nl  = number of border line segments,
#     ipl = integer array of dimension 3*nl containing the
#           point numbers of the end points of the border
#           line segments and their respective triangle
#           numbers,
#     pdd = array of dimension 5*ndp containing the partial
#           derivatives at the data points,
#     iti = triangle number of the triangle in which lies
#           the point for which interpolation is to be
#           performed,
#     xii,yii = x and y coordinates of the point for which
#           interpolation is to be performed.
# the output parameter is
#     zii = interpolated z value.
#     if miss is true, zii is na outside of original data
# declaration statements
logical miss
dimension   xd(100),yd(100),zd(100),ipt(585),ipl(300),pdd(500)
common/idpi/itpv
dimension   x(3),y(3),z(3),pd(15),zu(3),zv(3),zuu(3),zuv(3),zvv(3)
real        lu,lv
equivalence(p5,p50)
# preliminary processing
it0 = iti
ntl = nt+nl
if (it0<=ntl) {
# calculation of zii by interpolation.
# checks if the necessary coefficients have been calculated.
	if (it0!=itpv) {
# loads coordinate and partial derivative values at the
# vertexes.
		jipt = 3*(it0-1)
		jpd = 0
		do i = 1,3 {
			jipt = jipt+1
			idp = ipt(jipt)
			x(i) = xd(idp)
			y(i) = yd(idp)
			z(i) = zd(idp)
			jpdd = 5*(idp-1)
			do kpd = 1,5 {
				jpd = jpd+1
				jpdd = jpdd+1
				pd(jpd) = pdd(jpdd)
				}
			}
# determines the coefficients for the coordinate system
# transformation from the x-y system to the u-v system
# and vice versa.
		x0 = x(1)
		y0 = y(1)
		a = x(2)-x0
		b = x(3)-x0
		c = y(2)-y0
		d = y(3)-y0
		ad = a*d
		bc = b*c
		dlt = ad-bc
		ap = d/dlt
		bp = -b/dlt
		cp = -c/dlt
		dp = a/dlt
# converts the partial derivatives at the vertexes of the
# triangle for the u-v coordinate system.
		aa = a*a
		act2 = 2.0*a*c
		cc = c*c
		ab = a*b
		adbc = ad+bc
		cd = c*d
		bb = b*b
		bdt2 = 2.0*b*d
		dd = d*d
		do i = 1,3 {
			jpd = 5*i
			zu(i) = a*pd(jpd-4)+c*pd(jpd-3)
			zv(i) = b*pd(jpd-4)+d*pd(jpd-3)
			zuu(i) = aa*pd(jpd-2)+act2*pd(jpd-1)+cc*pd(jpd)
			zuv(i) = ab*pd(jpd-2)+adbc*pd(jpd-1)+cd*pd(jpd)
			zvv(i) = bb*pd(jpd-2)+bdt2*pd(jpd-1)+dd*pd(jpd)
			}
# calculates the coefficients of the polynomial.
		p00 = z(1)
		p10 = zu(1)
		p01 = zv(1)
		p20 = 0.5*zuu(1)
		p11 = zuv(1)
		p02 = 0.5*zvv(1)
		h1 = z(2)-p00-p10-p20
		h2 = zu(2)-p10-zuu(1)
		h3 = zuu(2)-zuu(1)
		p30 = 10.0*h1-4.0*h2+0.5*h3
		p40 = -15.0*h1+7.0*h2-h3
		p50 = 6.0*h1-3.0*h2+0.5*h3
		h1 = z(3)-p00-p01-p02
		h2 = zv(3)-p01-zvv(1)
		h3 = zvv(3)-zvv(1)
		p03 = 10.0*h1-4.0*h2+0.5*h3
		p04 = -15.0*h1+7.0*h2-h3
		p05 = 6.0*h1-3.0*h2+0.5*h3
		lu = sqrt(aa+cc)
		lv = sqrt(bb+dd)
		thxu = atan2(c,a)
		thuv = atan2(d,b)-thxu
		csuv = cos(thuv)
		p41 = 5.0*lv*csuv/lu*p50
		p14 = 5.0*lu*csuv/lv*p05
		h1 = zv(2)-p01-p11-p41
		h2 = zuv(2)-p11-4.0*p41
		p21 = 3.0*h1-h2
		p31 = -2.0*h1+h2
		h1 = zu(3)-p10-p11-p14
		h2 = zuv(3)-p11-4.0*p14
		p12 = 3.0*h1-h2
		p13 = -2.0*h1+h2
		thus = atan2(d-c,b-a)-thxu
		thsv = thuv-thus
		aa = sin(thsv)/lu
		bb = -cos(thsv)/lu
		cc = sin(thus)/lv
		dd = cos(thus)/lv
		ac = aa*cc
		ad = aa*dd
		bc = bb*cc
		g1 = aa*ac*(3.0*bc+2.0*ad)
		g2 = cc*ac*(3.0*ad+2.0*bc)
		h1 = -aa*aa*aa*(5.0*aa*bb*p50+(4.0*bc+ad)*p41)-cc*cc*cc*(5.0*cc*dd*p05+(4.0*ad+bc)*p14)
		h2 = 0.5*zvv(2)-p02-p12
		h3 = 0.5*zuu(3)-p20-p21
		p22 = (g1*h2+g2*h3-h1)/(g1+g2)
		p32 = h2-p22
		p23 = h3-p22
		itpv = it0
		}
# converts xii and yii to u-v system.
	dx = xii-x0
	dy = yii-y0
	u = ap*dx+bp*dy
	v = cp*dx+dp*dy
# evaluates the polynomial.
	p0 = p00+v*(p01+v*(p02+v*(p03+v*(p04+v*p05))))
	p1 = p10+v*(p11+v*(p12+v*(p13+v*p14)))
	p2 = p20+v*(p21+v*(p22+v*p23))
	p3 = p30+v*(p31+v*p32)
	p4 = p40+v*p41
	zii = p0+u*(p1+u*(p2+u*(p3+u*(p4+u*p5))))
	}
else if (miss)	# set points outside original data to na - do not extrapolate
	call setna(zii)
else {
	il1 = it0/ntl
	il2 = it0-il1*ntl
	if (il1==il2) {
# calculation of zii by extrapolation in the rectangle.
# checks if the necessary coefficients have been calculated.
		if (it0!=itpv) {
# loads coordinate and partial derivative values at the end
# points of the border line segment.
			jipl = 3*(il1-1)
			jpd = 0
			do i = 1,2 {
				jipl = jipl+1
				idp = ipl(jipl)
				x(i) = xd(idp)
				y(i) = yd(idp)
				z(i) = zd(idp)
				jpdd = 5*(idp-1)
				do kpd = 1,5 {
					jpd = jpd+1
					jpdd = jpdd+1
					pd(jpd) = pdd(jpdd)
					}
				}
# determines the coefficients for the coordinate system
# transformation from the x-y system to the u-v system
# and vice versa.
			x0 = x(1)
			y0 = y(1)
			a = y(2)-y(1)
			b = x(2)-x(1)
			c = -b
			d = a
			ad = a*d
			bc = b*c
			dlt = ad-bc
			ap = d/dlt
			bp = -b/dlt
			cp = -bp
			dp = ap
# converts the partial derivatives at the end points of the
# border line segment for the u-v coordinate system.
			aa = a*a
			act2 = 2.0*a*c
			cc = c*c
			ab = a*b
			adbc = ad+bc
			cd = c*d
			bb = b*b
			bdt2 = 2.0*b*d
			dd = d*d
			do i = 1,2 {
				jpd = 5*i
				zu(i) = a*pd(jpd-4)+c*pd(jpd-3)
				zv(i) = b*pd(jpd-4)+d*pd(jpd-3)
				zuu(i) = aa*pd(jpd-2)+act2*pd(jpd-1)+cc*pd(jpd)
				zuv(i) = ab*pd(jpd-2)+adbc*pd(jpd-1)+cd*pd(jpd)
				zvv(i) = bb*pd(jpd-2)+bdt2*pd(jpd-1)+dd*pd(jpd)
				}
# calculates the coefficients of the polynomial.
			p00 = z(1)
			p10 = zu(1)
			p01 = zv(1)
			p20 = 0.5*zuu(1)
			p11 = zuv(1)
			p02 = 0.5*zvv(1)
			h1 = z(2)-p00-p01-p02
			h2 = zv(2)-p01-zvv(1)
			h3 = zvv(2)-zvv(1)
			p03 = 10.0*h1-4.0*h2+0.5*h3
			p04 = -15.0*h1+7.0*h2-h3
			p05 = 6.0*h1-3.0*h2+0.5*h3
			h1 = zu(2)-p10-p11
			h2 = zuv(2)-p11
			p12 = 3.0*h1-h2
			p13 = -2.0*h1+h2
			p21 = 0.0
			p23 = -zuu(2)+zuu(1)
			p22 = -1.5*p23
			itpv = it0
			}
# converts xii and yii to u-v system.
		dx = xii-x0
		dy = yii-y0
		u = ap*dx+bp*dy
		v = cp*dx+dp*dy
# evaluates the polynomial.
		p0 = p00+v*(p01+v*(p02+v*(p03+v*(p04+v*p05))))
		p1 = p10+v*(p11+v*(p12+v*p13))
		p2 = p20+v*(p21+v*(p22+v*p23))
		zii = p0+u*(p1+u*p2)
		}
	else {
# calculation of zii by extrapolation in the triangle.
# checks if the necessary coefficients have been calculated.
		if (it0!=itpv) {
# loads coordinate and partial derivative values at the vertex
# of the triangle.
			jipl = 3*il2-2
			idp = ipl(jipl)
			x(1) = xd(idp)
			y(1) = yd(idp)
			z(1) = zd(idp)
			jpdd = 5*(idp-1)
			do kpd = 1,5 {
				jpdd = jpdd+1
				pd(kpd) = pdd(jpdd)
				}
# calculates the coefficients of the polynomial.
			p00 = z(1)
			p10 = pd(1)
			p01 = pd(2)
			p20 = 0.5*pd(3)
			p11 = pd(4)
			p02 = 0.5*pd(5)
			itpv = it0
			}
# converts xii and yii to u-v system.
		u = xii-x(1)
		v = yii-y(1)
# evaluates the polynomial.
		p0 = p00+v*(p01+v*p02)
		p1 = p10+v*p11
		zii = p0+u*(p1+u*p20)
		}
	}
return
end



subroutine  idsfft(md,ncp,ndp,xd,yd,zd,nxi,nyi,xi,yi,zi,iwk,wk,miss)
# this subroutine performs smooth surface fitting when the pro-
# jections of the data points in the x-y plane are irregularly
# distributed in the plane.
# *** modified by rab for
# linear interpolation if ncp=0
# na's returned outside convex hull of data if miss is true
# the input parameters are
#     md  = mode of computation (must be 1, 2, or 3),
#         = 1 for new ncp and/or new xd-yd,
#         = 2 for old ncp, old xd-yd, new xi-yi,
#         = 3 for old ncp, old xd-yd, old xi-yi,
#     ncp = number of additional data points used for esti-
#           mating partial derivatives at each data point
#           (must be 2 or greater, but smaller than ndp),
#            or else 0 implying linear interpolation
#     ndp = number of data points (must be 4 or greater),
#     xd  = array of dimension ndp containing the x
#           coordinates of the data points,
#     yd  = array of dimension ndp containing the y
#           coordinates of the data points,
#     zd  = array of dimension ndp containing the z
#           coordinates of the data points,
#     nxi = number of output grid points in the x coordinate
#           (must be 1 or greater),
#     nyi = number of output grid points in the y coordinate
#           (must be 1 or greater),
#     xi  = array of dimension nxi containing the x
#           coordinates of the output grid points,
#     yi  = array of dimension nyi containing the y
#           coordinates of the output grid points.
# the output parameter is
#     zi  = doubly-dimensioned array of dimension (nxi,nyi),
#           where the interpolated z values at the output
#           grid points are to be stored.
# the other parameters are
#     iwk = integer array of dimension
#              max0(31,27+ncp)*ndp+nxi*nyi
#           used internally as a work area,
#     wk  = array of dimension 5*ndp used internally as a
#           work area.
#  miss   = logical - should missing values be assigned if
#           interpolated point is outside of convex hull of data
# the very first call to this subroutine and the call with a new
# ncp value, a new ndp value, and/or new contents of the xd and
# yd arrays must be made with md=1.  the call with md=2 must be
# preceded by another call with the same ncp and ndp values and
# with the same contents of the xd and yd arrays.  the call with
# md=3 must be preceded by another call with the same ncp, ndp,
# nxi, and nyi values and with the same contents of the xd, yd,
# xi, and yi arrays.  between the call with md=2 or md=3 and its
# preceding call, the iwk and wk arrays must not be disturbed.
# use of a value between 3 and 5 (inclusive) for ncp is recom-
# mended unless there are evidences that dictate otherwise.
# this subroutine calls the idcldp, idgrid, idpdrv, idptip, and
# idtang subroutines.
# declaration statements
logical miss
dimension   xd(100),yd(100),zd(100),xi(101),yi(101),zi(10201),iwk(13301),wk(500)
common/idpi/itpv
# setting of some input parameters to local variables (for md=1,2,3)
md0 = md
ncp0 = ncp
ndp0 = ndp
nxi0 = nxi
nyi0 = nyi
# error check.  (for md=1,2,3)
if( md0<1 | md0>3 | ncp0<0 | ncp0==1 | ncp0>=ndp0 | ndp0<4 | nxi0<1 | nyi0<1)
	ERROR(Bad input values)
if (md0<2) {
	iwk(1) = ncp0
	iwk(2) = ndp0
	}
else {
	ncppv = iwk(1)
	ndppv = iwk(2)
	if (ncp0!=ncppv | ndp0!=ndppv) ERROR(Bad input values)
	}
if (md0<3) {
	iwk(3) = nxi0
	iwk(4) = nyi0
	}
else {
	nxipv = iwk(3)
	nyipv = iwk(4)
	if (nxi0!=nxipv | nyi0!=nyipv) ERROR(Bad input values)
	}
# allocation of storage areas in the iwk array.  (for md=1,2,3)
jwipt = 16
jwiwl = 6*ndp0+1
jwngp0 = jwiwl-1
jwipl = 24*ndp0+1
jwiwp = 30*ndp0+1
jwipc = 27*ndp0+1
jwigp0 = max0(31,27+ncp0)*ndp0
if (md0<=1) {	# triangulates the x-y plane.  (for md=1)
	call idtang(ndp0,xd,yd,nt,iwk(jwipt),nl,iwk(jwipl),iwk(jwiwl),iwk(jwiwp),wk)
	iwk(5) = nt
	iwk(6) = nl
	if (nt==0) return
	}
if (md0<=1&&ncp0!=0) {	# determines ncp points closest to each data point.  (for md=1)
	call idcldp(ndp0,xd,yd,ncp0,iwk(jwipc))
	if (iwk(jwipc)==0) return
	}
# sorts output grid points in ascending order of the triangle
# number and the border line segment number.  (for md=1,2)
if (md0!=3)
	call idgrid(xd,yd,nt,iwk(jwipt),nl,iwk(jwipl),nxi0,nyi0,xi,yi,iwk(jwngp0+1),iwk(jwigp0+1))
# estimates partial derivatives at all data points (for md=1,2,3)
if (ncp0!=0)
	call idpdrv(ndp0,xd,yd,zd,ncp0,iwk(jwipc),wk)
# interpolates the zi values.  (for md=1,2,3)
itpv = 0
jig0mx = 0
jig1mn = nxi0*nyi0+1
nngp = nt+2*nl
do jngp = 1,nngp {
	iti = jngp
	if (jngp>nt) {
		il1 = (jngp-nt+1)/2
		il2 = (jngp-nt+2)/2
		if (il2>nl)
			il2 = 1
		iti = il1*(nt+nl)+il2
		}
	jwngp = jwngp0+jngp
	ngp0 = iwk(jwngp)
	if (ngp0!=0) {
		jig0mn = jig0mx+1
		jig0mx = jig0mx+ngp0
		do jigp = jig0mn,jig0mx {
			jwigp = jwigp0+jigp
			izi = iwk(jwigp)
			iyi = (izi-1)/nxi0+1
			ixi = izi-nxi0*(iyi-1)
			if (ncp0==0)
				call idplin(xd,yd,zd,nt,iwk(jwipt),nl,iwk(jwipl),iti,xi(ixi),yi(iyi),zi(izi))
			else
				call idptip(xd,yd,zd,nt,iwk(jwipt),nl,iwk(jwipl),wk,iti,xi(ixi),yi(iyi),zi(izi),miss)
			}
		}
	jwngp = jwngp0+2*nngp+1-jngp
	ngp1 = iwk(jwngp)
	if (ngp1!=0) {
		jig1mx = jig1mn-1
		jig1mn = jig1mn-ngp1
		do jigp = jig1mn,jig1mx {
			jwigp = jwigp0+jigp
			izi = iwk(jwigp)
			iyi = (izi-1)/nxi0+1
			ixi = izi-nxi0*(iyi-1)
			if (ncp0==0)
				call idplin(xd,yd,zd,nt,iwk(jwipt),nl,iwk(jwipl),iti,xi(ixi),yi(iyi),zi(izi))
			else
				call idptip(xd,yd,zd,nt,iwk(jwipt),nl,iwk(jwipl),wk,iti,xi(ixi),yi(iyi),zi(izi),miss)
			}
		}
	}
return
end
