#include <stdio.h>
#include <signal.h>
#include <sys/param.h>
#include "S.h"
#ifdef ATT_UNIX
#include <sys/types.h>
#endif
#include <sys/times.h>
#include "y.tab.h"
#include "eval.h"
#include "options.h"
#include "math.h"

#define NOT_THERE(p) ((p)==NULL || (p)->mode==NULL || (p)->mode==MISSING)

/* structure describing a compressed index vector */
typedef struct ix {
	long len;		/* length of subset (for get) */
	long max;		/* largest element accessed (for set) */
	vector *i;		/* compressed index vector */
	vector *names;		/* names attribute of data (grows for set) */
	struct ix **d;		/* vector of ix's for array subscripting */
} ix;

vector *S_extract(), *S_replace(), *set_names(), *get_names(), *xact_comp();
vector *S_ext_extract();
int del_comp(), has_names();

static vector *get_mode(), *get_length(), *get_list_names();
static vector *get_dim(), *get_dollar(), *get_dimnames();
static vector *get_tsp(), *get_attr(), *get_levels(), *get_attrs();
static vector *set_mode(), *set_length(), *set_dim(), *set_dollar();
static vector *set_dimnames(), *set_tsp(), *set_levels(), *set_attrs();
static vector *subset(), *extract(), *vextract(), *replace(), *element();
static vector *subtree(), *attribute(), *add_nulls(), *Data();
static vector *ext_subset();
static ix null_ix, *make_index(), *vec_index(), *arr_index(), *alcix();
static ix *mat_index(), *lgl_index(), *int_index(), *neg_index();
static ix *char_index(), *full_index();
static long find_match(), get_length2(), max(), min();
static int int_cmp(), set_length2(), ok_dimnames(), overlap();
static char err_buf[80]; static void ok_levels();
static char *attr_name();

static char *attr_internal[] = {
	".Dim", ".Dimnames", ".Tsp", ".Names", ".Label", NULL
};
static char *attr_external[] = {
	"dim", "dimnames", "tsp", "names", "levels", NULL
};
static char *Extract_names[MAX_SUB_FUN] = {
	"[", "[", "[[", "attr", "mode", "length",
	"$", "names", "[", "dim", "dimnames", "tsp", 
	"levels", "attributes", "storage.mode"
};

vector *
S_extract(ent, arglist)
vector *ent, *arglist;
{
	long nargs = arglist->length;
	vector **args = arglist->value.tree, *value;

if(check){
	long n; int err=FALSE;
	for(n=0; n<nargs; n++)if(check_obj(args[n],NULL_STRING))err = n+1;
	if(err)
		PROBLEM "Bad object in argument %d", err RECOVER(ent);
}
	switch(sys_index) {
	case SUBSET_FUN:
		value = subset(args[0], &args[1], nargs-1, S_void);
		break;
	case DOLLAR_FUN:
		value = get_dollar(args[0], args[1]); break;
	case ELEMENT_FUN:
		value = element(args[0], &args[1], nargs-1, S_void);
		break;
	case ATTR_FUN:
		value = attribute(args[0], args[1], S_void); break;
	case MODE_FUN:
	case STORAGE_FUN:
		value = get_mode(args[0], sys_index==STORAGE_FUN); break;
	case LENGTH_FUN:
		value = get_length(args[0]); break;
	case NAMES_FUN:
		value = get_names(args[0]); break;
	case DIM_FUN:
		value = get_dim(args[0]); break;
	case DIMNAMES_FUN:
		value = get_dimnames(args[0]); break;
	case TSP_FUN:
		value = get_tsp(args[0]); break;
	case LEVELS_FUN:
		value = get_levels(args[0]); break;
	case ALL_ATTR_FUN:
		value = get_attrs(args[0]); break;
	default:
		MEANINGFUL(value);
		PROBLEM "Unknown internal index (%d) in extract", sys_index RECOVER(ent);
	}
	if(value->mode == STRUCTURE && value->length == 1)
		value = value->value.tree[0];
if(check) {
	if(check_obj(value,NULL_STRING))
		PROBLEM "Bad object in result" RECOVER(ent);
}
	return(value);
}

vector *
S_replace(ent, arglist)
vector *ent, *arglist;
{
	long nargs = arglist->length;
	vector **args = arglist->value.tree, *dataset = args[0];
	vector *repl_val = args[nargs-1], *value;
if(check){
	long n; int err=FALSE;
	for(n=0; n<nargs; n++)if(check_obj(args[n],NULL_STRING))err = n+1;
	if(err)
		PROBLEM "Bad object in argument %d", err RECOVER(ent);
}
	switch(sys_index) { /* some replacements protect the right side */
	case SUBSET_FUN: case ELEMENT_FUN:
	case MODE_FUN: case STORAGE_FUN: case LENGTH_FUN:
		break;
	default: /* the rest don't */
	 	if(repl_val && PRECIOUS(repl_val))
	 		repl_val = copy_data(repl_val, dataset->x.frame);
	}
	switch(sys_index) {
	case SUBSET_FUN:
		(void) subset(dataset, &args[1], nargs-2, repl_val); break;
	case DOLLAR_FUN:
		(void) set_dollar(dataset, args[1], repl_val); break;
	case ELEMENT_FUN:
		(void) element(dataset, &args[1], nargs-2, repl_val); break;
	case ATTR_FUN:
		if(nargs<3)
			PROBLEM "No attribute name specified in replacement" RECOVER(ent);
		(void) attribute(dataset, args[1], repl_val); break;
	case MODE_FUN:
	case STORAGE_FUN:
		(void) set_mode(dataset, repl_val); break;
	case LENGTH_FUN:
		(void) set_length(dataset, repl_val); break;
	case NAMES_FUN:
		(void) set_names(dataset, repl_val); break;
	case DIM_FUN:
		(void) set_dim(dataset, repl_val); break;
	case DIMNAMES_FUN:
		(void) set_dimnames(dataset, repl_val); break;
	case TSP_FUN:
		(void) set_tsp(dataset,repl_val); break;
	case LEVELS_FUN:
		(void) set_levels(dataset,repl_val); break;
	case ALL_ATTR_FUN:
		(void) set_attrs(dataset,repl_val); break;
	default:
		PROBLEM "Error in replacement: unknown index %d", sys_index RECOVER(ent);
	}
	if(dataset->mode == STRUCTURE && dataset->length == 1)
		dataset = dataset->value.tree[0];
if(check) {
	if(check_obj(dataset,NULL_STRING))
		PROBLEM "Bad object in result" RECOVER(ent);
}
	return(dataset);
}

static vector *
subset(data, subs, nsub, replacement)
vector *data, **subs, *replacement;
long nsub;
{
	ix *i;
	int drop;

	if(VOID(data))
		return(blt_in_NULL);
	drop = VOID(replacement) ? subs[--nsub]->value.Long[0] : 0;
	if(!VOID(replacement))
		set_levels(data, blt_in_NULL);
	switch(Data_mode(data)) {
	case NAME:
	case SYSTEM:
		PROBLEM "Subsets not meaningful for mode \"%s\"",
			token_name(data->mode) WARNING(data);
		return(data);
	}
	if(VOID(replacement) && nsub < 1)
		return(data);
	i = make_index(data, subs, nsub, !VOID(replacement));
	if(i == &null_ix)
		return(data);
	return(VOID(replacement)
			? extract(data, i, drop)
			: replace(data, i, replacement));
}

static ix *
make_index(data, subs, nsub, replace)
vector *data, **subs;
long nsub;
int replace;
{
	vector *vec, *names, *dim, *dimnames, *index;
	ix *i = &null_ix;

	vec = Data(data);
	if(VOID(vec))
		return(i);
	names = get_names(data);

	/* 0 subscripts -- equivalent to all the data */
	if(nsub == 0)
		i = full_index(vec->length);

	/* 1 subscript -- index for vector or matrix index for array */
	else if(nsub == 1) {
		index = subs[0];
		if(index->mode == STRUCTURE && data->mode == STRUCTURE)
			i = mat_index(vec->length, get_dim(data), index);
		if(i == &null_ix) {
			index = coevec(index, ANY, FALSE, CHECK_IT);
			if(VOID(index))
				return(&null_ix);
			i = vec_index(index, vec->length, names, replace);
		}
	}

	/* more than 1 subscript -- index for array */
	else {
		dim = get_dim(data);
		if(NOT_THERE(dim))
			PROBLEM "No dim attribute for array subset" RECOVER(data);
		if(nsub != dim->length)
			PROBLEM "Need %ld subscripts for array", dim->length RECOVER(dim);
		dimnames = get_dimnames(data);
		i = arr_index(subs, nsub, dim, dimnames, replace);
	}

	if(i != &null_ix && VOID(i->names))
		i->names = names;
	return(i);
}

static ix *
full_index(length)
long length;
{
	ix *new = alcix();

	new->len = length;
	new->max = length;
	if(length == 0)
		new->i = alcvec(INT, 0L);
	else {
		new->i = alcvec(INT, 2L);
		new->i->value.Long[0] = -length;
		new->i->value.Long[1] = 0;
	}
	return(new);
}
	
static ix *
vec_index(index, length, names, replace)
vector *index, *names;
long length;
int replace;
{
	switch(index->mode) {
	case MISSING:
		return(full_index(length));
	case CHAR:
		return(char_index(index, length, names, replace));
	case LGL:
		return(lgl_index(index, length));
	default:
		index = coevec(index, INT, FALSE, CHECK_IT);
		if(VOID(index))
			PROBLEM "Invalid mode \"%s\" for vector subscript",
				token_name(index->mode) RECOVER(NULL_ENTRY);
		return(int_index(index, length));
	}
}

#define RECORD		if(last>=0){span>1?(*to++= -span,*to++=start):(*to++=start);len+=span;}
#define NA_RECORD	RECORD na_set(to); to++; len++; last= -2;

static ix *
char_index(index, length, names, replace)
vector *index, *names;
long length;
int replace;
{
	ix *new = alcix();
	char **from;
	long *to, i, start, span, last = -2, cur;
	long ni = index->length, len = 0, max = length;
	MEANINGFUL(start); MEANINGFUL(span);

if(check) {sanity(index, "Vector for character subset");}

	new->i = PRECIOUS(index) ? alcvec(INT, ni) : index;
	new->i->mode = INT;
	from = index->value.Char;
	to = new->i->value.Long;
	if(VOID(names) || NOT_THERE(names))
		new->names = names = alcvec(CHAR, length);
	for(i = 0; i < ni; i++, from++) {
		cur = find_match(*from, names, replace);
		if(cur < 0) {
			if(PRECIOUS(names))
				new->names = names = copy_data(names, NULL_ENTRY);
			append_data(names, cur = max++, 1L, (char *)from);
		}
		if(cur == last+1)
			span++;
		else {
			RECORD
			start = cur;
			span = 1;
		}
		last = cur;
	}
	RECORD
	new->len = len;
	new->max = max;
	new->i->length = to - new->i->value.Long;
	return(new);
}

/*
 * Look up target in the table of character strings.
 * If an exact match is found, return its index in the table.
 * Otherwise, if exact is 1 then return -1 (no match).
 * Otherwise, if exactly 1 partial match is found return its index.
 * Otherwise, return -1.
 */
static long
find_match(target, table, exact)
char *target;
vector *table;
int exact;
{
	char **s = table->value.Char;
	long n = table->length, i, index = -1;
	int nt = strlen(target), npartial = 0;

	if(nt == 0)
		return(-1);
	for(i = 0; i < n; i++, s++)
		if(*s && strncmp(target, *s, nt) == 0) {
			if(strcmp(target, *s) == 0)
				return(i);
			index = i;
			npartial++;
		}
	return((exact || (npartial > 1)) ? -1 : index);
}

static ix *
lgl_index(index, length)
vector *index;
long length;
{
	ix *new = alcix();
	long *from, *to, i, start, span, last;
	long ni = index->length, len ;
	MEANINGFUL(start); MEANINGFUL(span);

	if(ni == 0)
		length = 0;
	else if(ni > length)
		length = ni;
	new->i = (ni < length || PRECIOUS(index)) ? alcvec(INT, length) : index;
	from = index->value.Long;
	to = new->i->value.Long;
	last = -2; len = 0; 
	for(i = 0; i < length; i++, from++) {
		if(i%ni == 0)
			from = index->value.Long;
		if(is_na(from)) {
			NA_RECORD
			continue;
		}
		if(*from)
			if(last >= 0)
				span++;
			else {
				last = start = i;
				span = 1;
			}
		else {
			RECORD
			last = -2;
		}
	}
	RECORD
	new->len = len;
	new->max = length;
	new->i->length = to - new->i->value.Long;
	return(new);
}

static ix *
int_index(index, length)
vector *index;
long length;
{
	ix *new = alcix();
	long *from, *to, i, start, span, cur, last = -2, nzero = 0, nneg = 0;
	long ni = index->length, len = 0, max = 0;
	MEANINGFUL(start); MEANINGFUL(span);

	new->i = PRECIOUS(index) ? copy_data(index, NULL_ENTRY) : index;
	from = index->value.Long;
	to = new->i->value.Long;
	for(i = 0; i < ni; i++, from++) {
		if(is_na(from)) {
			NA_RECORD
			continue;
		}
		cur = *from;
		if(cur == 0) {
			nzero++;
			continue;
		}
		if(cur < 0) {
			nneg++;
			*to++ = -cur - 1;
			continue;
		}
		if(cur > max)
			max = cur;
		if(cur == last+1)
			span++;
		else {
			RECORD
			start = cur - 1;
			span = 1;
		}
		last = cur;
	}
	RECORD
	new->len = len;
	new->max = max;
	new->i->length = to - new->i->value.Long;
	if(nneg > 0 && nneg+nzero != ni)
		PROBLEM "Only 0's may be mixed with negative subscripts" RECOVER(NULL_ENTRY);
	if(nneg == 0)
		return(new);
	return(neg_index(new, length));
}

static ix *
neg_index(index, length)
ix *index;
long length;
{
	ix *new = alcix();
	long *from, *to, i, start, span, cur, last;
	long ni = index->i->length, len = 0, max;
	MEANINGFUL(start); MEANINGFUL(span);

	QSORT(index->i->value.Long, ni, int_cmp);
	new->i = alcvec(INT, 2*(ni+1));	/* worst case */
	from = index->i->value.Long;
	to = new->i->value.Long; last = -1;
	for(i = 0; i < ni; i++, from++) {
		cur = *from;
		if(cur >= length)
			break;
		if(cur - last > 1){
			start = last + 1;
			span = cur - start;
			span>1?(*to++ = -span,*to++ = start):(*to++ = start);
			len += span;
		}
		last = cur;
	}
	max = start + span;
	if(last < length - 1) {
		start = last + 1;
		span = length - start;
		span>1?(*to++ = -span,*to++ = start):(*to++ = start);
		len += span;
		max = length;
	}
	new->len = len;
	new->max = max;
	new->i->length = to - new->i->value.Long;
	return(new);
}

static ix *
mat_index(length, dim, sub)
long length;
vector *dim, *sub;
{
	vector *sdim = get_dim(sub), *val;
	long *dims, *sdims, *svals, n, *vals, nrow, *rowstart, rc1;

	if(NOT_THERE(dim) || NOT_THERE(sdim) || sdim->length != 2)
		  return(&null_ix);
	sdims = sdim->value.Long;
	if(sdims[1] != dim->length)
		return(&null_ix);
	sub = Data(sub);
	switch(sub->mode) {
	case INT:
		break;
	case DOUBLE:
	case REAL:
		sub = coevec(sub, INT, FALSE, PRECIOUS(sub));
		break;
	default:
		return(&null_ix);
	}
	dims = dim->value.Long + dim->length - 1;
	rowstart = sub->value.Long;
	nrow = sdims[0];
	rc1 = nrow*(sdims[1]-1);
	val = alcvec(INT, nrow);
	vals = val->value.Long;
	for(n = 0; n < nrow; n++, rowstart++) {
		long i = 0, *dd = dims, *sd;
		for(sd = rowstart + rc1; sd >= rowstart; sd -= nrow) {
			if(*sd < 0)
				PROBLEM "Negative entry in a subscript matrix" RECOVER(NULL_ENTRY);
			if(*sd == 0) {
				i = -1;
				break;
			}
			if(is_na(sd)) {
				for(sd -= nrow; sd >= rowstart; sd -= nrow)
					if(*sd == 0) {
						i = -1;
						break;
					}
				if(i >= 0)
					na_set(&i);
				break;
			}
			i = ((*sd)-1) + (*dd--)*i;
		}
		if(is_na(&i)) {
			na_set(vals);
			vals++;
		} else if(++i > 0)
			*vals++ = i;
	}
	val->length = vals - val->value.Long;
	return(int_index(val, length));
}

static
int_cmp(i, j)
long *i, *j;
{
	return(*i - *j);
}

static ix *
arr_index(index, nindex, dim, dimnames, replace)
vector **index, *dim, *dimnames;
long nindex;
int replace;
{
	ix *new;
	vector *names, **dimnamess, *subi;
	long *dims, size = 1, max = 0, i, *to, last, bigspan, t;
	long **inds, **begins, *lens, *starts, *spans;
	long **ind, **begin, *len, *start, *span;
	long *dimend, **indend, *startend, *spanend, *d;

	/* collect subscript vectors for each dimension */
	new = alcix();
	new->d = (ix **)S_alloc(nindex, sizeof(ix *));
	dims = dim->value.Long;
	MEANINGFUL(dimnamess);
	if(!NOT_THERE(dimnames))
		dimnamess = dimnames->value.tree;
	for(i = 0; i < nindex; i++) {
		coeves(index[i], ANY, FALSE, FALSE, &subi);
		if(subi->mode == CHAR) {
			if(NOT_THERE(dimnames))
				PROBLEM "No dimnames attribute for array subset" RECOVER(NULL_ENTRY);
			if(dimnamess[i]->length == 0)
				PROBLEM "No dimnames for dimension %ld", i+1 RECOVER(NULL_ENTRY);
		}
		names = NOT_THERE(dimnames) ? blt_in_NULL : dimnamess[i];
		new->d[i] = vec_index(subi, dims[i], names, replace);
		if(new->d[i]->max > dims[i])
			PROBLEM "Array subscript (%d) out of bounds, should be at most %d", new->d[i]->max, dims[i] RECOVER(NULL_ENTRY);
		if(VOID(new->d[i]->names))
			new->d[i]->names = names;
		size *= new->d[i]->len;
	}

	/* convert tuples to linear indexes; use pointers for speed */
	inds = (long **)S_alloc(nindex, sizeof(long *));
	begins = (long **)S_alloc(nindex, sizeof(long));
	lens = (long *)S_alloc(nindex, sizeof(long));
	starts = (long *)S_alloc(nindex, sizeof(long));
	spans = (long *)S_alloc(nindex, sizeof(long));
	for(i = 0; i < nindex; i++) {
		inds[i] = begins[i] = new->d[i]->i->value.Long;
		lens[i] = new->d[i]->i->length;
	}
	dimend = &dims[nindex];
	indend = &inds[nindex];
	startend = &starts[nindex];
	spanend = &spans[nindex];

	/* conversion */
	new->i = alcvec(INT, size);
	to = new->i->value.Long;
	last = -2;
	span=spans; start=starts; ind=inds;
	for(; span<spanend; span++, start++, ind++){
		*start = *(*ind)++;
		*span = 1;
		if(!is_na(start) && *start<0) {
			*span = -*start;
			*start = *(*ind)++;
		}
	}
	MEANINGFUL(bigspan);
	for(i = 0; i < size; i++) {
		d = dimend; start = startend;
		t = 0;
		while(--start >= starts) {
			if(is_na(start)) { na_set(&t); break; }
			t *= *--d;
			t += *start;
		}
		if(is_na(&t)) {
			if(last>=0){
				if(bigspan > 1) *to++ = -bigspan;
				*to++ = last - bigspan + 1;
			}
			na_set(to);to++;
			last = -2;
		} else if(t == last+1){
			bigspan++;
			if(t > max) max = t;
			last = t;
		} else {
			if(last >= 0){
				if(bigspan > 1) *to++ = -bigspan;
				*to++ = last - bigspan + 1;
			}
			bigspan = 1;
			if(t > max) max = t;
			last = t;
		}

		(*spans)--;
		if(*spans>0) (*starts)++;
		ind = inds; begin = begins; len = lens;
		span = spans; start = starts;
		for(; *span==0 && ind < indend; ind++, begin++, len++, span++, start++) {
			if(*ind - *begin >= *len) {
				*ind = *begin;
				if(ind < indend-1) {
					(*(span+1))--;
					if(*(span+1)>0)
						(*(start+1))++;
				}
				else if(i != size-1)
					PROBLEM "Problem with subset" RECOVER(NULL_ENTRY);
			}
			*span = 1;
			*start = *(*ind)++;
			if(!is_na(start) && *start<0) {
				*span = -*start;
				*start = *(*ind)++;
			}
		}
	
	}

	if(last >= 0) {
		if(bigspan > 1)
			*to++ = -bigspan;
		*to++ = last - bigspan + 1;
	}
	new->len = size;
	new->max = max;
	new->i->length = to - new->i->value.Long;
	if(new->i->length > size)
		PROBLEM "Array subscripting failed" RECOVER(NULL_ENTRY);
	return(new);
}

static ix *
alcix()
{
	ix *a = (ix *)S_alloc(1L, sizeof(ix));

	a->i = a->names = S_void;
	a->d = (ix **)0;
	return(a);
}

static vector *
extract(data, index, drop)
vector *data;
ix *index;
int drop;
{
	vector *subset, *dim, *dimnames, *names, **t, **tt, *lev;
	ix **d;
	long ndim, i, *l, *ll, which;

	lev = get_levels(data);
	subset = vextract(coevec(data, ANY, FALSE, PRECIOUS(data)), index);
	if(subset->length == 0)
		return(subset);
	if(!NOT_THERE(index->names) && atomic_type(subset->mode))
		set_names(subset, vextract(add_nulls(index), index));
	if(index->d) {
		dim = get_dim(data);
		ndim = dim->length;
		if(ndim <= 1)
			goto wrapup;
		if(PRECIOUS(dim))
			dim = copy_data(dim, NULL_ENTRY);
		l = dim->value.Long;
		d = index->d;
		for(i = 0; i < ndim; i++, d++, l++)
			*l = (*d)->len;
		set_dim(subset, dim);
		dimnames = get_dimnames(data);
		if(!NOT_THERE(dimnames)) {
			if(PRECIOUS(dimnames))
				dimnames = copy_data(dimnames, NULL_ENTRY);
			t = dimnames->value.tree;
			d = index->d;
			for(i = 0; i < ndim; i++, d++, t++)
				if((*d)->len >= 1 && (*t) != NULL && (*t)->length > 0)
					*t = vextract(*t, *d);
			set_dimnames(subset, dimnames);
		}
		if(!drop)
			goto wrapup;
		l = ll = dim->value.Long;
		MEANINGFUL(t); MEANINGFUL(tt);
		if(!NOT_THERE(dimnames))
			t = tt = dimnames->value.tree;
		for(i = 0; i < ndim; i++) {
			if(*l > 1) {
				*ll++ = *l;
				if(!NOT_THERE(dimnames))
					*tt++ = *t;
			}
			l++;
			if(!NOT_THERE(dimnames))
				t++;
		}
		ndim = ll - dim->value.Long;
		if(ndim == dim->length)
			goto wrapup;
		set_dim(subset, blt_in_NULL);
		if(ndim > 1) {
			dim->length = ndim;
			set_dim(subset, dim);
			if(!NOT_THERE(dimnames)) {
				dimnames->length = ndim;
				set_dimnames(subset, dimnames);
			}
		} else if(!NOT_THERE(dimnames)) { /* try to copy dimnames to names */
			names = get_attr(subset, ".Names", 1);
			if(!NOT_THERE(names))
				goto wrapup;
			which = -1;
			if(ndim == 0) {
				ndim = dimnames->length;
				t = dimnames->value.tree;
				for(i = 0; i < ndim; i++, t++) {
					if((*t)->length == 0)
						continue;
					if(which >= 0) {
						which = -1;
						break;
					} else
						which = i;
				}
			} else
				if(dimnames->value.tree[0]->length > 0)
					which = 0;
			if(which >= 0)
				set_names(subset, dimnames->value.tree[which]);
		}
	}
wrapup:
	if(lev != blt_in_NULL)
		set_levels(subset, lev);
	return(subset);
}

#define EXTRACT(type, member, naaction) { \
	type *from = data->value.member; \
	type *to = subset->value.member; \
	long start, span, ncopy, n, nna; \
	long *I = index->i->value.Long, *Iend = I + index->i->length; \
 \
	while(I < Iend) { \
		if(is_na(I)) {naaction; I++; continue;} \
		if((n = *I++) >= 0) { \
			if(n < length){ \
				if(is_na(&from[n])){ na_set(to);to++;} \
				else *to++ = from[n]; \
			} else {\
				naaction; \
			}\
			continue; \
		} \
		span = -n; start = *I++; \
		ncopy = min(span, max(0L, length-start)); nna = span - ncopy; \
		if(ncopy) { \
			MEMCPY(to, from+start, ncopy); \
			to += ncopy; \
		} \
		while(nna--){ naaction;} \
	} \
	break; \
}

static vector *
vextract(data, index)
vector *data;
ix *index;
{
	vector *subset;
	int mode;
	long length;

	if(VOID(data) || NOT_THERE(data) || index == &null_ix)
		return(data);
	mode = data->mode;
	length = data->length;
	subset = alcvec(mode, index->len);
	switch(mode) {
	case LGL:
	case INT:
		EXTRACT(long,Long,(na_set(to),to++))
	case REAL:
		EXTRACT(float,Float,(na_set(to),to++))
	case DOUBLE:
		EXTRACT(double,Double,(na_set(to),to++))
	case COMPLEX:
		EXTRACT(complex,Complex,(na_set(to),to++))
	case CHAR:
		/* following copy (but not the default case) could go away if
		/* strings were hashed, so the copied string pointers wouldn't hurt */
		if(PRECIOUS(data))subset->x.frame = Local_data;
		/* assumes cur_frame == Nframe; if this changes, must be cleverer */
		EXTRACT(char *,Char,*to++=NA_STRING)
	default: {
	/* Special code included to handle repeated subscripts referring to
	/* non-precious elements. The pointer used is marked precious
	/* temporarily. If there is a repeated use
	/* of the same element, the future uses will see precious & copy.
	/* The frame pointers are removed at the end.  The strategy is
	/* conservative, in all el's are copied if *any* are precious,
	/* but in practice, preciousness should be all-or-none */
#define COPY_EL(n) ((any_precious || PRECIOUS(from[n])) ? copy_data(from[n],NULL_ENTRY) \
	: ( from[n]->x.frame = Local_data, from[n]))
#define NA_EL copy_data(blt_in_NULL, NULL_ENTRY)
	vector * *from = data->value.tree, **to = subset->value.tree;
	long start, span, ncopy, n, nna,ni; int any_precious = FALSE;
	long *I , *Iend;
	ni=index->i->length;
	for(I= index->i->value.Long, Iend=I+ni; I<Iend;) { /* are any from el's precious?*/
		if(is_na(I)) {I++; continue;}
		if((n = *I++) >= 0) {
			if( (n < length) && PRECIOUS(from[n]))any_precious = TRUE;
			continue;
		}
		span = -n; start = *I++;
		ncopy = min(span, max(0L, length-start));
		if(ncopy) {
			long i, end = start+ncopy;
			for(i=start; i<end; i++) if( PRECIOUS(from[i]))any_precious = TRUE;
		}
	}
	for(I= index->i->value.Long, Iend=I+ni; I<Iend;) {
		if(is_na(I)) {*to++ = NA_EL; I++; continue;}
		if((n = *I++) >= 0) {
			*to++ = (n < length) ? COPY_EL(n) : NA_EL;
			continue;
		}
		span = -n; start = *I++;
		ncopy = min(span, max(0L, length-start)); nna = span - ncopy;
		if(ncopy) {
			long i, end = start+ncopy;
			for(i=start; i<end; i++) *to++ = COPY_EL(i);
		}
		while(nna--) *to++ = NA_EL;
	}
	if(!any_precious) { /* remove the frame pointers used to force copy*/
		ncopy = to - subset->value.tree; to = subset->value.tree;
		for(n = 0; n < ncopy; n++, to++)
			(*to)->x.frame = NULL_ENTRY;
	}
	break;
}
	}
	return(subset);
}

#ifdef __STDC__
#define REPLACE(type, member) { \
	type *member##from = source->value.member; \
	type *member##end = member##from + source->length; \
	type *member##to = target->value.member; \
	I = index->i->value.Long; Iend = I + index->i->length; \
 \
	while(I < Iend) { \
		if(is_na(I)) {I++; continue;} \
		if((n = *I++) >= 0) { \
			if(is_na(member##from)) na_set(member##to + n); \
			else member##to[n] = *member##from; \
			if(++member##from >= member##end) member##from = source->value.member; \
			continue; \
		} \
		span = -n; start = *I++; \
		while(span > 0) { \
			ncopy = min(span, (long)(member##end-member##from)); \
			MEMCPY(member##to+start, member##from, ncopy); \
			if((member##from += ncopy) >= member##end) member##from = source->value.member; \
			span -= ncopy; start += ncopy; \
		} \
	} \
	if(member##from  != source->value.member ) \
		PROBLEM "Replacement length not a multiple of number of elements to replace" WARNING(NULL_ENTRY); \
	break; \
}
#else
#define REPLACE(type, member) { \
	type *member/**/from = source->value.member; \
	type *member/**/end = member/**/from + source->length; \
	type *member/**/to = target->value.member; \
	I = index->i->value.Long; Iend = I + index->i->length; \
 \
	while(I < Iend) { \
		if(is_na(I)) {I++; continue;} \
		if((n = *I++) >= 0) { \
			if(is_na(member/**/from)) na_set(member/**/to + n); \
			else member/**/to[n] = *member/**/from; \
			if(++member/**/from >= member/**/end) member/**/from = source->value.member; \
			continue; \
		} \
		span = -n; start = *I++; \
		while(span > 0) { \
			ncopy = min(span, (long)(member/**/end-member/**/from)); \
			MEMCPY(member/**/to+start, member/**/from, ncopy); \
			if((member/**/from += ncopy) >= member/**/end) member/**/from = source->value.member; \
			span -= ncopy; start += ncopy; \
		} \
	} \
	if(member/**/from  != source->value.member) \
		PROBLEM "Replacement length not a multiple of number of elements to replace" WARNING(NULL_ENTRY); \
	break; \
}
#endif
	
static vector *
replace(data, index, replacement)
vector *data, *replacement;
ix *index;
{
	vector *source, *target;
	vector **from, **to;
	int mode;
	long length, start, span, ncopy, n, i, nfrom, end;
	long *I, *Iend;
	char **names, *el_name;

	if(VOID(data) || index == &null_ix)
		return(S_void);
	source = coevec(replacement, ANY, FALSE, PRECIOUS(replacement));
	if(source->length <= 0)
		return(replacement);
	target = coevec(data, ANY, FALSE, FALSE);
	mode = target->mode;
	if(mode != source->mode) {
		mode = coerce_to(mode, source->mode);
		if(target->mode != mode)
			target = coevec(target, mode, FALSE, FALSE);
		if(source->mode != mode)
			source = coevec(source, mode, FALSE, PRECIOUS(replacement));
	}
	if((PRECIOUS(source) && !atomic_type(mode)) || overlap(target,source))
		source = copy_data(source,NULL_ENTRY);
	length = target->length;
	if(index->max > length)
		set_length2(target, index->max);
	switch(mode) {
	case LGL:
	case INT:
		REPLACE(long,Long)
	case REAL:
		REPLACE(float,Float)
	case DOUBLE:
		REPLACE(double,Double)
	case COMPLEX:
		REPLACE(complex,Complex)
	case CHAR:
		REPLACE(char *,Char)
	default:
		from = source->value.tree;
		nfrom = 0; end = source->length;
		I = index->i->value.Long;
		Iend = I + index->i->length;
		names = index->names == blt_in_NULL ? 0 : add_nulls(index)->value.Char;
		while(I < Iend) {
			if(is_na(I)) {I++; continue;}
			if((n = *I++) >= 0) {
				to = target->value.tree + n;
				el_name = n >= length
					? (names ? names[n] : "")
					: (*to)->name;
				*to = copy_data(from[nfrom], data->x.frame);
				if(IS_NULL_STRING((*to)->name) &&
				  !IS_NULL_STRING(el_name))
					(*to)->name = el_name;
				if(++nfrom >= end)
					nfrom = 0;
				continue;
			}
			span = -n; n = *I++; to = target->value.tree + n;
			while(span > 0) {
				ncopy = min(span, (long)(end-nfrom));
				for(i = 0; i < ncopy; i++) {
					el_name = n >= length
						? (names ? names[n] : "")
						: (*to)->name;
					(*to) = copy_data(from[nfrom], data->x.frame);
					if(IS_NULL_STRING((*to)->name) &&
					  !IS_NULL_STRING(el_name))
						(*to)->name = el_name;
					nfrom++; to++; n++;
				}
				if(nfrom >= end)
					nfrom = 0;
				span -= ncopy;
			}
		}
		if(nfrom != 0)
			PROBLEM "Replacement length not a multiple of number of elements to replace" WARNING(NULL_ENTRY);
		break;
	}
	if(NOT_RECURSIVE(mode) && !NOT_THERE(index->names))
		set_names(data, add_nulls(index));
	return(replacement);
}

long
x_which_comp(name,data)
char *name; vector *data;
{
	vector **p; long n;
	if(!data)return(0);
	for(n=0, p = data->value.tree; n<data->length; n++, p++)
		if(name_eq(name, (*p)->name))return(n+1);
	return(0);
}

vector *
xact_comp(data,name)
vector *data; char *name;
{
	vector **p; long n;
	if(!data)return(NULL_ENTRY);
	if(NOT_RECURSIVE(data->mode)) /* not data_mode(data) - assumes caller did it */
		PROBLEM "Component asked from atomic data" RECOVER(data);
	for(n=0, p = data->value.tree; n<data->length; n++, p++)
		if(name_eq(name, (*p)->name))return(*p);
	return(NULL_ENTRY);
}

static vector *
get_dollar(data, comp)
vector *data, *comp;
{
	char *name; long mode; vector *vec;
	mode = data->mode; vec = data;
	if(mode == STRUCTURE) {
		vec = xact_comp(data,".Data");
		mode = vec? vec->mode : MISSING;
	}
	if(NOT_RECURSIVE(mode))return(blt_in_NULL);
	name = string_value(comp);
	vec = find_comp(vec, name);
	if(VOID(vec)) return(blt_in_NULL);
	/* vec may be precious -- strictly, the semantic model
	/* would require us to copy here, but the assertion is
	/* that the preciousness will cause a copy later if needed */
	return(vec);
}		

static vector *
set_dollar(data, comp, repl_val)
vector *data, *comp, *repl_val;
{
	char *name; long mode, n; vector *vec;
	if(NOT_THERE(repl_val)){}
	else if(PRECIOUS(repl_val))repl_val = copy_data(repl_val,data->x.frame);
	else set_precious(repl_val,data->x.frame);
	mode = data->mode; vec = data;
	if(mode == STRUCTURE) {
		vec = xact_comp(data,".Data");
		mode = vec? vec->mode : MISSING;
	}
	if(NOT_RECURSIVE(mode))	{
		vector *frame = data->x.frame;
		if(NOT_THERE(repl_val))return(repl_val); /* no effect */
		coeves(data,LIST,FALSE,FALSE,&vec);
		set_precious(data,frame);
	}
	name = repl_val->name = string_value(comp);
	n = x_which_comp(name,vec);;
	if(NOT_THERE(repl_val))del_comp(vec,n);
	else if(n) vec->value.tree[n-1] = repl_val;
	else append_el(vec,(long)NOARG,repl_val);
	return(repl_val);
}

static vector *
element(data, subs, nsub, replacement)
vector *data, **subs, *replacement;
long nsub;
{
	ix *i;
	int mode, drop;
	long which;
	char *name;
	vector *vec = data;

	if(VOID(data))
		return(blt_in_NULL);
	mode = data->mode;
	drop = VOID(replacement) ? subs[--nsub]->value.Long[0] : 0;
	if(mode == STRUCTURE) {
		vec = xact_comp(data, ".Data");
		mode = vec ? vec->mode : MISSING;
	}

	/* special cases */
	switch(mode) {
	case NAME:
	case SYSTEM:
		PROBLEM "Can't take elements of objects with mode  \"%s\"",
			token_name(mode) RECOVER(NULL_ENTRY);
	}

	/* no subscripts or zero-length lists */
	if(nsub < 1)
		return(data);

	if(!atomic_type(mode) && NOT_RECURSIVE(mode))
		PROBLEM "Can't take elements of objects with mode  \"%s\"",
			token_name(mode) RECOVER(NULL_ENTRY);
	if(!atomic_type(mode) && nsub==1) {
		vector *sub = *subs;
		if(!atomic_type(sub->mode) || sub->length>1)
			return(subtree(data,sub,replacement));
	}
	i = make_index(data, subs, nsub, !VOID(replacement));
	if(i == &null_ix)
		return(data);
	if(i->len > 1)
		PROBLEM "More than one element not allowed" RECOVER(data);
	if(i->len < 1)
		PROBLEM "Less than one element not allowed" RECOVER(data);
	if(VOID(replacement)) {
		if(atomic_type(mode))
			return(extract(data, i, drop));
		which = i->i->value.Long[0];
		if(is_na(&which) || which >= vec->length)
			return(blt_in_NULL);
		else return(vec->value.tree[which]);
			/* above may be precious */
	}
	if(atomic_type(mode))
		return(replace(data, i, replacement));
	else if(NOT_RECURSIVE(mode))
		PROBLEM "Can't replace elements in object of mode \"%s\"",
			token_name(mode) RECOVER(NULL_ENTRY);
	which = i->i->value.Long[0];
	if(NOT_THERE(replacement)) {
		del_comp(vec, which+1);
		return(replacement);
	}
	else if(PRECIOUS(replacement))replacement = copy_data(replacement,data->x.frame);
	else set_precious(replacement,data->x.frame);
	if(which >= vec->length)
		set_length2(vec, which+1);
	if(!NOT_THERE(i->names))
		name = i->names->value.Char[which];
	else
		name = (vec->value.tree[which])->name;
	vec->value.tree[which] = replacement;
	(vec->value.tree[which])->name = name;
	return(replacement);
}

static vector *
attribute(data, which_attr, replacement)
vector *data, *which_attr, *replacement;
{
	vector **t, *v, *attr;
	long i;
	int found;
	char *name;

	if(VOID(data) || VOID(which_attr))
		return(blt_in_NULL);
	if(which_attr->mode != CHAR)
		PROBLEM "Attribute name must be of mode \"character\"" RECOVER(which_attr);
	if(which_attr->length != 1)
		PROBLEM "Exactly one attribute must be specified" RECOVER(which_attr);
	name = attr_name(which_attr->value.Char[0]);
	if(!name)
		PROBLEM "Invalid attribute name" RECOVER(which_attr);

	/* get an attribute */
	found = FALSE;
	if(VOID(replacement)) {
		MEANINGFUL(attr);
		switch(*name) {
		case 'm':
			if(found = !strcmp(name, "mode"))
				attr = get_mode(data, 0);
			break;
		case 'l':
			if(found = !strcmp(name, "length"))
				attr = get_length(data);
			else if(found = !strcmp(name, "levels"))
				attr = get_levels(data);
			break;
		case 'n':
			if(found = !strcmp(name, "names"))
				attr = get_names(data);
			break;
		case 'd':
			if(found = !strcmp(name, "dim"))
				attr = get_dim(data);
			else if(found = !strcmp(name, "dimnames"))
				attr = get_dimnames(data);
			break;
		case 't':
			if(found = !strcmp(name, "tsp"))
				attr = get_tsp(data);
			break;
		}
		if(!found) {
			if(data->mode != STRUCTURE)return(blt_in_NULL);
			attr = get_attr(data,name,0);
			if(!PRECIOUS(attr) && !VOID(replacement)) /* pass down protection */
				set_precious(attr, data->x.frame);
		}
		return(attr);
	}

	/* set an attribute */
	switch(*name) {
	case 'm':
		if(strcmp(name, "mode") == 0) {
			set_mode(data, replacement);
			return(replacement);
		}
		break;
	case 'l':
		if(strcmp(name, "length") == 0) {
			set_length(data, replacement);
			return(replacement);
		}
		else if(strcmp(name, "levels") == 0) {
			set_levels(data, replacement);
			return(replacement);
		}
		break;
	case 'n':
		if(strcmp(name, "names") == 0) {
			set_names(data, replacement);
			return(replacement);
		}
		break;
	case 'd':
		if(strcmp(name, "dim") == 0) {
			set_dim(data, replacement);
			return(replacement);
		}
		if(strcmp(name, "dimnames") == 0) {
			set_dimnames(data, replacement);
			return(replacement);
		}
		break;
	case 't':
		if(strcmp(name, "tsp") == 0) {
			set_tsp(data, replacement);
			return(replacement);
		}
		break;
	}
	replacement->name = name;
	if(data->mode == STRUCTURE) {
		i = x_which_comp(name,data);
		if(NOT_THERE(replacement))
			del_comp(data,i);
		else if(i > 0) {
			if(replacement!=data->value.tree[i-1])
				try_to_free(data->value.tree[i-1],TRUE);
			data->value.tree[i-1] = replacement;
			set_precious(replacement,data->x.frame);
		}
		else
			append_el(data, data->length, replacement);
		return(replacement);
	}
	if(NOT_THERE(replacement))return(replacement); /*no effect*/
	v = alcvec(STRUCTURE, 2L);
	t = v->value.tree;
	v->name  = data->name;
	t[0] = New_vector(); *(t[0]) = *data; t[0]->name = ".Data";
	t[1] = replacement;
	set_precious(v,data->x.frame);
	*data = *v; 
	return(replacement);
}

static vector *
get_mode(data, storage)
vector *data;
int storage;
{
	vector *v;
	int mode;
	char *name;

	v = alcvec(CHAR, 1L);
	if(data->mode == STRUCTURE)
		data = get_attr(data, ".Data", 1);
	mode = data->mode;
	if(storage)name = mode==DOUBLE ? "double" : token_name(mode);
	else switch(mode) {
	case DOUBLE: case INT: case REAL: name = "numeric"; break;
	default: name = token_name(mode);
	}
	v->value.Char[0] = name;
	return(v);
}

static vector *
set_mode(data, mode)
vector *data, *mode;
{
	int m;
	long n, i, k;
	vector *d, **dp;
	char **names;

	if(mode->mode != CHAR || mode->length != 1)
		PROBLEM "Replacement mode not character string" RECOVER(mode);
	m = mode_lookup(mode->value.Char[0]);
	if(m == UNKNOWN)
		PROBLEM "Unknown replacement mode: %s", mode->value.Char[0] RECOVER(NULL_ENTRY);
	if(data->length == 0) {
		if(min_length(m) > 0)
			PROBLEM "Trying to set too short a length for mode \"%s\"",
				token_name(m) RECOVER(NULL_ENTRY);
		data->mode = m;
	} else if(data->mode == STRUCTURE) {
		d = get_attr(data, ".Data", 1);
		n = d->length;
		coevec(d, m, FALSE, FALSE);
		if(d->length != n) { /* changed length; e.g., atomic to list */
			d->name = "";
			set_length2(d, n);
		}
		if(!NOT_RECURSIVE(m)) {
			k = x_which_comp(".Names", data);
			if(k > 0) {
				names = data->value.tree[k-1]->value.Char;
				dp = d->value.tree;
				for(i = 0; i < n; i++)
					(*dp++)->name = *names++;
			}
			del_comp(data, k);
		}
	}
	else coevec(data, m, FALSE, FALSE);
	return(mode);
}

static vector *
get_length(data)
vector *data;
{
	vector *v = alcvec(INT, 1L);

	v->value.Long[0] = get_length2(data);
	return(v);
}

static long
get_length2(data)
vector *data;
{
	if(data->mode == STRUCTURE)
		data = get_attr(data, ".Data", 1);
	return(data->length);
}

static vector *
set_length(data,length)
vector *data, *length;
{
	set_length2(data,long_value(length, NULL_ENTRY));
	return(length);
}

static
set_length2(data, l)
vector *data;
long l;
{
	char *vv = NA_STRING, *ns = "";
	vector *d = data, *names = blt_in_NULL;

	if(data->mode == STRUCTURE)
		d = get_attr(data, ".Data", 1);
	if(l == d->length)
		return;
	if(min_length(d->mode) > d->length)
		PROBLEM "Trying to set too short a length for mode \"%s\"",
			token_name(d->mode) RECOVER(NULL_ENTRY);
	if(data->mode == STRUCTURE) {
		names = get_attr(data, ".Names", 1);
		if(NOT_THERE(names)) { /* throw away structure */
			*data = *d;
			d = data;
		} else { /* remember only names */
			data->length = 2;
			data->value.tree[0] = d;
			data->value.tree[1] = names;
		}
	}
	if(l > d->length) {
		long ll, m = d->mode;
		na_set(&ll);
		if(m == CHAR) append_data(d,l-1,1L,(char *)&vv);
		else if(atomic_type(d->mode))append_data(d,l-1,1L,(char *)&ll);
		else if(NOT_RECURSIVE(d->mode))
			PROBLEM "Invalid to set length of this data" RECOVER(d);
		else append_el(d,l-1,S_void);
		if(names != blt_in_NULL)
			append_data(names,l-1,1L,(char *)&ns);
	} else {
		d->length = l;
		if(names != blt_in_NULL)
			names->length = l;
	}
}

vector *
get_names(data)
vector *data;
{
	vector *names, *t;

	if(data == NULL_ENTRY || NOT_RECURSIVE(data->mode))
		return(blt_in_NULL);
	if(data->mode == STRUCTURE) {
		names = get_attr(data, ".Names", 1);
		if(NOT_THERE(names)) {	/* recursive .Data with names? */
			t = get_attr(data, ".Data", 1);
			if(!NOT_RECURSIVE(t->mode))
				names = get_list_names(t);
		}
	} else
		names = get_list_names(data);
	if(!NOT_THERE(names)) {
		if(PRECIOUS(data))set_precious(names,data->x.frame);
		return(names);
	}
	return(blt_in_NULL);
}

static vector *
get_list_names(data)
vector * data;
{
	long n = data->length;
	char **strings, *name;
	vector **t, *names;

	if(NOT_RECURSIVE(data->mode) || n <= 0)
		return(blt_in_NULL);
	t = data->value.tree;
	do {
		name = (*t++)->name;
		if(name && *name)
			break;
	} while(--n);
	if(n <= 0)
		return(blt_in_NULL);
	n = data->length;
	t = data->value.tree;
	names = alcvec(CHAR, n);
	strings = names->value.Char;
	while(n--) {
		name = (*t++)->name;
		*strings++ = name ? name : "";
	}
	return(names);
}

has_names(data)
vector *data;
{
	vector **t;
	long n;

	if(data == NULL_ENTRY || NOT_RECURSIVE(data->mode))
		return(0);
	if(data->mode == STRUCTURE) {
		if(get_attr(data, ".Names", 1) != blt_in_NULL)
			return(1);
		return(has_names(get_attr(data, ".Data", 1)));
	}
	t = data->value.tree;
	n = data->length;
	while(n--) {
		char *s = (*t++)->name;
		if(s && *s)
			return(1);
	}
	return(0);
}

#define SET_NAME(p,text) {tt = *p; if(tt->x.frame==cons_frame)tt = *p = copy_data(*p,data->x.frame); tt->name = text;}
vector *
set_names(data, namesa)
vector *data, *namesa;
{
	vector *names, *v, **t, *dim, *tt;
	long n, i, dmode;
	char **strings;

	if(VOID(data))
		PROBLEM "No data to assign names attribute" RECOVER(NULL_ENTRY);
	dmode = data->mode; /* not data_mode(data) */
	if(NOT_THERE(namesa)) {
		if(dmode == STRUCTURE)
			del_comp(data, x_which_comp(".Names", data));
		else if(!NOT_RECURSIVE(dmode))
			for(n = data->length, t = data->value.tree; n>0; n--,t++)
				SET_NAME(t, "")
		return(namesa);
	}
	names = coevec(namesa, CHAR, FALSE, PRECIOUS(namesa));
	if(PRECIOUS(names))names = copy_data(names,NULL_ENTRY);
	if(atomic_type(dmode)) {
		if(names->length != data->length)
			PROBLEM "Invalid length for names attribute" RECOVER(data);
		v = alcvec(STRUCTURE, 2L);
		v->name  = data->name;
		t = v->value.tree;
		t[0] = New_vector(); *(t[0]) = *data; t[0]->name = ".Data";
		t[1] = names; names->name = ".Names";
		if(PRECIOUS(data))set_precious(v,data->x.frame);
		*data = *v;
	} else if(dmode == STRUCTURE) {
		v = get_attr(data, ".Data", 1);
		n = v->length;
		if(names->length != n)
			PROBLEM "Invalid length for names attribute" RECOVER(data);
		if(NOT_RECURSIVE(v->mode)) {
			names->name = ".Names";
			i = x_which_comp(".Names", data);
			if(i > 0) {
				if(PRECIOUS(data))set_precious(names, data->x.frame);
				data->value.tree[i-1] = names;
			}
			else
				append_el(data, data->length, names);
		} else {	/* change names of v */
			t = v->value.tree;
			strings = names->value.Char;
			for(i = 0; i < n; i++,t++,strings++)
				SET_NAME(t,  *strings)
		}
		dim = get_dim(data);
		if(!NOT_THERE(dim) && dim->length == 1) {
			vector *dimnames = get_dimnames(data);
			if(NOT_THERE(dimnames) || dimnames->value.tree[0]->length == 0) {
				vector *new = alcvec(LIST, 1L);
				new->value.tree[0] = names;
				set_dimnames(data, new);
			} else {
				char **s = dimnames->value.tree[0]->value.Char;
				char **t = names->value.Char;
				for(i = 0; i < n; i++)
					if(strcmp(*s++, *t++))
						PROBLEM "Incompatible names and dimnames for 1-d array" RECOVER(data);
			}
		}
			
	} else if(NOT_RECURSIVE(dmode))
		PROBLEM "Can't set names for mode \"%s\"",
			token_name(data->mode) RECOVER(NULL_ENTRY);
	else {
		n = data->length;
		if(names->length != n)
			PROBLEM "Invalid length for names attribute" RECOVER(data);
		t = data->value.tree;
		strings = names->value.Char;
		for(i = 0; i < n; i++, t++, strings++)
			SET_NAME(t, *strings)
	}
	return(names);
}

static vector *
get_dim(data)
vector *data;
{
	if(VOID(data) || data->mode != STRUCTURE)
		return(blt_in_NULL);
	return(get_attr(data, ".Dim", 1));
}

static vector *
set_dim(data, dim)
vector *data, *dim;
{
	long ndim, *d, n, i;
	vector **t, *v;

	if(NOT_THERE(dim)){
		if(data->mode == STRUCTURE) {
			del_comp(data, x_which_comp(".Dim", data));
			del_comp(data, x_which_comp(".Dimnames", data));
		}
		return(dim);
	}
	dim = coevec(dim, INT, FALSE, PRECIOUS(dim));
	if(VOID(dim) || dim->length < 1)
		PROBLEM "Invalid dimension vector" RECOVER(NULL_ENTRY);
	if(PRECIOUS(dim)) dim = copy_data(dim, NULL_ENTRY);
	if(PRECIOUS(data))set_precious(dim,data->x.frame);
	ndim = dim->length;
	if(ndim < 1)
		PROBLEM "Invalid length for dim attribute" RECOVER(data);
	d = dim->value.Long;
	n = 1;
	for(i = 0; i < ndim; i++, d++){
		if(is_na(d) || *d <= 0)
			PROBLEM "Invalid value for dimension %ld", i+1 RECOVER(dim);
		n *= *d;
	}
	if(data->mode == STRUCTURE) {
		v = get_attr(data, ".Data", 1);
		if(n != v->length)
			PROBLEM "Invalid dim attribute: lengths do not match" RECOVER(data);
		dim->name = ".Dim";
		i = x_which_comp(".Dim", data);
		if(i > 0) {
			if(dim!=data->value.tree[i-1])
				try_to_free(data->value.tree[i-1],TRUE);
			set_precious(dim,data->x.frame);
			data->value.tree[i-1] = dim;
		}
		else
			append_el(data, 1L, dim);
		i = x_which_comp(".Dimnames", data);
		if(i > 0 && !ok_dimnames(data, data->value.tree[i-1])) {
			long ii;
			for(ii = i; ii<data->length; ii++)
				data->value.tree[ii-1] = data->value.tree[ii];
			data->length--;
			PROBLEM "Invalid dimnames deleted" WARNING(NULL_ENTRY);
		}
	} else {
		if(n != data->length)
			PROBLEM "Length of data (%ld) doesn't match product of dimensions (%ld)", data->length, n RECOVER(NULL_ENTRY);
		v = alcvec(STRUCTURE, 2L);
		t = v->value.tree;
		v->name  = data->name;
		t[0] = New_vector(); *(t[0]) = *data; t[0]->name = ".Data";
		t[1] = dim; dim->name = ".Dim";
		set_precious(v,data->x.frame);
		*data = *v; 
	}
	i = x_which_comp(".Dimnames", data);
	n = has_names(data);
	if(dim->length == 1)	/* equate names and dimnames for 1-d array */
		if(i && !n) {
			v = data->value.tree[i-1]->value.tree[0];
			if(v->length > 0)
				set_names(data, v);
		} else if(!i && n) {
			v = alcvec(LIST, 1L);
			v->value.tree[0] = get_names(data);
			set_dimnames(data, v);
		}
	return(dim);
}

static vector *
get_dimnames(data)
vector *data;
{
	if(VOID(data) || data->mode != STRUCTURE)
		return(blt_in_NULL);
	return(get_attr(data, ".Dimnames", 1));
}

static vector *
set_dimnames(data, dimnames)
vector *data, *dimnames;
{
	vector *names;
	long  i;
	if(NOT_THERE(dimnames)) {
		if(data->mode == STRUCTURE)
			del_comp(data, x_which_comp(".Dimnames", data));
		return(dimnames);
	}
	dimnames = coevec(dimnames, LIST, FALSE, PRECIOUS(dimnames));
	if(PRECIOUS(dimnames))dimnames = copy_data(dimnames,data->x.frame);
	if(dimnames->length < 1)
		PROBLEM "Invalid dimnames vector" RECOVER(NULL_ENTRY);
	if(!ok_dimnames(data, dimnames))
		PROBLEM err_buf RECOVER(NULL_ENTRY);
	dimnames->name = ".Dimnames";
	i = x_which_comp(".Dimnames", data);
	if(i > 0) {
		if(dimnames!=data->value.tree[i-1])
			try_to_free(data->value.tree[i-1],TRUE);
		set_precious(dimnames,data->x.frame);
		data->value.tree[i-1] = dimnames;
	}
	else
		append_el(data, data->length, dimnames);
	if(dimnames->length == 1) {
		names = get_names(data);
		if(NOT_THERE(names))
			set_names(data, copy_data(dimnames->value.tree[0], NULL_ENTRY));
		else {
			char **s = names->value.Char;
			char **t = dimnames->value.tree[0]->value.Char;
			long n = names->length;
			for(i = 0; i < n; i++)
				if(strcmp(*s++, *t++))
					PROBLEM "Incompatible names and dimnames for 1-d array" RECOVER(data);
		}
	}
	return(dimnames);
}

static int
ok_dimnames(data,dimnames)
vector *data, *dimnames;
{
	long ndim, ndimnames, *d, n, i;
	vector *dim, **dn;
	if(dimnames->mode != LIST){
		sprintf(err_buf,"Invalid mode (%s) for dimnames attribute",
			token_name(dimnames->mode));
		return(FALSE);
	}
	if(data->mode!=STRUCTURE || (dim = get_dim(data))==S_void){
		sprintf(err_buf,"Cannot have dimnames for nonarray");
		return(FALSE);
	}
	ndim = dim->length;
	ndimnames = dimnames->length;
	if(ndimnames != ndim){
		sprintf(err_buf,"Invalid length for dimnames attribute");
		return(FALSE);
	}
	d = dim->value.Long;
	dn = dimnames->value.tree;
	for(i = 0; i < ndim; i++) {
		if((dn[i])->mode != CHAR) {
			if(PRECIOUS(dimnames)) {
				dimnames = copy_data(dimnames, NULL_ENTRY);
				dn = dimnames->value.tree;
			}
			coevec(dn[i],CHAR,FALSE,FALSE);
		}
		n = (dn[i])->length;
		if(n != 0 && n != d[i]) {
			sprintf(err_buf,"Component %ld of dimnames has length %ld, should be %ld", i+1, n, d[i]);
			return(FALSE);
		}
	}
	return(TRUE);
}

static vector *
get_attr(data, name, exact)
vector *data;
char *name;
int exact;
{
	int i;
	vector *attr;

	attr = exact ? xact_comp(data, name) : find_comp(data, name);
	if(attr == NULL_ENTRY)
		return(blt_in_NULL);
if(check) {
	if(PRECIOUS(data) && !PRECIOUS(attr))
		PROBLEM "Data is precious, but attribute \"%s\" is not", name WARNING(NULL_ENTRY);
	else if(!PRECIOUS(data) && PRECIOUS(attr))
		PROBLEM "Data is not precious, but attribute \"%s\" is", name WARNING(NULL_ENTRY);
}
	if(PRECIOUS(data))
		set_precious(attr, data->x.frame);
	return(attr);
}

static vector *
get_tsp(data)
vector *data;
{
	if(VOID(data) || data->mode != STRUCTURE)
		return(blt_in_NULL);
	return(get_attr(data, ".Tsp", 1));
}

#define NOTINT(x)	(fabs((x) - floor((x)+.5)) > (DOUBLE_EPS * fabs(x)))

static vector *
set_tsp(data, tsp)
vector *data, *tsp;
{
	long  i, n_implied, n_actual;
	double *dd, start, end, freq, floor();
	vector **t, *v;

	if(NOT_THERE(tsp)) {
		if(data->mode == STRUCTURE)
			del_comp(data, x_which_comp(".Tsp", data));
		return(tsp);
	}
	tsp = coevec(tsp, DOUBLE, FALSE, PRECIOUS(tsp));
	if(tsp->length < 3)
		PROBLEM "Need three values for time-series parameters" RECOVER(tsp);
	dd = tsp->value.Double;
	if(is_na(&dd[0]) || is_na(&dd[1]) || is_na(&dd[2]))
		PROBLEM "NAs not allowed in time-series parameters" RECOVER(tsp);
	start = dd[0]; end = dd[1]; freq = dd[2];
	if(start > end)
		PROBLEM "Start may not be larger than end in time-series parameters" RECOVER(tsp);
	if(freq <= 0 || NOTINT(freq))
		PROBLEM "Frequency for time-series must be a positive integer" RECOVER(tsp);
	dd[2] = floor(dd[2] + 0.1);
	if(NOTINT(start*freq))
		PROBLEM "Start*frequency for time-series must be an integer" RECOVER(tsp);
	n_implied = (end - start) * freq + 1.01;
	n_actual = data->mode == STRUCTURE
		 ? get_attr(data, ".Data", 1)->length
		 : data->length;
	if(n_implied != n_actual)
		PROBLEM "Length of data is %ld; time-series parameters imply %ld", n_actual, n_implied RECOVER(tsp);
	if(PRECIOUS(tsp))tsp = copy_data(tsp, NULL_ENTRY);
	tsp->length = 3; tsp->name = ".Tsp";
	if(data->mode == STRUCTURE) {
		i = x_which_comp(".Tsp", data);
		if(i > 0) {
			if(tsp!=data->value.tree[i-1])
				try_to_free(data->value.tree[i-1],TRUE);
			set_precious(tsp,data->x.frame);
			data->value.tree[i-1] = tsp;
		}
		else
			append_el(data, 1L, tsp);
	} else {
		v = alcvec(STRUCTURE, 2L);
		t = v->value.tree;
		v->name = data->name;
		t[0] = New_vector(); *(t[0]) = *data; t[0]->name = ".Data";
		t[1] = tsp;
		set_precious(v,data->x.frame);
		*data = *v; 
	}
	return(tsp);
}

static vector *
get_levels(data)
vector *data;
{
	if(VOID(data) || data->mode != STRUCTURE)
		return(blt_in_NULL);
	return(get_attr(data, ".Label", 1));
}

static vector *
set_levels(data, levels)
vector *data, *levels;
{
	long  i;
	vector **t, *v;

	if(NOT_THERE(levels)) {
		if(data->mode == STRUCTURE)
			del_comp(data, x_which_comp(".Label", data));
		return(levels);
	}
	levels = coevec(levels, CHAR, FALSE, PRECIOUS(levels));
	if(PRECIOUS(levels))levels = copy_data(levels,NULL_ENTRY);
	levels->name = ".Label";
	if(data->mode == STRUCTURE) {
		v = get_attr(data, ".Data", 1);
		if(v->mode != INT) coeves(data, INT, FALSE, FALSE, &v);
		ok_levels(v, levels->length);
		i = x_which_comp(".Label", data);
		if(i > 0) {
			if(levels!=data->value.tree[i-1])
				try_to_free(data->value.tree[i-1],TRUE);
			set_precious(levels,data->x.frame);
			data->value.tree[i-1] = levels;
		}
		else
			append_el(data, 1L, levels);
	} else {
		if(data->mode != INT) coevec(data, INT, FALSE, FALSE);
		ok_levels(data,levels->length);
		v = alcvec(STRUCTURE, 2L);
		t = v->value.tree;
		v->name  = data->name;
		t[0] = New_vector(); *(t[0]) = *data; t[0]->name = ".Data";
		t[1] = levels;
		set_precious(v,data->x.frame);
		*data = *v; 
	}
	return(levels);
}

static void
ok_levels(v, length)
vector *v;
long length;
{
	long n, *ll;
	for(n = v->length, ll = v->value.Long; n>0; n--, ll++) {
		if(is_na(ll)) continue;
		if(*ll < 1 || *ll > length)
			PROBLEM "Invalid data (%ld) in category; must be between 1 and %ld", *ll, length RECOVER(NULL_ENTRY);
	}
}

/* turn internal attribute name into external attribute name */
static char *
attr_name(name)
char *name;
{
	char **pp, **qq;

	if(*name != '.')
		return(name);
	if(name_eq(name, ".Data"))
		return(NULL);
	for(pp = attr_internal, qq = attr_external; *pp; pp++, qq++)
		if(name_eq(name, *pp))
			return(*qq);
	return(name);
}

static vector *
get_attrs(Temp)
vector *Temp;
{
	vector *val, *names, **from, **to;
	long n;
	char *name;

	if(Temp->mode == STRUCTURE) {
		val = alcvec(LIST,Temp->length);
		for(n=Temp->length, from = Temp->value.tree, to = val->value.tree;
		  n>0; n--, from++) {
			name = attr_name((*from)->name);
			if(!name) {	/* for .Data, use names if present */
				names = get_list_names(*from);
				if(!NOT_THERE(names)){
					(*to) = names;
					(*to)->name = "names";
					to++;
				} else val->length--;
			} else {
				(*to) = PRECIOUS(Temp) ? copy_data(*from, NULL_ENTRY): *from;
				(*to)->name = name;
				to++;
			}
		}
	}
	else {
		names = get_names(Temp);
		if(NOT_THERE(names)) val = alcvec(LIST, 0L);
		else {
			val = alcvec(LIST, 1L);
			*(val->value.tree) = names;
			(*(val->value.tree))->name = "names";
			val->x.frame = names->x.frame;
		}
	}
	return(val);
}

static vector *
set_attrs(data, attrs)
vector *data, *attrs;
{
	vector *name, **attr;
	char **s;
	long n, i, idim, idimnames;

	/* clear out old attributes */
	if(data->mode == STRUCTURE) {
		char *dataname = data->name;
		*data = *get_attr(data, ".Data", 1);
		data->name = dataname;
	}
	set_names(data, blt_in_NULL);

	/* special cases where no more work is needed */
	if(NOT_THERE(attrs) || attrs->length == 0)
		return(attrs);

	attrs = coevec(attrs, LIST, FALSE, TRUE);
	if(PRECIOUS(attrs))
		attrs = copy_data(attrs, NULL_ENTRY);

	/* find positions of dim and dimnames */
	n = attrs->length;
	attr = attrs->value.tree;
	idim = idimnames = -1;
	for(i = 0; i < n; i++, attr++) {
		char *attrname = (*attr)->name;
		if(strcmp(attrname, "dim") == 0)
			idim = i;
		if(strcmp(attrname, "dimnames") == 0)
			idimnames = i;
	}

	/* make sure we will set dim before dimnames */
	if(idimnames > -1 && idim > idimnames) {
		vector *v = attrs->value.tree[idim];
		attrs->value.tree[idim] = attrs->value.tree[idimnames];
		attrs->value.tree[idimnames] = v;
	}

	/* set attributes, one at a time */
	name = alcvec(CHAR, 1L);
	s = name->value.Char;
	attr = attrs->value.tree;
	for(i = 0; i < n; i++, attr++) {
		*s = (*attr)->name;
		(void)attribute(data, name, *attr);
	}
	return(attrs);
}

del_comp(data, which)
vector *data;
long which;
{
	vector **p;
	long n = data->length;

	if(which < 1 || which > n || NOT_RECURSIVE(data->mode))
		return;
	for(p = data->value.tree+which-1; which < n; which++, p++)
		p[0] = p[1];
	data->length--;
}

static vector *
add_nulls(index)
ix *index;
{
	vector *names = index->names;
	long length = names->length, i;
	char *ns = "", **p;

	if(length < index->max) {
		if(PRECIOUS(names))
			index->names = names = copy_data(names, NULL_ENTRY);
		append_data(names, index->max-1, 1L, (char *)&ns);
		p = names->value.Char + length;
		for(i = length; i < names->length; i++)
			*p++ = ns;
	}
	return(index->names);
}

static long
max(a, b)
long a, b;
{
	return(a > b ? a : b);
}

static long
min(a, b)
long a, b;
{
	return(a < b ? a : b);
}

static vector *
subtree(data,  path, replacement)
vector *data, *path, *replacement;
{
	ix *i;
	int mode, out;
	long which, n;
	char *name;
	vector *vec, **sub, *new;
	path = coevec(path,LIST,FALSE,PRECIOUS(path));
	for(n = path->length-1, sub = path->value.tree; n>=0; n--, sub++) {			vec = data; mode = data->mode;
		if(mode == STRUCTURE) {
			vec = xact_comp(data, ".Data");
			mode = vec ? vec->mode : MISSING;
		}
		/* special cases */
		switch(mode) {
		case NAME:
		case SYSTEM:
			PROBLEM "Can't take subtree of objects with mode \"%s\"",
				token_name(mode) RECOVER(NULL_ENTRY);
		}
		if(!atomic_type(mode) && NOT_RECURSIVE(mode))
			PROBLEM "Can't take subtree of objects with mode  \"%s\"",
				token_name(mode) RECOVER(NULL_ENTRY);
		i = make_index(data, sub, 1L, !VOID(replacement));
		if(i == &null_ix)
			return(data);
		if(i->len > 1)
			PROBLEM "More than one element not allowed" RECOVER(data);
		if(i->len < 1)
			PROBLEM "Less than one element not allowed" RECOVER(data);
		which = i->i->value.Long[0];
		out = is_na(&which) || which >= vec->length;
		if((n && !out) || VOID(replacement)) {
			if(mode==NULL)data = vec;
			else if(atomic_type(mode))
				data = extract(data, i, 0);
			else if(out)
				data = blt_in_NULL;
			else {
				vec = vec->value.tree[which];
				if(PRECIOUS(data))
					set_precious(vec,data->x.frame);
				data = vec;
			}
			continue;
		}
		new = n ? alcvec(LIST,0L) : replacement;
		if(PRECIOUS(new) && !atomic_type(mode))
			new = copy_data(new,NULL_ENTRY);
		if(atomic_type(mode))
			replace(data, i, new);
		else if(NOT_RECURSIVE(mode))
			PROBLEM "Can't replace subtree in object of mode \"%s\"",
				token_name(mode) RECOVER(NULL_ENTRY);
		if(NOT_THERE(new))
			del_comp(vec, which+1);
		if(which >= vec->length)
			set_length2(vec, which+1);
		if(!NOT_THERE(i->names))
			name = i->names->value.Char[which];
		else
			name = (vec->value.tree[which])->name;
		vec->value.tree[which] = new;
		(vec->value.tree[which])->name = name;
		data =  n ? (atomic_type(mode) ? extract(data, i, 0) :
			  vec->value.tree[which]) : replacement;
	}
	return(data);
}

static int
overlap(x,y)
vector *x, *y;
{
	long d;
	switch(x->mode) { /* asserted that x,y are vectors & of same mode */
	case DOUBLE:
		 d = y->value.Double - x->value.Double; break;
	case CHAR:
		 d = y->value.Char - x->value.Char; break;
	case LGL:
	case INT:
		 d = y->value.Long - x->value.Long; break;
	case REAL:
		 d = y->value.Float - x->value.Float; break;
	case COMPLEX:
		 d = y->value.Complex - x->value.Complex; break;
	default:
		 d = y->value.tree - x->value.tree; break;
	}
	return( (d<0 && y->length-d >= 0) || (d>=0 && x->length-d > 0));
}

static vector *
Data(ent)
vector *ent;
{
	return(ent->mode == STRUCTURE ? xact_comp(ent, ".Data") : ent);
}

static FILE *ext_file;

vector *
S_ext_extract(ent, arglist)
vector *ent, *arglist;
{
	long nargs = arglist->length;
	vector **args = arglist->value.tree, *value;

	value = ext_subset(args[0], &args[1], nargs-1, S_void);
	if(NOT_THERE(value))
		return(blt_in_NULL);
	if(value->mode == STRUCTURE && value->length == 1)
		value = value->value.tree[0];
	return(value);

}

static vector *
ext_subset(data, subs, nsub, replacement)
vector *data, **subs, *replacement;
long nsub;
{
	vector *origdata = data;
	char *basename, *filename, **paths, *path;
	long npath, i, namelength = 0, length;
	int mode;
	double copy_header;
	FILE *efile;

	if(VOID(data) || NOT_THERE(data))
		return(blt_in_NULL);
	if(data->mode != STRUCTURE)
		PROBLEM "Illegal object of class external" RECOVER(origdata);
	data = Data(data);
	if(NOT_THERE(data))
		PROBLEM "Weird object" RECOVER(origdata);
	if(data->mode != CHAR || data->length != 1)
		PROBLEM "Illegal object of class external" RECOVER(origdata);
	basename = data->value.Char[0];
	npath = Search_list->length;
	paths = Search_list->value.Char;
	for(i = 0; i < npath; i++)
		namelength = max(namelength, (long)strlen(paths[i]));
	path = S_alloc(namelength + 1 + strlen(basename) + 1, sizeof(char));
	for(i = 0; i < npath; i++) {
		strcpy(path, paths[i]);
		strcat(path, "/");
		strcat(path, basename);
		efile = fopen(path, "r");
		if(efile != NULL)
			break;
	}
	if(efile == NULL)
		PROBLEM "Cannot find %s", basename RECOVER(origdata);
	fprintf(stderr, "Found data in %s\n", path);
	if(!is_S_object(efile)) {
		fclose(efile);
		PROBLEM "%s is not an S object", path RECOVER(origdata);
	}
	if(FREAD(&mode, 1, efile) != 1) {
		fclose(efile);
		PROBLEM "Cannot read mode in %s", path RECOVER(origdata);
	}
	if(FREAD(&length, 1, efile) != 1) {
		fclose(efile);
		PROBLEM "Cannot read length in %s", path RECOVER(origdata);
	}
	return(blt_in_NULL);
}

is_S_object(f)
FILE *f;
{
	double copy_header;

	return(FREAD(&copy_header, 1, f) == 1 && copy_header == *data_header);
}
