/* 
 * Kodkod -- Copyright (c) 2005-2011, Emina Torlak
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kodkod.engine.ucore;

import kodkod.engine.fol2sat.TranslationLog;
import kodkod.engine.fol2sat.Translator;
import kodkod.engine.satlab.ReductionStrategy;
import kodkod.engine.satlab.ResolutionTrace;
import kodkod.util.ints.IntCollection;
import kodkod.util.ints.IntSet;
import kodkod.util.ints.IntTreeSet;
import kodkod.util.ints.Ints;

/**
* Naive Core Extraction is a strategy for generating unsat cores that are minimal at the logic level.  
 * Specifically, let C be a core that is minimal according to this strategy, 
 * and let F(C) be the top-level logic constraints
 * corresponding to C.  Then, this strategy guarantees that there is no clause
 * c in C such that F(C - c) is a strict subset of F(C). Furthermore, it also
 * guarantees that for all f in F(C), F(C) - f is satisfiable.  This is a stronger
 * guarantee than that of {@linkplain HybridStrategy}.  In general, using this strategy
 * is more expensive, timewise, than using {@linkplain HybridStrategy}.
 * 
 * <p>This implementation of NCE will work properly only on CNFs generated by the kodkod {@linkplain Translator}. </p>
* @author Emina Torlak
* @see HybridStrategy
*/
public final class NCEStrategy implements ReductionStrategy {
	private final IntCollection varsToTry;
	private final IntSet coreVars;
	
	/**
	 * Constructs an NCE strategy that will use the given translation
	 * log to relate the cnf clauses back to the logic constraints from 
	 * which they were generated. 
	 */
	public NCEStrategy(final TranslationLog log) {
		varsToTry = StrategyUtils.rootVars(log);
		coreVars = new IntTreeSet();//new IntTreeSet(varsToTry);
		coreVars.addAll(varsToTry);
	}
	
	/**
	 * {@inheritDoc}
	 * @see kodkod.engine.satlab.ReductionStrategy#next(kodkod.engine.satlab.ResolutionTrace)
	 */
	public IntSet next(ResolutionTrace trace) {
		if (varsToTry.isEmpty()) return Ints.EMPTY_SET;
		// if the last attempt at reduction was unsuccessful,
		// add the unit clauses that we tried to discard back to coreVars
		coreVars.addAll(StrategyUtils.coreTailUnits(trace));
		final int first = varsToTry.iterator().next();//varsToTry.min();
		varsToTry.remove(first);
		coreVars.remove(first);
		// get all axioms corresponding to the clauses that
		// form the translations of formulas identified by coreVars
		final IntSet relevantClauses = StrategyUtils.clausesFor(trace, coreVars); 
		assert !relevantClauses.isEmpty() && !relevantClauses.contains(trace.size()-1);
		return relevantClauses;
	}

}
