/*********************************************************************
 * File: ltxo_pri.c 
 * This file contains private routines for LTxo.  It contains various
 * private routines & the verbatim routines.  The fill mode routines
 * are in ltxo_fill.c
 * It contains routines written by John Barba '88 and Linh Giang '89.  
 *
 * Most of the answers given by these routines are held in global
 * variables declared in the beginning of the file ltxo.c.  The reason
 * behind this little break in modularity is because the private
 * routines are recursive and therefore the line by line information
 * of the text are held in variables declared outside of the routines.
 *********************************************************************/

#include <stdio.h>		/* C include files */
#include <strings.h>

#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>

#include <ltxo.h>	        /* Public LTxo include file */
#include <ltxo_pri.h>		/* Private LTxo include file */

/* extern variables defined in ltxo.c */
extern Display *LTx_dpy;
extern char *calloc();

/* tables defined in ltxo_tbl.c */
extern font_map font_tbl[][MAXSIZE]; 
extern sym_map sym_tbl[][MAXSYMSTRUCT]; 

/* default GCs created in LTxLoadFont in ltxo.c */
extern GC LTxText_gc;  

/* external private routines for verbatim mode call from ltxo.c */
char *text_width(), *text_height(), *text_out();

int ltxErrFlag;			/* use in ltxo.c too */

/* external private routines used in ltxo_fill.c as well */
char findmark();
void text_draw();
unsigned findbestsqrt();
	
/* The following extern variables are used by almost every routine in 
   this file which holds private routines.  The reason behind having
   these extern variables is that most of the width and height finding
   routines are recursive and therefore require extern variables to
   hold line by line information.  As a convention in LTxo, the extern
   variables have a capitalized first letter. */ 

extern Window Win;		/* window to draw text on */
extern XRectangle Box;          /* ltxo bounding box */
extern unsigned Boxwidth, Boxheight;   /* width & height of 
					  bounding box in pixels */
extern int X, Y;              /* x,y of bounding box in pixels - use 
			         for determining the x,y of baseline */

extern unsigned long Foregd, Backgd; /* holds preferred back & 
					foreground of Box rectangle */

extern int Xbase, Ybase;          /* x,y of baseline to draw text */

extern unsigned Current;	     /* current line index */
extern unsigned MaxCurrent;	/* the actual max lines of text output */

extern int *Linewidth, *Lineheight;   /* arrays holding each text 
				         line's width and height */

extern int *Lineascent;		/* array holding each text line's 
				   ascent - use for determining
				   the baseline. */

extern int *Linenumchar;	/* array holding num of chars per line */

extern int Currentlength;	/* length of current line -- use for 
				   fill_text_height & fill_text_out */

/* The following holds information about symbol & math formatting */
extern unsigned Count, MaxCount; 
extern int *Topwidth, *Botwidth; 
extern int *Topdescent, *Botascent;
extern int Overflag;



/**********************************************************************
 * Private Routine: text_width(fid, sid, total, text, endmark) 
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  total   = the running total of current line width.
 *  text    = the text to be displayed.
 *  endmark = the marker which tells us that the text using the 
 *            current font is finished. Endmarks are either 
 *            }, ], ), or >.                   
 * Description: 
 *  This routine computes the width of complex text and calls itself
 *  recursively to find the width of each individual line (set in
 *  Linewidth).  In order to display everything with no clipping, the
 *  bounding box width has to be equal or greater than the maximum of 
 *  all the line widths.
 **********************************************************************/

char *text_width(fid, sid, total, text, endmark) 
     unsigned fid, sid;
     int *total; 
     char *text, endmark; 
{ 
  unsigned i = 0;           /* index to the characters in text */
  unsigned tmpsid,bestsid;  /* temp size ids */ 
  unsigned tmpfid;	    /* temp font id */ 
  int offset;		    /* offset to in the text array */
  unsigned tmpi;	    /* temp index to characters */ 
  unsigned tmpskip;         /* number of blank lines requested */ 
  char tmpmark, tmpmark2;	/* temp endmarks */
  int toptotal, bottotal;	/* hold recursive calls widths */
  int thiscount, symwidth;	/* count to symbol arrays & width var.*/

#ifdef LTXO_DEBUG 
  printf("text_width(%d, %d, %d, %s, %c)\n",fid,sid,*total,text,endmark); 
#endif 

  /* The following while loop deciphers the characters in the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call text_width
     recursively to add up total width. */ 

  while(1) {
#ifdef LTXO_DEBUG 
    printf("Loop: i: %d text: %s \n", i, text);
#endif 
    if( *(text + i) == NULLCHAR ) {
      /* Came to end of text. Add up total line width of current line, 
         set it to Linewidth array and return rest of text. */
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	
	return(NULLCHAR);
      }
      if(i) 
	*total += XTextWidth(get_font_struct(fid, sid), text,(int)i);
      Linewidth[Current] = *total;
      MaxCurrent = Current;
      MaxCount = Count;
      
      return(text + i);
    }
    else if(*(text + i) == endmark) {
      /* Came to end of command. Add to total width so far, and return
	 rest of text skipping endmark. */
      if(i) 
	*total += XTextWidth(get_font_struct(fid, sid), text,(int)i);
      
      return(text + i + 1);
    }
    else if( *(text + i) == NL || *(text + i) == CR ) { 
      /* Came to an implicit line break. Get total width of current 
         line, point to next line, clear i to start at beginning of 
         next line, and set *total to zero to get width of next line. */
      if(i) 
	*total += XTextWidth(get_font_struct(fid, sid), text, (int)i); 
      Linewidth[Current++] = *total;
      text += i;
      i = *total = 0;
    }

    if( *(text + i) == '@' ) {	/* '@' command indicator */
      if(i) 
	*total += XTextWidth(get_font_struct(fid, sid), text,(int)i);
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:      /* user really wants an '@' character */
	text++;      
	i++;
	offset = NONE;	/* no recursive call necessary*/
	break;
      case COM_BOLD:   /* bold & check for recursive bold */
	tmpfid = ((fid == HELVET_BOLD || fid == TIMES_BOLD) ? fid :
		  findbold(fid)); 
	tmpsid = sid;
	offset = 3;
	break;
      case COM_ITA:   /* italics & check for recursive italics */
	tmpfid = ((fid == HELVET_ITALICS || fid == TIMES_ITALICS) ? 
		  fid : finditalic(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_GREEK:     /* greek */
	tmpfid = GREEK;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_HELVET:     /* helvetica*/
	tmpfid = HELVET;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_TIMES:     /* times */
	tmpfid = TIMES;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_FIX:     /* fixed */
	tmpfid = FIXED;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_CENTER:     /* center */
	tmpfid = fid;
	tmpsid = sid;
	offset = 8;
	break;
      case COM_C:          /* center */
      case COM_RIGHT:      /* right flush */
      case COM_U:          /* underline */
	tmpfid = fid;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_SUPER:		/* superscripts */
      case COM_SUB:		/* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	tmpfid = fid;
	offset = 3;
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_ALEPH][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_AND][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_ANGLE][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_APPROX][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AST:		/* asterisk symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_AST][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_BULLET][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EXISTS:		/* exists symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_EXISTS][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_EMPTYSET][tmpsid].width;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EQV:		/* equivalence symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_EQV][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_FORALL][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_GTE][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_IN:		/* IN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_IN][tmpsid].width;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_INFTY][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INTER:		/* INTER symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_INTER][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_LTE][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_MULT:		/* multiplication symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_MULT][tmpsid].width;
	text += 5;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_NABLA][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_NEQ][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_NOTIN][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_OR:		/* OR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_OR][tmpsid].width;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_PARTIAL][tmpsid].width;
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PM:		/* PM symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_PM][tmpsid].width;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_PRSUBSET][tmpsid].width;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_PRSUPSET][tmpsid].width;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_SIMILIAR][tmpsid].width;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_SUBSET][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_SUPSET][tmpsid].width;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UNION:		/* UNION symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_UNION][tmpsid].width;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_LEFTARROW][tmpsid].width;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UPARROW:		/* uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_UPARROW][tmpsid].width;
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_RIGHTARROW][tmpsid].width;
	text += 11;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DOWNARROW][tmpsid].width;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DBLEFTARROW][tmpsid].width;
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DBUPARROW][tmpsid].width;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DBRIGHTARROW][tmpsid].width;
	text += 13;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DBDOWNARROW][tmpsid].width;
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOT:	    /* dot - alternative to mult */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*total += sym_tbl[SYM_DOT][tmpsid].width;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case COM_VEC:           /* vector symbol over variable */
	thiscount = Count++;
	toptotal = 0;
	if ((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text = text_width(fid, sid, &toptotal, text + 5, tmpmark);
	Topwidth[thiscount] = toptotal;
	offset = NONE;
	break;
      case COM_OVER:		/* math - over */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if((tmpmark = findmark(*(text + 5))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 6;	
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );	
	while ( *text != tmpmark ) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 3))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+4, tmpmark2);
	    break;
	  case SUBCOM_DENOM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 5))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+6, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text); 
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal+2; /* give some space for line */
	Botwidth[thiscount] = bottotal+2;
	*total += max(toptotal, bottotal) + 2;
	offset = NONE;
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	if ((tmpmark = findmark(*(text + 5))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	toptotal = get_font_struct(fid, sid)->ascent;
	bottotal = get_font_struct(fid, sid)->descent;
	(void)text_height(fid,sid,&toptotal,&bottotal,text+6,tmpmark);
	bestsid = findbestsqrt(toptotal + bottotal);
	if ( bestsid >= BIG )
	  *total +=
	    XTextWidth(get_font_struct(SYMBOL, bestsid) ,BIGSQRT,1); 
	else
	  *total += sym_tbl[SYM_SQRT][bestsid].width;
	toptotal = 0;
	text = text_width(fid, sid, &toptotal, text+6, tmpmark);
	Topwidth[thiscount] = toptotal;
	*total += toptotal;
	offset = NONE;
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  *total += XTextWidth(get_font_struct(SYMBOL, sid), BIGINT,1);
	else
	  *total += sym_tbl[SYM_INT][sid].width;
	if ((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2); 
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*total += max(toptotal, bottotal);
	offset = NONE;
	break;
      case COM_SUM:		/* math - summation */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  symwidth = XTextWidth(get_font_struct(SYMBOL, sid), BIGSIG,1);
	else
	  symwidth = sym_tbl[SYM_SIG][sid].width;
	if((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2); 
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*total += max(symwidth, max(toptotal, bottotal));
	offset = NONE;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  symwidth = XTextWidth(get_font_struct(SYMBOL, sid), BIGPI,1);
	else
	  symwidth = sym_tbl[SYM_PI][sid].width;
	if ((tmpmark = findmark(*(text + 5))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2); 
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*total += max(symwidth, max(toptotal, bottotal));
	offset = NONE;
	break;
      case COM_SPACE:		/* explicit spacing */
	tmpi = tmpskip = 1;
	if (delimiter(*(text+2))) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi);
	*total += tmpskip*XTextWidth(get_font_struct(sid, fid)," ",1);
	text += tmpi+1;
	offset = NONE;
	break;
      case COM_NL:
	/* Came to an explicit line break. Check if there are more 
	   than one break line requested. Get total width of 
	   current line, point to next line, clear i to start at
	   beginning of next line, and set *total to zero to get
	   width of next line. */ 
	
	tmpi = tmpskip = 1;
	if (delimiter(*(text+2)))
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi);
	Linewidth[Current] = *total;
	Current += tmpskip;
	text += tmpi + 1;
	*total = 0;
	offset = NONE;
	break;
      default:
	ltxErrAlert("Unknown command!", text);
	
	return(NULLCHAR);
      }
      /* After deciphering the command, check if the offset is set.
	 If it is, make necessary recursive calls. */
      if (offset) {
	if ((tmpmark = findmark(*(text+(offset-1)))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	else
	  text = text_width(tmpfid, tmpsid, total, text+offset, tmpmark);
      }
    }
    else
      i++;			/* just continue deciphering on */
                                /* current mode */ 
  }
}



/**********************************************************************
 * Private Routine: text_height(fid, sid, atotal, dtotal, text, endmark)
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  atotal  = the highest total ascent of current line.
 *  dtotal  = the highest total descent of current line.
 *  text    = the text to be displayed.
 *  endmark = the marker which tells us that the text using the 
 *            current font is finished. Endmarks are either 
 *            }, ], ), or >.                   
 * Description: 
 *  This routine computes the complex text and calls itself
 *  recursively to find the height of each individual line (set in
 *  Lineheight and Lineascent) as well as the running height total of
 *  text (set in total).
 **********************************************************************/

char *text_height(fid, sid, atotal, dtotal, text, endmark) 
     unsigned fid, sid;
     int *atotal, *dtotal; 
     char *text, endmark; 
{ 
  unsigned i = 0;        /* index to the characters in text */
  unsigned tmpsid;	 /* temp size id being used */ 
  unsigned tmpfid;	 /* temp font id being used */ 
  int offset;		 /* offset to text array  */
  unsigned tmpi;	 /* temp index to characters */ 
  unsigned tmpskip;      /* number of blank lines requested */ 
  char tmpmark, tmpmark2;	/* temp endmarks */
  int tmp, thiscount, overoffset;
  int topatotal, topdtotal;	/* for recursive calls */
  int botatotal, botdtotal;
  int sqrtoffset = 2;

#ifdef LTXO_DEBUG 
  printf("text_height(%u, %u, %d, %d, %s, %c)\n", fid, sid, *atotal,
	 *dtotal, text, endmark);  
#endif 

  /* The following while loop deciphers the characters in the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call text_height
     recursively to determine maximum height needed for current line
     total */ 

  while(1) {
    if( *(text + i) == NULLCHAR ) {
      /* Came to end of text. Add up total line width of current line, 
         set it to line array and return rest of text. */
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	
	return(NULLCHAR);
      }
      Lineheight[Current] = *atotal + *dtotal;
      Lineascent[Current] = *atotal;
      MaxCurrent = Current;
      MaxCount = Count;
      
      return(text + i);
    }
    else if(*(text + i) == endmark) {
      /* Came to end of command. Just return rest of text skipping
         the endmark */
      
      return(text + i + 1);
    }
    else if( *(text + i) == NL || *(text + i) == CR ) { 
      /* Came to an implicit line break. Get total width of current 
         line, point to next line, clear i to start at beginning of 
         next line, and set *total to zero to get width of next line.*/
      Lineheight[Current] = *atotal + *dtotal;
      Lineascent[Current++] = *atotal;
      text += i;
      i = 0;
      *atotal = get_font_struct(fid, sid)->ascent;
      *dtotal = get_font_struct(fid, sid)->descent;
    }

    if(*(text + i) == '@') {
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:        /* user really wants an '@' character */
	text++;       
	i++;		
	offset = NONE;	/* no recursive call necessary*/
	break;
      case COM_BOLD:   /* bold & check for recursive bold */
	tmpfid = ((fid == HELVET_BOLD || fid == TIMES_BOLD) ? fid
		 : findbold(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_ITA:   /* italics & check for recursive italics */
	tmpfid = ((fid == HELVET_ITALICS || fid == TIMES_ITALICS) ? fid
		 : finditalic(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_GREEK:    /* greek */
	tmpfid = GREEK;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_HELVET:    /* helvetica*/
	tmpfid = HELVET;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_TIMES:    /* times */
	tmpfid = TIMES;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_FIX:    /* fixed */
	tmpfid = FIXED;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_CENTER:	/* center */
	tmpfid = fid;
	tmpsid = sid;
	offset = 8;
	break;
      case COM_C:	/* center */
      case COM_RIGHT:  	/* right flush */
      case COM_U:	/* underline */
	tmpfid = fid;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	  *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	  *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	  *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	  *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	  *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6 );
	if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
	  *atotal = get_font_struct(fid, tmpsid)->ascent;
	if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
	    *dtotal = get_font_struct(fid, tmpsid)->descent;
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1 );
	tmpfid = fid;
	offset = 4;
	  break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_SUPER:	/* superscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	topatotal = get_font_struct(fid, tmpsid)->ascent;  
	topdtotal = get_font_struct(fid, tmpsid)->descent;  
	if ((tmpmark = findmark(*(text + 2))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text = text_height(fid, tmpsid, &topatotal, &topdtotal, 
			   text+3, tmpmark);
	tmp = script_offset(fid,sid) + topatotal;
	if ( tmp > *atotal ) *atotal = tmp;
	offset = NONE;
	break;
      case COM_SUB:	/* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	botatotal = get_font_struct(fid, tmpsid)->ascent;  
	botdtotal = get_font_struct(fid, tmpsid)->descent;  
	if ((tmpmark = findmark(*(text + 2))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text = text_height(fid, tmpsid, &botatotal, &botdtotal, 
			   text+3, tmpmark);
	tmp = script_offset(fid,sid) + botdtotal;
	if ( tmp > *dtotal ) *dtotal = tmp;
	offset = NONE;
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AST:		/* asterisk symbol */
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EXISTS:		/* exists symbol */
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EQV:		/* equivalence symbol */
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_IN:		/* IN symbol */
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INTER:		/* INTER symbol */
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_MULT:		/* multiplication symbol */
	text += 5;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_OR:		/* OR symbol */
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PM:		/* PM symbol */
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UNION:		/* UNION symbol */
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UPARROW:		/* uparrow */
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	text += 11;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	text += 13;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOT:	    /* dot - alternative to mult */	
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case COM_VEC:	    /* vector symbol over variable */
	thiscount = Count++;
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	tmp = get_font_struct(fid, tmpsid)->ascent +
	      get_font_struct(fid, tmpsid)->descent +
	      get_font_struct(fid, sid)->ascent;    
	if ( tmp > *atotal ) *atotal = tmp;
	tmpfid = fid;
	tmpsid = sid;
	offset = 5;
	break;
      case COM_OVER:		/* math - over */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );	
	if (Overflag) 
	  overoffset = 0;
	else 
	  overoffset = (get_font_struct(fid, sid)->ascent +
			get_font_struct(fid, sid)->descent)/2;
	Overflag++;
	while ( *text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 3))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 4, tmpmark2);
	    tmp = topatotal + topdtotal + overoffset;
	    if ( tmp > *atotal ) *atotal = tmp;
	    break;
	  case SUBCOM_DENOM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 5))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 6, tmpmark2);
	    tmp = botatotal + botdtotal - overoffset;
	    if ( tmp > *dtotal ) *dtotal = tmp;
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	Overflag--;
	text++;		/* skips endmark */
	offset = NONE;
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	botatotal = get_font_struct(fid, sid)->ascent;
	topdtotal = get_font_struct(fid, sid)->descent;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text = text_height(fid, sid, &botatotal, &topdtotal,
			   text + 6, tmpmark);
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal + sqrtoffset;
	tmp = sqrtoffset + botatotal;
	if ( tmp > *atotal )
	  *atotal = tmp;
	offset = NONE;
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	tmp = script_offset(fid,sid) + topatotal;
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = script_offset(fid,sid) + botdtotal;
	if ( tmp > *dtotal ) *dtotal = tmp;
	offset = NONE;
	break;
      case COM_SUM:		/* math - summation */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	tmp = topatotal + topdtotal +
	  get_font_struct(SYMBOL, sid)->ascent; 
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = botatotal + botdtotal +
	  get_font_struct(SYMBOL, sid)->descent; 
	if ( tmp > *dtotal )  *dtotal = tmp;
	offset = NONE;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 5))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	tmp = topatotal + topdtotal +
	  get_font_struct(SYMBOL, sid)->ascent; 
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = botatotal + botdtotal +
	  get_font_struct(SYMBOL, sid)->descent; 
	if ( tmp > *dtotal ) *dtotal = tmp;
	offset = NONE;
	break;
      case COM_SPACE:		/* explicit spacing */
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi); 
	text += tmpi+1;
	offset = NONE;
	break;
      case COM_NL: 
	/* Came to an explicit line break. Check if there are more 
	   than one break line requested.  Get total height of
	   current line, point to next line, clear i to start at
	   beginning of next line. */
	
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) )
	    tmpskip = getnumskip(text, findmark(*(text + 2)), &tmpi);  
	Lineheight[Current] = *atotal + *dtotal;
	Lineascent[Current] = *atotal;
	Current += tmpskip;
	text += tmpi + 1;
	*atotal = get_font_struct(fid, sid)->ascent;
	*dtotal = get_font_struct(fid, sid)->descent;
	offset = NONE;
	break;
      default:
	ltxErrAlert("Unknown command!", text);
	
	return(NULLCHAR);
      }
      /* After deciphering the command, do the necessary recursive 
	 calls. */
      if (offset) {
	if ((tmpmark = findmark(*(text + (offset - 1)))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  
	  return(NULLCHAR);
	}
	else
	  text = text_height(tmpfid, tmpsid, atotal, dtotal, 
			   text + offset, tmpmark);
      }
    }
    else
      i++;                    /* just continue deciphering on */
                              /* current mode */ 
  }
}



/**********************************************************************
 * Private Routine: text_out(fid, sid, text, flags, endmark) 
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  text    = the text to be displayed.
 *  flags   = flags giving hints as to how to output text. 
 *  endmark = the marker which tells us that the text using the 
 *            current font is finished. Endmarks are either 
 *            }, ], ), or >.                   
 * Description: 
 *  This routine deciphers the complex text and calls another routine,
 *  text_draw, to actually draw the text onto the screen.  
 **********************************************************************/

char *text_out(fid, sid, text, flags, endmark)
     unsigned fid, sid;
     long flags;
     char *text, endmark;
{
  unsigned i = 0;            /* index to the characters in text */
  unsigned tmpsid,bestsid;   /* temp size ids */ 
  unsigned tmpfid,tmpi;	     /* temp index to characters */ 
  char tmpmark;	             /* temp endmark found */ 
  unsigned tmpskip;          /* number of blank lines requested */ 
  int savebase, tmp;		/* temp vars for Xbase & misc */
  int thiscount, width;
  int overoffset;

#ifdef LTXO_DEBUG
  printf("text_out(%d, %d, %s, %d, %c)\n",fid,sid,text,flags,endmark); 
#endif

  /* The following while loop deciphers the characters in the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call text_out
     recursively to output text under desired mode. */ 

  while(1) {
    if( *(text + i) == NULLCHAR ) {
       /* Came to end of text. If there are already some deciphered 
         text, display it first, and then return rest of text. */
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	
	return(NULLCHAR);
      }
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
      }
      
      return(text + i);
    }
    else if(*(text + i) == endmark) {
      /* Came to end of command.  If there are some deciphered text, 
         display it first, and then return the text after the endmark
	 symbol. */
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
	Xbase += width;  /* advance Xbase to continue output */
      }
      
      return(text + i + 1);  
    }
    else if( *(text + i) == NL || *(text + i) == CR ) { 
      /* Came to an implicit line break. Draw text of current line, 
         point to next line, clear i to start from beginning, and
	 increment Ybase to for the next baseline. */
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
      }
      Ybase += (Lineheight[Current] - Lineascent[Current]) +
                Lineascent[Current + 1];
      Current++;
      text += i;
      i = 0;
      if(flags & H_CENTER) 
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current]) / 2; 
      else if(flags & H_RIGHT) 
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current]); 
      else
        Xbase= Box.x + X;
    }

    if(*(text + i) == '@') {
      /* Output deciphered text that is in front of the '@' */
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
	Xbase += width;  /* advance Xbase to continue output */
      }
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:        /* user really wants an '@' character */
	text++;
	i++; 
	break;
      case COM_BOLD:   /* bold */
	/* already in bold face */
	tmpfid = ((flags & BOLD) ? fid : findbold(fid));
	text = text_out(tmpfid, sid, text + 3, flags | BOLD,
			findmark(*(text + 2))); 
	break;
      case COM_ITA:   /* italics */
	/* already in italic face */
	tmpfid = ((flags & ITALICS) ? fid : finditalic(fid));
	text = text_out(tmpfid, sid, text + 3, flags | ITALICS,
			findmark(*(text + 2))); 
	break;
      case COM_GREEK:      /* greek */
	text = text_out((unsigned)GREEK, sid, text + 3,
			flags | GRK, findmark(*(text + 2)));
	break;
      case COM_HELVET:      /* helvetica*/
	text = text_out((unsigned)HELVET, sid, text + 3,
			flags, findmark(*(text + 2)));
	break;
      case COM_TIMES:      /* times */
	text = text_out((unsigned)TIMES, sid, text + 3,
			flags, findmark(*(text + 2)));
	break;
      case COM_FIX:      /* fixed */
	if(flags & (ITALICS | BOLD)) {
	  ltxErrAlert("No italics or bold in fixed font",text);
	  
	  return(NULLCHAR);
	}	  
	text = text_out((unsigned)FIXED, sid, text + 3,
			flags, findmark(*(text + 2)));
	break;
      case COM_CENTER:	/* center */
	if(flags & H_CENTER) { /* already centering */
	  ltxErrAlert("Recursive centering!", text);
	  
	  return(NULLCHAR);
	}
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current])/2;
	text = text_out(fid, sid, text + 8,
			flags | H_CENTER, findmark(*(text + 7)));
	break;
      case COM_C:	/* center */
	if(flags & H_CENTER) { /* already centering */
	  ltxErrAlert("Recursive centering!", text);
	  
	  return(NULLCHAR);
	}
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current])/2;
	text = text_out(fid, sid, text + 3,
			flags | H_CENTER, findmark(*(text + 2)));
	break;
      case COM_RIGHT:   	/* right flush */
	if(flags & H_RIGHT) { /* already flushing right */
	  ltxErrAlert("Recursive right flush!", text);
	  
	  return(NULLCHAR);	  
	}
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current]);
	text = text_out(fid, sid, text + 3,
			flags | H_RIGHT, findmark(*(text + 2)));
	break;
      case COM_U:	/* underline */
	text = text_out(fid, sid, text + 3,
			flags | UNDERLINE, findmark(*(text + 2)));
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6 );
	text = text_out(fid, tmpsid, text + 4,
			flags, findmark(*(text + 3)));
	break;
      case COM_SUPER:	/* superscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	Ybase -= script_offset(fid, sid);
	text = text_out(fid, tmpsid, text + 3,
                          flags, findmark(*(text + 2)));
	Ybase += script_offset(fid, sid);
	break;
      case COM_SUB:	/* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	Ybase += script_offset(fid, sid);
	text = text_out(fid, tmpsid, text + 3,
			flags, findmark(*(text + 2)));
	Ybase -= script_offset(fid, sid);
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_ALEPH][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,ALEPH,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_AND][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,AND,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_ANGLE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,ANGLE,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_APPROX][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,APPROX,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_AST:		/* asterisk symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_AST][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,AST,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_BULLET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,BULLET,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_EXISTS:		/* exists symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EXISTS][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EXISTS,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EMPTYSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EMPTYSET,1,width,flags | SYM);
	Xbase += width;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_EQV:		/* equivalence symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EQV][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EQV,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_FORALL][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,FORALL,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_GTE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,GTE,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_IN:		/* IN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_IN][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,IN,1,width,flags | SYM);
	Xbase += width;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_INFTY][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,INF,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_INTER:		/* INTER symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_INTER][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,INTER,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_LTE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,LTE,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_MULT:		/* multiplication symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_MULT][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,MULT,1,width,flags | SYM);
	Xbase += width;
	text += 5;
	if (*text == SP) text++;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NABLA][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NABLA,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NEQ][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NEQ,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NOTIN][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NOTIN,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_OR:		/* OR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_OR][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,OR,1,width,flags | SYM);
	Xbase += width;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PARTIAL][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PAR,1,width,flags | SYM);
	Xbase += width;
	text += 8;
	if (*text == SP) text++;
	break;
      case SYM_PM:		/* PM symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PM][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PM,1,width,flags | SYM);
	Xbase += width;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PRSUBSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PRSUBSET,1,width,flags | SYM);
	Xbase += width;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PRSUPSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PRSUPSET,1,width,flags | SYM);
	Xbase += width;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SIMILIAR][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SIMILIAR,1,width,flags | SYM);
	Xbase += width;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SUBSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SUBSET,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SUPSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SUPSET,1,width,flags | SYM);
	Xbase += width;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_UNION:		/* UNION symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_UNION][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,UNION,1,width,flags | SYM);
	Xbase += width;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_LEFTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,LEFTARROW,1,width,flags | SYM);
	Xbase += width;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_UPARROW:		/* uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_UPARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,UPARROW,1,width,flags | SYM);
	Xbase += width;
	text += 8;
	if (*text == SP) text++;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_RIGHTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,RIGHTARROW,1,width,flags | SYM);
	Xbase += width;
	text += 11;
	if (*text == SP) text++;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DOWNARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DOWNARROW,1,width,flags | SYM);
	Xbase += width;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBLEFTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBLEFTARROW,1,width,flags | SYM);
	Xbase += width;
	text += 12;
	if (*text == SP) text++;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBUPARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBUPARROW,1,width,flags | SYM);
	Xbase += width;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBRIGHTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBRIGHTARROW,1,width,flags | SYM);
	Xbase += width;
	text += 13;
	if (*text == SP) text++;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBDOWNARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBDOWNARROW,1,width,flags | SYM);
	Xbase += width;
	text += 12;
	if (*text == SP) text++;
	break;
      case SYM_DOT:	    /* dot - alternative to mult */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DOT][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DOT,1,width,flags | SYM);
	Xbase += width;
	text += 4;
	if (*text == SP) text++;
	break;
      case COM_VEC:          /* vector symbol over variable */
	thiscount = Count++;
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	savebase = Xbase;
	text = text_out(fid, sid, text + 5,
			flags, findmark(*(text + 4)));
	Xbase = savebase;
	Ybase -= get_font_struct(fid, sid)->ascent;
	text_draw((unsigned)SYMBOL,tmpsid,RIGHTARROW,1,0,flags | SYM);
	Ybase += get_font_struct(fid, sid)->ascent;
	Xbase += Topwidth[thiscount];
	break;
      case COM_OVER:		/* math - over */
	thiscount = Count++;
	tmpmark = findmark(*(text + 5));
	text += 6;
	/* smaller size for fractions */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid ); 
	tmp = max(Topwidth[thiscount],Botwidth[thiscount]);
	savebase = Xbase;
	if (Overflag) 
	  overoffset = 0;
	else 
	  overoffset = (get_font_struct(fid, sid)->ascent +
			get_font_struct(fid, sid)->descent)/2;
	Overflag++;
	while ( *text != tmpmark ) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    Xbase += ((tmp - Topwidth[thiscount])/2) + 1;
	    Ybase -= Topdescent[thiscount] + overoffset;
	    text = text_out(fid, tmpsid, text + 4, flags,
			    findmark(*(text + 3))); 
	    Xbase = savebase;
	    Ybase += Topdescent[thiscount] + overoffset;
	    break;
	  case SUBCOM_DENOM:
	    Xbase += ((tmp - Botwidth[thiscount])/2) + 1;
	    Ybase += Botascent[thiscount] - overoffset;
	    text = text_out(fid, tmpsid, text + 6, flags,
			    findmark(*(text + 5))); 
	    Xbase = savebase;
	    Ybase -= Botascent[thiscount] - overoffset;
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Overflag--;
	XSetLineAttributes(LTx_dpy, LTxText_gc,
			   (sid >= BIG ? 3 : 1) , 
			   LineSolid,
			   CapButt, JoinMiter);

	XDrawLine(LTx_dpy, Win, 
		  LTxText_gc, 
		  Xbase, Ybase-overoffset-2, Xbase+tmp, 
		  Ybase-overoffset-2);  
	Xbase += tmp;
	text++;		/* skip endmark */
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	tmp = Botascent[thiscount]+Topdescent[thiscount];
	bestsid = findbestsqrt(tmp);
	Ybase += get_font_struct(SYMBOL, bestsid)->ascent -
	         Botascent[thiscount]; 
	if ( bestsid >= BIG ) {
	  width =
	    XTextWidth(get_font_struct(SYMBOL, bestsid),BIGSQRT,1);  
	  text_draw((unsigned)SYMBOL,bestsid,BIGSQRT,1,width,
		    flags | SYM); 
	}
	else {
	  width = sym_tbl[SYM_SQRT][bestsid].width;
	  text_draw((unsigned)SYMBOL,bestsid,SQRT,1,width,flags | SYM);
	}
	Ybase -= get_font_struct(SYMBOL, bestsid)->ascent -
	         Botascent[thiscount]; 
	Xbase += width;
	savebase = Xbase;
	text = text_out(fid, sid, text + 6, flags,
			findmark(*(text + 5)));
	XSetLineAttributes(LTx_dpy, LTxText_gc, 
			   (sid >= BIG ? 3 : 1), LineSolid,
			   CapButt, JoinMiter);
	XDrawLine(LTx_dpy, Win,
		  LTxText_gc,
		  savebase, Ybase-Botascent[thiscount], 
		  savebase+Topwidth[thiscount],
		  Ybase-Botascent[thiscount]); 
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	if ( sid >= BIG ) {
	  width = XTextWidth(get_font_struct(SYMBOL, sid), BIGINT,1); 
	  text_draw((unsigned)SYMBOL,sid,BIGINT,1,width,flags | SYM);
	}
	else {
	  width = sym_tbl[SYM_INT][sid].width;
	  text_draw((unsigned)SYMBOL,sid,INT,1,width,flags | SYM);
	}
	Xbase += width;
	savebase = Xbase;
	tmpmark = findmark(*(text + 4));
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Ybase -= script_offset((unsigned)SYMBOL, sid);
	    text = text_out(fid, tmpsid, text + 3,
			    flags, findmark(*(text + 2)));
	    Ybase += script_offset((unsigned)SYMBOL, sid);
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Ybase += script_offset((unsigned)SYMBOL, sid);
	    text = text_out(fid, tmpsid, text + 5,
			    flags, findmark(*(text + 4)));
	    Ybase -= script_offset((unsigned)SYMBOL, sid);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Xbase = savebase;
	text++;		/* skips endmark */
	Xbase += max(Topwidth[thiscount], Botwidth[thiscount]);
	break;
      case COM_SUM:		/* math - summation */
	thiscount = Count++;
	savebase = Xbase;
	if ( sid >= BIG ) 
	  width = XTextWidth(get_font_struct(SYMBOL, sid), BIGSIG,1);
	else 
	  width = sym_tbl[SYM_SIG][sid].width;
	tmp = max(width,
		  max(Topwidth[thiscount],Botwidth[thiscount]));
	Xbase += (tmp - width)/2;
	if ( sid >= BIG ) 
	  text_draw((unsigned)SYMBOL,sid,BIGSIG,1,width,flags | SYM);
	else
	  text_draw((unsigned)SYMBOL,sid,SIG,1,width,flags | SYM);
	tmpmark = findmark(*(text + 4));
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Xbase += (tmp - Topwidth[thiscount])/2;
	    Ybase -= get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    text = text_out(fid, tmpsid, text + 3, flags,
			    findmark(*(text + 2)));
	    Ybase += get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Xbase += (tmp - Botwidth[thiscount])/2;
	    Ybase += get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    text = text_out(fid, tmpsid, text + 5, flags,
			    findmark(*(text + 4)));
	    Ybase -= get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Xbase = savebase;
	Xbase += tmp;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	savebase = Xbase;
	if ( sid >= BIG ) 
	  width = XTextWidth(get_font_struct(SYMBOL, sid),BIGPI,1);
	else 
	  width = sym_tbl[SYM_PI][sid].width;
	tmp = max(width, max(Topwidth[thiscount],
				    Botwidth[thiscount])); 
	Xbase += (tmp - width)/2;
	if ( sid >= BIG ) 
	  text_draw((unsigned)SYMBOL,sid,BIGPI,1,width,flags | SYM);
	else 
	  text_draw((unsigned)SYMBOL,sid,PI,1,width,flags | SYM);
	tmpmark = findmark(*(text + 5));
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Xbase += (tmp - Topwidth[thiscount])/2;
	    Ybase -= get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    text = text_out(fid, tmpsid, text + 3, flags,
			    findmark(*(text + 2)));
	    Ybase += get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Xbase += (tmp - Botwidth[thiscount])/2;
	    Ybase += get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    text = text_out(fid, tmpsid, text + 5, flags,
			    findmark(*(text + 4)));
	    Ybase -= get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Xbase = savebase;
	Xbase += tmp;
	break;
      case COM_SPACE:		/* explicit spacing */
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi); 
	text += tmpi+1;
	Xbase += tmpskip*XTextWidth(get_font_struct(sid, fid)," ",1);
	break;
      case COM_NL: 
	/* Came to an explicit line break. Check if there are more 
	   than one break line requested. Point to next line, clear 
	   i to start from beginning, and increment Ybase to for the 
	   next baseline. */

	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi);

	Ybase += (Lineheight[Current] - Lineascent[Current]) +
	         ((tmpskip-1)*Lineheight[Current]) +  
		   Lineascent[Current+tmpskip]; 
	Current += tmpskip; 
	text += tmpi + 1;
	if(flags & H_CENTER) 
	  Xbase = Box.x + X + (Boxwidth - Linewidth[Current]) / 2; 
	else if(flags & H_RIGHT) 
	  Xbase = Box.x + X + (Boxwidth - Linewidth[Current]); 
	else
	  Xbase = Box.x + X;

	break;
      default:
	ltxErrAlert("Unknown command!", text);
	
	return(NULLCHAR);
      }
    }
    else
      i++;	       /* continue deciphering in current mode */
  }
}



/**********************************************************************
 * Private Routine: text_draw(f, s, text, i, width, flags)
 * Requires: 
 *  f  = the internal font id number used by LTxo.
 *  s  = the internal size id number used by LTxo.
 *  text = the text to be displayed.
 *  i = how many characters to draw.
 *  width = width of text to draw underline if underline mode is set.
 *  flags = flags giving hints as to how to output text and as to 
 *          which GC to use.
 * Description:
 *  This routine actually place the text on screen using the current
 *  mode. Depending on the hints given by "flags", text_draw will use
 *  the appropriate GC to draw the text. 
 **********************************************************************/
void text_draw(f, s, text, i, width, flags)
     unsigned f, s, i;
     char *text;
     int width;
     long flags;
{
  XGCValues vals;
#ifdef LTXO_DEBUG 
  printf("text_draw(%u, %u, %s, %d, %d, %d)\n",f,s,text,i,width,flags); 
#endif 

  XSetFont(LTx_dpy, LTxText_gc, get_font_struct(f, s)->fid);
  XDrawString(LTx_dpy, Win, LTxText_gc, Xbase, Ybase, text, (int)i);

  
  /* if in underline mode, draw the line */
  if(flags & UNDERLINE) {
	
    vals.line_width = s >= BIG ? 3 : 1;
    XChangeGC(LTx_dpy, LTxText_gc, GCLineWidth, &vals);

    XDrawLine(LTx_dpy, Win, LTxText_gc,
	      Xbase, Ybase, Xbase+width, Ybase);}

  
}


/**********************************************************************	
 * Private Routine: findmark(begin)
 * Requires:
 *  The beginning mark
 * Returns:
 *  The correct endmark or 'e' for error
 * Description: 
 *  It simply returns the matching symbol that parenthesizes the text
 *  after the command indicator '@'. 
 **********************************************************************/
char findmark(begin)
     char begin;
{
   char end;

   end = begin == '(' ? ')' : begin == '[' ? ']' : begin == '{' ? '}' :
     begin == '<' ? '>' : 'e';

   return(end); /* return the endmark character */
 }


/**********************************************************************
 * Private Routine: delimiter(begin)
 * Requires:
 *  begin = a character in text.
 * Returns:
 *  1 if char is a delimiter,
 *  0 otherwise.
 * Description:
 *  Checks if the character is a delimiter.
 **********************************************************************/
int delimiter(begin)
     char begin;
{
  int ans;

  ans = begin == '(' ? 1 : begin == '[' ? 1 : begin == '{' ? 1 : begin
    == '<' ? 1 : 0;
  
  return(ans);
}

/**********************************************************************
 * Private Routine: endlimiter(end)
 * Requires:
 *  end = a end delimiter
 * Returns:
 *  1 if char is a delimiter,
 *  0 otherwise.
 * Description:
 *  Checks if the character is a delimiter.
 **********************************************************************/
int endlimiter(end)
     char end;
{
  int ans;

  ans = end == ']' ? 1 : end == ')' ? 1 : end == '}' ? 1 : end
    == '>' ? 1 : 0;

  
  return(ans);
}

/**********************************************************************
 * Private Routine: script_offset(fid, sid)
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 * Returns:
 *  The calculated offset of baseline needed for sub and superscripts
 *  according to the font and size given.
 * Description:
 *  Calculates the needed offset for sub and superscripts.
 **********************************************************************/
int script_offset(fid, sid)
     unsigned fid, sid;
{
  return((get_font_struct(fid, sid)->ascent +
	  get_font_struct(fid, sid)->descent)/2);
}

/**********************************************************************
 * Private Routine: get_font_struct
 * Requires:
 *    fid = the internal font id number used by LTxo
 *    sid = the internal size id number used by LTxo
 * Description:
 *    Returns font structure
 **********************************************************************/

XFontStruct *get_font_struct(fid, sid)
     unsigned fid, sid;
{
  if (!font_tbl[fid][sid].fstruct)
    load_font(fid, sid);
  return(font_tbl[fid][sid].fstruct);
}

/**********************************************************************
 * Private Routine: load_font(fid, sid)
 * Requires:
 *  fid  = the internal font id number used by LTxo.
 *  sid  = the internal size id number used by LTxo.
 * Description:
 *  Just loads in the font requested with XLoadQueryFont()
 **********************************************************************/
void load_font(fid, sid)
     unsigned fid, sid;
{
  if (!font_tbl[fid][sid].fstruct) {
  if ((font_tbl[fid][sid].fstruct = 
       XLoadQueryFont(LTx_dpy, font_tbl[fid][sid].fontfile)) == NULL) {  
    printf("Cannot open font: %s\n", font_tbl[fid][sid].fontfile);
    exit(-1); 
  }}
  
}

/**********************************************************************
 * Private Routine: getcommand_id(str)
 * Requires: 
 *  str = text which has the command.
 * Returns:
 *  an integer that is the command_id.
 * Description:
 *  This routine returns the comamnd_id of a command which is
 *  delimited by either one of the delimiters or a blank space.
 **********************************************************************/
int getcommand_id(str)
     char *str;
{
  char token[MAXCHARCOM+10];
  int i = 0, flag = 1;

  /* This looks awful but it saves alot of typing -- let it stay here */
  /* for easy reference instead of in a .h file */
#define COMPARE(COMM,ID) {if (!(strcasecmp(token, (COMM)))) { \
			       \
			      return((ID)); }}

  /* LErr error trapping */
   

  /* Check for commands with optional delimiters or the no_command id */
  switch(*str) {
  case '@':                 /* really want an '@' */
    
    return(NO_COM);
    break;
  case '.':                 /* newline */
    
    return(COM_NL);
    break;
  case '#':                 /* explicit spacing */
    
    return(COM_SPACE);
    break;
  }

  /* get command word til delimited */
  for (i = 0; i <= MAXCHARCOM+1 ; i++) {
    switch (*(str+i)) {
    case '(':                 /* valid delimiters */
    case '{':
    case '[':
    case '<':
    case ')':
    case '}':
    case ']':
    case '>':
    case ' ':
      if (!i) {	/* special flush right bec. conflicts with delimiter */
	if (*str == '>') continue;
      }
      (void)strncpy(token, str, i);
      token[i] = '\0';		/* make sure null terminated */
      flag = 0;			/* <= MAXCHARCOM */
      i = MAXCHARCOM+2;		/* end loop */
      break;
    }
  }

  /* error if character is greater than MAXCHARCOM */
  if (flag) {
    
    return(ERROR_COM);
  }
  
  switch(*str) {
  case 'a': case 'A':
    COMPARE("ast", SYM_AST);
    COMPARE("approx", SYM_APPROX);
    COMPARE("aleph", SYM_ALEPH);
    COMPARE("and", SYM_AND);
    COMPARE("angle", SYM_ANGLE);
    break;
  case 'b': case 'B':
    COMPARE("b", COM_BOLD);
    COMPARE("bullet", SYM_BULLET);
    break;
  case 'c': case 'C':
    COMPARE("c", COM_C);
    COMPARE("center", COM_CENTER);
    break;
  case 'd': case 'D':
    COMPARE("denom", SUBCOM_DENOM);
    COMPARE("dot", SYM_DOT);
    COMPARE("downarrow", SYM_DOWNARROW);
    COMPARE("dbdownarrow", SYM_DBDOWNARROW);
    COMPARE("dbrightarrow", SYM_DBRIGHTARROW);
    COMPARE("dbleftarrow", SYM_DBLEFTARROW);
    COMPARE("dbuparrow", SYM_DBUPARROW);
    break;
  case 'e': case 'E':
    COMPARE("exists", SYM_EXISTS);
    COMPARE("eqv", SYM_EQV);
    COMPARE("emptyset", SYM_EMPTYSET);
    break;
  case 'f': case 'F':
    COMPARE("f", COM_FIX);
    COMPARE("forall", SYM_FORALL);
    COMPARE("from", SUBCOM_FROM);
    break;
  case 'g': case 'G':
    COMPARE("g", COM_GREEK);
    COMPARE("gte", SYM_GTE);
    break;
  case 'h': case 'H':
    COMPARE("h", COM_HELVET);
    break;
  case 'i': case 'I':
    COMPARE("i", COM_ITA);
    COMPARE("infty", SYM_INFTY);
    COMPARE("int", COM_INT);
    COMPARE("in", SYM_IN);
    COMPARE("inter", SYM_INTER);
    break;
  case 'l': case 'L':
    COMPARE("lte", SYM_LTE);
    COMPARE("leftarrow", SYM_LEFTARROW);
    break;
  case 'm': case 'M':
    COMPARE("mult", SYM_MULT);
    break;
  case 'n': case 'N':
    COMPARE("num", SUBCOM_NUM);
    COMPARE("nabla", SYM_NABLA);
    COMPARE("neq", SYM_NEQ);
    COMPARE("notin", SYM_NOTIN);
    break;
  case 'o': case 'O':
    COMPARE("over", COM_OVER);
    COMPARE("or", SYM_OR);
    break;
  case 'p': case 'P':
    COMPARE("partial", SYM_PARTIAL);
    COMPARE("prod", COM_PROD);
    COMPARE("pm", SYM_PM);
    COMPARE("prsubset", SYM_PRSUBSET);
    COMPARE("prsupset", SYM_PRSUPSET);
    break;
  case 'r': case 'R':
    COMPARE("rightarrow", SYM_RIGHTARROW);
    break;
  case 's': case 'S':
    COMPARE("sqrt", COM_SQRT);
    COMPARE("sum", COM_SUM);
    COMPARE("subset", SYM_SUBSET);
    COMPARE("supset", SYM_SUPSET);
    COMPARE("similiar", SYM_SIMILIAR);
    break;
  case 't': case 'T':
    COMPARE("to", SUBCOM_TO);
    break;
  case 'u': case 'U':
    COMPARE("u", COM_U);
    COMPARE("uparrow", SYM_UPARROW);
    COMPARE("union", SYM_UNION);
    break;
  case 'v': case 'V':
    COMPARE("v1", COM_DOWN1);
    COMPARE("v2", COM_DOWN2);
    COMPARE("vec", COM_VEC);
    COMPARE("v3", COM_DOWN3);
    COMPARE("v4", COM_DOWN4);
    COMPARE("v5", COM_DOWN5);
    COMPARE("v6", COM_DOWN6);
    break;
  case '^':
    COMPARE("^1", COM_UP1);
    COMPARE("^2", COM_UP2);
    COMPARE("^3", COM_UP3);
    COMPARE("^4", COM_UP4);
    COMPARE("^5", COM_UP5);
    COMPARE("^6", COM_UP6);
    break;
  case '>':
    COMPARE(">", COM_RIGHT);
    break;
  case '+':
    COMPARE("+", COM_SUPER);
    break;
  case '-':
    COMPARE("-", COM_SUB);
    break;
  }

  /* get to here --- there is an error --- return error_id */
  
  return(ERROR_COM);

}


/**********************************************************************
 * Private Routine: getnumskip(text, tmpmark, tmpi)
 * Requires:
 *  text = text to decipher.
 *  tmpmark = the delimiter to look for.
 *  tmpi = how many characters to skip from text to continue
 *         deciphering 
 * Returns:
 *  an integer indicating the number of lines or spaces to skip.
 * Description:
 **********************************************************************/
int getnumskip(text, tmpmark, tmpi)
     char *text, tmpmark;
     int *tmpi;
{
  int skip = 0;

  for ( *tmpi = 3; *(text + *tmpi) != tmpmark; (*tmpi)++ ) {
    if ( *(text + *tmpi) >= '0' && *(text + *tmpi) <= '9')
      skip = (10 * skip) + (*(text + *tmpi) - '0');
    else {
      XtWarning("Parameter between delimiter is not a number!");
      
      return(1);      /* just skip 1 line & continue if error */
    }
  }

  if ( skip ) {
    
    return(skip);
  }
  else {
    
    return(1);
  }
}


/**********************************************************************
 * Private Routine: findbestsqrt(height)
 * Requires:
 *  height = height of text that sqrt sign is going over.
 * Returns:
 *  a font size id indicating the best size.
 * Description:
 **********************************************************************/
unsigned findbestsqrt(height)
     int height;
{
  int sym_height[MAXSIZE];
  unsigned size;

  for (size = SMALLEST; size <= BIGGEST; size++) {
    sym_height[size] = get_font_struct(SYMBOL, size)->ascent +
      get_font_struct(SYMBOL, size)->descent; 
    if ( sym_height[size] >= height ) {
      if ( size != SMALLEST ) {
	if ( sym_height[size-1] >= (height-(.4)*(height))) {
	  
	  return( size - 1);
	}
	else {
	  
	  return ( size );
	}
      }
      
      return ( size );
    }
  }

  /* At this point, just return the biggest fitting sqrt */
  
  return(BIGGEST);
}

/***********************************************************************
 * Error Routine: ltxErrAlert(err, text)
 * Description:
 *  Prints out error and the rest of text in which error was found.
 *  Note that ltxErrAlert does not use LTxDrawText bec. it does not
 *  want the error text to be deciphered with the @ char which is what
 *  LTxDrawText does automatically.
 **********************************************************************/
void ltxErrAlert(err, text)
     char *err, *text;
{
  char errormess[100], errortext[50];

  ltxErrFlag = 1;

  *errormess = NULLCHAR;        /* prevent garbage in array */
  *errortext = NULLCHAR;
  (void)strcat(errormess, "LTXERR: ");
  (void)strcat(errortext, "FROM TEXT: ");
  (void)strcat(errormess, err);
  (void)strncat(errortext, text, 10);

  /* printf on tty since we can't count on windows to work */
  printf("%s %s\n",errormess, errortext);
  fflush(stdout);

}
