/* $Date: 92/02/16 08:41:21 $    $Revision: 1.4 $  by $Author: joe $  */
/* 
  Copyright (C) 1991 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */
/**************************************
 CmDialog.cc 
 contact: Marc Schaub (maschaub)

 The dialog box is an object that makes it easy for the applications
 programmer to get input fromt the user.  All that he does is include
 CmDialog.cc, create a dialog object, add items to it and show it.
 He can then get the values out in any of the assorted types available.
 **************************************/
#include <CmDialog.h>
#include <CmStringMotif.h>
#include <stream.h>
#include <stdlib.h>
#ifndef _CDSX
#define _CDSX
extern "C" {
#define class cc_class
#define new cc_new
#define delete cc_delete
#define virtual cc_virtual
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Shell.h>
#include <Xm/Xm.h>
#include <Xm/Text.h>
#include <Xm/DialogS.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/CascadeB.h>
#include <Xm/List.h>
#include <Xm/MessageB.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/Scale.h>
#include <Xm/ScrollBar.h>
#include <Xm/SeparatoG.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#include <Xm/Form.h>
#include <Xm/MainW.h>
#include <Xm/SelectioB.h>
#include <At/Label.h>
#include <stdarg.h>
#undef virtual
#undef delete
#undef new
#undef class
}
#endif

#define DIALOG_MIN_WIDTH 340

/**************************** 
 C procedure to do callbacks 
 ****************************/

void CmInterpretChangedValue(Widget, CmDialogItem* item, XtPointer) { 
  item->Dialog->ValueChanged(item->ItemName); 
}

void CmDialogHelpCallback(Widget, CmDialog *dialog, XtPointer)
{  dialog->DisplayHelp(); }
void CmDialogHelpBoxCallback(Widget, CmDialog *dialog, XtPointer)
{  dialog->UnDisplayHelp(); }
void CmDialogOkCallback(Widget, CmDialog *dialog, XtPointer)
{  dialog->PopdownDialog(); }
void CmDialogCancelCallback(Widget, CmDialog *dialog, XtPointer)
{  dialog->PopdownDialog(); }

/******************* CmDialogPane Member Functions **************************/
/*********
 Panes are rowcolumns with names of panename+Base,Title
*********/
CmDialogPane::CmDialogPane(Widget work_area, const CmString &pname,
			   Widget attach_widget, char attach_method,
			   Boolean display_pane)
{
  Arg arglist[MAXARGS];

  int count;
  PaneName = pname;
  count = 0;
  switch ( attach_method ) {
  case CMDIALOG_ATTACH_NONE:
    XtSetArg(arglist[count], XmNleftAttachment, XmATTACH_FORM); count++;
    XtSetArg(arglist[count], XmNtopAttachment, XmATTACH_FORM);count++;
    break;
  case CMDIALOG_ATTACH_LEFT:
    XtSetArg(arglist[count], XmNleftAttachment, XmATTACH_WIDGET); count++;
    XtSetArg(arglist[count], XmNleftWidget, attach_widget);count++;
    XtSetArg(arglist[count], XmNtopAttachment,
	     XmATTACH_OPPOSITE_WIDGET); count++;
    XtSetArg(arglist[count], XmNtopWidget, attach_widget);count++;
    break; 
  case CMDIALOG_ATTACH_TOP:
    XtSetArg(arglist[count], XmNleftAttachment,
	     XmATTACH_OPPOSITE_WIDGET); count++;
    XtSetArg(arglist[count], XmNleftWidget, attach_widget); count++;
    XtSetArg(arglist[count], XmNtopAttachment, XmATTACH_WIDGET);count++;
    XtSetArg(arglist[count], XmNtopWidget, attach_widget);count++;
    break;
  }
  XtSetArg(arglist[count], XmNmarginWidth, CMDIALOG_PANE_WIDTH); count++;
  XtSetArg(arglist[count], XmNmarginHeight, CMDIALOG_PANE_HEIGHT); count++;
  XtSetArg(arglist[count], XmNspacing, CMDIALOG_PANE_HEIGHT); count++;
  BasePane = XmCreateRowColumn(work_area,"BasePane", arglist, count);
  XtManageChild(BasePane);
  if ( display_pane ) {
    // create title
    count = 0;
    XmString label = CmToXmString(PaneName);
    
    Title = 
      XtVaCreateManagedWidget("PaneTitle", xmLabelWidgetClass, BasePane,
			      XmNlabelString,  label,
			      XmNalignment, XmALIGNMENT_CENTER,
			      XmNtopAttachment, XmATTACH_FORM,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNrightAttachment, XmATTACH_FORM, NULL);
    XmStringFree(label);
  }
}

CmDialogPane::~CmDialogPane() {
  XtDestroyWidget(BasePane);
}

 CmDialogItem::~CmDialogItem() { }
 CmDialogLabel::~CmDialogLabel() { }
 CmDialogPushButton::~CmDialogPushButton() { }

 CmDialogScale::~CmDialogScale() { }
 CmDialogSeparator::~CmDialogSeparator() { }
 CmDialogTextString::~CmDialogTextString() { }
 CmDialogToggleButton::~CmDialogToggleButton() { }

void CmDialogItem::AddOption(char, XtArgVal) { 
  cerr << "base AddOption member called\n"; 
}

void CmDialogItem::ManageItem() { 
  cerr << "base ManageItem member called\n"; 
}

CmString CmDialogItem::GetValue() {
  return (ItemName);
}

void CmDialogItem::SetValue(const CmString &) {
}

Widget CmDialogItem::GetWidget() {
  return (ItemWidget);
}

Widget CmDialogItem::GetBasePane() const {
  return (Dialog->GetPane(PaneName)->BasePane);
}

void CmDialogItem::AddCallback(Widget actionw, String callback_method) {
  if (callback_method != NULL)
    XtAddCallback(actionw, callback_method, 
		  (XtCallbackProc) CmInterpretChangedValue, 
		  (XtPointer) this);
}  

/******************* CmDialog Member Function****************************/

/********************************************************
 CmDialog constructor
 Requires: parentwidget to be non-NULL
 Effects:  creates a dialog shell, puts a form widget in it with 3 buttons
           and a rowcolumn to put the items into
 ***********************************************************/
CmDialog::CmDialog(Widget parentwidget, const CmString &title, char state) {
  /* copy parameters into private variables for later use */
  if (state == CMDIALOG_MODAL) 
    dialog_state = state;
  parent = parentwidget;

  helpdisplayed = CMDIALOG_FALSE;
  dialog_state = CMDIALOG_MODELESS;
  display_pane = True;
  // setup base shell and form for dialog bo
  shell = XtCreatePopupShell(title.Chars(), transientShellWidgetClass,
			     parent, NULL, 0);

  base_form = 
    XtCreateManagedWidget("cmDialogBaseForm", xmPanedWindowWidgetClass,
			  shell, NULL, 0);

  // create work area for items 
  work_area = 
    XtCreateManagedWidget("cmDialogWorkArea", xmFormWidgetClass,
			  base_form, NULL, 0);
  XmAddTabGroup(work_area);

  button_area = 
    XtCreateManagedWidget("cmDialogButtonArea", xmFormWidgetClass,
			  base_form, NULL, 0);


  /* add ok, cancel, help buttons to form with callbacks */
  ok_pbutton =
    XtVaCreateManagedWidget("ok", xmPushButtonGadgetClass, button_area,
			    XmNleftAttachment, XmATTACH_POSITION,
			    XmNleftPosition, 10,
			    XmNrightAttachment, XmATTACH_POSITION,
			    XmNrightPosition, 30,
			    XmNtopAttachment, XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM, NULL);
			    
  XtAddCallback(ok_pbutton, XmNactivateCallback, 
		(XtCallbackProc) CmDialogOkCallback, (XtPointer) this);

  cancel_pbutton = 
    XtVaCreateManagedWidget("cancel", xmPushButtonGadgetClass,
			     button_area,
			     XmNleftAttachment, XmATTACH_POSITION,
			     XmNleftPosition, 40,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition, 60,
			     XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
			     XmNtopWidget, ok_pbutton,
			     XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
			     XmNbottomWidget, ok_pbutton, NULL);
  XtAddCallback(cancel_pbutton, XmNactivateCallback, 
		(XtCallbackProc) CmDialogCancelCallback, (XtPointer) this);

  help_pbutton = 
    XtVaCreateManagedWidget("help", xmPushButtonGadgetClass,
			    button_area,
			    XmNleftAttachment, XmATTACH_POSITION,
			    XmNleftPosition, 70,
			    XmNrightAttachment, XmATTACH_POSITION,
			    XmNrightPosition, 90,
			    XmNtopAttachment, XmATTACH_OPPOSITE_WIDGET,
			    XmNtopWidget, cancel_pbutton,
			    XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
			    XmNbottomWidget, cancel_pbutton, NULL);

  XtAddCallback(help_pbutton, XmNactivateCallback, 
		(XtCallbackProc) CmDialogHelpCallback, (XtPointer) this);
  XmAddTabGroup(button_area);
}

CmDialog::~CmDialog() {
  for (item_list.MoveFirst(); item_list.HasCurrent(); item_list.MoveNext())
    delete (CmDialogItem*) item_list.GetCurrent();
  for (pane_list.MoveFirst(); pane_list.HasCurrent(); pane_list.MoveNext() )
    delete (CmDialogPane*) pane_list.GetCurrent();
  XtDestroyWidget(shell);
}

/****************** Display Routines **************************/

/*******************************************************
 PopupDialog: displays the dialog object on the screen
 *******************************************************/
void CmDialog::PopupDialog() {
  Dimension parent_x,parent_y,parent_width,parent_height;
  Dimension dialog_x,dialog_y,dialog_width,dialog_height;

  XtRealizeWidget(shell); // realize the shell but don't display it

  // place the dialog 
  XtVaSetValues(shell,
		XmNallowShellResize, True, NULL);

  XtVaGetValues(parent,
		XmNx, &parent_x,
		XmNy, &parent_y,
		XmNwidth, &parent_width,
		XmNheight, &parent_height, NULL);

  XtVaGetValues(shell,
		XmNwidth, &dialog_width,
		XmNheight, &dialog_height, NULL);

  if ( dialog_width < DIALOG_MIN_WIDTH )
    dialog_width = (Dimension)DIALOG_MIN_WIDTH;

  dialog_x = (parent_x + parent_width/2) - (dialog_width / 2);
  dialog_y = (parent_y + parent_height/2) - (dialog_height / 2);

  XtVaSetValues(shell, 
		XmNx, dialog_x,
		XmNy, dialog_y,
		XmNwidth, dialog_width, NULL);

  XtVaSetValues(shell, 
		XtNmappedWhenManaged, True, NULL);

  /* display the dialog box on the screen */  
  XtPopup(shell, 
	  dialog_state == CMDIALOG_MODELESS ? XtGrabNone : XtGrabExclusive);
}

/*******************************************************
 PopDownDialog: takes the dialog shell off the screen
 *******************************************************/
void CmDialog::PopdownDialog() {  
  XtPopdown(shell); /* take the dialog box off the screen */

  // take the help of the screen as well if not already done
  if (helpdisplayed == CMDIALOG_TRUE)
    { printf("help is displayed\n"); UnDisplayHelp(); }
  // Set the values to the inital values if MODAL
}

/*******************************************************
 DisplayPane: changes the boolean value that determines
 if the pane names are displayed
 *******************************************************/
void CmDialog::DisplayPane(Boolean display)
{ display_pane = display; }  

/****************************** 
 CmDialog AddPane
******************************/
void CmDialog::AddPane(const CmString &pname, 
		       const CmString &pattach, 
		       CmDialogAttachMethod attach_method) { 
/* If pattach is empty the base pane is NULL, otherwise get the
 * base pane through GetPane.
 */

  pane =  new CmDialogPane(work_area, pname, 
			   pattach.Empty() ? 
			   NULL : GetPane(pattach)->BasePane,  
			   attach_method,display_pane);
  pane_list.Append(pane);
  pane->AttachPane = pattach;
  pane->AttachMethod = attach_method;
}

/****************************** 
 CmDialog AddItem
******************************/
CmDialogItem *CmDialog::AddItem(const CmDialogItemType item_type,
				const CmString &pane_name, 
				const CmString &name) {
  if( GetPane(pane_name) != NULL ){
    switch ( item_type ) {
    case CMDIALOG_LABEL:
      item = new CmDialogLabel(); break;
    case CMDIALOG_LIST:
      item = new CmDialogList(); break;
    case CMDIALOG_PUSHBUTTON: 
      item = new CmDialogPushButton(); break;
    case CMDIALOG_RADIOBOX: 
      item = new CmDialogRadioBox(); break;
    case CMDIALOG_SCALE: 
      item = new CmDialogScale(); break;
    case CMDIALOG_SEPARATOR: 
      item = new CmDialogSeparator(); break;
    case CMDIALOG_TEXTSTRING: 
      item = new CmDialogTextString(); break;
    case CMDIALOG_TOGGLEBUTTON:
      item = new CmDialogToggleButton(); break;
    default:
      cerr << "Tried to add nondefined item";
      return NULL;
    }
    item_list.Append(item);
    item->Dialog = this;
    item->PaneName = pane_name;
    item->ItemName = name;
    item->ItemType = item_type;
    return item;
  } else {
    cerr << " pane " << pane_name << " doesn't exist\n";
    return (NULL);
  }
}

/****************************** 
 CmDialog AddOption
******************************/
void CmDialog::AddOption(const CmString &pane_name, 
			 const CmString &item_name, 
			 char opt, XtArgVal value) {
  GetItem(pane_name, item_name)->AddOption(opt,value);
}

/****************************** 
 CmDialog GetValue
******************************/
CmString CmDialog::GetValue(const CmString &pane_name, 
			    const CmString &item_name)  { 
  return GetItem(pane_name, item_name)->GetValue();
}

/****************************** 
 CmDialog SetValue
******************************/
void CmDialog::SetValue(const CmString &pane_name, 
			const CmString &item_name, 
			const CmString &value) {
  GetItem(pane_name, item_name)->SetValue(value);
}


/******************************
 CmDialog SetCallbacks
 ******************************/
void CmDialog::SetCallbacks(XtCallbackProc ok_callback, XtPointer ok_pointer,
		  XtCallbackProc cancel_callback, XtPointer cancel_pointer) {
  if (ok_callback)
    XtAddCallback(ok_pbutton, XmNactivateCallback, 
		  ok_callback, ok_pointer);
  if (cancel_callback)
    XtAddCallback(cancel_pbutton, XmNactivateCallback,
		  cancel_callback, cancel_pointer);
}

/****************************** 
 CmDialog GetWidget
******************************/
Widget CmDialog::GetWidget(const CmString &pane_name, const CmString &name) {
  return GetItem(pane_name, name)->GetWidget();
}

/****************************** Items ***************************/

/***************************** 
  Notes: The widget names have been made similar to be compatible with 
  resource files.  The unique names that used to be created are commented out 
  Widgetnames <CmDialogLabel, CmDialogList (RowCol, Name), 
  CmDialogPushButton, CmDialogRadioBox (RadioBox), 
  CmDialogScale, CmDialogSeparator, CmDialogTextString (RowCol, Name, Inupt), 
  ToggleButton>
  (the names in parenthesis are added to the
  base name to form the internal parts ie. CmDialogListRowCol) 
*******************************/

/***************************** 
  Label Class 
*******************************/
void CmDialogLabel::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_ALIGNMENT:
    char opt = (char)value;
    switch (opt) {
    case CMDIALOG_LEFT: 
      XtSetArg(labelargs[count], XmNalignment, XmALIGNMENT_BEGINNING); count++;
      break;
    case CMDIALOG_CENTER: 
      XtSetArg(labelargs[count], XmNalignment, XmALIGNMENT_CENTER); count++;
      break;
    case CMDIALOG_RIGHT: 
      XtSetArg(labelargs[count], XmNalignment, XmALIGNMENT_END); count++;
      break;
    }
    break;
  }
}

void CmDialogLabel::ManageItem() {
  XmString xm_string = CmToXmString(CmDialogItem::ItemName);
  XtSetArg(labelargs[count], XtNlabelString, xm_string); count++;

  CmDialogItem::ItemWidget = 
    XtCreateManagedWidget(widget_name.Chars(), atLabelWidgetClass, 
			CmDialogItem::GetBasePane(), labelargs, count);
  XmStringFree(xm_string);
}

void CmDialogLabel::SetValue(const CmString &label_string) {
  XtVaSetValues(CmDialogItem::ItemWidget, XmNlabelString, 
		label_string.Chars(), NULL);
}

//***************************** 
//  List Class 
//******************************/

void CmDialogList::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_NUMVISIBLE:
    XtSetArg(listargs[count], XmNvisibleItemCount, value); count++; break;
  case CMDIALOG_OPTION:
    if (num_options < MAXOPTIONS)
      options[num_options++] = (char*)value;
    else
      cerr << "Options to Listbox exceed maximum of " << MAXOPTIONS << "\n";
    break;
  case CMDIALOG_VALUE:
    selected = (char*)value;
  }
}

void CmDialogList::ManageItem() {
  Widget base_pane = CmDialogItem::GetBasePane();

  // create base widget
  CmString rowcolname = widget_name + "RowCol";
  rowcol = 
  XtVaCreateManagedWidget(rowcolname.Chars(), xmRowColumnWidgetClass, 
                         base_pane,
			 XmNspacing, 10,
			 XmNorientation, XmVERTICAL, NULL);

  // create label
  CmString labelname = widget_name + "Name";
  XmString label_string = CmToXmString(CmDialogItem::ItemName);
  label =
    XtVaCreateManagedWidget(labelname.Chars(), xmLabelWidgetClass,
			   base_pane, 
			   XmNlabelString, label_string, NULL);
  XmStringFree(label_string);

  // create list
  XtSetArg(listargs[count], XmNselectionPolicy, XmSINGLE_SELECT); count++;
  CmDialogItem::ItemWidget = 
  XmCreateScrolledList(base_pane, (char *) widget_name.Chars(),
		       listargs, count);
  Widget list = CmDialogItem::ItemWidget;
  XtManageChild(list);
  for (int i = 0; i < num_options; i++) {
    XmString item_string = CmToXmString(options[i]);
    XmListAddItem(list, item_string, 0);
    XmStringFree(item_string);
  }
  if ( !selected.Empty() ) {
    XmString item_string = CmToXmString(selected);
    XmListSelectItem(list, item_string,False);
    XmStringFree(item_string);
  }
  CmDialogItem::AddCallback(list, XmNsingleSelectionCallback);
}

CmString CmDialogList::GetValue()
{
  CmString item_value, multiplevalues;

  XmStringTable stringtable;
  int num_selected;

  count = 0;
  XtSetArg(listargs[count], XmNselectedItemCount, &num_selected); count++;
  XtSetArg(listargs[count], XmNselectedItems, &stringtable); count++;
  XtGetValues(CmDialogItem::ItemWidget, listargs, count);
  for (int i = 0; i < num_selected; i++) {
    item_value = XmToCmString(stringtable[i]);
    if (item_value.Length() > 1)
      multiplevalues += item_value;
  }
  if (num_selected == 0)
    return("");
  else
    return(multiplevalues);
}

void CmDialogList::SetValue(const CmString &item) {
  // reset list box if necessary
  if (item.Empty())
    XmListDeleteAllItems(CmDialogItem::ItemWidget);
  else {
    XmString xm_string = CmToXmString(item);
    XmListAddItem(CmDialogItem::ItemWidget, xm_string, 0);
    XmStringFree(xm_string);
  }
}


/***************************** 
  PushButton Class 
*******************************/

void CmDialogPushButton::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_ALIGNMENT:
    char opt = (char)value;
    switch (opt) {
    case CMDIALOG_LEFT: 
      XtSetArg(pbargs[count], XmNalignment, XmALIGNMENT_BEGINNING); count++;
      break;
    case CMDIALOG_CENTER: 
      XtSetArg(pbargs[count], XmNalignment, XmALIGNMENT_CENTER); count++;
      break;
    case CMDIALOG_RIGHT: 
      XtSetArg(pbargs[count], XmNalignment, XmALIGNMENT_END); count++;
      break;
    }
    break;
  }
}

void CmDialogPushButton::ManageItem() {
  XmString item_name = CmToXmString(CmDialogItem::ItemName);
  XtSetArg(pbargs[count], XmNlabelString, item_name); count++;
  CmDialogItem::ItemWidget = 
    XtCreateManagedWidget(widget_name.Chars(), xmPushButtonGadgetClass, 
			CmDialogItem::GetBasePane(), pbargs, count);
  XmStringFree(item_name);
  CmDialogItem::AddCallback(CmDialogItem::ItemWidget, XmNactivateCallback);
}

/***************************** 
  RadioBox Class 
*******************************/

void CmDialogRadioBox::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_RADIOBUTTON:
    rbutton_name[num_rbuttons++] = (char*)value; break;
  case CMDIALOG_SELECTED:
    selected = (char*)value; break;
  }
}

void CmDialogRadioBox::ManageItem() {
  // create radio box
  rbname = widget_name + "RadioBox";
  Arg arglist[3];
  int count = 0;
  XtSetArg(arglist[count], XmNindicatorType, XmONE_OF_MANY); count++;
  XtSetArg(arglist[count], XmNborderWidth, 2); count++;
  CmDialogItem::ItemWidget = 
    XmCreateRadioBox(CmDialogItem::GetBasePane(),
		     (char *) rbname.Chars(), arglist, count);
  XtManageChild(CmDialogItem::ItemWidget);
  // Adding buttons
  rbutton = new Widget[num_rbuttons];

  for (int i = 0; i < num_rbuttons; i++ ) {
    XmString xm_string = CmToXmString(rbutton_name[i]);

    rbutton[i] = XtVaCreateManagedWidget(rbuttonname.Chars(), 
				       xmToggleButtonGadgetClass, 
				       CmDialogItem::ItemWidget, 
					 XmNlabelString, xm_string, NULL);
    XmStringFree(xm_string);
    // set selected one to true
    if ( selected == rbutton_name[i] )
      XmToggleButtonGadgetSetState(rbutton[i],True,False);
  CmDialogItem::AddCallback(rbutton[i], XmNvalueChangedCallback);
  }
}

CmString CmDialogRadioBox::GetValue() {
  for (int i = 0; i < num_rbuttons; i++) {
    if ( XmToggleButtonGadgetGetState(rbutton[i]) )
      return(rbutton_name[i]);
  }
  cerr << "No buttons selected\n";
  return "";
}

CmDialogRadioBox::~CmDialogRadioBox() {
  delete rbutton;
}

/***************************** 
  Scale Class 
*******************************/
void CmDialogScale::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_DECIMAL_POINTS:
    XtSetArg(scaleargs[count], XmNdecimalPoints, (short)value); count++; break;
  case CMDIALOG_MIN:
    XtSetArg(scaleargs[count], XmNminimum, (int)value); count++; break;
  case CMDIALOG_MAX:
    XtSetArg(scaleargs[count], XmNmaximum, (int)value); count++;break;
  case CMDIALOG_ORIENTATION:
    if ( ((char)value) == CMDIALOG_VERTICAL)
      XtSetArg(scaleargs[count], XmNorientation, XmVERTICAL);
    else
      XtSetArg(scaleargs[count], XmNorientation, XmHORIZONTAL);
    count++; break;
  case CMDIALOG_PROCESS_DIRECTION:
    if ( ((char)value) == CMDIALOG_RIGHT_TO_LEFT)
      XtSetArg(scaleargs[count], XmNprocessingDirection, XmMAX_ON_LEFT);
    else
      XtSetArg(scaleargs[count], XmNprocessingDirection, XmMAX_ON_RIGHT);
    count++;break;
  case CMDIALOG_SHOW_VALUE:
    if ( ((char)value) == CMDIALOG_TRUE)
      XtSetArg(scaleargs[count], XmNshowValue, True);
    else
      XtSetArg(scaleargs[count], XmNshowValue, False);
    count++;break;
  case CMDIALOG_VALUE:
    XtSetArg(scaleargs[count], XmNvalue, (int)value); count++; break;
  }
}

void CmDialogScale::ManageItem() {
  // create scale
  XmString xm_string = CmToXmString(CmDialogItem::ItemName);
  XtSetArg(scaleargs[count], XmNtitleString, xm_string); count++;
  CmDialogItem::ItemWidget = 
    XtCreateManagedWidget(widget_name.Chars(), xmScaleWidgetClass, 
			  CmDialogItem::GetBasePane(), scaleargs, count);
  CmDialogItem::AddCallback(CmDialogItem::ItemWidget, XmNvalueChangedCallback);
  XmStringFree(xm_string);
}

CmString CmDialogScale::GetValue()
{ int i;
  CmString value;
  
  XmScaleGetValue(CmDialogItem::ItemWidget, &i);
  value = i;
  return(value);
}

/***************************** 
  Separator Class 
*******************************/

void CmDialogSeparator::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_ORIENTATION:
    if ((int)value == CMDIALOG_VERTICAL)
      XtSetArg(separgs[count], XmNorientation, XmVERTICAL);
    else
      XtSetArg(separgs[count], XmNorientation, XmHORIZONTAL);
    break;
  case CMDIALOG_SEPARATOR_TYPE:
    switch( (int)value ) {
    case CMDIALOG_SINGLE_LINE:
      XtSetArg(separgs[count], XmNseparatorType, XmSINGLE_LINE);
    case CMDIALOG_DOUBLE_LINE:
      XtSetArg(separgs[count], XmNseparatorType, XmDOUBLE_LINE);
    case CMDIALOG_SINGLE_DASHED_LINE:
      XtSetArg(separgs[count], XmNseparatorType, XmSINGLE_DASHED_LINE);
    case CMDIALOG_DOUBLE_DASHED_LINE:
      XtSetArg(separgs[count], XmNseparatorType, XmDOUBLE_DASHED_LINE);
    case CMDIALOG_NO_LINE:
      XtSetArg(separgs[count], XmNseparatorType, XmNO_LINE);
    case CMDIALOG_SHADOW_ETCHED_IN:
      XtSetArg(separgs[count], XmNseparatorType, XmSHADOW_ETCHED_IN);
    case CMDIALOG_SHADOW_ETCHED_OUT:
      XtSetArg(separgs[count], XmNseparatorType, XmSHADOW_ETCHED_OUT);
    }
    count++;
  }
}

void CmDialogSeparator::ManageItem() {
  // create separator
  XmString xm_string = CmToXmString(CmDialogItem::ItemName);
  XtSetArg(separgs[count], XmNtitleString, xm_string); count++;
  CmDialogItem::ItemWidget
    = XtCreateManagedWidget(widget_name.Chars(), xmSeparatorGadgetClass, 
			  CmDialogItem::GetBasePane(), separgs, count);
  XmStringFree(xm_string);
}
 
//*****************************/
//  TextString Class 
//******************************/

void CmDialogTextString::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_MAX_LENGTH:
    XtSetArg(textargs[textcount], XmNmaxLength, (int)value); 
    textcount++; break;
  case CMDIALOG_VALUE:
    default_value = (char*)value;
    XtSetArg(textargs[textcount], XmNvalue, default_value.Chars() );
    textcount++; break;
  case CMDIALOG_EDITMODE:
    if ( ((char)value) == CMDIALOG_MULTI_LINE )
      XtSetArg(textargs[textcount], XmNeditMode, XmMULTI_LINE_EDIT);
    else
      XtSetArg(textargs[textcount], XmNeditMode, XmSINGLE_LINE_EDIT);
    textcount++; break;
  case CMDIALOG_ROWS:
    XtSetArg(textargs[textcount], XmNrows, (int)value); textcount++; break;
  case CMDIALOG_COLUMNS:
    XtSetArg(textargs[textcount], XmNcolumns, (int)value); textcount++; break;
  case CMDIALOG_SCROLLED_TEXT:
    ScrollBars = True; break;
  }
}

void CmDialogTextValueChanged(Widget w,CmDialog*, XtPointer) {
  char *string = XmTextGetString(w);
  CmString value = string;
  XtFree(string);

  int oldcursorpos = (int) XmTextGetInsertionPosition(w);
  int cursorpos =  oldcursorpos - 1;

  if (cursorpos >= 0 && value[cursorpos] == ')' ) {
    // do parenthesis checking
    Boolean balanced = False;
    int depth = 1;
   while ( !balanced && (cursorpos > 0) ) {
     cursorpos--;
     if ( value[cursorpos] == '(' ) {
       depth--;
       if (depth == 0) {
	 balanced = True;
	 
	 XmTextSetHighlight(w,cursorpos,cursorpos+1,XmHIGHLIGHT_SELECTED);
	 XmTextShowPosition(w,cursorpos);
	 XSync(XtDisplay(w),False);
	 sleep(1);
	 XmTextSetHighlight(w,cursorpos,cursorpos+1,XmHIGHLIGHT_NORMAL);
	 XmTextShowPosition(w,oldcursorpos);
       }
     }
     if ( value[cursorpos] == ')' )
       depth++;

   }
 }
}

void CmDialogTextString::ManageItem() {
  rowcolname = widget_name + "RowCol";
 CmDialogItem::ItemWidget = 
   rowcol = XtCreateManagedWidget(rowcolname.Chars(), xmFormWidgetClass, 
				CmDialogItem::GetBasePane(),
				  rowcolargs,rowcolcount);
  // create label
  labelname = widget_name + "Name";
  XmString xm_string = CmToXmString(CmDialogItem::ItemName);
  XtSetArg(labelargs[labelcount], XmNlabelString, xm_string);
  labelcount++;
  XtSetArg(labelargs[labelcount], XmNleftAttachment, XmATTACH_FORM);
  labelcount++;
  label = XmCreateLabel(rowcol, (char *) labelname.Chars(), labelargs, labelcount);
  XtManageChild(label);
  XmStringFree(xm_string);

  // create text
  textname = widget_name + "Input";
  XtSetArg(textargs[textcount], XmNleftAttachment,XmATTACH_WIDGET);textcount++;
  XtSetArg(textargs[textcount], XmNleftWidget, label);textcount++;
  XtSetArg(labelargs[labelcount], XmNrightAttachment, XmATTACH_FORM);
  if (ScrollBars)
    text = XmCreateScrolledText(rowcol, (char *) textname.Chars(), textargs, textcount);
  else
    text = XmCreateText(rowcol, (char *) textname.Chars(), textargs, textcount);
  XtManageChild(text);

  XtAddCallback(text,XmNvalueChangedCallback,
		(XtCallbackProc) CmDialogTextValueChanged,(XtPointer)this);
  CmDialogItem::AddCallback(text, XmNactivateCallback);
}

CmString CmDialogTextString::GetValue() { 
  return(XmTextGetString(text)); 
}
void CmDialogTextString::SetValue(const CmString &text_string) { 
  XmTextSetString(text, (char *)text_string.Chars()); 
}

/***************************** 
  ToggleButton Class 
*******************************/

void CmDialogToggleButton::AddOption(char option, XtArgVal value) {
  switch ( option ) {
  case CMDIALOG_VALUE:
    button_set = ((int)value == CMDIALOG_TRUE ) ? True : False;
  }
}

void CmDialogToggleButton::ManageItem() {
  // create scale
  XmString label_string = CmToXmString(CmDialogItem::ItemName);
  CmDialogItem::ItemWidget = 
    XtVaCreateManagedWidget(widget_name.Chars(), xmToggleButtonGadgetClass, 
			    CmDialogItem::GetBasePane(), 
			    XmNlabelString, label_string,
			    XmNset, button_set, NULL);
  XmStringFree(label_string);
  CmDialogItem::AddCallback(CmDialogItem::ItemWidget, XmNvalueChangedCallback);
}

CmString CmDialogToggleButton::GetValue() {
  return XmToggleButtonGetState(CmDialogItem::ItemWidget) == True 
    ? "True" : "False";
}

/************************ Widget manipulation routines **************/

void CmDialog::ManageItems() {
  Boolean rightattach = True;
  Boolean bottomattach = True;

  // Find the last pane and attach it to the right
  for (pane_list.MoveLast(); pane_list.HasCurrent(); pane_list.MovePrev()) {
   pane = (CmDialogPane *) pane_list.GetCurrent();

   // bottom attachments
   if (bottomattach == True) {
     XtVaSetValues(pane->BasePane, XmNbottomAttachment, XmATTACH_FORM, NULL); 
     // cmdebug << "Set bottom attach to pane " << pane->PaneName << "\n";
     bottomattach = False;  // the bottom pane
   }

   // right attachments
   if (rightattach == True) {
     XtVaSetValues(pane->BasePane, XmNrightAttachment, XmATTACH_FORM, NULL);
     // cmdebug << "Set right attach to pane " << pane->PaneName << "\n";
   }
   if ( (pane->AttachMethod == CMDIALOG_ATTACH_LEFT) || 
        (pane->AttachMethod == CMDIALOG_ATTACH_NONE) )  {
     rightattach = False; // stop right attach
     bottomattach = True; // the next pane is the bottom one
   }
 }
  
  // draw the items
  for (item_list.MoveFirst(); item_list.HasCurrent(); item_list.MoveNext())
    ((CmDialogItem *) item_list.GetCurrent())->ManageItem();
}

int CmDialog::GetInteger(const CmString &pane_name, const CmString &name)
{ return(atoi(GetValue(pane_name, name))); }

float CmDialog::GetFloat(const CmString &pane_name, const CmString &name)
{ 
  float x;
  sscanf(GetValue(pane_name, name), "%f", &x);
  return(x);
}

double CmDialog::GetDouble(const CmString &pane_name, const CmString &name)
{ return(atof(GetValue(pane_name, name))); }

CmString CmDialog::GetString(const CmString &pane_name, const CmString &name)
{ return(GetValue(pane_name, name)); }

void CmDialog::GreyItem(const CmString &pane_name, const CmString &name)
{ XtSetSensitive(GetWidget(pane_name, name), False); }

void CmDialog::UnGreyItem(const CmString &pane_name, const CmString &name)
{ XtSetSensitive(GetWidget(pane_name, name), True); }
 
void CmDialog::SetHelp( const CmString &helpstring ) { help += helpstring; }

CmDialogPane* CmDialog::GetPane(const CmString &pane_name) {
  for (pane_list.MoveFirst(); pane_list.HasCurrent(); pane_list.MoveNext()) {
    pane = (CmDialogPane *) pane_list.GetCurrent();
    if ( pane->PaneName == pane_name ) return(pane);
  }
  cerr << "pane " << pane_name << " not found in GetPane\n";
  return(NULL);
}

CmDialogItem* CmDialog::GetItem(const CmString &pane_name, 
				const CmString &item_name ) { 
  for (item_list.MoveFirst(); item_list.HasCurrent(); item_list.MoveNext()) {
    item = (CmDialogItem *) item_list.GetCurrent();
    if ( (item->PaneName == pane_name) && 
	(item->ItemName == item_name) )
      return(item);
  }
  cerr << "Item " << pane_name << "-" << item_name << " not found \n";
  return(NULL);
}  

void CmDialog::ValueChanged( const CmString & ) { 
  /* Send a message to the recieving procedure if the dialog box is modeless */
/*
  if (dialog_state == CMDIALOG_MODELESS)
    (*receiver)(name);
*/
}


void CmDialog::DisplayHelp() {
  Dimension parent_x,parent_y,parent_width,parent_height;
  Dimension help_x,help_y,help_width,help_height;

  Widget helpwidget;
  /* check if help already displayed */
  if ( helpdisplayed == CMDIALOG_FALSE ) {
    helpdisplayed = CMDIALOG_TRUE;
    /* setup base form for dialog box */
    helpshell = XtCreatePopupShell("Help", transientShellWidgetClass,
                                   shell, NULL, 0);
    count = 0;
    XtSetArg(arglist[count], XmNmessageString,
             XmStringCreateLtoR((char *) help.Chars(), XmSTRING_DEFAULT_CHARSET));
    count++;
    helpwidget = XmCreateMessageBox(helpshell, "helpbox", arglist, count);
    XtManageChild(helpwidget);
    XtAddCallback(helpwidget, XmNokCallback,
		  (XtCallbackProc) CmDialogHelpBoxCallback, (XtPointer) this);
    XtUnmanageChild(XmMessageBoxGetChild(helpwidget, XmDIALOG_CANCEL_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(helpwidget, XmDIALOG_HELP_BUTTON));
    
    // setup helpshell
    XtRealizeWidget(helpshell);

    // place help
    count = 0;
    XtSetArg(arglist[count], XmNallowShellResize, True); count++;
    XtSetValues(helpshell,arglist,count);
    
    count = 0;
    XtSetArg(arglist[count], XmNx, &parent_x); count++;
    XtSetArg(arglist[count], XmNy, &parent_y); count++;
    XtSetArg(arglist[count], XmNwidth, &parent_width); count++;
    XtSetArg(arglist[count], XmNheight, &parent_height); count++;
    XtGetValues(parent,arglist,count);
    
    count = 0;
    XtSetArg(arglist[count], XmNwidth, &help_width); count++;
    XtSetArg(arglist[count], XmNheight, &help_height); count++;
    XtGetValues(helpshell,arglist,count);
    
    if ( help_width < DIALOG_MIN_WIDTH )
      help_width = (Dimension)DIALOG_MIN_WIDTH;
    
    help_x = (parent_x + parent_width/2) - (help_width / 2);
    help_y = (parent_y + parent_height/2) - (help_height / 2);
    
    count = 0;
    XtSetArg(arglist[count], XmNx, help_x); count++;
    XtSetArg(arglist[count], XmNy, help_y); count++;
    XtSetArg(arglist[count], XmNwidth, help_width); count++;
    XtSetValues(helpshell, arglist, count);
  }
  // display popup
  XtPopup(helpshell,XtGrabNone);

}

void CmDialog::UnDisplayHelp() { 
  XtPopdown(helpshell); 
  helpdisplayed = CMDIALOG_FALSE;
}

/****************************** Dialog Convenience routines ****************/

void CmDialog::XmDialogCreate(CmDialogXmDialogType type_of_dialog, 
			      const CmString &text) {
  Arg arglist[MAXARGS];
  int count = 0;
  xm_dialog_type = type_of_dialog;
  char *text_chars = (char *) text.Chars();
  XmString xm_chars = CmToXmString(text);

  XtSetArg(arglist[count], XmNmessageString, xm_chars); count++;
 

  switch( xm_dialog_type ) {
  case CMDIALOG_ERROR:
    xm_dialog = XmCreateErrorDialog(parent, text_chars, arglist,count); 
    break;
  case CMDIALOG_INFORMATION:
    xm_dialog = XmCreateInformationDialog(parent, text_chars, 
				       arglist,count); break;
  case CMDIALOG_MESSAGE:
    xm_dialog = XmCreateMessageDialog(parent, text_chars, arglist,count); 
    break;
  case CMDIALOG_QUESTION:
    xm_dialog = XmCreateQuestionDialog(parent, text_chars, arglist,count); 
    break;
  case CMDIALOG_WARNING:
    xm_dialog = XmCreateWarningDialog(parent, text_chars, arglist,count); 
    break;
  case CMDIALOG_WORKING:
    xm_dialog = XmCreateWorkingDialog(parent, text_chars, arglist,count); 
    break;
  }
  XmStringFree(xm_chars);
}
  
void CmDialog::XmDialogRemoveButton(char type_of_button) {
  switch( type_of_button ) {
  case CMDIALOG_OK:
    XtUnmanageChild(XmMessageBoxGetChild(xm_dialog,XmDIALOG_OK_BUTTON)); 
    break;
  case CMDIALOG_CANCEL:
    XtUnmanageChild(XmMessageBoxGetChild(xm_dialog,XmDIALOG_CANCEL_BUTTON)); 
    break;
  case CMDIALOG_HELP:
    XtUnmanageChild(XmMessageBoxGetChild(xm_dialog,XmDIALOG_HELP_BUTTON)); 
    break; 
  }
}

Widget CmDialog::XmDialogGetWidget() { return(xm_dialog); }

void CmDialog::XmDialogManage() { XtManageChild(xm_dialog); }

