/*****************************************************************************
 Cmmodel.cc - Model Code
 Contact: sniyogi
 *****************************************************************************/
#include <CDS.h>
#include <CmCDS.h>
#include <CmPage.h>
#include <CmFigure.h>
#include <CmFigureEditor.h>
#include <CmModel.h>
#include <CmModelConnection.h>
#include <CmModelEvaluator.h>
#include <CmMenu.h>
#include <CmNavigator.h>

// There should be NO system dependent code in this file
// Code which depends on MOTIF goes in CmModelEvaluator.cc
// Code which depends on BSD UNIX goes in CmModelConnection.cc


static void CmModelReceiveInput(CmModel *model, int *, XtInputId *) {
  model->ReceiveInput();
}

static void CmModelEvaluatorShow(Widget, CmModelEvaluator* evaluator, 
				 XtPointer) {
  evaluator->Show();
}

static void CmModelEvaluate(Widget, CmModel *model, XtPointer) {
  model->Evaluate();
}

void CmModelInterrupt(Widget, CmModel *model, XtPointer) {
  if (model->connection)
    model->connection->Interrupt();
  model->command_buffer = "";
}

/*****************************************************************************
 CmModel Constructor
  Effects: None as of yet
 *****************************************************************************/
CmModel::CmModel(CmCDS* const c, const Widget widget) {
  cds = c;
  evaluator = new CmModelEvaluator(widget,
				   (XtCallbackProc) CmModelEvaluate,
				   (XtPointer) this);
  evaluator->AddButton("evaluator_interrupt",
		       (XtCallbackProc) CmModelInterrupt,
		       (XtPointer) this);
  
  cds->Menu()->AddItem("Options", "show_evaluator",
		       (XtCallbackProc) CmModelEvaluatorShow, 
		       (XtPointer) evaluator);
  connection = NULL;
  SetTraceLevel(1);
}

/*****************************************************************************
CmModel Destructor
  *****************************************************************************/
CmModel::~CmModel() {
  delete evaluator;
  if (connection)
    delete connection;
}

/**********
CmModel::ReceiveInput
requires: fid - ptr to file descriptor
effects: reads
returns: nothing
**********/

static CmString escape = "!cds!";
static CmRegex escape_regex = "!cds!.*\n";
static CmRegex begin_command = "^begin";
static CmRegex end_command = "^end";
static CmString buffer; // This really should be in receive input

void CmModel::ReceiveInput() {
  char input[BUFSIZ * 8];
  int nbytes;

  if (connection) {
    if((nbytes = connection->Read(input, BUFSIZ * 8)) == -1) {
// If we get an I/O error, this usually means that the process which should
// at the other end of the connection isn't there.  If this is the case, the
// connection should be closed so that we don't get a whole bunch of these
// messages.
      perror("get_file_input");
      delete connection;
      connection = NULL;
      return;
    }
  
    if (nbytes) {
      input[nbytes] = '\000';
      buffer += input;
      ParseModelCommands(buffer);
      if (buffer.Contains(escape)) {
	if (trace_level > 0) 
	  (*evaluator) << (CmString) buffer.Before(escape);
	buffer = buffer.From(escape);
      } else {
	if (trace_level > 0)
	  (*evaluator) << (CmString) buffer;
	buffer = "";
      }
    }
  }
}

void CmModel::Evaluate(CmString message) {
  // If the message contains #!exectuable
  // The model will execute #! executable

  static const CmRegex CommandString("^#!.*");
  static const CmRegex TrailingNewlines("\n*$");

  while (message.Contains(CommandString)) {
    Evaluate(message.Before(CommandString));
    CmString cds_exec = message.At(CommandString);
    if (connection)
      delete connection;
    connection = NULL;
    if (trace_level > 0)
      (*evaluator) << "Executing Command: " << cds_exec << "\n";
    connection = new CmModelConnection(cds_exec.After("#!"), 
				       cds->AppContext());
    connection->AddInput((XtInputCallbackProc) CmModelReceiveInput, 
			 (XtPointer) this);
    message = message.After(CommandString);
  }

  if  (connection) {
    // Replace page name
    message.Gsub("$CDS_THIS_PAGE", cds->Editor()->Page->Name);
    message.Gsub("$CDS_CONFIG_DIR", cds->ConfigDir);

    // Replace figure name
    const CmRegex figure_replace = 
      "${[A-za-z_-][A-Za-z0-9_.-]*:[A-za-z_-][A-Za-z0-9_.,-]*}";
    while(message.Contains(figure_replace)) {
      CmString match = message.At(figure_replace);
      match.Del('}');
      CmString widget = match.Before(':',2);
      CmString resource = match.After(':'); 
      CmString options;
      if (resource.Contains(',')) {
	resource = resource.Before(',');
	options = match.After(',');
      }
   
      if (trace_level > 2)
	(*evaluator) << "Matching" <<  match << "\n"
	             << "Widget: " <<  widget << "\n" 
	             << "Resource: " << resource << "\n"
		     << "Options: " << options << "\n";
      CmPage *current_page = cds->Editor()->Page;
      CmFigure *cmw = 
	current_page ? GetCmFigure(current_page, widget) : NULL;
      const CmString no_such_figure = "CDS_NO_SUCH_FIGURE";
      CmString replace;
      if (cmw) 
	replace = cmw->GetResource(resource);
      else
	replace = no_such_figure;
      while (!options.Empty()) {
	CmString option = options.Before(',');
	if (option.Empty()) {
	  option = options; options = "";
	} else
	  options = options.After(',');

	if (option == "escnl") {
	  replace.Gsub("\n", "\\\n");
	  replace.Gsub("\r", "");
	} else if (option == "quote") {
	  replace.Gsub("\"", "\\\"");
	  replace.Prepend("\"");  
	  replace += "\"";
	}
      }
      message.At(figure_replace) = replace;
    }
    // Make sure only one newline at the end
    message.At(TrailingNewlines) = '\n';
    connection->Write(message);
  }
}

void CmModel::Evaluate() {
  CmString string;
  (*evaluator) >> string;
  Evaluate(string);
}

void CmModel::Error(const CmString &s){
  cout << s;
}

/**********
CmModel::GetCmFigure
     requires: page_name and figure_name;
     effects: none
     returns: figure if found, NULL if not
There are TWO problems with this procedure, the first is that it calls the
NavigatorHeap directly.  The second is that it will not find a figure if
the page has not been widgetized
 **********/

CmFigure* CmModel::GetCmFigure(const char *&arg) {
  CmString page_name = cds->NextToken(arg);
  CmString widget_name = cds->NextToken(arg);
  CmPage* page = cds->Navigator()->GetPageByName(page_name);
  return page ? GetCmFigure(page, widget_name) : NULL;
}

CmFigure* CmModel::GetCmFigure(CmPage *page,
			       const CmString &widget_name) {
  CmFigure* cmw = page->PageCmFigure;
  return cmw ? cmw->NameToCmFigure(widget_name) : NULL;
}

void CmModel::ParseModelCommands(CmString &buffer) {
  while (buffer.Contains(escape_regex)) {
    CmString command_line = buffer.At(escape_regex);

    command_line.Del(0,escape.Length());
   
    if (command_line.Contains(begin_command)) {
      if (!command_buffer.Empty()) {
	(*evaluator) << "Premature end to CDS command\n";
	const char *command_chars = command_buffer.Chars();
	if (trace_level > 1)
	  (*evaluator) << "\nBegin CDS Command\n" << 
	    command_buffer << "\nEnd CDS Command\n";
	CmModelMethodProc cmm = FindMethod(cds->NextToken(command_chars));
	if ( cmm )
	  cmm(command_chars, this);
      }
      command_buffer = "";  
    } else if (command_line.Contains(end_command)) {
      const char *command_chars = command_buffer.Chars();
      if (trace_level > 1)
	(*evaluator) << "\nBegin CDS Command\n" << 
	  command_buffer << "\nEnd CDS Command\n";
      CmModelMethodProc cmm = FindMethod(cds->NextToken(command_chars));
      if ( cmm )
	cmm(command_chars, this);
      command_buffer = "";  
    } else
      command_buffer += command_line;
    buffer.Del(escape_regex);
  }
}

// Reset the trace level
void CmModel::SetTraceLevel(int trace) {
  trace_level = trace;
  connection->SetEcho(trace_level > 0 ? True : False);
}
