/* $Header: /afs/athena.mit.edu/astaff/project/atdev/src/plotter/RCS/Scale.c,v 3.3 91/03/17 20:01:37 crcraig Exp Locker: dot $ */

/*******************************************************************
  Copyright (C) 1990 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

***************************************************************** */

#include <math.h>
#include <stdio.h>

#ifdef _AtDevelopment_
#include "Scale.h"
#else
#include <At/Scale.h>
#endif

#define New(t) ((t *) XtMalloc(sizeof(t)))

static char *error_messages[] = {
  "AtScale:  No Error.",
  "AtScaleCalc: high bound is less than low bound.",
  "AtScaleCalc: high bound and low bound are too close together.",
  "AtScaleCalc: logarithmic scale must have positive bounds",
  "AtScaleCalc: unknown transform type.",
};

double _AtScaleAlmostZero = 1.0e-12;

static void AtScaleCalc(AtScale *s)
{
  s->errorcode = SCALEERROR_NONE;

  if (s->high < s->low) {
    s->errorcode = SCALEERROR_BOUNDLESS;
    s->transform = AtTransformINVALID;
    return;
  }

  if (s->high - s->low < _AtScaleAlmostZero) {
    s->errorcode = SCALEERROR_BOUNDCLOSE;
    s->transform = AtTransformINVALID;
    return;
  }
    
  if ((s->transform == AtTransformLOG) && (s->low <= 0.0)) {
    s->errorcode = SCALEERROR_LOGNEGATIVE;
    s->transform = AtTransformINVALID;
    return;
  }

  switch (s->transform) {
  case AtTransformLINEAR:
    s->mult = (s->highpix - s->lowpix)/(s->high - s->low);
    s->offset = s->low;
    break;
  case AtTransformLOG:
    s->offset = log10(s->low);
    s->mult = (s->highpix - s->lowpix)/(log10(s->high) - s->offset);
    break;
  default:
    s->errorcode = SCALEERROR_NOTRANSFORM;
    break;
  }
  return;
}
    

AtScale *AtScaleCreate(double low, double high, int lowpix, int highpix,
		       AtTransform transform)
{
  AtScale *scale;

  scale = New(AtScale);
  scale->low = low;
  scale->high = high;
  scale->lowpix = lowpix;
  scale->highpix = highpix;
  scale->transform = transform;

  AtScaleCalc(scale);
  return scale;
}

AtScale *AtScaleCopy(AtScale *s)
{
  AtScale *new;
  new = New(AtScale);
  *new = *s;
  return new;
}

void AtScaleDestroy(AtScale *s)
{
  XtFree(s);
}

int AtScaleUserToPixel(AtScale *s, double x)
{
  switch (s->transform) {
  case AtTransformLINEAR:
    return (int)((x - s->offset) * s->mult) + s->lowpix;
  case AtTransformLOG:
    return (int)((log10(x) - s->offset) * s->mult) + s->lowpix;
  case AtTransformINVALID:
    return 0;
  default:
    s->errorcode = SCALEERROR_NOTRANSFORM;
    return 0;
    }
}


double AtScalePixelToUser(AtScale *s, int i)
{
  switch (s->transform) {
  case AtTransformLINEAR:
    return (double)((i - s->lowpix) / s->mult) + s->offset;
  case AtTransformLOG:
    return pow(((i - s->lowpix) / s->mult) + s->offset, 10.0);
  case AtTransformINVALID:
    return 0.0;
  default:
    s->errorcode = SCALEERROR_NOTRANSFORM;
    return 0.0;
  }
}

void AtScaleResize(AtScale *s, int new_lowpix, int new_highpix)
{
  s->lowpix = new_lowpix;
  s->highpix = new_highpix;
  (void) AtScaleCalc(s);
}

void AtScaleRescale(AtScale *s, double new_low, double new_high)
{
  s->low = new_low;
  s->high = new_high;
  (void) AtScaleCalc(s);
}

void AtScaleChangeTransform(AtScale *s, AtTransform t)
{
  s->transform = t;
  (void) AtScaleCalc(s);
}

char *AtScaleGetErrorMessage(AtScale *s)
{
  if (s->errorcode < XtNumber(error_messages))
    return error_messages[s->errorcode];
  else
    return error_messages[0];
}


Boolean _AtStringToTransformRegistered = False;

void AtCvtStringToTransform(XrmValue *args, Cardinal num_args,
				XrmValue *from, XrmValue *to)
{
  static AtTransform transform;

  transform = AtTransformINVALID;
    
  if (strcasecmp(from->addr, "linear") == 0)
    transform = AtTransformLINEAR;
  else if ((strcasecmp(from->addr, "log") == 0) ||
	   (strcasecmp(from->addr, "logarithmic") == 0))
    transform = AtTransformLOG;

  if (transform == AtTransformINVALID)
    XtStringConversionWarning(from->addr, XtRTransform);
  else {
    to->addr = (caddr_t) &transform;
    to->size = sizeof(AtTransform);
  }
}
	
