/*********************************************************************
 * File: ltxo_fill.c 
 * This file contains private routines for fill mode in LTxo.  
 * It contains routines written by Linh Giang '89.  
 *
 * Most of the answers given by these routines are held in global
 * variables declared in the beginning of the file ltxo.c.  The reason
 * behind this little break in modularity is because the private
 * routines are recursive and therefore the line by line information
 * of the text are held in variables declared outside of the routines.
 *********************************************************************/

#include <stdio.h>		/* C include files */
#include <strings.h>

#include <X11/Xlib.h>           /* Xlib include files */
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#include <ltxo.h>	        /* Public LTxo include file */
#include <ltxo_pri.h>		/* Private LTxo include file */

/* extern variables defined in ltxo.c */
extern Display *LTx_dpy;
extern char *calloc();

/* tables defined in ltxo_tbl.c */
extern font_map font_tbl[][MAXSIZE]; 
extern sym_map sym_tbl[][MAXSYMSTRUCT]; 
extern GC LTxText_gc;

/* external private routines defined here but call from ltxo.c */
char *fill_text_width(), *fill_text_height(), *fill_text_out();

/* declared in ltxo_pri.c for alert box */
extern int ltxErrFlag;

/* extern private routines defined in ltxo_pri.c */
extern char *text_width(), *text_height();
extern char *text_out(), findmark();
extern void text_draw();
extern unsigned findbestsqrt();
	
/* The following extern variables are used by almost every routine in 
   this file which holds private routines.  The reason behind having
   these extern variables is that most of the width and height finding
   routines are recursive and therefore require extern variables to
   hold line by line information.  As a convention in LTxo, the extern
   variables have a capitalized first letter. */ 

extern Window Win;		     /* window to draw text */
extern XRectangle Box;		     /* LTxo bounding box */
extern unsigned Boxwidth, Boxheight;   /* width & height of 
					  bounding box in pixels */
extern int X, Y;              /* x,y of bounding box in pixels - use 
			         for determining the x,y of baseline */

extern unsigned long Foregd, Backgd; /* holds preferred back & 
					foreground of Box window */

extern int Xbase, Ybase;          /* x,y of baseline to draw text */

extern unsigned Current;	     /* current line index */
extern unsigned MaxCurrent;	/* the actual max lines of text output */

extern int *Linewidth, *Lineheight;   /* arrays holding each text 
				         line's width and height */

extern int *Lineascent;		/* array holding each text line's 
				   ascent - use for determining
				   the baseline. */

extern int *Linenumchar;	/* array holding num of chars per line */

extern int Currentlength;	/* length of current line -- use for 
				   fill_text_height & fill_text_out */

/* The following holds information about symbol & math formatting */
extern unsigned Count, MaxCount; 
extern int *Topwidth, *Botwidth; 
extern int *Topdescent, *Botascent;
extern int Overflag;



/**********************************************************************
 * Private Routine: fill_text_width(fid, sid, wordtotal, numchar, text,
 *                                  endmark) 
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  text    = the text to be displayed.
 *  numchar = temporarily holds the number of chars per word.
 *  wordtotal = the length of a word.
 *  endmark = the marker which tells us that the text using the 
 *            current font is finished. Endmarks are either 
 *            }, ], ), or >.                   
 * Description: 
 *  This routine computes the width of complex text in a filled
 *  environment and calls itself recursively to find the width of each
 *  individual line (set in Linewidth).  The routine fills in the text
 *  according to the Boxwidth of the bounding box and the given size
 *  of the font. 
 **********************************************************************/

char *fill_text_width(fid, sid, wordtotal, numchar, text, endmark) 
     unsigned fid, sid;
     int *wordtotal, *numchar; 
     char *text, endmark; 
{ 
  unsigned i = 0;          /* index to the characters in text */
  unsigned tmpsid,bestsid; /* temp size ids */ 
  unsigned tmpfid;	   /* temp font id */ 
  int offset;		   /* offset to the text array */
  unsigned tmpi;	   /* temp index to characters */ 
  char tmpmark, tmpmark2;    /* temp endmarks */ 
  unsigned tmpskip;          /* # of blank lines requested */ 
  int thiscount, symwidth;	/* width & count of symbols */
  int toptotal, bottotal;	/* for recursize calls */


#ifdef LTXO_DEBUG 
  printf("fill_text_width(%d, %d, %d, %d, %s, %c)\n", fid, sid,
	 *wordtotal, *numchar, text, endmark); 
#endif 

  /* The following loop deciphers each character of the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call
     fill_text_width() recursively to add up total width. */  

  while(1) {
#ifdef LTXO_DEBUG 
    printf("Filloop: i: %d text: %s\n", i, text);
#endif 
    if( *(text + i) == NULLCHAR ) {
      /* Came to end of text. Increment Linewidth & Linenumchar arrays
	 for fill_text_height() and fill_text_out() to use. */
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	;
	return(NULLCHAR);
      }
      Linewidth[Current] += *wordtotal;
      Linenumchar[Current] += *numchar;
      MaxCurrent = Current;
      MaxCount = Count;
      ;
      return(text + i);
    }

    else if( *(text + i) == endmark ) {
      /* Came to end of command. Just return the rest of text
	 skipping the endmark. */
      ;
      return(text + i + 1);
    }

    else if( *(text + i) == NL || *(text + i) == CR || *(text + i) ==
	    SP) { 
      /* Came to a spacing where possible break can occur in a 
	 filled environment. Thus, can add up running *wordtotal. */
      *wordtotal += XTextWidth(get_font_struct(fid, sid), (text+i),1); 
      Linewidth[Current] += *wordtotal;
      Linenumchar[Current] += *numchar + 1;
      text += i + 1;
      *wordtotal = *numchar = i = 0;
      while ( *text == SP ) text++;
    }

    else if(*(text + i) == '@') {
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:        /* user really wants an '@' character */
	*wordtotal += XTextWidth(get_font_struct(fid, sid),(text+1),1);
	(*numchar)++;
	text++;
	i++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0)
	  Current++;
	offset = NONE;	/* no recursive call necessary*/
	break;
      case COM_BOLD:   /* bold & check for recursive bold */
	tmpfid = ((fid == HELVET_BOLD || fid == TIMES_BOLD) ? fid :
		  findbold(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_ITA:   /* italics & check for recursive italics */
	tmpfid = ((fid == HELVET_ITALICS || fid == TIMES_ITALICS) ? fid
		 : finditalic(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_GREEK:     /* greek */
	tmpfid = GREEK;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_HELVET:     /* helvetica*/
	tmpfid = HELVET;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_TIMES:     /* times */
	tmpfid = TIMES;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_FIX:     /* fixed */
	tmpfid = FIXED;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_CENTER:	  /* center */
	tmpfid = fid;
	tmpsid = sid;
	offset = 8;
	break;
      case COM_C:	  /* center */
      case COM_RIGHT:     /* right flush */
      case COM_U:	  /* underline */
	tmpfid = fid;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6 );
	tmpfid = fid;
	offset = 4;
	break;
      case COM_SUPER:		/* superscripts */
      case COM_SUB:		/* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	tmpfid = fid;
	offset = 3;
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_ALEPH][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_AND][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_ANGLE][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_APPROX][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_AST:		/* asterisk symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_AST][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_BULLET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 ) Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EXISTS:		/* exists symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_EXISTS][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_EMPTYSET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_EQV:		/* equivalence symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_EQV][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_FORALL][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_GTE][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_IN:		/* IN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_IN][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_INFTY][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_INTER:		/* INTER symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_INTER][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_LTE][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_MULT:		/* multiplication symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_MULT][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 5;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_NABLA][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_NEQ][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_NOTIN][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_OR:		/* OR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_OR][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_PARTIAL][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PM:		/* PM symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_PM][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 3;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_PRSUBSET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_PRSUPSET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_SIMILIAR][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 9;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_SUBSET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_SUPSET][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 7;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UNION:		/* UNION symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_UNION][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 6;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_LEFTARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_UPARROW:		/* uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_UPARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 8;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_RIGHTARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 11;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DOWNARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DBLEFTARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DBUPARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 10;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DBRIGHTARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 13;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DBDOWNARROW][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 12;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case SYM_DOT:	      /* dot - alternative to mult */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	*wordtotal += sym_tbl[SYM_DOT][tmpsid].width;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )  Current++;
	text += 4;
	if (*text == SP) text++;
	offset = NONE;
	break;
      case COM_VEC:           /* vector symbol over variable */
	thiscount = Count++;
	toptotal = 0;
	if ((tmpmark = findmark(*(text + 4))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text = text_width(fid,sid,&toptotal,text+5,tmpmark);
	Topwidth[thiscount] = toptotal;
        offset = NONE;
        break;
      case COM_OVER:          /* math - over */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid != SMALLEST ? sid-1 : sid );
	while ( *text != tmpmark ) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 3))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+4, tmpmark2);
	    break;
	  case SUBCOM_DENOM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 5))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+6, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Topwidth[thiscount] = toptotal+2; /* give some for line */
	Botwidth[thiscount] = bottotal+2;
	*wordtotal += max(toptotal, bottotal)+2;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	text++;         /* skips endmark */
	offset = NONE;
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	toptotal = get_font_struct(fid, sid)->ascent;
	bottotal = get_font_struct(fid, sid)->descent;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	(void)text_height(fid, sid, &toptotal, &bottotal,
			  text + 6, tmpmark);
	bestsid = findbestsqrt(toptotal + bottotal);
	if ( bestsid >= BIG )
	  *wordtotal +=
	    XTextWidth(get_font_struct(SYMBOL, bestsid),BIGSQRT,1); 
	else
	  *wordtotal += sym_tbl[SYM_SQRT][bestsid].width;
	toptotal = 0;
	text = text_width(fid, sid, &toptotal, text + 6,
			  tmpmark);
	Topwidth[thiscount] = toptotal;
	*wordtotal += toptotal;
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	offset = NONE;
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  *wordtotal += XTextWidth(get_font_struct(SYMBOL, sid),BIGINT,1);
	else
	  *wordtotal += sym_tbl[SYM_INT][sid].width;
	if ((tmpmark = findmark(*(text + 4))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*wordtotal += max(toptotal, bottotal);
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	offset = NONE;
	break;
      case COM_SUM:		/* math - summation */
        thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  symwidth = XTextWidth(get_font_struct(SYMBOL, sid),BIGSIG,1);
	else
	  symwidth = sym_tbl[SYM_SIG][sid].width;
	if ((tmpmark = findmark(*(text + 4))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
        text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2); 
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*wordtotal += max(symwidth, max(toptotal, bottotal));
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	offset = NONE;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	toptotal = bottotal = 0;
	if ( sid >= BIG )
	  symwidth = XTextWidth(get_font_struct(SYMBOL, sid),BIGPI,1);
	else
	  symwidth = sym_tbl[SYM_PI][sid].width;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++; 
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    toptotal = 0;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &toptotal, text+3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    bottotal = 0;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') {
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_width(fid, tmpsid, &bottotal, text+5, tmpmark2); 
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topwidth[thiscount] = toptotal;
	Botwidth[thiscount] = bottotal;
	*wordtotal += max(symwidth, max(toptotal, bottotal));
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	offset = NONE;
	break;
      case COM_SPACE:         /* explicit spacing */
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text, findmark(*(text + 2)), &tmpi); 
	*wordtotal += tmpskip*XTextWidth(get_font_struct(sid, fid),
					 " ",1);
	(*numchar)++;
	if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	    Linewidth[Current] != 0 )
	  Current++;
	text += tmpi+1;
	offset = NONE;
	break;
      case COM_NL: 
	/* Came to an explicit line break. Check if there are more 
	   than one break line requested. Get total width of 
	   current line, point to next line, clear i to start at
	   beginning of next line, and set *wordtotal to zero to get
	   width of next word. */ 
	
	Linewidth[Current] += *wordtotal;
	Linenumchar[Current] += *numchar;

	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) )
	  tmpskip = getnumskip(text,findmark(*(text+2)), &tmpi);

	Current += tmpskip;
	text += tmpi + 1;
	*wordtotal = *numchar = 0;
	while ( *text == SP ) text++;
	offset = NONE;
	break;
      default:
	ltxErrAlert("Unknown command!", text);
	return(NULLCHAR);
      }
      /* After deciphering the command, do the necessary recursive 
         call. */
      if (offset) {
	if ((tmpmark = findmark(*(text + offset - 1))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}  
	else
	  text = fill_text_width(tmpfid, tmpsid, wordtotal, numchar, 
                                 (text + offset), tmpmark);
      }
    }
    else {
      *wordtotal += XTextWidth(get_font_struct(fid, sid),(text+i),1);
      (*numchar)++;
      i++;

      /* Check if line is full, if it is, point to next line. 
	 Linewidth[Current] equal to zero means that one single word 
	 is longer than the Boxwidth.  In that case, continue 
	 deciphering knowing tha the word will be clipped when 
	 displayed. */  

      if ((Linewidth[Current] + *wordtotal) > Boxwidth &&
	  Linewidth[Current] != 0 )
	Current++;
    }
  }
}



/**********************************************************************
 * Private Routine: fill_text_height(fid, sid, atotal, dtotal, text,
 *                                   endmark) 
 * Requires:
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  atotal  = the highest total ascent of current line.
 *  dtotal  = the highest total descent of current line.
 *  text    = the text to be displayed.
 *  endmark = the marker which tells us that the text using the 
 *            current font is finished. Endmarks are either 
 *            }, ], ), or >.                   
 * Description: 
 *  This routine computes each individual line height of complex text
 *  in a filled environment.  It calls itself recursively to find the
 *  height of each individual line (held in Lineheight & Lineascent). 
 **********************************************************************/

char *fill_text_height(fid, sid, atotal, dtotal, text, endmark)
     unsigned fid, sid;
     int *atotal, *dtotal; 
     char *text, endmark; 
{ 
  unsigned i = 0;        /* index to the characters in text */
  unsigned tmpsid;	 /* temp size id */ 
  unsigned tmpfid;	 /* temp font id */ 
  int offset;		 /* offset to the text array */
  unsigned tmpi;	 /* temp index to characters */ 
  char tmpmark, tmpmark2;    /* temporary endmark found */ 
  unsigned tmpskip;      /* # of blank lines requested */
  int tmp, thiscount;
  int topatotal, topdtotal;	/* for recursive calls */
  int botatotal, botdtotal;
  int sqrtoffset = 2, overoffset;

#ifdef LTXO_DEBUG 
  printf("fill_text_height(%u, %u, %d, %d, %s, %c)\n", fid, sid, *atotal,
	 *dtotal, text, endmark); 
#endif 

  /* The following while loop deciphers characters of the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call fill_text_height
     recursively to determine maximum height needed for current line
     total. */  

  while(1) {
#ifdef LTXO_DEBUG 
    printf("FHloop: i: %d text: %.1s Cur: %d Curlen: %d\n",
	   i,(text+i),Current,Currentlength);  
#endif 
    if( *(text + i) == NULLCHAR ) {
      /* Came to end of text. Set height and ascent needed for the 
         last line array and return rest of text. */ 
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	;
	return(NULLCHAR);
      }
      Lineheight[Current] = *atotal + *dtotal;
      Lineascent[Current] = *atotal;
      MaxCurrent = Current;
      MaxCount = Count;
      ;
      return(text + i);
    }

    else if(*(text + i) == endmark) {
      /* Came to end of command.  Just return rest of text skipping 
	 the endmark. */
      ;
      return(text + i + 1);  
    }

    else if( *(text + i) == NL || *(text + i) == CR || *(text + i) ==
	    SP) { 
      Currentlength--;
      i++;
#ifdef LTXO_DEBUG 
    printf("(SP): i: %d text: %.1s Cur: %d Curlen: %d\n",
	   i,(text+i),Current,Currentlength);  
#endif 
      text += i;
      i = 0;
      while ( *text == SP ) text++;
      if (Currentlength == 0) {
	if ( strncmp(text,"@.",2) ) {
	  Lineheight[Current] = *atotal + *dtotal;
	  Lineascent[Current] = *atotal;
	  Current++;
	  *atotal = get_font_struct(fid, sid)->ascent;
	  *dtotal = get_font_struct(fid, sid)->descent;
	  Currentlength = Linenumchar[Current];
#ifdef LTXO_DEBUG 
    printf("(NEW): i: %d text: %.1s Cur: %d Curlen: %d\n",
	   i,(text+i),Current,Currentlength);  
#endif 
	}
      }
    }

    else if(*(text + i) == '@') {
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:        /* user really wants an '@' character */
	Currentlength--;
	text++;
	i++;		
	offset = NONE;	/* no recursive call necessary*/
	break;
      case COM_BOLD:   /* bold & check for recursive bold */
	tmpfid = ((fid == HELVET_BOLD || fid == TIMES_BOLD) ? fid
		 : findbold(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_ITA:  /* italics & check for recursive italics */
	tmpfid = ((fid == HELVET_ITALICS || fid == TIMES_ITALICS) ? fid
		 : finditalic(fid));
	tmpsid = sid;
	offset = 3;
	break;
      case COM_GREEK:     /* greek */
	tmpfid = GREEK;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_HELVET:     /* helvetica*/
	tmpfid = HELVET;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_TIMES:     /* times */
	tmpfid = TIMES;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_FIX:    /* fixed */
	tmpfid = FIXED;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_CENTER:	/* center */
	tmpfid = fid;
	tmpsid = sid;
	offset = 8;
	break;
      case COM_C:	/* center */
      case COM_RIGHT:   /* right flush */
      case COM_U:	/* underline */
	tmpfid = fid;
	tmpsid = sid;
	offset = 3;
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
        break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
        break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
        break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
        break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6);
        if ( get_font_struct(fid, tmpsid)->ascent > *atotal )
          *atotal = get_font_struct(fid, tmpsid)->ascent;
        if ( get_font_struct(fid, tmpsid)->descent > *dtotal )
          *dtotal = get_font_struct(fid, tmpsid)->descent;
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6);
        tmpfid = fid;
        offset = 4;
        break;
      case COM_SUPER:	    /* superscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	topatotal = get_font_struct(fid, tmpsid)->ascent;  
	topdtotal = get_font_struct(fid, tmpsid)->descent;  
	if ((tmpmark = findmark(*(text + 2))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text = fill_text_height(fid, tmpsid, &topatotal, &topdtotal, 
			   text+3, tmpmark);
	tmp = script_offset(fid,sid) + topatotal;
	if ( tmp > *atotal ) *atotal = tmp;
	offset = NONE;
	break;
      case COM_SUB:	    /* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	botatotal = get_font_struct(fid, tmpsid)->ascent;  
	botdtotal = get_font_struct(fid, tmpsid)->descent;  
	if ((tmpmark = findmark(*(text + 2))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text = fill_text_height(fid, tmpsid, &botatotal, &botdtotal, 
			   text+3, tmpmark);
	tmp = script_offset(fid,sid) + botdtotal;
	if ( tmp > *dtotal ) *dtotal = tmp;
	offset = NONE;
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_AST:		/* asterisk symbol */
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_EXISTS:		/* exists symbol */
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	text += 9;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_EQV:		/* equivalence symbol */
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_IN:		/* IN symbol */
	text += 3;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_INTER:		/* INTER symbol */
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_MULT:		/* multiplication symbol */
	text += 5;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_OR:		/* OR symbol */
	text += 3;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	text += 8;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_PM:		/* PM symbol */
	text += 3;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	text += 9;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	text += 9;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	text += 9;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	text += 7;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_UNION:		/* UNION symbol */
	text += 6;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	text += 10;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_UPARROW:		/* uparrow */
	text += 8;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	text += 11;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	text += 10;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	text += 12;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	text += 10;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	text += 13;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	text += 12;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case SYM_DOT:	      /* dot - alternative to mult */	
	text += 4;
	if (*text == SP) text++;
	Currentlength--;
	offset = NONE;
	break;
      case COM_VEC:           /* vector symbol over variable */
	thiscount = Count++;
	tmpsid = ( sid >= BIG ? MEDBIG : sid);
        tmp  = get_font_struct(fid, tmpsid)->ascent +
	        get_font_struct(fid, tmpsid)->descent +
	        get_font_struct(fid, sid)->ascent;  
        if ( tmp > *atotal ) *atotal = tmp;
        tmpfid = fid;
        tmpsid = sid;
        offset = 5;
        break;
      case COM_OVER:		/* math - over */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid != SMALLEST ? sid-1 : sid );
	if (Overflag) 
	  overoffset = 0;
	else 
	  overoffset = (get_font_struct(fid, sid)->ascent +
			get_font_struct(fid, sid)->descent)/2;
	Overflag++;
	while ( *text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 3))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 4, tmpmark2);
	    tmp = topatotal + topdtotal + overoffset;
	    if ( tmp > *atotal ) *atotal = tmp;
	    break;
	  case SUBCOM_DENOM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 5))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 6, tmpmark2);
	    tmp = botatotal + botdtotal - overoffset;
	    if ( tmp > *dtotal ) *dtotal = tmp;
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	Overflag--;
	text++;		/* skips endmark */
	Currentlength--;
	offset = NONE;
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	topatotal = get_font_struct(fid, sid)->ascent;
	topdtotal = get_font_struct(fid, sid)->descent;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text = text_height(fid, sid, &topatotal, &topdtotal,
			   text + 6, tmpmark);
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = topatotal + sqrtoffset;
	tmp = sqrtoffset + topatotal;
	if ( tmp > *atotal )
	  *atotal = tmp;
	Currentlength--;
	offset = NONE;
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 4))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	tmp = script_offset(fid,sid) + topatotal;
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = script_offset(fid,sid) + botdtotal;
	if ( tmp > *dtotal ) *dtotal = tmp;
	Currentlength--;
	offset = NONE;
	break;
      case COM_SUM:		/* math - summation */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 4))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	tmp = topatotal + topdtotal +
	  get_font_struct(SYMBOL, sid)->ascent; 
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = botatotal + botdtotal +
	  get_font_struct(SYMBOL, sid)->descent; 
	if ( tmp > *dtotal ) *dtotal = tmp;
	Currentlength--;
	offset = NONE;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	topatotal = topdtotal = botatotal = botdtotal = 0;
	if ((tmpmark = findmark(*(text + 5))) == 'e') { 
	  ltxErrAlert("Invalid delimiter!", text);
	  ;
	  return(NULLCHAR);
	}
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    topatotal = get_font_struct(fid, tmpsid)->ascent;
	    topdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 2))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &topatotal, &topdtotal,
			       text + 3, tmpmark2);
	    break;
	  case SUBCOM_FROM:
	    botatotal = get_font_struct(fid, tmpsid)->ascent;
	    botdtotal = get_font_struct(fid, tmpsid)->descent;
	    if ((tmpmark2 = findmark(*(text + 4))) == 'e') { 
	      ltxErrAlert("Invalid delimiter!", text);
	      ;
	      return(NULLCHAR);
	    }
	    text = text_height(fid, tmpsid, &botatotal, &botdtotal,
			       text + 5, tmpmark2);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    ;
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Topdescent[thiscount] = topdtotal;
	Botascent[thiscount] = botatotal;
	tmp = topatotal + topdtotal +
	  get_font_struct(SYMBOL, sid)->ascent; 
	if ( tmp > *atotal ) *atotal = tmp;
	tmp = botatotal + botdtotal +
	  get_font_struct(SYMBOL, sid)->descent; 
	if ( tmp > *dtotal ) *dtotal = tmp;
	Currentlength--;
	offset = NONE;
	break;
      case COM_SPACE:         /* explicit spacing */
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi); 
	text += tmpi+1;
	Currentlength--;
	offset = NONE;
	break;
      case COM_NL: 
	/* Came to an explicit line break. Check if there are more 
	   than one break line requested.  Get total height of
	   current line, point to next line, clear i to start at
	   beginning of next line. */

	Lineheight[Current] = *atotal + *dtotal;
	Lineascent[Current] = *atotal;

        tmpi = tmpskip = 1;
        if ( delimiter(*(text + 2)) ) 
          tmpskip = getnumskip(text,findmark(*(text + 2)),&tmpi);

	Current += tmpskip;
	text += tmpi + 1;	
	*atotal = get_font_struct(fid, sid)->ascent;
	*dtotal = get_font_struct(fid, sid)->descent;
	Currentlength = Linenumchar[Current];
#ifdef LTXO_DEBUG 
	printf("(NL): i: %d text: %.1s Cur: %d Curlen: %d\n",
	       i,(text+i),Current,Currentlength);  
#endif 
	while ( *text == SP ) text++;
	offset = NONE;
	break;
      default:
	ltxErrAlert("Unknown command!", text);	
	;
	return(NULLCHAR);
	}
      /* After deciphering the command, do the necessary recursive 
         call. */
      if (offset) {
	if ((tmpmark = findmark(*(text + offset - 1))) == 'e') {
	  ltxErrAlert("Invalid delimiter!", text);
	  return(NULLCHAR);
	}
	else
	  text = fill_text_height(tmpfid, tmpsid, atotal, dtotal, 
				  (text + offset), tmpmark);
      }
    }
    else {
      Currentlength--;          /* just continue deciphering in */
      i++;                      /* current mode */ 
    }
  }
}



/**********************************************************************
 * Private Routine: fill_text_out(fid, sid, text, flags, endmark) 
 * Requires: 
 *  fid     = the internal font id number used by LTxo.
 *  sid     = the internal size id number used by LTxo.
 *  text   =   the text to be displayed.  
 *  flags  = flags giving hints as to how to output text.  
 *  endmark = the marker which tells us that the text using the 
 *           current font is finished. Endmarks are either 
 *           }, ], ), or >.   
 * Description: 
 *  This routine deciphers the complex text in a filled environment
 *  and calls another routine, text_draw, to actually draw the text
 *  onto the screen. 
 **********************************************************************/

char *fill_text_out(fid, sid, text, flags, endmark)
     unsigned fid, sid;
     long flags;
     char *text, endmark;
{
  unsigned i = 0;          /* index to the characters in text */
  unsigned tmpsid,bestsid; /* temp size ids */ 
  unsigned tmpfid,tmpi;	   /* temp font id & index  */ 
  char tmpmark;	           /* temp endmark found */ 
  unsigned tmpskip;        /* # of blank lines requested */ 
  int savebase, tmp; 
  int thiscount, width;
  int overoffset;
  XGCValues vals;

#ifdef LTXO_DEBUG
  printf("fill_text_out(%d, %d, %s, %d, %c)\n", fid, sid, text, flags,
	 endmark); 
#endif

  /* The following while loop deciphers the characters in the text and 
     checks for special cases. If the next character is the command
     indicator, decipher what the command is and call fill_text_out
     recursively to output text under desired mode. */ 

  while(1) {
    if( *(text + i) == NULLCHAR ) {
      /* Came to end of text. If there are already some deciphered 
         text, display it first, and then return rest of text. */
      if ( endlimiter(endmark) && !(ltxErrFlag)) {
	ltxErrAlert("Unexpected end of text! Expects a delimiter."," ");
	
	return(NULLCHAR);
      }
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
	Xbase += width;         /* advance Xbase to continue output */
      }
      
      return(text + i);
    }

    else if(*(text + i) == endmark) {
      /* Came to end of command. If there are already some deciphered 
	 text, display it first, and then return rest of text skipping
	 the endmark. */
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
	Xbase += width;         /* advance Xbase to continue output */
      }
      
      return(text + i + 1);  
    }

    else if( *(text + i) == NL || *(text + i) == CR || *(text + i) ==
	    SP) { 
      /* Came to an implicit line break. If no more text can fit on 
	 current line, draw rest of text in line and point to next
	 line, increment text & clear i to start from beginning, and
	 increment Ybase to for the next baseline. */ 
      Currentlength--;
      i++;
      width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
      text_draw(fid, sid, text, i, width, flags);
      Xbase += width;         /* advance Xbase to continue output */
      text += i;
      i = 0;
      while ( *text == SP ) text++;
      if (Currentlength == 0 ) {
	if ( strncmp(text,"@.",2) ) {
	  Ybase += (Lineheight[Current] - Lineascent[Current]) +
	            Lineascent[Current + 1];
	  Current++;
	  /* Now, set Xbase to the proper place */
	  if(flags & H_CENTER) 
	    Xbase = Box.x + X + (Boxwidth - Linewidth[Current]) / 2; 
	  else if(flags & H_RIGHT) 
	    Xbase = Box.x + X + (Boxwidth - Linewidth[Current]); 
	  else
	    Xbase = Box.x + X;

	  Currentlength = Linenumchar[Current];
	}
      }
    }
      
    else if(*(text + i) == '@') {
      if(i) {
	width = XTextWidth(get_font_struct(fid, sid), text, (int)i);
	text_draw(fid, sid, text, i, width, flags);
	Xbase += width;         /* advance Xbase to continue output */
      }
      text += i;
      i = 0;
      switch( getcommand_id(text+1) ) {
      case NO_COM:        /* user really wants an '@' character */
	Currentlength--;
	text++;         /* skips first @ */
	i++;		/* continue to next character */
	break;
      case COM_BOLD:     /* bold */ 
	/* already in bold face */	
	tmpfid = ((flags & BOLD) ? fid : findbold(fid));
	text = fill_text_out(tmpfid, sid, text + 3, flags | BOLD, 
			     findmark(*(text + 2)));
	break;
      case COM_ITA:    /* italics */
	/* already in italic face */
	tmpfid = ((flags & ITALICS) ? fid : finditalic(fid));
	text = fill_text_out(tmpfid, sid, text + 3, flags | ITALICS, 
			     findmark(*(text + 2)));
	break;
      case COM_GREEK:  /* greek */
	text = fill_text_out((unsigned)GREEK, sid, text + 3,
			     flags | GRK, findmark(*(text + 2)));
	break;
      case COM_HELVET:  /* helvetica*/
	text = fill_text_out((unsigned)HELVET, sid, text + 3,
			     flags, findmark(*(text + 2)));
	break;
      case COM_TIMES:  /* times */
	text = fill_text_out((unsigned)TIMES, sid, text + 3,
			     flags, findmark(*(text + 2)));
	break;
      case COM_FIX:  /* fixed */
	if(flags & (ITALICS | BOLD)) {
	  ltxErrAlert("No italics or bold in fixed font",text);
	  
	  return(NULLCHAR);
	}	  
	text = fill_text_out((unsigned)FIXED, sid, text + 3,
			     flags, findmark(*(text + 2)));
	break;
      case COM_CENTER:	/* center */
	if(flags & H_CENTER) { /* already centering */
	  ltxErrAlert("Recursive centering!", text);
	  
	  return(NULLCHAR);
	}	  
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current])/2;
	text = fill_text_out(fid, sid, text + 8,
			     flags | H_CENTER, 
			     findmark(*(text + 7)));
	break;
      case COM_C:	/* center */
	if(flags & H_CENTER) { /* already centering */
	  ltxErrAlert("Recursive centering!", text);
	  
	  return(NULLCHAR);
	}	  
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current])/2;
	text = fill_text_out(fid, sid, text + 3,
			     flags | H_CENTER, 
			     findmark(*(text + 2)));
	break;
      case COM_RIGHT:    	/* right flush */
	if(flags & H_RIGHT) {  /* already flushing right */
	  ltxErrAlert("Recursive flush right!", text);
	  
	  return(NULLCHAR);
	}
	Xbase = Box.x + X + (Boxwidth - Linewidth[Current]);
	text = fill_text_out(fid, sid, text + 3,
			     flags | H_RIGHT, 
			     findmark(*(text + 2)));
	break;
      case COM_U:	  /* underline */
	text = fill_text_out(fid, sid, text + 3,
			     flags | UNDERLINE, 
			     findmark(*(text + 2)));
	break;
      case COM_UP1:       /* increasing size by 1 */
	tmpsid = ( sid >= BIGGEST ? sid : sid + 1);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_UP2:       /* increasing size by 2 */
	tmpsid = ( sid >= BIG ? sid : sid + 2);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_UP3:       /* increasing size by 3 */
	tmpsid = ( sid >= MEDBIG ? sid : sid + 3);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_UP4:       /* increasing size by 4 */
	tmpsid = ( sid >= MEDIUM ? sid : sid + 4);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_UP5:       /* increasing size by 5 */
	tmpsid = ( sid >= MEDSMALL ? sid : sid + 5);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_UP6:       /* increasing size by 6 */
	tmpsid = ( sid >= SMALL ? sid : sid + 6);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN1:       /* decreasing size by 1 */
	tmpsid = ( sid == SMALLEST ? sid : sid - 1);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN2:       /* decreasing size by 2 */
	tmpsid = ( sid <= SMALL ? sid : sid - 2);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN3:       /* decreasing size by 3 */
	tmpsid = ( sid <= MEDSMALL ? sid : sid - 3);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN4:       /* decreasing size by 4 */
	tmpsid = ( sid <= MEDIUM ? sid : sid - 4);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN5:       /* decreasing size by 5 */
	tmpsid = ( sid <= MEDBIG ? sid : sid - 5);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_DOWN6:       /* decreasing size by 6 */
	tmpsid = ( sid <= BIG ? sid : sid - 6);
	text = fill_text_out(fid, tmpsid, text + 4,
			     flags, findmark(*(text + 3)));
        break;
      case COM_SUPER:	  /* superscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	Ybase -= script_offset(fid, sid);
	text = fill_text_out(fid, tmpsid, text + 3,
			     flags, findmark(*(text + 2)));
	Ybase += script_offset(fid, sid);
	break;
      case COM_SUB:	   /* subscripts */
	tmpsid = ( sid != SMALLEST ? sid - 1 : sid );
	Ybase += script_offset(fid, sid);
	text = fill_text_out(fid, tmpsid, text + 3,
			     flags, findmark(*(text + 2)));
	Ybase -= script_offset(fid, sid);
	break;
      case SYM_ALEPH:         /* aleph symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_ALEPH][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,ALEPH,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_AND:         /* 'logical and' symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_AND][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,AND,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_ANGLE:         /* angle symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_ANGLE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,ANGLE,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_APPROX:         /* approximate symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_APPROX][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,APPROX,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_AST:		/* asterisk symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_AST][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,AST,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_BULLET:         /* bullet symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_BULLET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,BULLET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_EXISTS:		/* exists symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EXISTS][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EXISTS,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_EMPTYSET:	/* emptyset symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EMPTYSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EMPTYSET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_EQV:		/* equivalence symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_EQV][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,EQV,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_FORALL:		/* FORALL symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_FORALL][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,FORALL,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_GTE:	       /* Greater than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_GTE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,GTE,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_IN:		/* IN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_IN][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,IN,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_INFTY:         /* infinity symbol */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_INFTY][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,INF,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_INTER:		/* INTER symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_INTER][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,INTER,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_LTE:	       /* Less than & equal symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_LTE][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,LTE,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_MULT:		/* multiplication symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_MULT][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,MULT,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 5;
	if (*text == SP) text++;
	break;
      case SYM_NABLA:		/* NABLA symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NABLA][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NABLA,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_NEQ:		/* NEQ symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NEQ][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NEQ,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case SYM_NOTIN:		/* NOTIN symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_NOTIN][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,NOTIN,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_OR:		/* OR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_OR][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,OR,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_PARTIAL:       /* partial symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PARTIAL][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PAR,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 8;
	if (*text == SP) text++;
	break;
      case SYM_PM:		/* PM symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PM][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PM,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 3;
	if (*text == SP) text++;
	break;
      case SYM_PRSUBSET:	/* PRSUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PRSUBSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PRSUBSET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_PRSUPSET:	/* PRSUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_PRSUPSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,PRSUPSET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_SIMILIAR:       /* SIMILIAR symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SIMILIAR][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SIMILIAR,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 9;
	if (*text == SP) text++;
	break;
      case SYM_SUBSET:		/* SUBSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SUBSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SUBSET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_SUPSET:		/* SUPSET symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_SUPSET][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,SUPSET,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 7;
	if (*text == SP) text++;
	break;
      case SYM_UNION:		/* UNION symbol */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_UNION][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,UNION,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 6;
	if (*text == SP) text++;
	break;
      case SYM_LEFTARROW:	/* leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_LEFTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,LEFTARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_UPARROW:		/* uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_UPARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,UPARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 8;
	if (*text == SP) text++;
	break;
      case SYM_RIGHTARROW:	/* rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_RIGHTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,RIGHTARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 11;
	if (*text == SP) text++;
	break;
      case SYM_DOWNARROW:	/* downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DOWNARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DOWNARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_DBLEFTARROW:	/* double leftarrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBLEFTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBLEFTARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 12;
	if (*text == SP) text++;
	break;
      case SYM_DBUPARROW:	/* double uparrow */
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBUPARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBUPARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 10;
	if (*text == SP) text++;
	break;
      case SYM_DBRIGHTARROW:	/* double rightarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBRIGHTARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBRIGHTARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 13;
	if (*text == SP) text++;
	break;
      case SYM_DBDOWNARROW:	/* double downarrow */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DBDOWNARROW][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DBDOWNARROW,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 12;
	if (*text == SP) text++;
	break;
      case SYM_DOT:	      /* dot - alternative to mult */	
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	width = sym_tbl[SYM_DOT][tmpsid].width;
	text_draw((unsigned)SYMBOL,tmpsid,DOT,1,width,flags | SYM);
	Xbase += width;
	Currentlength--;
	text += 4;
	if (*text == SP) text++;
	break;
      case COM_VEC:           /* vector symbol over variable */
	thiscount = Count++;
	savebase = Xbase;
	text = text_out(fid, sid, text + 5,
			flags, findmark(*(text + 4)));
	tmpsid = ( sid >= BIG ? MEDBIG : sid );
	Xbase = savebase;
	Ybase -= get_font_struct(fid, sid)->ascent;
	text_draw((unsigned)SYMBOL,tmpsid,RIGHTARROW,1,0,flags | SYM);
	Ybase += get_font_struct(fid, sid)->ascent;
	Xbase += Topwidth[thiscount];
	break;
      case COM_OVER:		/* math - over */
	thiscount = Count++;
	tmpmark = findmark(*(text + 5));
	text += 6;
	tmpsid = ( sid != SMALLEST ? sid-1 : sid );
	tmp = max(Topwidth[thiscount],Botwidth[thiscount]);
	savebase = Xbase;
	if (Overflag) 
	  overoffset = 0;
	else 
	  overoffset = (get_font_struct(fid, sid)->ascent +
			get_font_struct(fid, sid)->descent)/2;
	Overflag++;
	while ( *text != tmpmark ) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_NUM:
	    Xbase += ((tmp - Topwidth[thiscount])/2) + 1;
	    Ybase -= Topdescent[thiscount] + overoffset;
	    text = text_out(fid, tmpsid, text + 4, flags,
			    findmark(*(text + 3))); 
	    Xbase = savebase;
	    Ybase += Topdescent[thiscount] + overoffset;
	    break;
	  case SUBCOM_DENOM:
	    Xbase += ((tmp - Botwidth[thiscount])/2) + 1;
	    Ybase += Botascent[thiscount] - overoffset;
	    text = text_out(fid, tmpsid, text + 6, flags,
			    findmark(*(text + 5)));
	    Xbase = savebase;
	    Ybase -= Botascent[thiscount] - overoffset;
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	Overflag--;
	
	vals.line_width = sid >= BIG ? 3 : 1;
	XChangeGC(LTx_dpy, LTxText_gc, GCLineWidth, &vals);

	XDrawLine(LTx_dpy, Win, LTxText_gc,
		  Xbase, Ybase-overoffset-2,
		  Xbase+tmp, Ybase-overoffset-2);
	Xbase += tmp;
	text++;		/* skip endmark */
	Currentlength--;
	break;
      case COM_SQRT:		/* math - squareroot */
	thiscount = Count++;
	tmp = Botascent[thiscount]+Topdescent[thiscount];
	bestsid = findbestsqrt(tmp);
	Ybase += get_font_struct(SYMBOL, bestsid)->ascent -
	         Botascent[thiscount]; 
	if ( bestsid >= BIG ) {
	  width =
	    XTextWidth(get_font_struct(SYMBOL, bestsid) ,BIGSQRT,1);  
	  text_draw((unsigned)SYMBOL,bestsid,BIGSQRT,1,width,
		    flags | SYM); 
	}
	else {
	  width = sym_tbl[SYM_SQRT][bestsid].width;
	  text_draw((unsigned)SYMBOL,bestsid,SQRT,1,width,flags | SYM);
	}
	Ybase -= get_font_struct(SYMBOL, bestsid)->ascent -
	         Botascent[thiscount]; 
	Xbase += width;
	savebase = Xbase;
	text = text_out(fid, sid, text + 6, flags,
			findmark(*(text + 5)));

	vals.line_width = sid >= BIG ? 3 : 1;
	XChangeGC(LTx_dpy, LTxText_gc, GCLineWidth, &vals);

	XDrawLine(LTx_dpy, Win, LTxText_gc,
		  savebase,
		  Ybase-Botascent[thiscount], 
		  savebase+Topwidth[thiscount],
		  Ybase-Botascent[thiscount]); 
	Currentlength--;
	break;
      case COM_INT:		/* math - integration */
	thiscount = Count++;
	if ( sid >= BIG ) {
	  width = XTextWidth(get_font_struct(SYMBOL, sid),BIGINT,1); 
	  text_draw((unsigned)SYMBOL,sid,BIGINT,1,width,flags | SYM);
	}
	else {
	  width = sym_tbl[SYM_INT][sid].width;
	  text_draw((unsigned)SYMBOL,sid,INT,1,width,flags | SYM);
	}
	Xbase += width;
	savebase = Xbase;
	tmpmark = findmark(*(text + 4));
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Ybase -= script_offset((unsigned)SYMBOL, sid);
	    text = text_out(fid, tmpsid, text + 3,
			    flags, findmark(*(text + 2)));
	    Ybase += script_offset((unsigned)SYMBOL, sid);
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Ybase += script_offset((unsigned)SYMBOL, sid);
	    text = text_out(fid, tmpsid, text + 5,
			    flags, findmark(*(text + 4)));
	    Ybase -= script_offset((unsigned)SYMBOL, sid);
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Xbase = savebase;
	Xbase += max(Topwidth[thiscount], Botwidth[thiscount]);
	Currentlength--;
	break;
      case COM_SUM:		/* math - summation */
	thiscount = Count++;
	savebase = Xbase;
	if ( sid >= BIG ) 
	  width = XTextWidth(get_font_struct(SYMBOL, sid),BIGSIG,1);
	else 
	  width = sym_tbl[SYM_SIG][sid].width;
	tmp = max(width, max(Topwidth[thiscount],
			     Botwidth[thiscount])); 
	Xbase += (tmp - width)/2;
	if ( sid >= BIG ) 
	  text_draw((unsigned)SYMBOL,sid,BIGSIG,1,width,flags | SYM);
	else 
	  text_draw((unsigned)SYMBOL,sid,SIG,1,width,flags | SYM);
	tmpmark = findmark(*(text + 4));
	text += 5;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Xbase += (tmp - Topwidth[thiscount])/2;
	    Ybase -= get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    text = text_out(fid, tmpsid, text + 3, flags,
			    findmark(*(text + 2)));
	    Ybase += get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Xbase += (tmp - Botwidth[thiscount])/2;
	    Ybase += get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    text = text_out(fid, tmpsid, text + 5, flags,
			    findmark(*(text + 4)));
	    Ybase -= get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Xbase = savebase;
	Xbase += tmp;
	Currentlength--;
	break;
      case COM_PROD:		/* math - product */
	thiscount = Count++;
	savebase = Xbase;
	if ( sid >= BIG ) 
	  width = XTextWidth(get_font_struct(SYMBOL, sid),BIGPI,1);
	else 
	  width = sym_tbl[SYM_PI][sid].width;
	tmp = max(width, max(Topwidth[thiscount],
				    Botwidth[thiscount])); 
	Xbase += (tmp - width)/2;
	if ( sid >= BIG ) 
	  text_draw((unsigned)SYMBOL,sid,BIGPI,1,width,flags | SYM);
	else 
	  text_draw((unsigned)SYMBOL,sid,PI,1,width,flags | SYM);
	tmpmark = findmark(*(text + 5));
	text += 6;
	tmpsid = ( sid > SMALL ? sid - 2 : sid );
	while (*text != tmpmark) {
	  while (*text == SP || *text == COMMA) text++;
	  switch ( getcommand_id(text) ) {
	  case SUBCOM_TO:
	    Xbase = savebase;
	    Xbase += (tmp - Topwidth[thiscount])/2;
	    Ybase -= get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    text = text_out(fid, tmpsid, text + 3, flags,
			    findmark(*(text + 2)));
	    Ybase += get_font_struct(SYMBOL, sid)->ascent +
	      Topdescent[thiscount];
	    break;
	  case SUBCOM_FROM:
	    Xbase = savebase;
	    Xbase += (tmp - Botwidth[thiscount])/2;
	    Ybase += get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    text = text_out(fid, tmpsid, text + 5, flags,
			    findmark(*(text + 4)));
	    Ybase -= get_font_struct(SYMBOL, sid)->descent +
	      Botascent[thiscount];
	    break;
	  default:
	    ltxErrAlert("Unknown subcommand or invalid delimiter!",text);
	    
	    return(NULLCHAR);
	  }
	  while (*text == SP) text++; 
	}
	text++;		/* skips endmark */
	Xbase = savebase;
	Xbase += tmp;
	Currentlength--;
	break;
      case COM_SPACE:         /* explicit spacing */
	tmpi = tmpskip = 1;
	if ( delimiter(*(text + 2)) ) 
	  tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi); 
	text += tmpi+1;
	Xbase += tmpskip*XTextWidth(get_font_struct(sid, fid)," ",1);
	Currentlength--;
	break;
      case COM_NL: 
	/* Came to an explicit line break. Check if there is more than 
	   one break line requested. Point to next line, increment
	   text & clear i to start from beginning, and increment Ybase
	   to for the next baseline. */ 
	
        tmpi = tmpskip = 1;
        if ( delimiter(*(text + 2)) )
          tmpskip = getnumskip(text,findmark(*(text+2)),&tmpi);

	text += tmpi + 1;
	Ybase += (Lineheight[Current] - Lineascent[Current]) +
	         ((tmpskip-1)*Lineheight[Current]) + 
		   Lineascent[Current+tmpskip];
	Current += tmpskip; 
	Currentlength = Linenumchar[Current];
	
	/* Now, set Xbase to the proper place */
	if(flags & H_CENTER) 
	  Xbase = Box.x + X + (Boxwidth - Linewidth[Current]) / 2; 
	else if(flags & H_RIGHT) 
	  Xbase = Box.x + X + (Boxwidth - Linewidth[Current]); 
	else
	  Xbase = Box.x + X;

	while ( *text == SP ) text++;
	break;
      default:
	ltxErrAlert("Unknown command!", text);
	return(NULLCHAR);
      }
    }
    else {
      Currentlength--;		/* just continue to decipher on */
      i++;			/* current mode */
    }
  }
}



