/* $Date: 91/12/13 10:01:05 $    $Revision: 1.1 $  by $Author: joe $  */
/* 
  Copyright (C) 1991 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */
/**************************************************************
  CmFigureHandler.cc - Figure Handler (under CmFigureEditor)
  Contact: maschaub

  CmFigureHandler: The editing (double click, drag, etc.) of a figure is done 
    by adding an XtAddEventHandler to each editable figure of the page.
    If activated, CmFigureActivated is called, and all resizing, 
    moving, and editing options is done by a CmFigureHandler, which
    goes ahead and asks CmFigureParser to change the parameters as necessary.
 **************************************************************/

#include <CmFigureHandler.h>
#include <CmFigureEditor.h>
#include <CmFigure.h>
#include <CmPage.h>
#include <CmCDS.h>
#include <CmDialog.h>
#include <CmMenu.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <Xm/ToggleBG.h>
#include <math.h>

/********** constants *************/
#define CM_MINIMUM_FIGURE_WIDTH  10
#define CM_MINIMUM_FIGURE_HEIGHT 10
// grab range
#define CM_GR 7

/**************************************************************
 CmFigureActivated C Procedure
   Added as Callback with XtAddEventHandler to ALL editable figures
   Called when button is pressed.
   Calls the figure handler, and asks it to handle the figure
    activation.
 **************************************************************/
void CmFigureActivated(const Widget w, CmFigureHandler * const figure_handler, 
		       XEvent * const event) {
  figure_handler->FigureActivated(w, event); 
}

/**************************************************************
 CmFigureHandler setup
 setups up cursors and GC
 **************************************************************/

void CmFigureHandler::setup(const Widget w)
{
  if (!display) {
    display = XtDisplay(w);
    workarea_gc = XCreateGC(display, XtWindow(XtParent(w)), 0, NULL);
    XSetSubwindowMode(display, workarea_gc, IncludeInferiors);
    XSetForeground(display, workarea_gc, XtScreen(w)->black_pixel);
    XSetFunction(display, workarea_gc, GXxor);

    top_left_corner = XCreateFontCursor(display, XC_top_left_corner);
    top_side = XCreateFontCursor(display, XC_top_side);
    top_right_corner =XCreateFontCursor(display, XC_top_right_corner); 
    left_side = XCreateFontCursor(display, XC_left_side);
    right_side = XCreateFontCursor(display, XC_right_side);
    bottom_left_corner =  XCreateFontCursor(display, XC_bottom_left_corner);
    bottom_side = XCreateFontCursor(display, XC_bottom_side);
    bottom_right_corner = XCreateFontCursor(display, XC_bottom_right_corner);
    Move = XCreateFontCursor(display, XC_fleur);
    Norm = XCreateFontCursor(display, XC_left_ptr);
  }
}

/***************
   rounds the value to the nearest scale 
   ie. (24 to nearest 10 gives 20)
 **********************/
int CmFigureHandler::round_to_nearest( int value, int scale ) {
  int sdiv2 = (int)(scale/2);
  int cutoff = (int)(scale*rint(value/scale)+sdiv2);
  if (value > cutoff)
    return(cutoff + sdiv2);
  else
    return(cutoff - sdiv2);
}

/**************************************************************
 CmFigureHandler should_we_resize
   Called when user clicks on an editable figure by FigureActivated
   Changes cursor to match the resize position
    has been pressed, and changes the cursor appropriately.
   Returns true if you are close enough to the edge to be in 
   resize mode, otherwise it returns false
 **************************************************************/
Boolean CmFigureHandler::should_we_resize(const Widget w, XEvent *e) {
  
  mouse_x = (e->xbutton.x+border_width);
  mouse_y = (e->xbutton.y+border_width);
  /* get boolean values for position */
  left = ( mouse_x < CM_GR );
  top  = ( mouse_y < CM_GR );
  right = ( abs_width - mouse_x < CM_GR );
  bottom = ( abs_height - mouse_y < CM_GR );
  /* dispatch off of them */
  if ( left || right || top || bottom ) {
    if ( left && top ) XDefineCursor(display, XtWindow(w), top_left_corner); 
    else 
     if ( left && bottom ) XDefineCursor(XtDisplay(w), XtWindow(w), 
					 bottom_left_corner); else
     if ( right && top ) XDefineCursor(XtDisplay(w), XtWindow(w), 
				       top_right_corner); else 
     if ( right && bottom ) XDefineCursor(XtDisplay(w), XtWindow(w), 
					  bottom_right_corner); else
     if ( left ) XDefineCursor(XtDisplay(w), XtWindow(w), left_side); else
     if ( right ) XDefineCursor(XtDisplay(w), XtWindow(w), right_side); else
     if ( top ) XDefineCursor(XtDisplay(w), XtWindow(w), top_side); else
     if ( bottom ) XDefineCursor(XtDisplay(w), XtWindow(w), bottom_side); 
     return(True);
   } else return(False);
}

/**************************************************************
 CmFigureHandler get_absolute_geometry
 converts position geometry of figure to pixel geometry
 **************************************************************/
void CmFigureHandler::get_absolute_geometry(const Widget w) {  
  int leftpos, toppos, rightpos, botpos, border_width;
  int fraction_base;
  
  XtVaGetValues(w,
		XmNrightPosition, &rightpos,
		XmNbottomPosition, &botpos,
		XmNleftPosition, &leftpos,
		XmNtopPosition, &toppos,
		XmNborderWidth, &border_width, NULL);

  XtVaGetValues(XtParent(w),
		XmNwidth, &parent_width,
		XmNheight, &parent_height, 
		XmNfractionBase, &fraction_base, NULL);
    
  abs_to_xposition = (float)fraction_base/(float)parent_width;
  xposition_to_abs = (float)parent_width/(float)fraction_base;
  abs_to_yposition = (float)fraction_base/(float)parent_height;
  yposition_to_abs = (float)parent_height/(float)fraction_base;
  abs_x = (int)((float)(leftpos) * xposition_to_abs + 0.5);
  abs_y = (int)((float)(toppos) * yposition_to_abs + 0.5);
  abs_width = (int)((float)(rightpos-leftpos) * xposition_to_abs + 0.5);
  abs_height = (int)((float)(botpos-toppos) * yposition_to_abs + 0.5);
}

/**************************************************************
 CmFigureHandler set_position_geometry
 sets the figure's position geometry based on absolute geometry
 that was used to draw the rectangle that moves
 **************************************************************/
void CmFigureHandler::set_position_geometry(const Widget) {  
//  XSync(display, False); // wait for figure to move
 }

/**************************************************************
 CmFigureHandler handle_resize
 **************************************************************/
void CmFigureHandler::handle_resize(const Widget w, XEvent *event) {
   const int oldx = abs_x, oldy = abs_y;
   const int oldwidth = abs_width, oldheight = abs_height;
   int off_x, off_y;
   int x1, y1, x2, y2;
   Position root_x, root_y;
   Position parent_x, parent_y;
   XtTranslateCoords(w, event->xbutton.x, event->xbutton.y,
			  &root_x, &root_y);
   XtTranslateCoords(XtParent(w), 0, 0,
			  &parent_x, &parent_y);
   off_x = root_x - parent_x;
   off_y = root_y - parent_y;

   if (Snap) {
     off_x = (int) (round_to_nearest((int)(off_x *abs_to_xposition + .5), 
				     Xscale) * xposition_to_abs + .5);
     off_y = (int) (round_to_nearest((int)(off_y *abs_to_yposition + .5), 
				     Yscale) * yposition_to_abs + .5);
   }
   
   if ( left ) {  
     x1 = off_x;  
     x2 = oldwidth+oldx;
     if ( x1 < 0 ) x1 = 0; 
     if ( x1 > x2 - CM_MINIMUM_FIGURE_WIDTH) 
       x1 = x2 - CM_MINIMUM_FIGURE_WIDTH;
   } else if ( right ) {
     x1 = oldx;
     x2 = off_x;
     if ( x2 < x1 + CM_MINIMUM_FIGURE_WIDTH) 
       x2 = x1 + CM_MINIMUM_FIGURE_WIDTH;
     if ( x2 > parent_width - CM_GR ) 
       x2 = parent_width - CM_GR;
   } else {
     x1 = oldx;
     x2 = oldx + oldwidth;
   }
   if ( top ) {
     y1 = off_y;
     y2 = oldheight+oldy;
     if ( y1 < 0 ) y1 = 0;
     if ( y1 > y2 - CM_MINIMUM_FIGURE_HEIGHT ) 
       y1 = y2 - CM_MINIMUM_FIGURE_HEIGHT;
   } else if ( bottom ) {
     y1 = oldy;
     y2 = off_y;
     if ( y2 < y1 + CM_MINIMUM_FIGURE_HEIGHT ) 
       y2 = y1 + CM_MINIMUM_FIGURE_HEIGHT;
     if ( y2 > parent_height - CM_GR ) 
       y2 = parent_height - CM_GR;
   } else { 
     y1 = oldy;
     y2 = oldy + oldheight;
   }
   XDrawRectangle(XtDisplay(w), 
		  XtWindow(XtParent(w)), workarea_gc, 
		  oldx, oldy, oldwidth, oldheight);
     
   XDrawRectangle(XtDisplay(w), 
		  XtWindow(XtParent(w)), workarea_gc, 
		  x1, y1, x2-x1, y2-y1);
   abs_x = x1;
   abs_y = y1;
   abs_width = x2-x1;
   abs_height = y2-y1;
}


/**************************************************************
 CmFigureHandler finish_resize
  Requires: that the user is resizing a figure
  Modifies: the figure's position and size.
 **************************************************************/
void CmFigureHandler::finish_resize(const Widget w, XEvent *) {  

  // erase last rectangle drawn
  XDrawRectangle(display, XtWindow(XtParent(w)), workarea_gc, 
		 abs_x, abs_y, abs_width, abs_height);
  
  // set values in figure
  CmFigure * const cw = editor->Page->PageCmFigure->WidgetToCmFigure(w);
  cw->Resize((int)(abs_x * abs_to_xposition + 0.5), 
	     (int) (abs_y * abs_to_yposition + 0.5),
	     (int) (abs_width * abs_to_xposition + 0.5), 
	     (int) (abs_height * abs_to_yposition + 0.5));

  XDefineCursor(display, XtWindow(w), Norm);
  resizing = False;
}

/**************************************************************
 CmFigureHandler handle_drag
  Called by FigureActivated
  Erases the grips at the old abs_x, abs_y and draws in the new one
  at the place where the cursor was moved to (x_drag,y_drag)
 **************************************************************/
void CmFigureHandler::handle_drag(const Widget w, XEvent *event) {
  Position root_x, root_y;
  Position parent_x, parent_y;
  
  const int oldx = abs_x, oldy = abs_y;
  XtTranslateCoords(w, event->xbutton.x, event->xbutton.y,
			 &root_x, &root_y);
  XtTranslateCoords(XtParent(w), 0, 0,
			 &parent_x, &parent_y);
  abs_x = root_x - parent_x;
  abs_y = root_y - parent_y;
  
  // adjust for cursor being in the center of the figure
  abs_x -= drag_x;   
  abs_y -= drag_y;
  if (Snap) {
    abs_x = (int) (round_to_nearest((int)(abs_x * abs_to_xposition + .5), 
				    Xscale) * xposition_to_abs + .5); 
    abs_y = (int) (round_to_nearest((int)(abs_y * abs_to_yposition + .5), 
				    Yscale) * yposition_to_abs + .5); 
  }
  /* this area needs to be corrected for the size of the figure */
  if ( abs_x < 0 ) abs_x = 0;
  if ( abs_y < 0 ) abs_y = 0;
  if ( abs_x > parent_width - abs_width ) 
    abs_x = parent_width - abs_width;
  if ( abs_y > parent_height - abs_height ) 
    abs_y = parent_height - abs_height;
  
  // erase old rectangle
  XDrawRectangle(display, XtWindow(XtParent(w)), workarea_gc, 
		 oldx, oldy, abs_width, abs_height);
  // draw new one
  XDrawRectangle(display, XtWindow(XtParent(w)), workarea_gc, 
		 abs_x, abs_y, abs_width, abs_height);
}

/**************************************************************
 CmFigureHandler finish_drag
  Requires: that the user is dragging a figure
  Modifies: the position of the figure
 **************************************************************/
void CmFigureHandler::finish_drag(const Widget w, XEvent *event) { 
  Position root_x, root_y;
  Position parent_x, parent_y;

  dragging = False;
  // erase last rectangle drawn
  XDrawRectangle(display, XtWindow(XtParent(w)), workarea_gc, 
		 abs_x, abs_y, abs_width, abs_height);
  XtTranslateCoords(w, event->xbutton.x, event->xbutton.y,
			 &root_x, &root_y);
  XtTranslateCoords(XtParent(w), 0, 0,
			 &parent_x, &parent_y);
  abs_x = root_x - parent_x;
  abs_y = root_y - parent_y;

  // adjust for the cursor being in the center of the figure
  abs_x -= drag_x;
  abs_y -= drag_y;

  // check to see if new position is in the work area
  if ( abs_x < 0 ) abs_x = 0;   
  if ( abs_y < 0 ) abs_y = 0;
  if ( abs_x > parent_width - abs_width ) 
    abs_x = parent_width - abs_width;   
  if ( abs_y > parent_height - abs_height ) 
    abs_y = parent_height - abs_height;
  
  
  // set values in figure
  CmFigure * const cw = editor->Page->PageCmFigure->WidgetToCmFigure(w);
  cw->Resize((int)(abs_x * abs_to_xposition + 0.5), 
	     (int) (abs_y * abs_to_yposition + 0.5),
	     (int) (abs_width * abs_to_xposition + 0.5), 
	     (int) (abs_height * abs_to_yposition + 0.5));

  XDefineCursor(display, XtWindow(w), Norm);
}
  
/**************************************************************
 CmFigureHandler FigureActivated
  Modifies: 
   The position and size of the figure activated are modified
    if the user single clicks.
   The attributes of the figure are modified if the user double
    clicks.
  Effects:
   Called by CmFigureActivated, when a figure is activated.
   If a grip is pressed, then it sets resizing, and calls
    handle_resize each time the user moves until he lets go,
    and finish_resize is called.
   If not, then it set dragging, and calls handle_drag each time
    the user moves the cursor until he lets go, and finish_drag
    is called.
   If a double click occurs, then a dialog box will appear
    and prompt the user to change the figure's attributes.
 **************************************************************/
void CmFigureHandler::FigureActivated(Widget w, XEvent *event) {  

  if ( editor->Editing == False ) return; // return if not in edit mode
  setup(w);
  switch ( event->type ) {
  case ButtonPress:     
    if ( event->xbutton.button == Button2 &&
	event->xbutton.state && ShiftMask != 0) {
      button_pressed = True;
      dragging = False; 
      get_absolute_geometry(w);  // get geometry of new figure
      editor->
	SetHighlightedCmFigure(editor->Page->PageCmFigure->WidgetToCmFigure(w));

      resizing = should_we_resize(w, event); 
      // set coords to determine if we should move or not
      if ( resizing == False ) {
	// correct for borderwidth since 0,0 is the inner border
	drag_x = event->xbutton.x + border_width;
	drag_y = event->xbutton.y + border_width;
      } else // draw initial rectangle
	XDrawRectangle(display, XtWindow(XtParent(w)), workarea_gc, 
		       abs_x, abs_y, abs_width, abs_height);
      
    } 
    break;
  case MotionNotify:
    if ( button_pressed )
      if ( resizing == True ) handle_resize(w, event);
      else if ( dragging == True ) handle_drag(w, event);
      else if ( (dragging == False) && 
	       // correct for borderwidth since 0,0 is the inner border
	       (( (event->xbutton.x+border_width) - drag_x > CM_GR) ||
		( (event->xbutton.x+border_width) - drag_x < -CM_GR) ||
		( (event->xbutton.y+border_width) - drag_y > CM_GR)  ||
		( (event->xbutton.y+border_width) - drag_y < -CM_GR)))
	{ 
	  /* If you are not dragging and the pointer is moved more than
	   CM_GR then dragging should be turned on */
	  dragging = True;
	  XDefineCursor(display, XtWindow(w), Move);
	  // correct for borderwidth since 0,0 is the inner border
	  drag_x = (event->xbutton.x + border_width);
	  drag_y = (event->xbutton.y + border_width);
	  // draw initial rectangle to get drag going
	  XDrawRectangle(display, 
			 XtWindow(XtParent(w)), workarea_gc, 
			 abs_x, abs_y, abs_width, abs_height);
	  handle_drag(w, event);
	}
    break;
  case ButtonRelease:
    if  ( event->xbutton.button == Button2 &&
	event->xbutton.state && ShiftMask != 0) {
      button_pressed = False;
       if (event->xbutton.time - timePrevPress < MSDOUBLECLICK) 
	 editor->EditFigure(w);
       else timePrevPress = event->xbutton.time;
      if (dragging) finish_drag(w, event);
      if ( resizing ) finish_resize(w, event);
      break;
    }
  }
}

/**************************************************************
 CmFigureHandler GetHighlightedCmFigure
   Called by cut and copy from CmFigureEditor
   Returns the CmFigure that is presently highlighted
 **************************************************************/
CmFigure* CmFigureHandler::GetHighlightedCmFigure()
{ return(highlighted); }

/**************************************************************
 CmFigureHandler GetOldHighlightedCmFigure
 called by CmFigure to make links between figures
 returns the last selected CmFigure
 **************************************************************/
CmFigure* CmFigureHandler::GetOldHighlightedCmFigure()
{ if ( old_highlighted != NULL )
    return(old_highlighted);
  cerr << "There is no previously highlighted CmFigure\n";
  return(NULL);
}

/**************************************************************
 CmFigureHandler SetHighlightedCmFigure
   Called by CmCreateFigure in CmFigureEditor
   Sets the highlighted figure
 **************************************************************/
void CmFigureHandler::SetHighlightedCmFigure(CmFigure* cw) { 
  // do nothing if clicked on same figure
  if (highlighted == cw) 
    return;

  old_highlighted = highlighted;
  
  highlighted = cw;
  if (cw == NULL ) return;
  // get new border width
  border_width = atoi(cw->GetResource("borderWidth"));
  setup(cw->GetWidget());
  XRaiseWindow(display, XtWindow(highlighted->GetWidget()));
  // updated highlighted figure title
  editor->SetStatusBox(highlighted->Name);
}

void CmToggleSnap(Widget w, CmFigureHandler* handler, caddr_t) { 
  if (XmToggleButtonGadgetGetState(w))
    handler->Snap = True;
  else
    handler->Snap = False;
}
/************ routines for snap resolution dialog box ****************/
CmHandlerWad::CmHandlerWad() {}
CmHandlerWad::~CmHandlerWad() {}

void CmPopDownSnap(const Widget, CmHandlerWad* hwad, XtPointer) {
  hwad->dialog->PopdownDialog();
}

void CmUpdateSnap(const Widget, CmHandlerWad* hwad, XtPointer) { 
  hwad->handler->Xscale = hwad->dialog->GetInteger("resolution", "X Snap");
  hwad->handler->Yscale = hwad->dialog->GetInteger("resolution", "Y Snap");
  cmdebug << "updated snaps " << hwad->handler->Xscale
    << ", " << hwad->handler->Yscale << "\n";
  hwad->dialog->PopdownDialog();
}


void CmSnapResolution(Widget, CmFigureHandler* handler, XtPointer) { 
  CmDialog* resdialog = new CmDialog(handler->editor->CDSPtr()->TopLevel(),
			   "Snap Resolution", CMDIALOG_MODELESS);
  resdialog->DisplayPane(False);
  resdialog->AddPane("resolution","", CMDIALOG_ATTACH_NONE);
  resdialog->AddItem(CMDIALOG_TEXTSTRING,"resolution","X Snap");
  resdialog->AddItem(CMDIALOG_TEXTSTRING,"resolution","Y Snap");
  CmString xsnap, ysnap;
  xsnap = (form("%d",handler->Xscale));
  ysnap = (form("%d",handler->Yscale));
  cmdebug << xsnap << ", " << ysnap << "\n";

  resdialog->AddOption("resolution","X Snap",
			    CMDIALOG_VALUE,
			    (XtArgVal)( xsnap.Chars() ));
  resdialog->AddOption("resolution","Y Snap",
			    CMDIALOG_VALUE,
			    (XtArgVal)( ysnap.Chars() ));
  
  resdialog->ManageItems();
  resdialog->PopupDialog();

  CmHandlerWad* hwad = new CmHandlerWad();
  hwad->handler = handler;
  hwad->dialog = resdialog;
  resdialog->SetCallbacks((XtCallbackProc) CmUpdateSnap, 
			  (XtPointer)hwad, 
			  (XtCallbackProc) CmPopDownSnap,
			  (XtPointer)hwad);
}

/**************************************************************
 CmFigureHandler Constructor
  Effects:
   Set up all cursors to be used in resizing and dragging efforts.
   Resets dragging, resizing, highlighted, etc. to appropriate values.
 **************************************************************/
CmFigureHandler::CmFigureHandler(CmFigureEditor* e) {  
  highlighted = NULL;
  old_highlighted = NULL;
  display = NULL;
  workarea_gc = NULL;
  left = False; right = False; top = False; bottom = False;
  dragging = False; resizing = False;
  Snap = True;
  editor = e;
  Xscale = 16; Yscale = 16;
  editor->CDSPtr()->Menu()->AddToggle("Options","toggle_snap",
		  (XtCallbackProc) CmToggleSnap, (caddr_t)this);
  CmToggleSnap(editor->CDSPtr()->Menu()->GetItem("Options", "toggle_snap"),
	       this, NULL);
  editor->CDSPtr()->Menu()->AddItem("Options","snap_resolution",
				    (XtCallbackProc) CmSnapResolution, (caddr_t)this);		       
}

/**************************************************************
 CmFigureHandler Destructor
  Requires: the GC be valid
  Effects: Destroys the valid GC.
 **************************************************************/
CmFigureHandler::~CmFigureHandler() {   
  if (display)
    XFreeGC(display, workarea_gc);
}


