/*************************************************************************
 CmMenu.h - Menu Object Code
 Contact: sniyogi

 NOTE: Needs to be more robust.

 CmMenu: was created to avoid tedious Xt calls and merely encapsulate 
  all redundant information into a C++ object and its definitions.  
  Also, this abstraction can allow us to switch menu methods completely
  if we wish to do so.

   File  Edit    Options <-- these are all CmMenuPanes
  ------ ----    -------
   New   Create  <--- these are all CmMenuItems
 *************************************************************************/
//WARNING:  The XtDestroyWidgets have all been commented out because they
//seem to cause cds to send the error message
//Warning: Attempt to remove non-existant passive grab

#include <CDS.h>
#include <CmMenu.h>
#include <Xm/CascadeBG.h>
#include <Xm/PushBG.h>
#include <Xm/ToggleBG.h>
#include <Xm/RowColumn.h>


/**************************************************************
 CmMenuItem Constructor
  Effects:
   Take name and store it in itemname, a CmMenuItem member.
   Create the widget item, of type PushButton.
 **************************************************************/
CmMenuItem::CmMenuItem(const CmString &name, Widget pane, WidgetClass type) {
  itemname = name;
  
  item = XtCreateManagedWidget(itemname.Chars(), type, pane, NULL, 0);
}

/**************************************************************
 CmMenuItem Destructor
  Effects:
   Do-nothing destructor - implicitly calls itemname's
 **************************************************************/
CmMenuItem::~CmMenuItem() {
}

/**************************************************************
 CmMenuItem: ItemnameToWidget
  Effects:
   If the key given is the string item, returns the widget, otherwise nothing.
   i.e.  ItemnameToWidget("Save As..") will return the widget inside CmMenuItem
   if and only if this is the "Save As..." item.
 **************************************************************/
Widget CmMenuItem::ItemnameToWidget(const CmString &key) {
   if (key == itemname)
     return(item);
   else return(NULL);
}

/**************************************************************
 CmMenuPane Constructor: 
  Effects:
   panename = title  (i.e. "File") to be referred to later on.
   panewidgetname = title + "pane"  ---> Creates pane widget (PulldownMenu)
   cascadewidgetname = title + "cascade"  ---> Creates cascade widget (CascadeButton)
 **************************************************************/
CmMenuPane::CmMenuPane(const CmString &title, Widget bar) { 
  int count = 0;
  Arg arglist[CDS_ARGLIST_SIZE];
  
  items = 0;
  panename = title;
  
  // Create pane widget, of type PulldownMenu
  panewidgetname = title + "pane";
  pane = XmCreatePulldownMenu(bar, (char *) panewidgetname.Chars(), 
			      NULL, 0);

// Create cascade widget, of type CascadeButton
   cascadewidgetname = title;
   count = 0;
   XtSetArg(arglist[count], XmNsubMenuId, pane);  count++;
   cascade = XmCreateCascadeButtonGadget(bar, (char *) cascadewidgetname.Chars(), 
					 arglist, count);
   XtManageChild(cascade);

// Items represents the number of CmMenuItems this CmPane contains.
   items = 0;
}

/**************************************************************
 CmMenuPane Destructor
  Effects:
   Destroys strings created in constructor.
   Destroys CmMenuItems created by AddItem.
 **************************************************************/
CmMenuPane::~CmMenuPane() {  
  int i;
  for (i=0; i<items; i++)
    delete item[i];
//  XtDestroyWidget(cascade);
}

/**************************************************************
 CmMenuPane: AddItem
  Effects: Add iname to our pane  ( i.e. AddItem("Save As...") when our CmMenuPane represent "File" )
 **************************************************************/
void CmMenuPane::AddItem(const CmString &name) {
  AddItem(name, NULL);
}

void CmMenuPane::AddItem(const CmString &iname, 
			 WidgetClass type) {
  if (!type)
    type = xmPushButtonGadgetClass;
  item[items++] = new CmMenuItem(iname, pane, type);
}

/**************************************************************
 CmMenuPane: PanenameToCmMenuPane
  Effects: If string given matches the id of the CmMenuPane, return self.
 **************************************************************/
CmMenuPane* CmMenuPane::PanenameToCmMenuPane(const CmString &pname) {
  if (pname == panename) 
    return(this);
  else return(NULL);
}

/**************************************************************
 CmMenuItem: Pane
 returns the pane widget
 **************************************************************/
Widget CmMenuPane::Pane()
{ return(pane); }


/**************************************************************
 CmMenuPane: ItemnameToWidget
  Effects:
   If iname is in our pane, return the widget associated with it.
   i.e.  ItemnameToWidget("Save As...") would return the PushButton
     which contains "Save As..."
   If iname is not in the pane, NULL is returned.
 **************************************************************/
Widget CmMenuPane::ItemnameToWidget(const CmString &iname)
{  Widget w = NULL;
   for (int i=0; i<items; i++) {
     w = item[i]->ItemnameToWidget(iname);
     if ( w ) return(w);
   }
   return(w);
}

/**************************************************************
 CmMenu Constructor
  Effects: Adds a menubar to the window given.
 **************************************************************/
CmMenu::CmMenu(Widget window) {  
  panes = 0;
  menubar = XmCreateMenuBar(window, "menubar", NULL, 0);
  XtManageChild(menubar);
}

/**************************************************************
 CmMenu Destructor
  Effects: Deletes the panes added by AddPane.
 **************************************************************/
CmMenu::~CmMenu() { 
  for (int i=0; i<panes; i++)
    delete pane[i];
}

/**************************************************************
 CmMenu: AddPane
  Effects: Add the pane to our menu. (i.e. AddPane("File"))
 **************************************************************/
void CmMenu::AddPane(const CmString &title)
{
  if (!(FindPane(title))) {
    pane[panes] = new CmMenuPane(title, menubar);
    panes++;
  }
}

/**************************************************************
 CmMenu: FindPane
  Effects: Given a pane id, returns the pane associated with the id
 **************************************************************/
CmMenuPane* CmMenu::FindPane(const CmString &key)
{  CmMenuPane* p;
   for (int i=0; i<panes; i++) {
     p = pane[i]->PanenameToCmMenuPane(key);
     if ( p ) return(p);
   }
   return(NULL);
}

/**************************************************************
 CmMenu: AddItem
  Effects:
   Given a pane name pname and an item name item, adds
   the item to the menu.  i.e. AddItem("File", "Save As...")
 **************************************************************/
void CmMenu::AddItem(const CmString &pname, const CmString &iname)
{  CmMenuPane* p;
   if (p = FindPane(pname))
     if (!(p->ItemnameToWidget(iname)))
       p->AddItem(iname);
}

void CmMenu::AddToggle(const CmString &pname, const CmString &iname)
{  CmMenuPane* p;
   if (p = FindPane(pname))
     if (!(p->ItemnameToWidget(iname)))
       p->AddItem(iname, xmToggleButtonGadgetClass);
}

/**************************************************************
  CmMenu: AddItem
   Effects:
    Given a pane name pname and an item name item, adds
    the item to the menu.  i.e. AddItem("File", "Save As...")
    Takes the callback and the data and automatically add the callback
    to the item itself.  So if "Save As..." is selected, func is called.
 **************************************************************/
void CmMenu::AddItem(const CmString &pname, const CmString &iname, 
		     XtCallbackProc func, caddr_t closure)
{  CmMenuPane* p;
   if (p = FindPane(pname)) {
     if (!(p->ItemnameToWidget(iname))) {
       p->AddItem(iname);
       if ( func ) 
	 XtAddCallback(p->ItemnameToWidget(iname), XmNactivateCallback, 
		       func, closure); 
     }
   }
 }

void CmMenu::AddToggle(const CmString &pname, const CmString &iname, 
		     XtCallbackProc func, caddr_t closure)
{  CmMenuPane* p;
   if (p = FindPane(pname)) {
     if (!(p->ItemnameToWidget(iname))) {
       p->AddItem(iname, xmToggleButtonGadgetClass);
       if ( func ) 
	 XtAddCallback(p->ItemnameToWidget(iname), XmNvalueChangedCallback, 
		       func, closure); 
     }
   }
 }

/**************************************************************
 CmMenu: GetItem
  Effects:
   Returns the widget which is under pane pname and has the text iname
   i.e. GetItem("File", "Save As...") would return the actual PushButton widget
 **************************************************************/
Widget CmMenu::GetItem(const CmString &pname, const CmString &iname) {
  CmMenuPane* p = FindPane(pname);
  if (!p)
    return (NULL);
  else
    return(p->ItemnameToWidget(iname));
}

/**************************************************************
 CmMenu: GetPane
   Returns the widget which is under pane pname
   i.e. GetPane("File") would return the actual PushButton widget
 **************************************************************/
Widget CmMenu::GetPane(const CmString &pname)
{  CmMenuPane* p;
   p = FindPane(pname);
   return(p->Pane());
}

Boolean CmMenu::GetToggleState(const CmString &pname, const CmString &iname) {
  Widget widget = GetItem(pname, iname);
  if (widget && XtIsSubclass(widget, xmToggleButtonGadgetClass))
    return XmToggleButtonGadgetGetState(widget);
  else
    return False;
}


void CmMenu::SetToggleState(const CmString &pname, 
			    const CmString &iname, Boolean state) {
  Widget widget = GetItem(pname, iname);
  if (widget && XtIsSubclass(widget, xmToggleButtonGadgetClass))
    XmToggleButtonGadgetSetState(widget, state, True);
}


void CmMenu::SetSensitive(const CmString &pname, 
			  const CmString &iname, Boolean state) {
  Widget widget = GetItem(pname, iname);
  if (widget)
    XtSetSensitive(widget, state);
}
