/* $Date: 91/12/11 16:35:42 $    $Revision: 1.1 $  by $Author: joe $  */
/* 
  Copyright (C) 1991 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */
/*********************************************************
 CmNavigator.cc - Navigator Code
 Contact: joe, maschaub
 *********************************************************/
#include <CmNavigator.h>
#include <CmNavigatorHeap.h>
#include <CmNavigatorOutliner.h>
#include <CmNavigatorParser.h>
#include <CmDialog.h>
#include <CmMenu.h>
#include <CmCDS.h>
#include <CmFigureEditor.h>
#include <CmPage.h>
#include <CmStringMotif.h>
#include <Xm/PushBG.h>

#include <sys/file.h>
#include <ctype.h>
#include <sys/stat.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/FileSB.h>
#include <Xm/List.h>
#include <Xm/ScrolledW.h>


struct NavDialog { /** Need data type to send info to ok callback **/
  CmNavigator* nav;
  CmDialog* dialog;
  CmString string;
  NavDialog (CmNavigator* n, CmDialog *d, const CmString &s) {
    nav = n; dialog = d; string = s;
  };
  NavDialog (CmNavigator* n, CmDialog *d) {
    nav = n; dialog = d;
  };
  ~NavDialog() {
    delete dialog;
  };
};


static void CmCancel(Widget, NavDialog* const nd, XtPointer) {
  delete nd;
}

static void PopupDialog(CmNavigator *navigator,
			CmDialogXmDialogType type,
			const CmString & output_string,
			XtCallbackProc ok_callback,
			const CmString &callback_string) {
  CmDialog* box = 
    new CmDialog(navigator->GetDialogParent(), "dialog_box", CMDIALOG_MODAL);
  box->XmDialogCreate(type, output_string);

  Widget widget = box->XmDialogGetWidget();
  NavDialog* nd = new NavDialog(navigator, box, callback_string);

  XtAddCallback(widget, XmNcancelCallback, 
		(XtCallbackProc)CmCancel, (XtPointer) nd);
  XtAddCallback(widget, XmNokCallback, 
		(XtCallbackProc) ok_callback, (XtPointer) nd);
  box->XmDialogManage();
}

static CmBool is_directory(const CmString &);

static void CmAnyPageCallback(Widget, 
			      CmNavigator* const nav,
			      XmListCallbackStruct* const call_data) {
  nav->Navigate("any", call_data->item_position);  
}


/*** PREV BUTTON ***/
static void CmPrevPageCallback(Widget, CmNavigator* const nav, 
			       XmListCallbackStruct*) {
  nav->Navigate("prev");
}


/*** NEXT BUTTON ***/
static void CmNextPageCallback(Widget, CmNavigator* const nav, 
			       XmListCallbackStruct*) {
  nav->Navigate("next");
}


// File Handling Callbacks (called underneath "File" in menu)

/*** NEW ***/
static void CmOKNewFile(Widget, NavDialog* const nd, XtPointer) { 
  nd->nav->NewFile(nd->string); 
  delete nd;
}

static void CmNewFile(Widget, CmNavigator* const navigator, XtPointer)  { 
  PopupDialog(navigator, CMDIALOG_WARNING, 
	      "If you OK a new file, the changes you have made will be lost forever.",
	      (XtCallbackProc) CmOKNewFile, "untitled");

}

static void CmToggleTitles(Widget, CmNavigator* navigator, 
			XmToggleButtonCallbackStruct *) {
  navigator->ToggleNameAndTitles();
}

/*** OPEN ***/

static void CmCancelOpen(const Widget w, XtPointer, XtPointer) { 
  XtUnmanageChild(w);
}

static void CmOpenFileCallback(const Widget w, 
			       CmNavigator* const navigator,
			       XmFileSelectionBoxCallbackStruct *
			       callback_struct) {
  XtUnmanageChild(w);
  navigator->OpenFile(XmToCmString(callback_struct->value));
}

static void CmOpenFile(Widget, Widget file_box, XtPointer) {  
  XtManageChild(file_box);
}

/*** save_as***/

/****************************************************************************
CmCancel callback - gives user option to cancel the save if it would
     overwrite another file. Only called from the save_asdialog box.
****************************************************************************/

static void CmOKSave(Widget, NavDialog* const nd, XtPointer) {
  nd->nav->SaveFile(nd->string);
  delete nd;
}  

/****************************************************************************
CmSaveAsFileOK callback - called from the save_as dialog box. This routine
  generates another dialog warning box only if the filename already exists
  in memory. This second dialog's callback is CmCancel.
****************************************************************************/
static void CmSaveAsFileOK(const Widget w, 
			   NavDialog* const nd, 
			   const XtPointer call_data) {
  const CmString fname(nd->dialog->GetString("save_as", "filename"));

  if (!access(fname.Chars(),W_OK) && !is_directory(fname)) {
    CmString msg(fname + "\"?");
    msg.Prepend("Overwrite current version of \"");

    PopupDialog(nd->nav, CMDIALOG_WARNING, msg, 
		(XtCallbackProc) CmOKSave, fname);
    delete nd;
  }
  else {
    nd->string = fname;
    CmOKSave(w,nd,call_data);
  }
}

/****************************************************************************
CmSaveAsFile - calls up a dialog box to get the filename to save in. The
  dialog box's callback is CmSaveAsOK, for the OK button.
****************************************************************************/
static void CmSaveAsFile(Widget, CmNavigator* const navigator, XtPointer) {
  
  char dir[CDS_STRING_SIZE];
  CmString dirname(getwd(dir)); // Gets directory name for dialog box.
  dirname += "/"; // Adds on a slash to the end of the directory name.
  
  CmDialog* const box = new CmDialog(navigator->GetDialogParent(),
				     "save_as", CMDIALOG_MODAL);
  
  box->DisplayPane(False);
  box->AddPane("save_as", "", CMDIALOG_ATTACH_NONE);
  box->SetHelp("Enter the filename in which to save your changes.\n");
  box->AddItem(CMDIALOG_TEXTSTRING, "save_as", "filename");
  box->AddOption("save_as","filename",
		 CMDIALOG_VALUE,(XtArgVal)dirname.Chars());
  box->AddOption("save_as","filename",CMDIALOG_COLUMNS,
		 (XtArgVal)40);
  box->ManageItems();
  box->PopupDialog();

  NavDialog* const nd = new NavDialog(navigator, box);
  nd->dialog->SetCallbacks((XtCallbackProc)CmSaveAsFileOK, (XtPointer) nd,
			   (XtCallbackProc)CmCancel, (XtPointer) nd);
}

/*** SAVE ***/
/****************************************************************************
CmSaveFile - saves a file, as normal, unless the current filename is the
 default "untitled", which means a name has not yet been selected. In this 
 case, call the save_as callback instead.
****************************************************************************/
static void CmSaveFile(Widget, CmNavigator* const navigator, XtPointer) {
  if (navigator->CurrentFilename == "untitled")
    CmSaveAsFile((Widget)NULL, navigator, (XtPointer) NULL);
  else
    navigator->SaveFile(navigator->CurrentFilename);
}

/*** EXIT ***/
/****************************************************************************
CmExit - If there are unsaved changes, asks the user if he/she wants to save
 them. Always double checks that the user really wants to exit.
****************************************************************************/

static void CmExitOK(Widget, NavDialog* const nd, XtPointer) {
  nd->nav->ExitCDS();
}

static void CmExit(Widget, CmNavigator* const navigator, XtPointer) {
  PopupDialog(navigator, CMDIALOG_QUESTION,
	      "Are you sure you want to exit?",
	      (XtCallbackProc)CmExitOK, "");
}

// Miscellaneous Callbacks (called underneath "Options" in menu)

/*** APPEND PAGE ***/
static void CmAppendPageOK(Widget, NavDialog* const nd, XtPointer) {
  CmString where = nd->dialog->GetString("Where", "Where");
  CmString command = nd->dialog->GetString("Append Page", "Copy");
  CmString named_page = nd->dialog->GetString("list", "Page name:");

  if (command == "Move named page")
    nd->nav->AppendNewPage(named_page, where, False); 
  else
    if (nd->dialog->GetString("Append Page","Copy") == 
	"Copy the current page.")
      nd->nav->AppendNewPage(nd->nav->GetCurrentPage()->Name,
			     where, True);
  else
    nd->nav->AppendNewPage("", where, False);
}

static void CmAppendPage(Widget, CmNavigator* const navigator, XtPointer) {
  CmDialog* const box = new CmDialog(navigator->GetDialogParent(),
				     "Append Page...",CMDIALOG_MODAL);
  box->DisplayPane(False);
  box->AddPane("Append Page", "", CMDIALOG_ATTACH_NONE);
  box->SetHelp("Check the top button if you want the new page\n");
  box->SetHelp("to look like the current page. Check the bottom\n");
  box->SetHelp("button if you want a new, blank page.");
  box->AddItem(CMDIALOG_RADIOBOX,"Append Page","Copy");
  box->AddOption("Append Page","Copy",CMDIALOG_RADIOBUTTON,
		 (XtArgVal)"Copy the current page.");
  box->AddOption("Append Page","Copy",CMDIALOG_RADIOBUTTON,
		 (XtArgVal)"Make a new blank page.");
  box->AddOption("Append Page","Copy",CMDIALOG_RADIOBUTTON,
		 (XtArgVal)"Move named page");
  box->AddOption("Append Page","Copy",CMDIALOG_SELECTED,
		 (XtArgVal)"Make a new blank page.");
  box->AddPane("list", "Append Page", CMDIALOG_ATTACH_TOP);
  box->AddItem(CMDIALOG_LIST, "list", "Page name:");
  box->AddOption("list", "Page name:", CMDIALOG_NUMVISIBLE,
		 (XtArgVal) 5);
  navigator->SetListOfHeap(box, "list", "Page name:");
  box->AddPane("Where", "Append Page", CMDIALOG_ATTACH_LEFT);
  box->AddItem(CMDIALOG_RADIOBOX, "Where", "Where");
  box->AddOption("Where", "Where", CMDIALOG_RADIOBUTTON,
		 (XtArgVal) "first");
  box->AddOption("Where", "Where", CMDIALOG_RADIOBUTTON,
		 (XtArgVal) "before");
  box->AddOption("Where", "Where", CMDIALOG_RADIOBUTTON,
		 (XtArgVal) "after");
  box->AddOption("Where", "Where", CMDIALOG_RADIOBUTTON,
		 (XtArgVal) "last");
  box->AddOption("Where", "Where", CMDIALOG_SELECTED,
		 (XtArgVal) "last");


  box->ManageItems(); box->PopupDialog();
  NavDialog* const nd = new NavDialog(navigator, box);

  box->SetCallbacks((XtCallbackProc)CmAppendPageOK, (XtPointer) nd,
		    (XtCallbackProc)CmCancel, (XtPointer) nd);
}


static void CmRemovePage(Widget, CmNavigator* const navigator, XtPointer) { 
  navigator->RemovePage(navigator->GetCurrentPage()); 
}

static void CmDestroyPageOK (Widget, NavDialog *nd, XtPointer){
  nd->nav->DestroyPage(nd->nav->GetCurrentPage());
  delete nd;
}

static void CmDestroyPage(Widget, CmNavigator* const navigator, XtPointer) {
  PopupDialog(navigator, CMDIALOG_WARNING, 
	      "Are you sure you want to destroy this page?",
	      (XtCallbackProc) CmDestroyPageOK, "");
}

/*** CHANGE TITLE ***/
static void CmChangeNameAndTitleOK(Widget, NavDialog* const nd, XtPointer) {
  nd->nav->ChangeNameAndTitle(nd->dialog->GetString("Title", "name"),
		       nd->dialog->GetString("Title", "title"));
  delete nd;
}

static void CmChangeNameAndTitle(Widget, CmNavigator* const navigator, XtPointer) {
  CmDialog* const box = 
    new CmDialog(navigator->GetDialogParent(),
		 "Title", CMDIALOG_MODAL);
  box->DisplayPane(False);
  box->AddPane("Title", "", CMDIALOG_ATTACH_NONE);
  box->SetHelp("The new title that you set will appear in the titlebar.\n");
  box->AddItem(CMDIALOG_TEXTSTRING, "Title", "name");
  box->AddItem(CMDIALOG_TEXTSTRING, "Title", "title");
  box->AddOption("Title", "name", CMDIALOG_VALUE, 
		 (XtArgVal) navigator->GetCurrentPage()->Name.Chars());
  box->AddOption("Title", "title", CMDIALOG_VALUE, 
		 (XtArgVal) navigator->GetCurrentPage()->Title.Chars());

  box->ManageItems();
  box->PopupDialog();
  NavDialog* const nd = new NavDialog(navigator, box);
  box->SetCallbacks((XtCallbackProc)CmChangeNameAndTitleOK, (XtPointer) nd,
		    (XtCallbackProc)CmCancel, (XtPointer) nd);
}

/*************************************
 CmNavigator constructor
  Adds New, Open,Save, Save As, Exit, Append Page and Remove Page to 
  menu with callbacks to procedures here
**************************************/
CmNavigator::CmNavigator(CmCDS* const t, const Widget outline) {
  cds = t;

  // create widgets
  prev = XmCreatePushButtonGadget(outline, "prev", NULL, 0);    
  next = XmCreatePushButtonGadget(outline, "next", NULL,0);
  scroll = XmCreateScrolledWindow(outline, "scroll", NULL, 0);
  any = XmCreateList(scroll, "any", NULL, 0);

  // create submodules of navigator
  outliner = new CmNavigatorOutliner(this, any);
  heap = new CmNavigatorHeap();
  parser = new CmNavigatorParser(cds, outliner, heap);
 
  // add callbacks
  XtAddCallback(any, XmNsingleSelectionCallback,
                (XtCallbackProc) CmAnyPageCallback, (XtPointer) this);
  XtAddCallback(prev, XmNactivateCallback,
                (XtCallbackProc) CmPrevPageCallback, (XtPointer) this);
  XtAddCallback(next, XmNactivateCallback,
                (XtCallbackProc) CmNextPageCallback, (XtPointer) this);
  XtManageChild(prev);
  XtManageChild(next);
  XtManageChild(scroll);
  XtManageChild(any);


  file_box = 
    XmCreateFileSelectionDialog(GetDialogParent(), "Filebox", NULL, 0);
  XtAddCallback(file_box, XmNokCallback, 
		(XtCallbackProc) CmOpenFileCallback, (XtPointer) this);
  XtAddCallback(file_box, XmNcancelCallback, 
		(XtCallbackProc) CmCancelOpen, (XtPointer) this);

  // add menu items
  cds->Menu()->AddItem("File", "new", 
		       (XtCallbackProc) CmNewFile, (XtPointer) this);
  cds->Menu()->AddItem("File", "open", 
		       (XtCallbackProc) CmOpenFile, (XtPointer)  file_box);
  cds->Menu()->AddItem("File", "save", 
		       (XtCallbackProc) CmSaveFile, (XtPointer) this);
  cds->Menu()->AddItem("File", "save_as", 
		       (XtCallbackProc) CmSaveAsFile, (XtPointer) this);
  cds->Menu()->AddItem("File", "exit", 
		       (XtCallbackProc) CmExit, (XtPointer) this);
  cds->Menu()->AddItem("Page", "append_page", 
		       (XtCallbackProc) CmAppendPage, (XtPointer)this);
  cds->Menu()->AddItem("Page", "remove_page", 
		       (XtCallbackProc) CmRemovePage, (XtPointer)this);
  cds->Menu()->AddItem("Page", "destroy_page", 
		       (XtCallbackProc) CmDestroyPage, (XtPointer)this);     
  cds->Menu()->AddItem("Page", "change_title",
		       (XtCallbackProc) CmChangeNameAndTitle, (XtPointer)this);
  cds->Menu()->AddToggle("Page", "toggle_titles", 
			 (XtCallbackProc) CmToggleTitles, (XtPointer) this);
  ToggleNameAndTitles();  // Sets the title and name scheme
                          // to match what is stated in menu
  NewFile("untitled");
}

/*************************************
 CmNavigator destructor
  deletes the parser
**************************************/
CmNavigator::~CmNavigator() {
  XtDestroyWidget(prev);
  XtDestroyWidget(next);
  XtDestroyWidget(file_box);
  XtDestroyWidget(scroll);

  delete parser;
  delete outliner;
  delete heap;
}

void CmNavigator::NewFile(const CmString &filename) {
  CurrentFilename = filename;

  cds->Editor()->Page = NULL;
  (void) outliner->GetPageName("clear");
  heap->ClearAllPages();

  // create first_page
  // Do NOT copy else you will get seg fault as no pages exist
  AppendNewPage("", "last", False);
}


/*************************************
 CmNavigator OpenFile
  Called when Open button is pressed
  Opens a file, loads it in and sets the editing page to the file page
**************************************/
void CmNavigator::OpenFile(const CmString &fname) {
  if (access(fname.Chars(), R_OK) || is_directory(fname)) {
    const CmString msg = fname + ": file not found.\nDo you wish to create?";
    PopupDialog(this, CMDIALOG_QUESTION, msg, (XtCallbackProc) CmOKNewFile, fname);
  } else {
    cds->Editor()->Page = NULL;
    (void) outliner->GetPageName("clear");
    heap->ClearAllPages();
    parser->ParseFile(fname);
    Navigate("update");
    Navigate("first");
    CurrentFilename = fname;
  }
}

/*************************************
 CmNavigator SaveFile
   Starts at root page, successively saves every page to filename given.
   If saves successfully, sets the current Nav filename its argument.
**************************************/
void CmNavigator::SaveFile(const CmString &fname) {
  filebuf f1;
  CurrentFilename = fname;

  if (!f1.open(fname.Chars(), output)) {
    CmString msg(fname + "\".");
    msg.Prepend("Cannot write to filename \"");

    PopupDialog(this, CMDIALOG_ERROR, msg, (XtCallbackProc) CmCancel, "");
  } else {
    // Change cursor
    cds->ShowWaitCursor();

    // Save the file
    ostream to(&f1);
    to << outliner << heap;
    f1.close();

    // Change the cursor back
    
    CmString filemode = "chmod 664 ";
    filemode += fname;
    system(filemode.Chars()); // change protections
    cds->ShowNormCursor();
  }
}

void CmNavigator::AppendNewPage(const CmString &page_name,
				const CmString &where,
				Boolean copy_current) {
  CmString new_page_name;
  if (page_name.Empty()) {
    new_page_name = heap->MakePageName();
    heap->AppendPageToHeap(new CmPage(new_page_name));
  } else if (copy_current) {
    new_page_name = heap->MakePageName();
    heap->AppendPageToHeap(new CmPage(new_page_name, GetCurrentPage()));
  }
   else
     new_page_name = page_name;

// The outliner call should be made after the page is added to the heap so
// that the outliner can find the title of the page when it updates its
// list

  outliner->AddItemToOutline(new_page_name, where);  
  Navigate("goto", new_page_name);
}

void CmNavigator::RemovePage(CmPage *const page) {
  Navigate("goto", outliner->GetPageName("remove", page->Name));
}

void CmNavigator::DestroyPage(CmPage *const page) {
  CmString page_to_move = outliner->GetPageName("remove", page->Name); 
  if (!page_to_move.Empty()) {
    heap->DestroyPageInHeap(page);
    Navigate("goto", page_to_move);
  } else 
    PopupDialog(this, CMDIALOG_WARNING,
		"Unable to destroy page.",
		(XtCallbackProc) CmCancel, "");
}

static CmBool is_directory(const CmString &fname) {
  /*** system call stuff to check if for a directory ***/ 
  struct stat buf; 
  stat(fname.Chars(), &buf);
  return((buf.st_mode & S_IFMT) == S_IFDIR);
}

void CmNavigator::ExitCDS() {
  cds->StopEventLoop();
}

void CmNavigator::ChangeNameAndTitle(const CmString &new_name,
				     const CmString &new_title) {
  // Check if name is valid
  if (!new_name.Matches(CmRegex("[A-za-z_][A-za-z0-9_.-]*"))) {
    PopupDialog(this, CMDIALOG_WARNING, 
		"No changes made\nName contains invalid characters",
		(XtCallbackProc) CmCancel, "");
    return;
  }

  // Check if other node has the requested name
  if (new_name != GetCurrentPage()->Name &&  
      heap->GetPageByName(new_name))
    PopupDialog(this, CMDIALOG_WARNING,
		"No changes made\nName is not unique",
		(XtCallbackProc) CmCancel, "");

  GetCurrentPage()->Name = new_name;
  GetCurrentPage()->Title = new_title;
  outliner->GetPageName("change_to", new_name);
  outliner->GetPageName("update");
}

void CmNavigator::ChangeTitle(const CmString &new_title) {
  GetCurrentPage()->Title = new_title;
  outliner->GetPageName("update");
}
  
Widget CmNavigator::GetDialogParent() {
  return cds->TopLevel();
}

CmPage *CmNavigator::GetCurrentPage() {
  return cds->Editor()->Page;
}

CmPage *CmNavigator::GetPageByName(const CmString &name) {
  return heap->GetPageByName(name);
}

void CmNavigator::ToggleNameAndTitles() {
  outliner->ShowTitles(cds->Menu()->GetToggleState("Page", "toggle_titles"));
}

void CmNavigator::Navigate(const CmString &command, 
			   const CmString &page) {
  CmString page_name = outliner->GetPageName(command, page);
  if (!page_name.Empty()) {
    CmPage *page = heap->GetPageByName(page_name);
    if (!page) {
      page = new CmPage(page_name);
      heap->AppendPageToHeap(page);
    }
  cds->Editor()->ChangePage(page);
  }
}

void CmNavigator::Navigate(const CmString &command, const int position) {
  CmString page_name = outliner->GetPageName(command, position);
  if (!page_name.Empty()) {
    CmPage *page = heap->GetPageByName(page_name);
    if (!page) {
      page = new CmPage(page_name);
      heap->AppendPageToHeap(page);
    }
  cds->Editor()->ChangePage(page);
  }
}

void CmNavigator::SetListOfHeap(CmDialog* box, const CmString &pane,
				const CmString &item) {
  heap->SetListOfHeap(box, pane, item);
}
