// This may look like C code, but it is really -*- C++ -*-
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#if defined(SHORT_NAMES) || defined(VMS)
#define re_compile_pattern	recmppat
#define re_pattern_buffer	repatbuf
#define re_registers		reregs
#endif

#ifndef _String_h
#ifdef __GNUG__
//#pragma once
//#pragma interface
#endif
#define _String_h 1

#include <stream.h>
#include <CmRegex.h>

struct CmStrRep                     // internal CmString representations
{
  unsigned short    len;         // CmString length 
  unsigned short    sz;          // allocated space
  char              s[1];        // the CmString starts here 
                                 // (at least 1 char for trailing null)
                                 // allocated & expanded via non-public fcts
};

// primitive ops on StrReps -- nearly all CmString fns go through these.

CmStrRep*     Salloc(CmStrRep*, const char*, int, int);
CmStrRep*     Scopy(CmStrRep*, CmStrRep*);
CmStrRep*     Sresize(CmStrRep*, int);
CmStrRep*     Scat(CmStrRep*, const char*, int, const char*, int);
CmStrRep*     Scat(CmStrRep*, const char*, int,const char*,int, const char*,int);
CmStrRep*     Sprepend(CmStrRep*, const char*, int);
CmStrRep*     Sreverse(CmStrRep*, CmStrRep*);
CmStrRep*     Supcase(CmStrRep*, CmStrRep*);
CmStrRep*     Sdowncase(CmStrRep*, CmStrRep*);
CmStrRep*     Scapitalize(CmStrRep*, CmStrRep*);

// These classes need to be defined in the order given

class CmString;
class CmSubString;

class CmSubString
{
  friend class      CmString;
protected:

  CmString&           S;        // The CmString I'm a CmSubString of
  unsigned short    pos;      // starting position in S's rep
  unsigned short    len;      // length of CmSubString

  void              assign(CmStrRep*, const char*, int = -1);
                    CmSubString(CmString& x, int p, int l);
                    CmSubString(const CmSubString& x);

public:

// Note there are no public constructors. SubStrings are always
// created via CmString operations

                   ~CmSubString();

  void              operator =  (const CmString&     y);
  void              operator =  (const CmSubString&  y);
  void              operator =  (const char* t);
  void              operator =  (char        c);

// return 1 if target appears anywhere in CmSubString; else 0

  int               Contains(char        c) const;
  int               Contains(const CmString&     y) const;
  int               Contains(const CmSubString&  y) const;
  int               Contains(const char* t) const;
  int               Contains(const CmRegex&       r) const;

// return 1 if target matches entire CmSubString

  int               Matches(const CmRegex&  r) const;

// IO 

  friend ostream&   operator<<(ostream& s, const CmSubString& x);

// status

  int               Length() const;
  int               Empty() const;
  const char*       Chars() const;

  int               OK() const; 

};


class CmString
{
  friend class      CmSubString;

protected:
  CmStrRep*           rep;   // Strings are pointers to their representations

// some helper functions

  int               Search(int, int, const char*, int = -1) const;
  int               Search(int, int, char) const;
  int               Match(int, int, int, const char*, int = -1) const;
  int               _gsub(const char*, int, const char* ,int);
  int               _gsub(const CmRegex&, const char*, int);
  CmSubString         _substr(int, int);

public:

// constructors & assignment

                    CmString();
                    CmString(const CmString& x);
                    CmString(const CmSubString&  x);
                    CmString(const char* t);
                    CmString(const char* t, int len);
                    CmString(char c);

                    ~CmString();

  void              operator =  (const CmString&     y);
  void              operator =  (const char* y);
  void              operator =  (char        c);
  void              operator =  (const CmSubString&  y);

// concatenation

  void              operator += (const CmString&     y); 
  void              operator += (const CmSubString&  y);
  void              operator += (const char* t);
  void              operator += (char        c);

  void              Prepend(const CmString&     y); 
  void              Prepend(const CmSubString&  y);
  void              Prepend(const char* t);
  void              Prepend(char        c);


// procedural versions:
// concatenate first 2 args, store result in last arg

  friend void     Cat(const CmString&, const CmString&, CmString&);
  friend void     Cat(const CmString&, const CmSubString&, CmString&);
  friend void     Cat(const CmString&, const char*, CmString&);
  friend void     Cat(const CmString&, char, CmString&);

  friend void     Cat(const CmSubString&, const CmString&, CmString&);
  friend void     Cat(const CmSubString&, const CmSubString&, CmString&);
  friend void     Cat(const CmSubString&, const char*, CmString&);
  friend void     Cat(const CmSubString&, char, CmString&);

  friend void     Cat(const char*, const CmString&, CmString&);
  friend void     Cat(const char*, const CmSubString&, CmString&);
  friend void     Cat(const char*, const char*, CmString&);
  friend void     Cat(const char*, char, CmString&);

// double concatenation, by request. (yes, there are too many versions, 
// but if one is supported, then the others should be too...)
// Concatenate first 3 args, store in last arg

  friend void     Cat(const CmString&,const CmString&, const CmString&,CmString&);
  friend void     Cat(const CmString&,const CmString&,const CmSubString&,CmString&);
  friend void     Cat(const CmString&,const CmString&, const char*, CmString&);
  friend void     Cat(const CmString&,const CmString&, char, CmString&);
  friend void     Cat(const CmString&,const CmSubString&,const CmString&,CmString&);
  friend void     Cat(const CmString&,const CmSubString&,const CmSubString&,CmString&);
  friend void     Cat(const CmString&,const CmSubString&, const char*, CmString&);
  friend void     Cat(const CmString&,const CmSubString&, char, CmString&);
  friend void     Cat(const CmString&,const char*, const CmString&,    CmString&);
  friend void     Cat(const CmString&,const char*, const CmSubString&, CmString&);
  friend void     Cat(const CmString&,const char*, const char*, CmString&);
  friend void     Cat(const CmString&,const char*, char, CmString&);

  friend void     Cat(const char*, const CmString&, const CmString&,CmString&);
  friend void     Cat(const char*,const CmString&,const CmSubString&,CmString&);
  friend void     Cat(const char*,const CmString&, const char*, CmString&);
  friend void     Cat(const char*,const CmString&, char, CmString&);
  friend void     Cat(const char*,const CmSubString&,const CmString&,CmString&);
  friend void     Cat(const char*,const CmSubString&,const CmSubString&,CmString&);
  friend void     Cat(const char*,const CmSubString&, const char*, CmString&);
  friend void     Cat(const char*,const CmSubString&, char, CmString&);
  friend void     Cat(const char*,const char*, const CmString&,    CmString&);
  friend void     Cat(const char*,const char*, const CmSubString&, CmString&);
  friend void     Cat(const char*,const char*, const char*, CmString&);
  friend void     Cat(const char*,const char*, char, CmString&);


// searching & matching

// return position of target in CmString or -1 for failure

  int               Index(char        c, int startpos = 0) const;      
  int               Index(const CmString&     y, int startpos = 0) const;      
  int               Index(const CmSubString&  y, int startpos = 0) const;      
  int               Index(const char* t, int startpos = 0) const;  
  int               Index(const CmRegex&      r, int startpos = 0) const;       

// return 1 if target appears anyhere in CmString; else 0

  int               Contains(char        c) const;
  int               Contains(const CmString&     y) const;
  int               Contains(const CmSubString&  y) const;
  int               Contains(const char* t) const;
  int               Contains(const CmRegex&      r) const;

// return 1 if target appears anywhere after position pos 
// (or before, if pos is negative) in CmString; else 0

  int               Contains(char        c, int pos) const;
  int               Contains(const CmString&     y, int pos) const;
  int               Contains(const CmSubString&  y, int pos) const;
  int               Contains(const char* t, int pos) const;
  int               Contains(const CmRegex&      r, int pos) const;

// return 1 if target appears at position pos in CmString; else 0

  int               Matches(char        c, int pos = 0) const;
  int               Matches(const CmString&     y, int pos = 0) const;
  int               Matches(const CmSubString&  y, int pos = 0) const;
  int               Matches(const char* t, int pos = 0) const;
  int               Matches(const CmRegex&      r, int pos = 0) const;

//  return number of occurences of target in CmString

  int               Freq(char        c) const; 
  int               Freq(const CmString&     y) const;
  int               Freq(const CmSubString&  y) const;
  int               Freq(const char* t) const;

// CmSubString extraction

// Note that you can't take a CmSubString of a const CmString, since
// this leaves open the possiblility of indirectly modifying the
// CmString through the CmSubString

  CmSubString         At(int         pos, int len);
  CmSubString         operator () (int         pos, int len); // synonym for at

  CmSubString         At(const CmString&     x, int startpos = 0); 
  CmSubString         At(const CmSubString&  x, int startpos = 0); 
  CmSubString         At(const char* t, int startpos = 0);
  CmSubString         At(char        c, int startpos = 0);
  CmSubString         At(const CmRegex&      r, int startpos = 0); 

  CmSubString         Before(int          pos);
  CmSubString         Before(const CmString&      x, int startpos = 0);
  CmSubString         Before(const CmSubString&   x, int startpos = 0);
  CmSubString         Before(const char*  t, int startpos = 0);
  CmSubString         Before(char         c, int startpos = 0);
  CmSubString         Before(const CmRegex&       r, int startpos = 0);

  CmSubString         Through(int          pos);
  CmSubString         Through(const CmString&      x, int startpos = 0);
  CmSubString         Through(const CmSubString&   x, int startpos = 0);
  CmSubString         Through(const char*  t, int startpos = 0);
  CmSubString         Through(char         c, int startpos = 0);
  CmSubString         Through(const CmRegex&       r, int startpos = 0);

  CmSubString         From(int          pos);
  CmSubString         From(const CmString&      x, int startpos = 0);
  CmSubString         From(const CmSubString&   x, int startpos = 0);
  CmSubString         From(const char*  t, int startpos = 0);
  CmSubString         From(char         c, int startpos = 0);
  CmSubString         From(const CmRegex&       r, int startpos = 0);

  CmSubString         After(int         pos);
  CmSubString         After(const CmString&     x, int startpos = 0);
  CmSubString         After(const CmSubString&  x, int startpos = 0);
  CmSubString         After(const char* t, int startpos = 0);
  CmSubString         After(char        c, int startpos = 0);
  CmSubString         After(const CmRegex&      r, int startpos = 0);


// deletion

// delete len chars starting at pos
  void              Del(int         pos, int len);

// delete the first occurrence of target after startpos

  void              Del(const CmString&     y, int startpos = 0);
  void              Del(const CmSubString&  y, int startpos = 0);
  void              Del(const char* t, int startpos = 0);
  void              Del(char        c, int startpos = 0);
  void              Del(const CmRegex&      r, int startpos = 0);

// global substitution: substitute all occurrences of pat with repl

  int               Gsub(const CmString&     pat, const CmString&     repl);
  int               Gsub(const CmSubString&  pat, const CmString&     repl);
  int               Gsub(const char* pat, const CmString&     repl);
  int               Gsub(const char* pat, const char* repl);
  int               Gsub(const CmRegex&      pat, const CmString&     repl);

// friends & utilities

// split CmString into array res at separators; return number of elements

  friend int        Split(const CmString& x, CmString res[], int maxn, 
                          const CmString& sep);
  friend int        Split(const CmString& x, CmString res[], int maxn, 
                          const CmRegex&  sep);

  friend CmString     CommonPrefix(const CmString& x, const CmString& y, 
                                  int startpos = 0);
  friend CmString     CommonSuffix(const CmString& x, const CmString& y, 
                                  int startpos = -1);
  friend CmString     Replicate(char        c, int n);
  friend CmString     Replicate(const CmString&     y, int n);
  friend CmString     Join(CmString src[], int n, const CmString& sep);

// simple builtin transformations

  friend CmString     Reverse(const CmString& x);
  friend CmString     Upcase(const CmString& x);
  friend CmString     Downcase(const CmString& x);
  friend CmString     Capitalize(const CmString& x);

// in-place versions of above

  void              Reverse();
  void              Upcase();
  void              Downcase();
  void              Capitalize();

// element extraction

  char&             operator [] (int i);
  char              Elem(int i) const;
  char              Firstchar() const;
  char              Lastchar() const;

// conversion

                    operator const char*() const;
  const char*       Chars() const;


// IO

  friend ostream&   operator<<(ostream& s, const CmString& x);
  friend ostream&   operator<<(ostream& s, const CmSubString& x);
  friend istream&   operator>>(istream& s, CmString& x);

  friend int        Readline(istream& s, CmString& x, 
                             char terminator = '\n',
                             int discard_terminator = 1);

// status

  int               Length() const;
  int               Empty() const;

// preallocate some space for CmString
  void              Alloc(int newsize);

// report current Allocation (not length!)

  int               Allocation() const;


  volatile void     Error(const char* msg) const;

  int               OK() const;
};

typedef CmString StrTmp; // for backward compatibility

// other externs

int        Compare(const CmString&    x, const CmString&     y);
int        Compare(const CmString&    x, const CmSubString&  y);
int        Compare(const CmString&    x, const char* y);
int        Compare(const CmSubString& x, const CmString&     y);
int        Compare(const CmSubString& x, const CmSubString&  y);
int        Compare(const CmSubString& x, const char* y);
int        Fcompare(const CmString&   x, const CmString&     y); // ignore case

extern CmStrRep  _nilStrRep;
extern CmString _nilString;

// other inlines

CmString operator + (const CmString& x, const CmString& y);
CmString operator + (const CmString& x, const CmSubString& y);
CmString operator + (const CmString& x, const char* y);
CmString operator + (const CmString& x, char y);
CmString operator + (const CmSubString& x, const CmString& y);
CmString operator + (const CmSubString& x, const CmSubString& y);
CmString operator + (const CmSubString& x, const char* y);
CmString operator + (const CmSubString& x, char y);
CmString operator + (const char* x, const CmString& y);
CmString operator + (const char* x, const CmSubString& y);

int operator==(const CmString& x, const CmString& y); 
int operator!=(const CmString& x, const CmString& y);
int operator> (const CmString& x, const CmString& y);
int operator>=(const CmString& x, const CmString& y);
int operator< (const CmString& x, const CmString& y);
int operator<=(const CmString& x, const CmString& y);
int operator==(const CmString& x, const CmSubString&  y);
int operator!=(const CmString& x, const CmSubString&  y);
int operator> (const CmString& x, const CmSubString&  y);
int operator>=(const CmString& x, const CmSubString&  y);
int operator< (const CmString& x, const CmSubString&  y);
int operator<=(const CmString& x, const CmSubString&  y);
int operator==(const CmString& x, const char* t);
int operator!=(const CmString& x, const char* t);
int operator> (const CmString& x, const char* t);
int operator>=(const CmString& x, const char* t);
int operator< (const CmString& x, const char* t);
int operator<=(const CmString& x, const char* t);
int operator==(const CmSubString& x, const CmString& y);
int operator!=(const CmSubString& x, const CmString& y);
int operator> (const CmSubString& x, const CmString& y);
int operator>=(const CmSubString& x, const CmString& y);
int operator< (const CmSubString& x, const CmString& y);
int operator<=(const CmSubString& x, const CmString& y);
int operator==(const CmSubString& x, const CmSubString&  y);
int operator!=(const CmSubString& x, const CmSubString&  y);
int operator> (const CmSubString& x, const CmSubString&  y);
int operator>=(const CmSubString& x, const CmSubString&  y);
int operator< (const CmSubString& x, const CmSubString&  y);
int operator<=(const CmSubString& x, const CmSubString&  y);
int operator==(const CmSubString& x, const char* t);
int operator!=(const CmSubString& x, const char* t);
int operator> (const CmSubString& x, const char* t);
int operator>=(const CmSubString& x, const char* t);
int operator< (const CmSubString& x, const char* t);
int operator<=(const CmSubString& x, const char* t);

#if defined(__OPTIMIZE__) || defined(USE_LIBGXX_INLINES)


// status reports, needed before defining other things

inline int         CmString::Length() const {  return rep->len; }
inline int         CmString::Empty() const { return rep->len == 0; }
inline const char* CmString::Chars() const { return &(rep->s[0]); }
inline int         CmString::Allocation() const { return rep->sz; }
inline void        CmString::Alloc(int newsize) { rep = Sresize(rep, newsize); }

inline int         CmSubString::Length() const { return len; }
inline int         CmSubString::Empty() const { return len == 0; }
inline const char* CmSubString::Chars() const { return &(S.rep->s[pos]); }


// constructors

inline CmString::CmString() 
  : rep(&_nilStrRep) {}
inline CmString::CmString(const CmString& x) 
  : rep(Scopy(0, x.rep)) {}
inline CmString::CmString(const char* t) 
  : rep(Salloc(0, t, -1, -1)) {}
inline CmString::CmString(const char* t, int tlen)
  : rep(Salloc(0, t, tlen, tlen)) {}
inline CmString::CmString(const CmSubString& y)
  : rep(Salloc(0, y.Chars(), y.Length(), y.Length())) {}
inline CmString::CmString(char c) 
  : rep(Salloc(0, &c, 1, 1)) {}

inline CmString::~CmString() { if (rep != &_nilStrRep) delete rep; }

inline CmSubString::CmSubString(const CmSubString& x)
  :S(x.S), pos(x.pos), len(x.len) {}
inline CmSubString::CmSubString(CmString& x, int first, int l)
  :S(x), pos(first), len(l) {}

inline CmSubString::~CmSubString() {}

// assignment

inline void CmString::operator =  (const CmString& y)
{ 
  rep = Scopy(rep, y.rep);
}

inline void CmString::operator=(const char* t)
{
  rep = Salloc(rep, t, -1, -1); 
}

inline void CmString::operator=(const CmSubString&  y)
{
  rep = Salloc(rep, y.Chars(), y.Length(), y.Length());
}

inline void CmString::operator=(char c)
{
  rep = Salloc(rep, &c, 1, 1); 
}


inline void CmSubString::operator = (const char* ys)
{
  assign(0, ys);
}

inline void CmSubString::operator = (char ch)
{
  assign(0, &ch, 1);
}

inline void CmSubString::operator = (const CmString& y)
{
  assign(y.rep, y.Chars(), y.Length());
}

inline void CmSubString::operator = (const CmSubString& y)
{
  assign(y.S.rep, y.Chars(), y.Length());
}

// Zillions of cats...

inline void Cat(const CmString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y, -1);
}

inline void Cat(const CmString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const CmSubString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmSubString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmSubString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), y, -1);
}

inline void Cat(const CmSubString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const char* x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, x, -1, y.Chars(), y.Length());
}

inline void Cat(const char* x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, x, -1, y.Chars(), y.Length());
}

inline void Cat(const char* x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, x, -1, y, -1);
}

inline void Cat(const char* x, char y, CmString& r)
{
  r.rep = Scat(r.rep, x, -1, &y, 1);
}

inline void Cat(const CmString& a, const CmString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const CmString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const CmString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y, -1);
}

inline void Cat(const CmString& a, const CmString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const CmString& a, const CmSubString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const CmSubString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const CmSubString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), y, -1);
}

inline void Cat(const CmString& a, const CmSubString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const CmString& a, const char* x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x, -1, y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const char* x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x, -1, y.Chars(), y.Length());
}

inline void Cat(const CmString& a, const char* x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x, -1, y, -1);
}

inline void Cat(const CmString& a, const char* x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a.Chars(), a.Length(), x, -1, &y, 1);
}


inline void Cat(const char* a, const CmString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const char* a, const CmString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const char* a, const CmString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y, -1);
}

inline void Cat(const char* a, const CmString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const char* a, const CmSubString& x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const char* a, const CmSubString& x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y.Chars(), y.Length());
}

inline void Cat(const char* a, const CmSubString& x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), y, -1);
}

inline void Cat(const char* a, const CmSubString& x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x.Chars(), x.Length(), &y, 1);
}

inline void Cat(const char* a, const char* x, const CmString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y.Chars(), y.Length());
}

inline void Cat(const char* a, const char* x, const CmSubString& y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y.Chars(), y.Length());
}

inline void Cat(const char* a, const char* x, const char* y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y, -1);
}

inline void Cat(const char* a, const char* x, char y, CmString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, &y, 1);
}


// operator versions

inline void CmString::operator +=(const CmString& y)
{
  Cat(*this, y, *this);
}

inline void CmString::operator +=(const CmSubString& y)
{
  Cat(*this, y, *this);
}

inline void CmString::operator += (const char* y)
{
  Cat(*this, y, *this);
}

inline void CmString:: operator +=(char y)
{
  Cat(*this, y, *this);
}

// constructive concatenation

#if defined(__GNUG__) && !defined(NO_NRV)

inline CmString operator + (const CmString& x, const CmString& y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmString& x, const CmSubString& y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmString& x, const char* y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmString& x, char y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmSubString& x, const CmString& y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmSubString& x, const CmSubString& y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmSubString& x, const char* y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const CmSubString& x, char y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const char* x, const CmString& y) return r;
{
  Cat(x, y, r);
}

inline CmString operator + (const char* x, const CmSubString& y) return r;
{
  Cat(x, y, r);
}

inline CmString Reverse(const CmString& x) return r;
{
  r.rep = Sreverse(x.rep, r.rep);
}

inline CmString Upcase(const CmString& x) return r;
{
  r.rep = Supcase(x.rep, r.rep);
}

inline CmString Downcase(const CmString& x) return r;
{
  r.rep = Sdowncase(x.rep, r.rep);
}

inline CmString Capitalize(const CmString& x) return r;
{
  r.rep = Scapitalize(x.rep, r.rep);
}

#else /* NO_NRV */

inline CmString operator + (const CmString& x, const CmString& y)
{
  CmString r;  Cat(x, y, r);  return r;
}

inline CmString operator + (const CmString& x, const CmSubString& y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmString& x, const char* y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmString& x, char y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmSubString& x, const CmString& y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmSubString& x, const CmSubString& y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmSubString& x, const char* y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const CmSubString& x, char y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const char* x, const CmString& y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString operator + (const char* x, const CmSubString& y) 
{
  CmString r; Cat(x, y, r); return r;
}

inline CmString Reverse(const CmString& x) 
{
  CmString r; r.rep = Sreverse(x.rep, r.rep); return r;
}

inline CmString Upcase(const CmString& x) 
{
  CmString r; r.rep = Supcase(x.rep, r.rep); return r;
}

inline CmString Downcase(const CmString& x) 
{
  CmString r; r.rep = Sdowncase(x.rep, r.rep); return r;
}

inline CmString Capitalize(const CmString& x) 
{
  CmString r; r.rep = Scapitalize(x.rep, r.rep); return r;
}

#endif

// prepend

inline void CmString::Prepend(const CmString& y)
{
  rep = Sprepend(rep, y.Chars(), y.Length());
}

inline void CmString::Prepend(const char* y)
{
  rep = Sprepend(rep, y, -1); 
}

inline void CmString::Prepend(char y)
{
  rep = Sprepend(rep, &y, 1); 
}

inline void CmString::Prepend(const CmSubString& y)
{
  rep = Sprepend(rep, y.Chars(), y.Length());
}

// misc transformations


inline void CmString::Reverse()
{
  rep = Sreverse(rep, rep);
}


inline void CmString::Upcase()
{
  rep = Supcase(rep, rep);
}


inline void CmString::Downcase()
{
  rep = Sdowncase(rep, rep);
}


inline void CmString::Capitalize()
{
  rep = Scapitalize(rep, rep);
}

// element extraction

inline char&  CmString::operator [] (int i) 
{ 
  if (((unsigned)i) >= Length()) Error("invalid index");
  return rep->s[i];
}

inline char  CmString::Elem (int i) const
{ 
  if (((unsigned)i) >= Length()) Error("invalid index");
  return rep->s[i];
}

inline char  CmString::Firstchar() const
{ 
  return Elem(0);
}

inline char  CmString::Lastchar() const
{ 
  return Elem(Length() - 1);
}

// searching

inline int CmString::Index(char c, int startpos) const
{
  return Search(startpos, Length(), c);
}

inline int CmString::Index(const char* t, int startpos) const
{   
  return Search(startpos, Length(), t);
}

inline int CmString::Index(const CmString& y, int startpos) const
{   
  return Search(startpos, Length(), y.Chars(), y.Length());
}

inline int CmString::Index(const CmSubString& y, int startpos) const
{   
  return Search(startpos, Length(), y.Chars(), y.Length());
}

inline int CmString::Index(const CmRegex& r, int startpos) const
{
  int unused;  return r.Search(Chars(), Length(), unused, startpos);
}

inline int CmString::Contains(char c) const
{
  return Search(0, Length(), c) >= 0;
}

inline int CmString::Contains(const char* t) const
{   
  return Search(0, Length(), t) >= 0;
}

inline int CmString::Contains(const CmString& y) const
{   
  return Search(0, Length(), y.Chars(), y.Length()) >= 0;
}

inline int CmString::Contains(const CmSubString& y) const
{   
  return Search(0, Length(), y.Chars(), y.Length()) >= 0;
}

inline int CmString::Contains(char c, int p) const
{
  return Match(p, Length(), 0, &c, 1) >= 0;
}

inline int CmString::Contains(const char* t, int p) const
{
  return Match(p, Length(), 0, t) >= 0;
}

inline int CmString::Contains(const CmString& y, int p) const
{
  return Match(p, Length(), 0, y.Chars(), y.Length()) >= 0;
}

inline int CmString::Contains(const CmSubString& y, int p) const
{
  return Match(p, Length(), 0, y.Chars(), y.Length()) >= 0;
}

inline int CmString::Contains(const CmRegex& r) const
{
  int unused;  return r.Search(Chars(), Length(), unused, 0) >= 0;
}

inline int CmString::Contains(const CmRegex& r, int p) const
{
  return r.Match(Chars(), Length(), p) >= 0;
}


inline int CmString::Matches(const CmSubString& y, int p) const
{
  return Match(p, Length(), 1, y.Chars(), y.Length()) >= 0;
}

inline int CmString::Matches(const CmString& y, int p) const
{
  return Match(p, Length(), 1, y.Chars(), y.Length()) >= 0;
}

inline int CmString::Matches(const char* t, int p) const
{
  return Match(p, Length(), 1, t) >= 0;
}

inline int CmString::Matches(char c, int p) const
{
  return Match(p, Length(), 1, &c, 1) >= 0;
}

inline int CmString::Matches(const CmRegex& r, int p) const
{
  int l = (p < 0)? -p : Length() - p;
  return r.Match(Chars(), Length(), p) == l;
}


inline int CmSubString::Contains(const char* t) const
{   
  return S.Search(pos, pos+len, t) >= 0;
}

inline int CmSubString::Contains(const CmString& y) const
{   
  return S.Search(pos, pos+len, y.Chars(), y.Length()) >= 0;
}

inline int CmSubString::Contains(const CmSubString&  y) const
{   
  return S.Search(pos, pos+len, y.Chars(), y.Length()) >= 0;
}

inline int CmSubString::Contains(char c) const
{
  return S.Search(pos, pos+len, 0, c) >= 0;
}

inline int CmSubString::Contains(const CmRegex& r) const
{
  int unused;  return r.Search(Chars(), len, unused, 0) >= 0;
}

inline int CmSubString::Matches(const CmRegex& r) const
{
  return r.Match(Chars(), len, 0) == len;
}


inline int CmString::Gsub(const CmString& pat, const CmString& r)
{
  return _gsub(pat.Chars(), pat.Length(), r.Chars(), r.Length());
}

inline int CmString::Gsub(const CmSubString&  pat, const CmString& r)
{
  return _gsub(pat.Chars(), pat.Length(), r.Chars(), r.Length());
}

inline int CmString::Gsub(const CmRegex& pat, const CmString& r)
{
  return _gsub(pat, r.Chars(), r.Length());
}

inline int CmString::Gsub(const char* pat, const CmString& r)
{
  return _gsub(pat, -1, r.Chars(), r.Length());
}

inline int CmString::Gsub(const char* pat, const char* r)
{
  return _gsub(pat, -1, r, -1);
}


inline CmString::operator const char*() const
{ 
  return str(Chars());
}

inline  ostream& operator<<(ostream& s, const CmString& x)
{
#ifdef VMS
   s << x.Chars(); return s;
#else
#ifdef __GNUG__
  s.put(x.Chars()); return s;
#else
	   s << x.Chars(); return s;
#endif /* __GNUG__ */
#endif


}

// a zillion comparison operators

inline int operator==(const CmString& x, const CmString& y) 
{
  return Compare(x, y) == 0; 
}

inline int operator!=(const CmString& x, const CmString& y)
{
  return Compare(x, y) != 0; 
}

inline int operator>(const CmString& x, const CmString& y)
{
  return Compare(x, y) > 0; 
}

inline int operator>=(const CmString& x, const CmString& y)
{
  return Compare(x, y) >= 0; 
}

inline int operator<(const CmString& x, const CmString& y)
{
  return Compare(x, y) < 0; 
}

inline int operator<=(const CmString& x, const CmString& y)
{
  return Compare(x, y) <= 0; 
}

inline int operator==(const CmString& x, const CmSubString&  y) 
{
  return Compare(x, y) == 0; 
}

inline int operator!=(const CmString& x, const CmSubString&  y)
{
  return Compare(x, y) != 0; 
}

inline int operator>(const CmString& x, const CmSubString&  y)      
{
  return Compare(x, y) > 0; 
}

inline int operator>=(const CmString& x, const CmSubString&  y)
{
  return Compare(x, y) >= 0; 
}

inline int operator<(const CmString& x, const CmSubString&  y) 
{
  return Compare(x, y) < 0; 
}

inline int operator<=(const CmString& x, const CmSubString&  y)
{
  return Compare(x, y) <= 0; 
}

inline int operator==(const CmString& x, const char* t) 
{
  return Compare(x, t) == 0; 
}

inline int operator!=(const CmString& x, const char* t) 
{
  return Compare(x, t) != 0; 
}

inline int operator>(const CmString& x, const char* t)  
{
  return Compare(x, t) > 0; 
}

inline int operator>=(const CmString& x, const char* t) 
{
  return Compare(x, t) >= 0; 
}

inline int operator<(const CmString& x, const char* t)  
{
  return Compare(x, t) < 0; 
}

inline int operator<=(const CmString& x, const char* t) 
{
  return Compare(x, t) <= 0; 
}

inline int operator==(const CmSubString& x, const CmString& y) 
{
  return Compare(y, x) == 0; 
}

inline int operator!=(const CmSubString& x, const CmString& y)
{
  return Compare(y, x) != 0;
}

inline int operator>(const CmSubString& x, const CmString& y)      
{
  return Compare(y, x) < 0;
}

inline int operator>=(const CmSubString& x, const CmString& y)     
{
  return Compare(y, x) <= 0;
}

inline int operator<(const CmSubString& x, const CmString& y)      
{
  return Compare(y, x) > 0;
}

inline int operator<=(const CmSubString& x, const CmString& y)     
{
  return Compare(y, x) >= 0;
}

inline int operator==(const CmSubString& x, const CmSubString&  y) 
{
  return Compare(x, y) == 0; 
}

inline int operator!=(const CmSubString& x, const CmSubString&  y)
{
  return Compare(x, y) != 0;
}

inline int operator>(const CmSubString& x, const CmSubString&  y)      
{
  return Compare(x, y) > 0;
}

inline int operator>=(const CmSubString& x, const CmSubString&  y)
{
  return Compare(x, y) >= 0;
}

inline int operator<(const CmSubString& x, const CmSubString&  y) 
{
  return Compare(x, y) < 0;
}

inline int operator<=(const CmSubString& x, const CmSubString&  y)
{
  return Compare(x, y) <= 0;
}

inline int operator==(const CmSubString& x, const char* t) 
{
  return Compare(x, t) == 0; 
}

inline int operator!=(const CmSubString& x, const char* t) 
{
  return Compare(x, t) != 0;
}

inline int operator>(const CmSubString& x, const char* t)  
{
  return Compare(x, t) > 0; 
}

inline int operator>=(const CmSubString& x, const char* t) 
{
  return Compare(x, t) >= 0; 
}

inline int operator<(const CmSubString& x, const char* t)  
{
  return Compare(x, t) < 0; 
}

inline int operator<=(const CmSubString& x, const char* t) 
{
  return Compare(x, t) <= 0; 
}


// a helper needed by at, before, etc.

inline CmSubString CmString::_substr(int first, int l)
{
  if (first < 0 || (unsigned)(first + l) > Length()) 
    return CmSubString(_nilString, 0, 0) ;
  else 
    return CmSubString(*this, first, l);
}


#endif

#endif
