/* $Header: /afs/athena.mit.edu/astaff/project/atdev/src/plotter/RCS/AxisPS.c,v 3.2 91/07/15 23:12:21 crcraig Exp Locker: dot $ */

/*******************************************************************
  Copyright (C) 1990 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

***************************************************************** */

#include <stdio.h>
#include <math.h>    

#ifdef _AtDevelopment_
#include "AxisP.h"
#include "FontFamilyP.h"    
#else
#include <At/AxisP.h>
#include <At/FontFamilyP.h>
#endif

/* 1 point is about 1 pixel on our displays, so for now, tic lengths and
   margin lengths etc. are okay as is.
*/

/*
 * returns the "width" (the height if horizontal) of the passed axis.
 * XXX assumes maxnumwidth is pixels is approximately correct.
 */
int AtAxisWidthPS(AtAxisObject w)
{
  if (w->axis.vertical) {
    return (w->axis.labelText ? AtTextPSWidth(w->axis.labelText) : 0) +
      (w->axis.drawNumbers && w->axis.numbersOutside ?
       w->axis.maxnumwidth : 0) +
	MIN_MARGIN + w->axis.ticLength + 1;
  } else {
    return (w->axis.labelText ? 
	    AtTextPSAscent(w->axis.labelText) +
	    AtTextPSDescent(w->axis.labelText) : 0) +
	      (w->axis.drawNumbers && w->axis.numbersOutside ?
	       w->axis.numberFont->ascent + w->axis.numberFont->descent : 0) +
		 MIN_MARGIN + w->axis.ticLength + 1;
  }
}
			   


void AtAxisDrawPS(FILE *f, AtAxisObject axis,
		  int x1, int y1, int x2, int y2, int g1, int g2)
{
    AtAxisPart *a = &axis->axis;
    AtScale *scale;
    int i, j;
    short st1, st2, bt1, bt2; /* where to draw small and big tics */
    short tmp;
    double subtic;
    double *tics;
    int maxnumwidth = 0;

    if (a->vertical)
	scale = AtScaleCreate(a->axis_min, a->axis_max, 
			      y1*100, y2*100, a->transform);
    else
	scale = AtScaleCreate(a->axis_min, a->axis_max, 
			      x1*100, x2*100, a->transform);

    fprintf(f, "%%%% BeginObject: AtAxis\n");
    fprintf(f, "gsave\n");

    /* draw the line */
    fprintf(f, "%d %d M %d %d L\n", x1, y1, x2, y2);    

#define swap(a,b) {tmp = a; a = b; b = tmp;}    
    /* figure out how to draw the tics */
    st1 = st2 = 0;
    bt1 = bt2 = 0;
    if (a->vertical) {
	if (a->ticsOutside) {st1 = a->subticLength; bt1 = a->ticLength;}
	if (a->ticsInside)  {st2 = a->subticLength; bt2 = a->ticLength;}
	if (a->mirror) { swap(st1, st2); swap(bt1,bt2); }
    } else {
	if (a->ticsOutside) {st2 = a->subticLength; bt2 = a->ticLength;}
	if (a->ticsInside)  {st1 = a->subticLength; bt1 = a->ticLength;}
	if (a->mirror) { swap(st1, st2); swap(bt1,bt2); }
    }
#undef swap

    tics = (double *) XtMalloc(a->ntics * sizeof(double));
    
    /* draw the tics */
    if (a->vertical) 
	for(i = 0; i < a->ntics; i++) {
	    tics[i] = AtScaleUserToPixel(scale, a->ticvalues[i]) / 100.0;
	    fprintf(f, "%d %g M %d %g L\n",
		    x1-bt1, tics[i], x1+bt2, tics[i]);
	}
    else
	for(i=0; i<a->ntics; i++) {
	    tics[i] = AtScaleUserToPixel(scale, a->ticvalues[i]) / 100.0;
	    fprintf(f, "%g %d M %g %d L\n",
		    tics[i], y1+bt1, tics[i], y1-bt2);
	}
    fprintf(f, "stroke\n");

    /* draw the subtics */
    for(i = 0; i < a->ntics-1; i++) {
	if (a->transform == AtTransformLINEAR)
	    for(j=0; j < a->subtics; j++) {
		subtic = AtScaleUserToPixel(scale, a->ticvalues[i] +
			         a->ticInterval/(a->subtics+1)*(j+1)) / 100.0;
		if (a->vertical)
		    fprintf(f, "%d %g M %d %g L\n",
			    x1-st1, subtic, x1+st2, subtic);
		else
		    fprintf(f, "%g %d M %g %d L\n",
			    subtic, y1+st1, subtic, y1-st2);
	    }
	else {
	    int decade;
	    double ticval, logticval;
	    ticval = a->ticvalues[i];
	    logticval = log10(ticval);
	    decade = (int)floor(logticval);
	    if (logticval == (double)decade) /* if first tic of a decade */
		ticval = 0.0;

	    for(j=0; j < a->subtics; j++) {
		subtic =
		    AtScaleUserToPixel(scale, ticval + 
				       (a->ticInterval/(a->subtics+1)*(j+1)) *
				       pow(10.0, (double)decade)) / 100.0;
		if (a->vertical)
		    fprintf(f, "%d %g M %d %g L\n",
			    x1-st1, subtic, x1+st2, subtic);
		else
		    fprintf(f, "%g %d M %g %d L\n",
			    subtic, y1+st1, subtic, y1-st2);
	    }
	}
    }
    fprintf(f, "stroke\n");

    /* draw the grid */
    if (a->drawGrid) {
	fprintf(f, "gsave\n");
	fprintf(f, ".5 setlinewidth\n");
	fprintf(f, "[.5 .75] 0 setdash\n");
	for(i = 0; i<a->ntics; i++) {
	    if (a->vertical) 
		fprintf(f, "%d %g M %d %g L\n",
			g1, tics[i], g2, tics[i]);
	    else
		fprintf(f, "%g %d M %g %d L\n",
			tics[i], g1, tics[i], g2);
	}
	fprintf(f, "stroke\n");
	fprintf(f, "grestore\n");
    }


    XtFree(tics);
    
#define ascent(a) AtFontPSAscent(a->numberFF, a->numberStyle, a->numberSize)
#define descent(a) AtFontPSDescent(a->numberFF, a->numberStyle, a->numberSize)

    /* draw the numbers */
    fprintf(f, "%d /%s F\n", AtFontPointSize(a->numberFF, a->numberSize),
	    AtFontPSName(a->numberFF, a->numberStyle));
    maxnumwidth = 0;
    if (a->drawNumbers) {
	for(i=0; i < a->ntics; i++) {
	    int x,y,wid,len;
	    len = strlen(a->ticlabels[i]);
	    wid = AtFontPSTextWidth(a->numberFF, a->numberStyle, a->numberSize,
				    a->ticlabels[i], len);
	    if (wid > maxnumwidth) maxnumwidth = wid;
	    if (a->vertical) {
		x = x1;
		/* y = a->ticcoords[i];*/
		y = AtScaleUserToPixel(scale, a->ticvalues[i]) / 100;
		if (!a->mirror ^ !!a->numbersOutside) 
		    x += bt2 + 2;
		else
		    x -= bt1 + 2 + wid;
		y -= ascent(a)/2 - descent(a)/2;
	    }
	    else {
		/*x = a->ticcoords[i];*/
		x = AtScaleUserToPixel(scale, a->ticvalues[i]) / 100;
		y = y1;
		if (!a->mirror ^ !!a->numbersOutside)
		    y += bt1 + 2 + descent(a);
		else
		    y -= bt2 + 2 + ascent(a);
		x -= wid/2;
	    }
	    fprintf(f, "(%s) %d %d S\n", a->ticlabels[i], x, y);
	}
    }

    /* draw the label */
    if (a->labelText) {
	int x,y;
	x = (x1 + x2)/2;
	y = (y1 + y2)/2;
	if (a->vertical) {
	    if (a->mirror) 
		x += bt2 + 2 + maxnumwidth + 2;
	    else
		x -= bt1 + maxnumwidth + 2 + AtTextPSWidth(a->labelText);
	    y -=AtTextPSAscent(a->labelText)/2-AtTextPSDescent(a->labelText)/2;
	}
	else {
	    if (a->mirror) {
		y += bt1 + 2 +  AtTextPSDescent(a->labelText);
		if (a->drawNumbers)
		    y += 2 + ascent(a) + descent(a);
	    }
	    else {
		y -= bt2 + 3 + AtTextPSAscent(a->labelText);
		if (a->drawNumbers)
		    y -=  2 + ascent(a) + descent(a);
	    }
	    x -= AtTextPSWidth(a->labelText)/2;
	}
	AtTextPSDraw(f, a->labelText, x, y);
    }
    fprintf(f, "grestore\n");
    fprintf(f, "%%%% EndObject\n");
}
#undef ascent
#undef descent

    
