#include <X11/copyright.h>
/*
 * Copyright 1988 Massachusetts Institute of Technology
 * Copyright 1987 by Digital Equipment Corporation, Maynard, Massachusetts.
 *
 *                         All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Digital Equipment
 * Corporation not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.
 *
 *
 * DIGITAL DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * DIGITAL BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* Tekproc.c */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <TekP.h>
#include <Tekparse.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>


/* Tek defines */

#define	BEL		07
#define	CANCEL		030
#define	DOTDASHEDLINE	2
#define	DOTTEDLINE	1
#define	EAST		01
#define	ETX		03
#define	LINEMASK	07
#define	LONGDASHEDLINE	4
#define	MARGIN1		0
#define	MARGIN2		1
#define MAX_PTS		150
#define MAX_VTX		300
#define	NAK		025
#define	NORTH		04
#define	PENDOWN		1
#define	PENUP		0
#define	SHORTDASHEDLINE	3
#define	SOLIDLINE	0
#define	SOUTH		010
#define	TEKBOTTOMPAD	23
#define	TEKHOME		0
#define	TEKTOPPAD	34
#define	WEST		02

#define	TekMove(tw, x,y)	tw->tek.cur_X = x; tw->tek.cur_Y = y

#define NBOX 4

static struct Tek_Char {
  int hsize;	/* in Tek units */
  int vsize;	/* in Tek units */
  int charsperline;
  int nlines;
} TekChar[TEKNUMFONTS] = {
	{56, 88, 74, 35},	/* large */
	{51, 82, 81, 38},	/* #2 */
	{34, 53, 121, 58},	/* #3 */
	{31, 48, 133, 64},	/* small */
};

extern int Talptable[];
extern int Tbestable[];
extern int Tbyptable[];
extern int Tesctable[];
extern int Tipltable[];
extern int Tplttable[];
extern int Tpttable[];
extern int Tspttable[];

static int *curstate = Talptable;
static int *Tparsestate = Talptable;

static Dimension tek_def_height = 565;
static Dimension tek_def_width = 750;
static Dimension tek_height = 3072;
static Dimension tek_width = 4096;

static XtResource resources[] = {
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     XtOffset(Widget, core.width), XtRDimension, (caddr_t)&tek_def_width},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     XtOffset(Widget, core.height), XtRDimension, (caddr_t)&tek_def_height},
  {XtNtekWidth, XtCWidth, XtRDimension, sizeof(Dimension),
     XtOffset(TekWidget, tek.width), XtRDimension, (XtPointer) &tek_width},
  {XtNtekHeight, XtCHeight, XtRDimension, sizeof(Dimension),
     XtOffset(TekWidget, tek.height), XtRDimension, (XtPointer) &tek_height},
  {XtNtekString, XtCString, XtRString, sizeof(String),
     XtOffset(TekWidget, tek.string), XtRString, (XtPointer) NULL},
  {XtNfontLarge, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     XtOffset(TekWidget, tek.Tfont[TEK_FONT_LARGE]),
     XtRString, "9x15"},
  {XtNfont2, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     XtOffset(TekWidget, tek.Tfont[TEK_FONT_2]),
     XtRString, "6x13"},
  {XtNfont3, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     XtOffset(TekWidget, tek.Tfont[TEK_FONT_3]),
     XtRString, "8x13"},
  {XtNfontSmall, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
     XtOffset(TekWidget, tek.Tfont[TEK_FONT_SMALL]),
     XtRString, "6x10"},
  {XtNinitialFont, XtCInitialFont, XtRString, sizeof(char *),
     XtOffset(TekWidget, tek.initial_font),
     XtRString, "large"},
  {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
     XtOffset(TekWidget, tek.foreground_pixel), 
     XtRString, "XtDefaultForeground"},
  {XtNtekFile, XtCFile, XtRString, sizeof(char *),
     XtOffset(TekWidget, tek.file),
     XtRString, (XtPointer) NULL}
};

static void TekInitialize(), TekConfigure();
static void AddToDraw(), TekFlush();
static void TekExpose(), TCursorToggle(), TekRealize();
static Boolean TekSetValues();
static void TekLoadFile();

static TekClassRec tekWidgetClassRec = {
  {
/* core_class fields */	
    /* superclass	  */	(WidgetClass) &widgetClassRec,
    /* class_name	  */	"Tek",
    /* widget_size	  */	sizeof(TekRec),
    /* class_initialize   */    NULL,
    /* class_part_initialize */ NULL,
    /* class_inited       */	FALSE,
    /* initialize	  */	TekInitialize,
    /* initialize_hook    */    NULL,	
    /* realize		  */	TekRealize,
    /* actions		  */	NULL,
    /* num_actions	  */	0,
    /* resources	  */	resources,
    /* num_resources	  */	XtNumber(resources),
    /* xrm_class	  */	NULLQUARK,
    /* compress_motion	  */	TRUE,
    /* compress_exposure  */	TRUE,
    /* compress_enterleave */   TRUE,
    /* visible_interest	  */	FALSE,
    /* destroy		  */	NULL,
    /* resize		  */	TekConfigure,
    /* expose		  */	TekExpose,
    /* set_values	  */	TekSetValues,
    /* set_values_hook    */    NULL,
    /* set_values_almost  */    NULL,
    /* get_values_hook    */    NULL,
    /* accept_focus	  */	NULL,
    /* version            */    XtVersion,
    /* callback_offsets   */    NULL,
    /* tm_table           */    NULL,
    /* query_geometry     */    XtInheritQueryGeometry,
    /* display_accelerator*/    XtInheritDisplayAccelerator,
    /* extension          */    NULL
  },
  {
    /* empty              */    NULL,
  }
};


WidgetClass tekWidgetClass = (WidgetClass)&tekWidgetClassRec;

static char TekGetChar();
static void TekUnputChar();
static void TCursorBack(), TCursorForward(), TCursorUp(), TCursorDown();
static void TekFlush(), TekPage(), TekSetGCFont(), TekDraw();
static int getpoint();


static void Tekparse(tw)
     TekWidget tw;
{
  register int c, x, y;
  char ch;
  tw->tek.string_done = False;
  tw->tek.string_pos = 0;
  tw->tek.line_num = 0;
  tw->tek.line_cur = tw->tek.line_buffer;

  for( ; ; ) {
    c = TekGetChar(tw);
    if (tw->tek.string_done)
      return;
    switch(Tparsestate[c]) {
    case CASE_REPORT:
      break;

    case CASE_VT_MODE:
      return;

    case CASE_SPT_STATE:
      /* Enter Special Point Plot mode */
      Tparsestate = curstate = Tspttable;
      break;
      
    case CASE_GIN:
      Tparsestate = Tbyptable;	/* Bypass mode */
      break;
      
    case CASE_BEL:
      /* BEL */
      XBell(XtDisplay(tw), 50);
      Tparsestate = curstate;	/* clear bypass condition */
      break;
      
    case CASE_BS:
      /* BS */
      Tparsestate = curstate;	/* clear bypass condition */
      TCursorBack(tw);
      break;

    case CASE_PT_STATE:
      /* Enter Tek Point Plot mode */
      Tparsestate = curstate = Tpttable;
      break;
      
    case CASE_PLT_STATE:
      /* Enter Tek Plot mode */
      Tparsestate = curstate = Tplttable;
      if((c = TekGetChar(tw)) == BEL)
	tw->tek.pen_down = True;
      else {
	TekUnputChar(tw, c);
	tw->tek.pen_down = False;
      }
      break;

    case CASE_TAB:
      /* HT */
      Tparsestate = curstate;	/* clear bypass condition */
      TCursorForward(tw);
      break;
      
    case CASE_IPL_STATE:
      /* Enter Tek Incremental Plot mode */
      Tparsestate = curstate = Tipltable;
      break;
      
    case CASE_ALP_STATE:
      /* Enter Tek Alpha mode from any other mode */
      /* if in one of graphics states, move alpha cursor */
      if(tw->tek.line_num > 0)	/* flush line Tbuffer */
	TekFlush(tw);
      Tparsestate = curstate = Talptable;
      break;

    case CASE_UP:
      /* cursor up */
      Tparsestate = curstate;	/* clear bypass condition */
      TCursorUp(tw);
      break;
      
    case CASE_COPY:
      /* make copy */
      break;
      
    case CASE_PAGE:
      /* Page Function */
      TekPage(tw);	/* clear bypass condition */
      break;
      
    case CASE_BES_STATE:
      /* Byp: an escape char */
      Tparsestate = Tbestable;
      break;
      
    case CASE_BYP_STATE:
			/* set bypass condition */
      Tparsestate = Tbyptable;
      break;
      
    case CASE_IGNORE:
			/* Esc: totally ignore CR, ESC, LF, ~ */
      break;
      
    case CASE_ASCII:
      /* Select ASCII char set */
      /* ignore for now */
      Tparsestate = curstate;
      break;
      
    case CASE_APL:
      /* Select APL char set */
      /* ignore for now */
      Tparsestate = curstate;
      break;
      
    case CASE_CHAR_SIZE: 
      /* character size selector */
      TekSetGCFont (tw, tw->tek.cur.fontsize = (c & 03));
      Tparsestate = curstate;
      break;
      
    case CASE_BEAM_VEC:
      /* beam and vector selector */
      /* only line types */
      if((c &= LINEMASK) != tw->tek.cur.linetype) {
	if(tw->tek.line_num > 0)
	  TekFlush(tw);
	tw->tek.cur.linetype = c;
      }
      Tparsestate = curstate;
			break;
      
    case CASE_CURSTATE:
      Tparsestate = curstate;
      break;
      
    case CASE_PENUP:
      /* Ipl: penup */
      tw->tek.pen_down = False;
      break;
      
    case CASE_PENDOWN:
      /* Ipl: pendown */
      tw->tek.pen_down = True;
      break;
      
    case CASE_IPL_POINT:
      /* Ipl: point */
      x = tw->tek.cur_X;
      y = tw->tek.cur_Y;
      if(c & NORTH)
	y++;
      else if(c & SOUTH)
	y--;
      if(c & EAST)
	x++;
      else if(c & WEST)
	x--;
      if(tw->tek.pen_down)
	TekDraw(tw, x, y);
      else
	TekMove(tw, x, y);
      break;
      
    case CASE_PLT_VEC:
      /* Plt: vector */
      TekUnputChar(tw, c);
      if(getpoint(tw)) {
	if(tw->tek.pen_down)
	  TekDraw(tw, tw->tek.cur.x, tw->tek.cur.y);
	else
	  TekMove(tw, tw->tek.cur.x, tw->tek.cur.y);
	tw->tek.pen_down = True;
      }
      break;

    case CASE_PT_POINT:
      /* Pt: point */
      TekUnputChar(tw, c);
      if(getpoint(tw)) {
	TekMove(tw, tw->tek.cur.x, tw->tek.cur.y);
	TekDraw(tw, tw->tek.cur.x, tw->tek.cur.y);
      }
      break;

    case CASE_SPT_POINT:
      /* Spt: point */
      /* ignore intensity character in c */
      if(getpoint(tw)) {
	TekMove(tw, tw->tek.cur.x, tw->tek.cur.y);
	TekDraw(tw->tek.cur.x, tw->tek.cur.y);
      }
      break;
      
    case CASE_CR:
      /* CR */
      if(tw->tek.line_num > 0)	/* flush line Tbuffer */
	TekFlush(tw);
      tw->tek.cur_X = tw->tek.margin == MARGIN1 ? 0 :
	tw->tek.width / 2;
      Tparsestate = curstate = Talptable;
      break;
      
    case CASE_ESC_STATE:
      /* ESC */
      Tparsestate = Tesctable;
      break;
      
    case CASE_LF:
      /* LF */
      TCursorDown(tw);
      break;
      
    case CASE_SP:
      /* SP */
      TCursorForward(tw);
      break;
      
    case CASE_PRINT:
      /* printable character */
      ch = c;
      c = tw->tek.cur.fontsize;
      
      XDrawString(XtDisplay(tw),
		  XtWindow(tw), 
		  tw->tek.TnormalGC,
		  (int)(tw->tek.cur_X * tw->tek.scale) + tw->core.border_width,
		  (int)((tw->tek.height + TEKTOPPAD - tw->tek.cur_Y) * 
			tw->tek.scale) + tw->core.border_width,
		  &ch,
		  1);
      TCursorForward(tw);
      break;
    case CASE_OSC:
      /* do osc escape */
      Tparsestate = curstate;
      break;
    }
  }			
}

static void TekSetGCFont (tw, size)
     TekWidget tw;
    int size;  /* TEK_FONT_{LARGE,2,3,SMALL} */
{
    Font fid = tw->tek.Tfont[size]->fid;
    XSetFont (XtDisplay(tw), tw->tek.TnormalGC, fid);
  }



/* this should become the Tek Widget's Resize proc */
static void TekConfigure(tw)
    TekWidget tw;
{
  register int border = 2 * tw->core.border_width;
  register double d;
  int TWidth, THeight;

  if (XtWindow(tw)) XClearWindow(XtDisplay(tw), XtWindow(tw));
  TWidth = tw->core.width - border;
  THeight = tw->core.height - border;
  tw->tek.scale = (double)TWidth / tw->tek.width;
  if((d = (double)THeight / (tw->tek.height + TEKTOPPAD + TEKBOTTOMPAD))
     < tw->tek.scale)
    tw->tek.scale = d;
}

/* this should become the Tek Widget's Expose proc */
/* need to use compress_events = TRUE so you don't need to 
   look at the "count" in the exposure event ! */
/*ARGSUSED*/
static void TekExpose(w, event, region)
Widget w;
XExposeEvent *event;
Region region;
{
  TekWidget tw = (TekWidget) w;

  TCursorToggle(tw, CLEAR);
  tw->tek.cur_X = 0;
  tw->tek.cur_Y = TEKHOME;
  TekSetGCFont (tw, tw->tek.cur.fontsize);
  tw->tek.margin = MARGIN1;
  Tparsestate = curstate = Talptable;
  Tekparse(tw);
}

static void TekPage(tw)
     TekWidget tw;
{
  XClearWindow(XtDisplay(tw), XtWindow(tw));
  tw->tek.cur_X = 0;
  tw->tek.cur_Y = TEKHOME;
  tw->tek.margin = MARGIN1;
  Tparsestate = curstate = Talptable;	/* Tek Alpha mode */
}

#define	EXTRABITS	017
#define	FIVEBITS	037
#define	HIBITS		(FIVEBITS << SHIFTHI)
#define	LOBITS		(FIVEBITS << SHIFTLO)
#define	SHIFTHI		7
#define	SHIFTLO		2
#define	TWOBITS		03

static int getpoint(tw)
     TekWidget tw;
{
  register int c, x, y, e, lo_y = 0;
  
  x = tw->tek.cur.x;
  y = tw->tek.cur.y;
  for( ; ; ) {
    if((c = TekGetChar(tw)) < ' ') {	/* control character */
      TekUnputChar(tw, c);
      return(0);
    }
    if(c < '@') {	/* Hi X or Hi Y */
      if(lo_y) {	/* seen a Lo Y, so this must be Hi X */
	x &= ~HIBITS;
	x |= (c & FIVEBITS) << SHIFTHI;
	continue;
      }
      /* else Hi Y */
      y &= ~HIBITS;
      y |= (c & FIVEBITS) << SHIFTHI;
      continue;
		}
    if(c < '`') {	/* Lo X */
      x &= ~LOBITS;
      x |= (c & FIVEBITS) << SHIFTLO;
      tw->tek.cur.x = x;
      tw->tek.cur.y = y;
      return(1);	/* OK */
    }
    /* else Lo Y */
    if(lo_y) {	/* seen a Lo Y, so other must be extra bits */
      e = (y >> SHIFTLO) & EXTRABITS;
      x &= ~TWOBITS;
      x |= e & TWOBITS;
      y &= ~TWOBITS;
      y |= (e >> SHIFTLO) & TWOBITS;
    }
    y &= ~LOBITS;
    y |= (c & FIVEBITS) << SHIFTLO;
    lo_y++;
  }
}

static void TCursorBack(tw)
     TekWidget tw;
{
  register struct Tek_Char *t;
  register int x, l;
  
  x = ( tw->tek.cur_X -=
       (t = &TekChar[tw->tek.cur.fontsize])->hsize
       );
  
  if(tw->tek.margin == MARGIN1 && x < 0 || tw->tek.margin == MARGIN2
     && x < tw->tek.width / 2) {
    if((l = (tw->tek.cur_Y + (t->vsize - 1)) / t->vsize + 1) >=
       t->nlines) {
      tw->tek.margin = !tw->tek.margin;
      l = 0;
    }
    tw->tek.cur_Y = l * t->vsize;
    tw->tek.cur_X = (t->charsperline - 1) * t->hsize;
  }
}

static void TCursorForward(tw)
     TekWidget tw;
{
  register struct Tek_Char *t;
  register int l;
  
  if( ( tw->tek.cur_X +=
       ( t = &TekChar[tw->tek.cur.fontsize])->hsize
       ) > tw->tek.width
     ) {
    if((l = tw->tek.cur_Y / t->vsize - 1) < 0) {
      tw->tek.margin = !tw->tek.margin;
      l = t->nlines - 1;
    }
    tw->tek.cur_Y = l * t->vsize;
    tw->tek.cur_X = tw->tek.margin == MARGIN1 ? 0 : tw->tek.width / 2;
  }
}

static void TCursorUp(tw)
     TekWidget tw;
{
  register struct Tek_Char *t;
  register int l;
  
  t = &TekChar[tw->tek.cur.fontsize];
  
  if((l = (tw->tek.cur_Y + (t->vsize - 1)) / t->vsize + 1) >= t->nlines) {
    l = 0;
    if((tw->tek.margin = !tw->tek.margin) != MARGIN1) {
      if(tw->tek.cur_X < tw->tek.width / 2)
	tw->tek.cur_X += tw->tek.width / 2;
    } else if(tw->tek.cur_X >= tw->tek.width / 2)
      tw->tek.cur_X -= tw->tek.width / 2;
  }
  tw->tek.cur_Y = l * t->vsize;
}

static void TCursorDown(tw)
     TekWidget tw;
{
  register struct Tek_Char *t;
  register int l;
  
  t = &TekChar[tw->tek.cur.fontsize];
  
  if((l = tw->tek.cur_Y / t->vsize - 1) < 0) {
    l = t->nlines - 1;
    if((tw->tek.margin = !tw->tek.margin) != MARGIN1) {
      if(tw->tek.cur_X < tw->tek.width / 2)
	tw->tek.cur_X += tw->tek.width / 2;
    } else if(tw->tek.cur_X >= tw->tek.width / 2)
      tw->tek.cur_X -= tw->tek.width / 2;
  }
  tw->tek.cur_Y = l * t->vsize;
}

static void TekDraw (tw, x, y)
     TekWidget tw;
     int x, y;
{
  if(tw->tek.line_num == 0 || tw->tek.last_x != tw->tek.cur_X 
     || tw->tek.last_y != tw->tek.cur_Y) {
    /*
     * We flush on each unconnected line segment if the line
     * type is not solid.  This solves a bug in X when drawing
     * points while the line type is not solid.
     */
    if(tw->tek.line_num > 0 && tw->tek.cur.linetype != SOLIDLINE)
      TekFlush(tw);
  }
  AddToDraw(tw, tw->tek.cur_X, tw->tek.cur_Y, x, y);
  tw->tek.last_x = tw->tek.cur_X = x;
  tw->tek.last_y = tw->tek.cur_Y = y;
}

static void AddToDraw(tw, x1, y1, x2, y2)
     TekWidget tw;
     int x1, y1, x2, y2;
{
  register XSegment *lp;
  
  if(tw->tek.line_num >= MAX_PTS) {
    TekFlush(tw);
  }
  lp = tw->tek.line_cur++;
  lp->x1 = x1 = x1 * tw->tek.scale + tw->core.border_width;
  lp->y1 = y1 = (tw->tek.height + TEKTOPPAD - y1) * tw->tek.scale +
    tw->core.border_width;
  lp->x2 = x2 = x2 * tw->tek.scale + tw->core.border_width;
  lp->y2 = y2 = (tw->tek.height + TEKTOPPAD - y2) * tw->tek.scale +
    tw->core.border_width;
  tw->tek.line_num++;
}

static void TekFlush (tw) 
     TekWidget tw;
{  
  XDrawSegments(XtDisplay(tw), XtWindow(tw), 
		((tw->tek.cur.linetype == SOLIDLINE)?  tw->tek.TnormalGC :
		 tw->tek.linepat[tw->tek.cur.linetype - 1]),
		tw->tek.line_buffer, tw->tek.line_num);
  tw->tek.line_num = 0;
  tw->tek.line_cur = tw->tek.line_buffer;
}

#define DOTTED_LENGTH 2
#define DOT_DASHED_LENGTH 4
#define SHORT_DASHED_LENGTH 2
#define LONG_DASHED_LENGTH 2


static int dash_length[TEKNUMLINES] = {
	DOTTED_LENGTH,
	DOT_DASHED_LENGTH,
	SHORT_DASHED_LENGTH,
	LONG_DASHED_LENGTH,
};

static unsigned char dotted[DOTTED_LENGTH] = {3, 1};
static unsigned char dot_dashed[DOT_DASHED_LENGTH] = {3, 4, 3, 1};
static unsigned char short_dashed[SHORT_DASHED_LENGTH] = {4, 4};
static unsigned char long_dashed[LONG_DASHED_LENGTH] = {4, 7};

static unsigned char *dashes[TEKNUMLINES] = {
	dotted,
	dot_dashed,
	short_dashed,
	long_dashed,
};



/*
 * The following is called the create the tekWidget
 */

static void TekInitialize(request, new)
    Widget request, new;
{
  TekWidget tw = (TekWidget) new;
  FILE *file;

  tw->tek.line_buffer = (XSegment*) XtMalloc(MAX_VTX * sizeof(XSegment));

  tw->tek.cur_X = 0;
  tw->tek.cur_Y = TEKHOME;
  tw->tek.cur.fontsize = 0;  
  tw->tek.cur.x = 0;
  tw->tek.cur.y = 0;
  tw->tek.cur.linetype = 0;
  TekConfigure(tw);

  if (tw->tek.string == NULL) {
    tw->tek.string_pos = 0;
    tw->tek.string_length = 0;
  }

  if (tw->tek.file != NULL) {
    file = fopen(tw->tek.file, "r");
    if (file) {
      TekLoadFile(new, file);
      fclose(file);
    }
    tw->tek.file = NULL;
  }
}

static void TekRealize (w, value_mask, attributes)
     Widget w;
     XtValueMask *value_mask;
     XSetWindowAttributes *attributes;
{
  TekWidget tw = (TekWidget) w;
  XGCValues gcv;
  int i;

  XtCreateWindow (w, InputOutput, CopyFromParent, 
		  *value_mask, attributes);

  gcv.graphics_exposures = TRUE;      /* default */
  gcv.font = tw->tek.Tfont[tw->tek.cur.fontsize]->fid;
  gcv.foreground = tw->tek.foreground_pixel;
  gcv.background = tw->core.background_pixel;

  /* if font wasn't successfully opened, then gcv.font will contain
     the Default GC's ID, meaning that we must use the server default font.
  */   
  tw->tek.TnormalGC = XCreateGC (XtDisplay(tw), XtWindow(tw),
				 (GCFont|GCGraphicsExposures|
				  GCForeground|GCBackground), &gcv);
 
  gcv.foreground = tw->tek.foreground_pixel;
  gcv.line_style = LineOnOffDash;

  for (i = 0 ; i < TEKNUMLINES ; i++) {
    tw->tek.linepat[i] = XCreateGC (XtDisplay(tw), XtWindow(tw),
				    (GCForeground|GCLineStyle), &gcv);
    XSetDashes (XtDisplay(tw), tw->tek.linepat[i], 0,
		(char *) dashes[i], dash_length[i]);
  }

  tw->tek.cur_X = 0;
  tw->tek.cur_Y = TEKHOME;
}


void TekSetFontSize (tw, newitem)
    TekWidget tw;
    int newitem;
{
    int oldsize = tw->tek.cur.fontsize;
    int newsize = newitem;
    
    if (oldsize == newsize) return;
    TCursorToggle(tw, TOGGLE);
    TekSetGCFont (tw, newsize);
    tw->tek.cur.fontsize = newsize;
    TCursorToggle(tw, TOGGLE);
}

/*
 * Toggles cursor on or off at cursor position in screen.
 */
static void TCursorToggle(tw, toggle)
     TekWidget tw;
     int toggle;
{
  register int c, x, y;
  unsigned int cellwidth, cellheight;

  c = tw->tek.cur.fontsize;
  cellwidth = (unsigned) tw->tek.Tfont[c]->max_bounds.width;
  cellheight = (unsigned) (tw->tek.Tfont[c]->ascent + 
			   tw->tek.Tfont[c]->descent);

  x = tw->tek.cur_X * tw->tek.scale + tw->core.border_width;
  y = (tw->tek.height + TEKTOPPAD - tw->tek.cur_Y) * tw->tek.scale +
    tw->core.border_width - tw->tek.tobaseline[c];
  if (toggle)
       XFillRectangle(XtDisplay(tw), XtWindow(tw), tw->tek.TcursorGC,
		      x, y, cellwidth, cellheight);
  else 
      XClearArea(XtDisplay(tw), XtWindow(tw), x, y,
		 cellwidth, cellheight, FALSE);
}

void TekSimulatePageButton (tw, reset)
     TekWidget tw;
     Bool reset;
{

    if (reset)
      bzero ((char *) &tw->tek.cur, sizeof tw->tek.cur);

    TekPage (tw);
    tw->tek.cur_X = 0;
    tw->tek.cur_Y = TEKHOME;
}

static char TekGetChar(tw)
     TekWidget tw;
{
  if (tw->tek.string_pos >= 0 && 
      tw->tek.string_pos < tw->tek.string_length) {
    return (tw->tek.string[tw->tek.string_pos++]);
  } else {
    tw->tek.string_done = True;
    return(0);
  }
}

static void TekUnputChar(tw, c)
     TekWidget tw;
     char c;
{
  tw->tek.string_pos--;
  tw->tek.string[tw->tek.string_pos] = c;
}

static Boolean TekSetValues(current, request, new) 
     TekWidget current;
     TekWidget request;
     TekWidget new;
{
  FILE *file;
  if (current->tek.string != new->tek.string) {
    if (current->tek.string)
      XtFree(current->tek.string);
    new->tek.string = XtNewString(new->tek.string);
    new->tek.string_length = strlen(new->tek.string);
  }

  if (new->tek.file != NULL) {
    file = fopen(new->tek.file, "r");
    if (file) {
      TekLoadFile(new, file);
      fclose(file);
    }
    new->tek.file = NULL;
  }
    
  return(True);
}

static void TekLoadFile(w, file)
     Widget w;
     FILE *file;
{
  unsigned fsze;
  struct stat fstats;
  TekWidget tw = (TekWidget) w;

  fstat(fileno(file), &fstats);

  fsze = fstats.st_size;
  if (tw->tek.string)
    XtFree(tw->tek.string);
  tw->tek.string = (char*) XtMalloc(fsze);
  fread(tw->tek.string, 1, fsze, file);

  tw->tek.string_length = strlen(tw->tek.string);
  if (XtIsRealized(tw))
    XClearArea(XtDisplay(tw), XtWindow(tw), 0, 0, 0, 0, True);
}


