/*
 * gznola.c: A GNOME-based xznola
 * David Maze <dmaze@mit.edu>
 * $Id: gznola.c,v 1.5 2000/01/09 21:52:16 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zgroup.h"
#include "anyone.h"
#include "gtkanyone.h"
#include "gnome.h"

/* Some global constants */
#define MYNAME "gznola"

/* Open a .anyone file, and display a tree for it in the main window. */
void znol_open(GString *file);

/* Close and then re-open the current file. */
void znol_reopen(void);

/* Update all of the users in the current file. */
void znol_update(void);

/* Menu item callbacks */
void file_open(gpointer data);
void file_reopen(gpointer data);
void file_update(gpointer data);
void file_preferences(gpointer data);
void file_exit(gpointer data);

void help_about(gpointer data);

/* Definition of the main menu */
static GnomeUIInfo menu_file[] = {
  GNOMEUIINFO_MENU_OPEN_ITEM(file_open, NULL),
  GNOMEUIINFO_MENU_REVERT_ITEM(file_reopen, NULL),
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Update"),
    N_("Find out which users are logged in currently"),
    file_update, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
    GDK_U, GDK_CONTROL_MASK, NULL },
  { GNOME_APP_UI_ITEM, N_("_Locate"),
    N_("Locate a particular user"),
    NULL, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH,
    GDK_L, GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PREFERENCES_ITEM(file_preferences, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM(file_exit, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_help[] = {
  GNOMEUIINFO_HELP(MYNAME),
  GNOMEUIINFO_MENU_ABOUT_ITEM(help_about, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_main[] = {
  GNOMEUIINFO_MENU_FILE_TREE(menu_file),
  GNOMEUIINFO_MENU_HELP_TREE(menu_help),
  GNOMEUIINFO_END
};

/* Definition of the toolbar */
static GnomeUIInfo toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Update"),
			 N_("Find out which users are logged in currently"),
			 file_update,
			 GNOME_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_ITEM_STOCK(N_("Locate"),
			 N_("Locate a particular user"),
			 NULL /* file_locate */,
			 GNOME_STOCK_PIXMAP_SEARCH),
  GNOMEUIINFO_ITEM_STOCK(N_("Revert"),
			 N_("Re-open the current file"),
			 file_reopen,
			 GNOME_STOCK_PIXMAP_REVERT),
  GNOMEUIINFO_END
};

static GtkWidget *app, *appbar;
static GtkAnyone *anyone;

int main(int argc, char **argv)
{
  GString *file;

  ZInitialize();
  gnome_init(MYNAME, VERSION, argc, argv);

  app = gnome_app_new(MYNAME, MYNAME);
  gnome_app_create_menus(GNOME_APP(app), menu_main);
  gnome_app_create_toolbar(GNOME_APP(app), toolbar);
  appbar = gnome_appbar_new(TRUE, FALSE, GNOME_PREFERENCES_NEVER);
  gnome_app_set_statusbar(GNOME_APP(app), appbar);
  /* Let the window shrink itself */
  gtk_window_set_policy(GTK_WINDOW(app), TRUE, TRUE, TRUE);
  gtk_widget_show(app);

  /* Open the default .anyone file. */
  anyone = NULL;
  file = default_anyone();
  znol_open(file);
  g_string_free(file, TRUE);

  znol_update();
  
  gtk_main();
  return 0;
}

/**
 * znol_open - open a named file
 * @file: GString containing the name of the file to open
 *
 * znol_open() tries to create a &GtkAnyone widget, which reads in
 * @file.  If it fails, nothing happens; otherwise, the existing
 * &GtkAnyone structure is destroyed and replaced with the new one.
 */
void znol_open(GString *file)
{
  GtkWidget *my_anyone = gtk_anyone_open(file);
  GtkAnyone *last_anyone = anyone;
  
  /* If the open failed, don't do anything; just leave the existing UI up. */
  if (!my_anyone)
    {
      g_message("Couldn't open file %s", file->str);
      return;
    }
  
  /* Otherwise close the old UI and set up the new one. */
  anyone = GTK_ANYONE(my_anyone);
  gnome_app_set_contents(GNOME_APP(app), my_anyone);
  gtk_widget_show(my_anyone);

  /* Get rid of the old UI, if appropriate. */
  if (last_anyone)
    gtk_widget_destroy(GTK_WIDGET(last_anyone));
}

/**
 * znol_reopen - reopen the current file
 *
 * znol_reopen() creates a new &gtk_anyone widget for the same file as
 * the current one, as in znol_open().  This effectively causes the
 * current file to be reread.
 */
void znol_reopen(void)
{
  GString *file;
  
  /* This just shouldn't work without an open file. */
  g_assert(anyone != NULL);
  
  file = gtk_anyone_get_filename(anyone);
  znol_open(file);
  g_string_free(file, TRUE);
}

/**
 * znol_update - update all users
 *
 * znol_update() attempts to locate all of the users listed in the current
 * &GtkAnyone object.  The visible user interface is updated accordingly.
 */
void znol_update(void)
{
  if (anyone)
    gtk_anyone_update(anyone, gnome_appbar_get_progress(GNOME_APPBAR(appbar)));
}

/**
 * file_open - File/Open menu callback
 * @data: unused
 */
void file_open(gpointer data)
{
}

/**
 * file_reopen - File/Revert menu callback
 * @data: unused
 */
void file_reopen(gpointer data)
{
  znol_reopen();
}

/**
 * file_update - File/Update menu callback
 * @data: unused
 */
void file_update(gpointer data)
{
  znol_update();
}

/**
 * file_preferences - File/Preferences menu callback
 * @data: unused
 */
void file_preferences(gpointer data)
{
}

/**
 * file_exit - File/Exit menu callback
 * @data: unused
 */
void file_exit(gpointer data)
{
  gtk_exit(0);
}

/**
 * help_about - Help/About menu callback
 * @data: unused
 */
void help_about(gpointer data)
{
  GtkWidget *widget;
  
  static const char *authors[] = { "David Maze <dmaze@mit.edu>", NULL };
  
  widget = gnome_about_new(MYNAME, VERSION,
			   "(C) 1999, David Z. Maze", authors,
			   "Displays a heirarchical tree of users "
			   "visible over Zephyr.", NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(widget), GTK_WINDOW(app));
  gtk_widget_show(widget);
  gnome_dialog_run_and_close(GNOME_DIALOG(widget));
}
