/*
 * zgroup.c: Describe groups of users
 * David Maze <dmaze@mit.edu>
 * $Id: zgroup.c,v 1.3 1999/10/05 14:47:31 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zgroup.h"

struct _zgroup
{
  GString *name;
  GString *label;
  GList *groups; /* List of child zgroups */
  GList *users; /* List of GStrings */
  zusertable *table;
  int expanded;
};

static gint g_string_unequal(gconstpointer v, gconstpointer v2)
/* requires: v and v2 are GStrings
 *  effects: Runs g_str_equal() on the string parts of v and v2 */
{
  return !g_str_equal(((GString *)v)->str, ((GString *)v2)->str);
}

zgroup *zgroup_new(const GString *name, const GString *label,
		   zusertable *table)
{
  zgroup *group = g_new(zgroup, 1);
  group->name = g_string_new(name->str);
  group->label = g_string_new(label->str);
  group->groups = NULL; /* Equivalent to an empty list */
  group->users = NULL;
  group->table = table;
  group->expanded = TRUE;
  return group;
}

zgroup *zgroup_new_child(zgroup *group,
			 const GString *name, const GString *label)
{
  zgroup *child = zgroup_new(name, label, group->table);
  zgroup_append_group(group, child);
  return child;
}

GString *zgroup_name(const zgroup *group)
{
  g_assert(group != NULL);
  g_assert(group->name != NULL);
  return g_string_new(group->name->str);
}

GString *zgroup_label(const zgroup *group)
{
  g_assert(group != NULL);
  g_assert(group->label != NULL);
  return g_string_new(group->label->str);
}

zusertable *zgroup_table(const zgroup *group)
{
  return group->table;
}

GList *zgroup_groups(zgroup *group)
{
  g_assert(group != NULL);
  return group->groups;
}

GList *zgroup_usernames(zgroup *group)
{
  g_assert(group != NULL);
  return group->users;
}

GList *zgroup_users(const zgroup *group)
{
  GList *names, *users;
  zuser *user;
  
  g_assert(group != NULL);

  names = group->users;
  users = NULL;
  
  for (names = group->users; names; names = g_list_next(names))
  {
    user = zusertable_lookup(group->table, names->data);
    if (user)
      users = g_list_append(users, user);
    else
      g_error("Couldn't find zuser object for '%s'",
	      ((GString *)names->data)->str);
  }

  return users;
}

GList *zgroup_find_user(const zgroup *group, zuser *user)
{
  GList *rtn, *list = zgroup_users(group);
  
  rtn = g_list_find(list, user);
  if (!rtn)
    g_list_free(list);

  return rtn;
}

zgroup *zgroup_append_group(zgroup *group, zgroup *added)
{
  group->groups = g_list_append(group->groups, added);
  return added;
}

const GString *zgroup_append_username(zgroup *group, const GString *name)
{
  GString *myname;

  /* Check that they aren't in the user list already. */
  if (g_list_find_custom(group->users, (gpointer)name, g_string_unequal))
    return name;

  /* Otherwise clone the name and add them to our user list. */
  myname = g_string_new(name->str);
  group->users = g_list_append(group->users, myname);

  /* Make sure they're in the table, too. */
  zusertable_add_new(group->table, myname);

  return name;
}

zuser *zgroup_append_user(zgroup *group, zuser *user)
{
  GString *myname = zuser_name(user);
  zgroup_append_username(group, myname);
  g_string_free(myname, TRUE);
  return user;
}

zgroup *zgroup_find(zgroup *group, const GString *name)
{
  int i;
  zgroup *g;

  if (g_str_equal(group->name->str, name->str))
    return group;
  
  for (i = 0; i < g_list_length(group->groups); i++)
  {
    g = zgroup_find((zgroup *)g_list_nth_data(group->groups, i), name);
    if (g)
      return g;
  }
  
  return NULL;
}

int zgroup_set_expanded(zgroup *group, int expanded)
{
  int old = group->expanded;
  group->expanded = expanded;
  return old;
}

int zgroup_get_expanded(const zgroup *group)
{
  return group->expanded;
}
