/*
 * zusertable.c: A table mapping usernames to user objects
 * David Maze <dmaze@mit.edu>
 * $Id: zusertable.c,v 1.3 1999/10/05 14:47:57 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zusertable.h"

struct _zusertable
{
  GHashTable *hash;
};

struct iterdata
{
  zuseriter iter;
  gpointer data;
};

static void cleanup(gpointer key, gpointer value, gpointer user_data);
static void iteriter(gpointer key, gpointer value, gpointer user_data);

zusertable *zusertable_new()
{
  zusertable *table = g_new(zusertable, 1);
  /* hash is a string-keyed hash table. */
  table->hash = g_hash_table_new(g_str_hash, g_str_equal);
  return table;
}

void zusertable_delete(zusertable *table)
{
  /* Ask the hash table to have us delete everything stored in it. */
  g_hash_table_foreach(table->hash, cleanup, table);
  
  /* Now burn the hash table. */
  g_hash_table_destroy(table->hash);

  /* Finally, free the object state. */
  g_free(table);
}

static void cleanup(gpointer key, gpointer value, gpointer user_data)
{
  gchar *skey = key;
  zuser *user = value;
  zusertable *table = user_data;
  
  /* The zpointer object owns all of the keys and values, so they should
   * both be freed before the object itself goes away.  First, remove the
   * object from the hash table: */
  g_hash_table_remove(table->hash, skey);
  
  /* Now free the memory the table object was using. */
  g_free(skey);
  zuser_delete(user);
}

zuser *zusertable_add(zusertable *table, GString *name, zuser *user)
{
  GString *newname;
  zuser *old;
  
  old = g_hash_table_lookup(table->hash, name->str);
  if (old)
  {
    /* Something's already in the table; let it override the new entry. */
    if (user != old) /* If it's different, delete the new entry. */
      zuser_delete(user);
    return old;
  }
  
  newname = g_string_new(name->str);
  g_hash_table_insert(table->hash, newname->str, user);
  g_string_free(newname, FALSE);
  
  return user;
}

zuser *zusertable_add_new(zusertable *table, GString *name)
{
  zuser *user = zuser_new(name);
  return zusertable_add(table, name, user);
}

GString *zusertable_remove(zusertable *table, GString *name)
{
  zuser *user;
  GString *realname;

  if(g_hash_table_lookup_extended(table->hash, name->str,
				  (gpointer *)&realname, (gpointer *)&user))
  {
    g_hash_table_remove(table->hash, name);
    g_string_free(realname, TRUE);
    zuser_delete(user);
  }
    
  return name;
}

zuser *zusertable_lookup(zusertable *table, GString *name)
{
  return g_hash_table_lookup(table->hash, name->str);
}

void zusertable_foreach(zusertable *table, zuseriter iter, gpointer data)
{
  struct iterdata idata;
  idata.iter = iter;
  idata.data = data;
  g_hash_table_foreach(table->hash, iteriter, &idata);
}

static void iteriter(gpointer key, gpointer value, gpointer data)
{
  zuser *user = value;
  struct iterdata *idata = data;
  (idata->iter)(user, idata->data);
}

int zusertable_count(const zusertable *table)
{
  return g_hash_table_size(table->hash);
}
