/*
 * gtk_anyone.c: Show the user tree as a Gtk+ widget
 * David Maze <dmaze@mit.edu>
 * $Id: gtkanyone.c,v 1.13 2000/04/20 18:11:14 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gtkanyone.h"
#include "anyone.h"

/* Local function prototypes */
static void gtk_anyone_class_init(GtkAnyoneClass *klass);
static void gtk_anyone_init(GtkAnyone *anyone);

/* Populating the object
 * 
 * build_children() is called by gtk_anyone_construct(); it adds every
 * possible GtkTreeItem for both member users and child groups to this.
 * gtk_anyone_show_user() and gtk_anyone_hide_user() show and hide
 * relevant children.  This implementation may be changed, mostly because
 * of Gtk sucking (the tree looks wrong if the last visible item is not the
 * last item). */
static void build_children(GtkAnyone *anyone);
static void show_user_callback(GtkWidget *child, zuser *target);
static void show_the_user(GtkWidget *child, zuser *user);
static void hide_user_callback(GtkWidget *child, zuser *target);

/* Get some help with updating */
struct update_iter_data
{
  GtkAnyone *anyone;
  GtkProgress *progress;
  int count;
  int total;
};
static void update_iter(zuser *user, gpointer data);

static void gtk_anyone_class_init(GtkAnyoneClass *klass)
{
  GtkWidget *item;
  
  /* Create the popup menu (once). */
  klass->popup_menu = gtk_menu_new();
  
  /* Create the update item and add it. */
  item = gtk_menu_item_new_with_label("Update");
  gtk_menu_append(GTK_MENU(klass->popup_menu), item);
}

static void gtk_anyone_init(GtkAnyone *anyone)
{
  anyone->groupmap = NULL;
  anyone->group = NULL;
  anyone->filename = NULL;
  anyone->children = NULL;
  anyone->tips = NULL;
}

GtkType gtk_anyone_get_type(void)
{
  static GtkType anyone_type = 0;
  
  if (!anyone_type)
  {
    static const GtkTypeInfo anyone_info =
    {
      "GtkAnyone",
      sizeof(GtkAnyone),
      sizeof(GtkAnyoneClass),
      (GtkClassInitFunc) gtk_anyone_class_init,
      (GtkObjectInitFunc) gtk_anyone_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL
    };
    
    anyone_type = gtk_type_unique(gtk_tree_get_type(), &anyone_info);  
  }

  return anyone_type;
}

GtkWidget *gtk_anyone_new(zgroup *group)
{
  GtkWidget *anyone;
  
  g_return_val_if_fail(group != NULL, NULL);
  
  anyone = gtk_type_new(gtk_anyone_get_type());
  gtk_anyone_construct(GTK_ANYONE(anyone), group, NULL);
  return anyone;
}

GtkWidget *gtk_anyone_open(GString *filename)
{
  GtkWidget *anyone;
  zgroup *group;

  g_return_val_if_fail(filename != NULL, NULL);

  group = read_anyone(filename);
  if (!group)
    return NULL;
  
  anyone = gtk_type_new(gtk_anyone_get_type());
  gtk_anyone_construct(GTK_ANYONE(anyone), group, filename);
  return anyone;
}

void gtk_anyone_construct(GtkAnyone *anyone, zgroup *group, GString *filename)
{
  g_return_if_fail(anyone != NULL);
  g_return_if_fail(group != NULL);
  /* filename can be NULL */

  anyone->group = group;
  if (filename)
    anyone->filename = g_string_new(filename->str);
  else
    anyone->filename = NULL;
  anyone->tips = gtk_tooltips_new();

  build_children(anyone);
}

static void build_children(GtkAnyone *anyone)
{
  GList *list, *base;
  GString *name;
  GtkWidget *child;
  GtkWidget *subtree;
  zgroup *group;
  zuser *user;

  g_return_if_fail(anyone != NULL);

  /* Add all of the group's users. */
  base = zgroup_users(anyone->group);
  for (list = base; list; list = g_list_next(list))
  {
    /* Get the child user object... */
    user = list->data;
    
    /* Build a GtkTreeItem for it... */
    name = zuser_name(user);
    child = gtk_tree_item_new_with_label(name->str);
    g_string_free(name, TRUE);

    /* Set its user data to the relevant zuser... */
    gtk_object_set_user_data(GTK_OBJECT(child), user);
    
    /* Add the tree item to the tree. */
    gtk_tree_append(GTK_TREE(anyone), child);
  }

  /* Recursively add all of the group's child groups. */
  base = zgroup_groups(anyone->group);
  for (list = base; list; list = g_list_next(list))
  {
    /* Build the child group object... */
    group = list->data;
    subtree = gtk_anyone_new(group);

    /* Build a GtkTreeItem for it... */
    name = zgroup_label(group);
    child = gtk_tree_item_new_with_label(name->str);
    g_string_free(name, TRUE);

    /* Have to attach the child to the parent before we tweak things
     * like subtrees.  Remember, kids, documentation is for wusses, and
     * invariant checking doubly so.  *sigh* */
    gtk_tree_append(GTK_TREE(anyone), child);

    /* Make sure we show the children. */
    gtk_widget_show(GTK_WIDGET(child));
    gtk_widget_show(GTK_WIDGET(subtree));

    /* Attach the two, and deal with expansion and such. */
    gtk_tree_item_set_subtree(GTK_TREE_ITEM(child), subtree);
    if (zgroup_get_expanded(group))
      gtk_tree_item_expand(GTK_TREE_ITEM(child));
    else
      gtk_tree_item_collapse(GTK_TREE_ITEM(child));

    /* Save the child in our list of children. */
    anyone->children = g_list_append(anyone->children, subtree);
  }
}

zgroup *gtk_anyone_group(const GtkAnyone *anyone)
{
  return anyone->group;
}

GString *gtk_anyone_get_filename(const GtkAnyone *anyone)
{
  if (!anyone->filename)
    return NULL;
  return g_string_new(anyone->filename->str);
}

void gtk_anyone_update(GtkAnyone *anyone, GtkProgress *progress)
{
  struct update_iter_data uidata;
  zusertable *table;
  
  /* Assume the entire tree uses the same user table. */
  table = zgroup_table(anyone->group);
  
  /* Ask the table to update all of our users for us. */
  uidata.anyone = anyone;
  uidata.progress = progress;
  uidata.count = 0;
  uidata.total = zusertable_count(table);
  zusertable_foreach(table, update_iter, &uidata);
}

static void update_iter(zuser *user, gpointer data)
{
  struct update_iter_data *uidata = data;

  /* Figure out where they are. */
  zuser_update(user);

  /* Show or hide the user widget as appropriate. */
  gtk_anyone_show_or_hide_user(uidata->anyone, user);

  /* Update the progress bar. */
  uidata->count++;
  if (uidata->progress)
  {
    gtk_progress_set_percentage(uidata->progress, 
				((gfloat)uidata->count)/uidata->total);
    while (gtk_events_pending())
      gtk_main_iteration();
  }
}

void gtk_anyone_update_user(GtkAnyone *anyone, zuser *user)
{
  /* If this isn't a root tree, and we have a root tree that is a GtkAnyone,
   * do the update there instead. */
  if (!GTK_IS_ROOT_TREE(anyone) && IS_GTK_ANYONE(GTK_TREE_ROOT_TREE(anyone)))
  {
    gtk_anyone_update_user(GTK_ANYONE(GTK_TREE_ROOT_TREE(anyone)), user);
    return;
  }

  /* Update the user and show them. */
  zuser_update(user);
  gtk_anyone_show_or_hide_user(anyone, user);
}

void gtk_anyone_show_or_hide_user(GtkAnyone *anyone, zuser *user)
{
  if (zuser_count_locs(user) > 0)
    gtk_anyone_show_user(anyone, user);
  else
    gtk_anyone_hide_user(anyone, user);
}

void gtk_anyone_show_user(GtkAnyone *anyone, zuser *user)
{
  GList *base, *list;
  
  /* Use a callback to conditionally show each user. */
  gtk_container_foreach(GTK_CONTAINER(anyone),
			(GtkCallback)show_user_callback,
			user);

  /* Recurse into subgroups. */
  base = anyone->children;
  for (list = base; list; list = g_list_next(list))
    gtk_anyone_show_user(GTK_ANYONE(list->data), user);
}

static void show_user_callback(GtkWidget *child, zuser *target)
{
  zuser *user;
  
  /* Get the user from the child's user data. */
  user = gtk_object_get_user_data(GTK_OBJECT(child));
  
  /* If it matches, show the child. */
  if (user == target)
    show_the_user(child, user);
}

static void show_the_user(GtkWidget *child, zuser *user)
{
  const ZLocations_t *loc;
  int n;
  GString *tip;
  GtkWidget *parent;
  GtkAnyone *anyone;
  
  /* Get the parent of child.  This had better be a GtkAnyone... */
  parent = child->parent; /* Abstraction, anyone? */
  g_return_if_fail(parent != NULL);
  anyone = GTK_ANYONE(parent);
  
  /* Show the widget. */
  gtk_widget_show(child);

  /* While we're at it, set a tooltip for the child widget. */
  tip = zuser_name(user);
  g_string_append(tip, "\n");
  
  /* Also pull the locations out of the user object. */
  for (n = 0; n < zuser_count_locs(user); n++)
  {
    loc = zuser_get_loc(user, n);
    g_string_sprintfa(tip, "%s (%s): %s\n", loc->host, loc->tty, loc->time);
  }

  /* Set the tooltip. */
  gtk_tooltips_set_tip(anyone->tips, child, tip->str, tip->str);
  g_string_free(tip, TRUE);
}

void gtk_anyone_hide_user(GtkAnyone *anyone, zuser *user)
{
  GList *base, *list;
  
  /* Use a callback to conditionally hide each user. */
  gtk_container_foreach(GTK_CONTAINER(anyone),
			(GtkCallback)hide_user_callback,
			user);

  /* Recurse into subgroups. */
  base = anyone->children;
  for (list = base; list; list = g_list_next(list))
    gtk_anyone_hide_user(GTK_ANYONE(list->data), user);
}

static void hide_user_callback(GtkWidget *child, zuser *target)
{
  zuser *user;
  
  /* Get the user from the child's user data. */
  user = gtk_object_get_user_data(GTK_OBJECT(child));
  
  /* If it matches, hide the child. */
  if (user == target)
    gtk_widget_hide(child);
}

