/*
 * gznola.c: A GNOME-based xznola
 * David Maze <dmaze@mit.edu>
 * $Id: gznola.c,v 1.9 2000/04/18 20:01:12 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zgroup.h"
#include "anyone.h"
#include "gtkanyone.h"
#include "gtksaneprogressbar.h"

/* Deal with bizarre name conflicts between des.h and GNOME. */
#undef _
#include "gnome.h"

/* Some global constants */
#define MYNAME "gznola"

/* Open a .anyone file, and display a tree for it in the main window. */
void znol_open(GString *file);

/* Close and then re-open the current file. */
void znol_reopen(void);

/* Update all of the users in the current file. */
void znol_update(void);

/* Menu item callbacks */
void file_open(gpointer data);
void file_reopen(gpointer data);
void file_update(gpointer data);
void file_locate(gpointer data);
void file_preferences(gpointer data);
void file_exit(gpointer data);

void help_about(gpointer data);

void do_file_open(gpointer data);
void do_file_locate(gchar *string, gpointer data);

/* Definition of the main menu */
static GnomeUIInfo menu_file[] = {
  GNOMEUIINFO_MENU_OPEN_ITEM(file_open, NULL),
  GNOMEUIINFO_MENU_REVERT_ITEM(file_reopen, NULL),
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Update"),
    N_("Find out which users are logged in currently"),
    file_update, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
    GDK_U, GDK_CONTROL_MASK, NULL },
  { GNOME_APP_UI_ITEM, N_("_Locate"),
    N_("Locate a particular user"),
    file_locate, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH,
    GDK_L, GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PREFERENCES_ITEM(file_preferences, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM(file_exit, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_help[] = {
  GNOMEUIINFO_HELP(MYNAME),
  GNOMEUIINFO_MENU_ABOUT_ITEM(help_about, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_main[] = {
  GNOMEUIINFO_MENU_FILE_TREE(menu_file),
  GNOMEUIINFO_MENU_HELP_TREE(menu_help),
  GNOMEUIINFO_END
};

/* Definition of the toolbar */
static GnomeUIInfo toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Update"),
			 N_("Find out which users are logged in currently"),
			 file_update,
			 GNOME_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_ITEM_STOCK(N_("Locate"),
			 N_("Locate a particular user"),
			 file_locate,
			 GNOME_STOCK_PIXMAP_SEARCH),
  GNOMEUIINFO_ITEM_STOCK(N_("Revert"),
			 N_("Re-open the current file"),
			 file_reopen,
			 GNOME_STOCK_PIXMAP_REVERT),
  GNOMEUIINFO_END
};

static GtkWidget *app, *progbar;
static GtkAnyone *anyone;

int main(int argc, char **argv)
{
  GString *file;

  ZInitialize();
  gnome_init(MYNAME, VERSION, argc, argv);

  app = gnome_app_new(MYNAME, MYNAME);
  gnome_app_create_menus(GNOME_APP(app), menu_main);
  gnome_app_create_toolbar(GNOME_APP(app), toolbar);
  progbar = gtk_sane_progress_bar_new();
  gnome_app_set_statusbar(GNOME_APP(app), progbar);
  /* Let the window shrink itself */
  gtk_window_set_policy(GTK_WINDOW(app), TRUE, TRUE, TRUE);
  gtk_widget_show(app);

  /* Open the default .anyone file. */
  anyone = NULL;
  file = default_anyone();
  znol_open(file);
  g_string_free(file, TRUE);

  znol_update();
  
  gtk_main();
  return 0;
}

/**
 * znol_open - open a named file
 * @file: GString containing the name of the file to open
 *
 * znol_open() tries to create a &GtkAnyone widget, which reads in
 * @file.  If it fails, nothing happens; otherwise, the existing
 * &GtkAnyone structure is destroyed and replaced with the new one.
 */
void znol_open(GString *file)
{
  GtkWidget *my_anyone = gtk_anyone_open(file);
  
  /* If the open failed, don't do anything; just leave the existing UI up. */
  if (!my_anyone)
    {
      g_message("Couldn't open file %s", file->str);
      return;
    }
  
  /* Otherwise close the old UI and set up the new one. */
  anyone = GTK_ANYONE(my_anyone);
  gnome_app_set_contents(GNOME_APP(app), my_anyone);
  gtk_widget_show(my_anyone);
}

/**
 * znol_reopen - reopen the current file
 *
 * znol_reopen() creates a new &gtk_anyone widget for the same file as
 * the current one, as in znol_open().  This effectively causes the
 * current file to be reread.
 */
void znol_reopen(void)
{
  GString *file;
  
  /* This just shouldn't work without an open file. */
  g_assert(anyone != NULL);
  
  file = gtk_anyone_get_filename(anyone);
  znol_open(file);
  g_string_free(file, TRUE);
}

/**
 * znol_update - update all users
 *
 * znol_update() attempts to locate all of the users listed in the current
 * &GtkAnyone object.  The visible user interface is updated accordingly.
 */
void znol_update(void)
{
  if (anyone)
    gtk_anyone_update(anyone, GTK_PROGRESS(progbar));
}

/**
 * file_open - File/Open menu callback
 * @data: unused
 */
void file_open(gpointer data)
{
  GtkWidget *selection;
  selection = gtk_file_selection_new("Open .anyone");
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(selection), ".anyone");
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->ok_button),
     "clicked", GTK_SIGNAL_FUNC(do_file_open), (gpointer)selection);
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->cancel_button),
     "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)selection);
  gtk_window_position(GTK_WINDOW(selection), GTK_WIN_POS_MOUSE);
  gtk_widget_show(selection);
}

/**
 * do_file_open - Open a file based on information from a file selection
 * @data: pointer to GtkFileSelection
 */
void do_file_open(gpointer data)
{
  GtkWidget *selection = data;
  gchar *filename =
    gtk_file_selection_get_filename(GTK_FILE_SELECTION(selection));
  GString *str = g_string_new(filename);
  znol_open(str);
  g_string_free(str, TRUE);
  znol_update();
}

/**
 * file_reopen - File/Revert menu callback
 * @data: unused
 */
void file_reopen(gpointer data)
{
  znol_reopen();
  znol_update();
}

/**
 * file_update - File/Update menu callback
 * @data: unused
 */
void file_update(gpointer data)
{
  znol_update();
}

/**
 * file_locate - File/Locate menu callback
 * @data: unused
 */
void file_locate(gpointer data)
{
  gnome_request_dialog(FALSE,
		       _("Locate which user?"),
		       NULL,
		       256,
		       do_file_locate,
		       NULL,
		       GTK_WINDOW(app));
}

/**
 * do_file_locate - Locate a user and display their location in a dialog
 * @string: User to locate
 * @data: unused
 */
void do_file_locate(gchar *string, gpointer data)
{
  zuser *user;
  GString *str = g_string_new(string);
  int n;
  const ZLocations_t *loc;

  /* Create and update a user object. */
  user = zuser_new(str);
  g_string_free(str, TRUE);
  zuser_update(user);

  /* Create a message string. */
  str = zuser_name(user);
  g_string_append(str, ":\n");
  
  /* If the user isn't there, indicate this. */
  if (zuser_count_locs(user) == 0)
  {
    g_string_append(str, "Not logged in (or no such user)");
  }
  /* Otherwise, construct the message string. */
  else
  {
    for (n = 0; n < zuser_count_locs(user); n++)
    {
      loc = zuser_get_loc(user, n);
      g_string_sprintfa(str, "%s (%s): %s\n", loc->host, loc->tty, loc->time);
    }
  }

  /* Display the message. */
  gnome_ok_dialog_parented(str->str, GTK_WINDOW(app));
  
  /* Clean up. */
  g_string_free(str, TRUE);
}

/**
 * file_preferences - File/Preferences menu callback
 * @data: unused
 */
void file_preferences(gpointer data)
{
}

/**
 * file_exit - File/Exit menu callback
 * @data: unused
 */
void file_exit(gpointer data)
{
  gtk_exit(0);
}

/**
 * help_about - Help/About menu callback
 * @data: unused
 */
void help_about(gpointer data)
{
  GtkWidget *widget;
  
  static const char *authors[] = { "David Maze <dmaze@mit.edu>", NULL };
  
  widget = gnome_about_new(MYNAME, VERSION,
			   "(C) 1999-2000, David Z. Maze", authors,
			   "Displays a heirarchical tree of users "
			   "visible over Zephyr.\n"
			   "Project home page: "
			   "http://www.mit.edu/~dmaze/xznola/\n",
			   NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(widget), GTK_WINDOW(app));
  gtk_widget_show(widget);
  gnome_dialog_run_and_close(GNOME_DIALOG(widget));
}
