/*
 * gzw.c: A simple GNOME-based Zephyr window
 * David Maze <dmaze@mit.edu>'
 * $Id: gzw.c,v 1.1 2001/10/16 17:49:16 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zgroup.h"
#include "anyone.h"
#include "gtkanyone.h"

#undef _
#include "gnome.h"
#include "zvt/zvtterm.h"
#include <math.h>

/* Some global constants */
#define MYNAME "gzw"

struct gzw_settings
{
  /* Inferior settings */
  char *inferior;
  int set_wgfile;
  int exit_on_death;

  /* Terminal settings */
  int cursor_blinks;
  char *font_name;
  gushort color_red[18];
  gushort color_green[18];
  gushort color_blue[18];
} settings;

void znol_open(GString *file);
void znol_reopen(void);
void znol_update(void);

void read_settings(void);
void read_color(const char *name, int num, int r, int g, int b);
void write_settings(void);
void write_color(const char *name, int num);

void subs_to_clist(GtkWidget *clist);

void file_open(gpointer data);
void file_reopen(gpointer data);
void file_update(gpointer data);
void file_exit(gpointer data);

void settings_zephyr(gpointer data);
void settings_term(gpointer data);
void settings_term_take(gpointer data);
void settings_term_put(gpointer data);
void settings_term_apply(GnomePropertyBox *pbox, gint arg1, gpointer data);
void settings_inferior(gpointer data);
void settings_inferior_take(gpointer data);
void settings_inferior_put(gpointer data);
void settings_inferior_apply(GnomePropertyBox *pbox, gint arg1, gpointer data);
void settings_adjustment_integral(GtkAdjustment *object, gpointer data);
void settings_any_changed(GtkObject *object, gpointer data);
void settings_any_destroy(GtkObject *object, gpointer data);

void help_about(gpointer data);
void do_file_open(gpointer data);
void child_died_event(gpointer data);

/* Definition of the main menu */
static GnomeUIInfo menu_file[] = {
  GNOMEUIINFO_MENU_OPEN_ITEM(file_open, NULL),
  GNOMEUIINFO_MENU_REVERT_ITEM(file_reopen, NULL),
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Update"),
    N_("Find out which users are logged in currently"),
    file_update, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
    GDK_U, GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_MENU_EXIT_ITEM(file_exit, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_settings[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("_Zephyr..."),
			 N_("Change general Zephyr settings"),
			 settings_zephyr,
			 GNOME_STOCK_PIXMAP_JUMP_TO),
  GNOMEUIINFO_ITEM_STOCK(N_("_Terminal..."),
			 N_("Change terminal settings"),
			 settings_term,
			 GNOME_STOCK_PIXMAP_INDEX),
  GNOMEUIINFO_ITEM_STOCK(N_("_Inferior..."),
			 N_("Change the program run inside " MYNAME),
			 settings_inferior,
			 GNOME_STOCK_PIXMAP_EXEC),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_help[] = {
  GNOMEUIINFO_HELP(MYNAME),
  GNOMEUIINFO_MENU_ABOUT_ITEM(help_about, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_main[] = {
  GNOMEUIINFO_MENU_FILE_TREE(menu_file),
  GNOMEUIINFO_MENU_SETTINGS_TREE(menu_settings),
  GNOMEUIINFO_MENU_HELP_TREE(menu_help),
  GNOMEUIINFO_END
};

/* Definition of the toolbar */
static GnomeUIInfo toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Update"),
			 N_("Find out which users are logged in currently"),
			 file_update,
			 GNOME_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_ITEM_STOCK(N_("Revert"),
			 N_("Re-open the current file"),
			 file_reopen,
			 GNOME_STOCK_PIXMAP_REVERT),
  GNOMEUIINFO_END
};

static GtkWidget *app, *zvt, *frame, *anyone, *appbar, *progbar;

int main(int argc, char **argv)
{
  GString *file;
  GtkWidget *dockitem;
  GdkFont *font;

  ZInitialize();
  gnome_init(MYNAME, VERSION, argc, argv);
  read_settings();

  app = gnome_app_new(MYNAME, MYNAME);
  gnome_app_create_menus(GNOME_APP(app), menu_main);
  gnome_app_create_toolbar(GNOME_APP(app), toolbar);
  appbar = gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_USER);
  progbar = GTK_WIDGET(gnome_appbar_get_progress(GNOME_APPBAR(appbar)));
  gnome_app_set_statusbar(GNOME_APP(app), appbar);
  gnome_app_install_menu_hints(GNOME_APP(app), menu_main);
  
  zvt = zvt_term_new_with_size(132, 50);
  zvt_term_set_color_scheme(ZVT_TERM(zvt), settings.color_red,
			    settings.color_green, settings.color_blue);
  font = gdk_font_load(settings.font_name);
  zvt_term_set_fonts(ZVT_TERM(zvt), font, NULL);
  gtk_widget_show(zvt);

  // Start the inferior process now.
  if (zvt_term_forkpty(ZVT_TERM(zvt), FALSE) == 0)
    {
      if (settings.set_wgfile)
	{
	  /* Yes, there is a possible race condition here: a malicious
	   * user could guess the name for WGFILE and create it in
	   * between calling tempnam() and zwgc starting. */
          char *wgname = tempnam(NULL, "wg.");
	  gchar *wgenv = g_strdup_printf("WGFILE=%s", wgname);
	  free(wgname);
	  putenv(wgenv);
	  /* Don't free wgenv, since some putenv() implementations require
	   * it to stick around */
	}
      execl(settings.inferior, MYNAME "-inferior", NULL);
      exit(0);
    }
  gtk_signal_connect(GTK_OBJECT(zvt), "child_died",
		     GTK_SIGNAL_FUNC(child_died_event), NULL);

  gnome_app_set_contents(GNOME_APP(app), zvt);

  // Put the GtkAnyone widget inside a dock item.
  dockitem = gnome_dock_item_new("GtkAnyone",
				 GNOME_DOCK_ITEM_BEH_NEVER_HORIZONTAL);
  gtk_widget_show(dockitem);
  gnome_app_add_dock_item(GNOME_APP(app), GNOME_DOCK_ITEM(dockitem),
			  GNOME_DOCK_RIGHT, 0, 0, 0);

  // Put the GtkAnyone widget inside a frame inside that.
  frame = gtk_frame_new(NULL);
  gtk_widget_show(frame);
  gtk_container_add(GTK_CONTAINER(dockitem), frame);

  // Open the default .anyone file. */
  anyone = NULL;
  file = default_anyone();
  znol_open(file);
  g_string_free(file, TRUE);

  gtk_widget_show(app);
  znol_update();

  gtk_main();

  zvt_term_killchild(ZVT_TERM(zvt), 15);
  return 0;
}

/**
 * znol_open - open a named file
 * @file: GString containing the name of the file to open
 *
 * znol_open() tries to create a &GtkAnyone widget, which reads in
 * @file.  If it fails, nothing happens; otherwise, the existing
 * &GtkAnyone structure is destroyed and replaced with the new one.
 */
void znol_open(GString *file)
{
  GtkWidget *my_anyone = gtk_anyone_open(file);
  
  /* If the open failed, don't do anything; just leave the existing UI up. */
  if (!my_anyone)
    {
      g_message("Couldn't open file %s", file->str);
      return;
    }
  
  /* Otherwise close the old UI and set up the new one. */
  if (anyone) gtk_container_remove(GTK_CONTAINER(frame), anyone);
  anyone = my_anyone;
  gtk_container_add(GTK_CONTAINER(frame), anyone);
  gtk_widget_show(anyone);
}

/**
 * znol_reopen - reopen the current file
 *
 * znol_reopen() creates a new &gtk_anyone widget for the same file as
 * the current one, as in znol_open().  This effectively causes the
 * current file to be reread.
 */
void znol_reopen(void)
{
  GString *file;
  
  /* This just shouldn't work without an open file. */
  g_assert(anyone != NULL);
  
  file = gtk_anyone_get_filename(GTK_ANYONE(anyone));
  znol_open(file);
  g_string_free(file, TRUE);
}

/**
 * znol_update - update all users
 *
 * znol_update() attempts to locate all of the users listed in the current
 * &GtkAnyone object.  The visible user interface is updated accordingly.
 */
void znol_update(void)
{
  if (anyone)
    gtk_anyone_update(GTK_ANYONE(anyone), GTK_PROGRESS(progbar));
}

/**
 * read_settings - get gzw_settings from the config file
 */
void read_settings(void)
{
  settings.inferior = gnome_config_get_string_with_default
    ("/" MYNAME "/settings/inferior=/bin/sh", NULL);
  settings.set_wgfile = gnome_config_get_int_with_default
    ("/" MYNAME "/settings/set_wgfile=1", NULL);
  settings.exit_on_death = gnome_config_get_int_with_default
    ("/" MYNAME "/settings/exit_on_death=0", NULL);

  settings.cursor_blinks = gnome_config_get_int
    ("/" MYNAME "/terminal/cursor_blinks=1");
  /* Use the XFree86 font name, again because infrastructure
     (here GnomeFontPicker) sucks. */
  settings.font_name = gnome_config_get_string
    ("/" MYNAME "/terminal/font_name=-misc-fixed-medium-r-semicondensed--13-120-75-75-c-60-iso8859-1");
  /* These are default colors out of my .Xresources file.
   * They aren't the ZvtTerm default colors, since there's no way
   * to get the current/default colors out of the widget.
   * TODO: Write an X terminal emulator that doesn't suck. */
  read_color("color0", 0, 0x0000, 0x0000, 0x0000); /* black */
  read_color("color1", 1, 0xCCCC, 0x0000, 0x0000); /* red3 */
  read_color("color2", 2, 0x0000, 0xCCCC, 0x0000); /* green3 */
  read_color("color3", 3, 0xCCCC, 0xCCCC, 0x0000); /* yellow3 */
  read_color("color4", 4, 0x0000, 0x0000, 0xCCCC); /* blue3 */
  read_color("color5", 5, 0xCCCC, 0x0000, 0xCCCC); /* magenta3 */
  read_color("color6", 6, 0x0000, 0xCCCC, 0xCCCC); /* cyan3 */
  read_color("color7", 7, 0xCCCC, 0xCCCC, 0xCCCC); /* was gray80 */
  read_color("color8", 8, 0x4444, 0x4444, 0x4444); /* gray30 */
  read_color("color9", 9, 0xFFFF, 0x0000, 0x0000); /* red */
  read_color("color10", 10, 0x0000, 0xFFFF, 0x0000); /* green */
  read_color("color11", 11, 0xFFFF, 0xFFFF, 0x0000); /* yellow */
  read_color("color12", 12, 0x0000, 0x0000, 0xFFFF); /* blue */
  read_color("color13", 13, 0xFFFF, 0x0000, 0xFFFF); /* magenta */
  read_color("color14", 14, 0x0000, 0xFFFF, 0xFFFF); /* cyan */
  read_color("color15", 15, 0xFFFF, 0xFFFF, 0xFFFF); /* was gray80 */
  /* Since I'm writing this, use my favored default colors (white-on-black). */
  read_color("foreground", 16, 0xCCCC, 0xCCCC, 0xCCCC);
  read_color("background", 17, 0x0000, 0x0000, 0x0000);
}

void read_color(const char *name, int num, int r, int g, int b)
{
  char *setting;

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_red=%d", name, r);
  settings.color_red[num] = gnome_config_get_int(setting);
  g_free(setting);

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_green=%d", name, g);
  settings.color_green[num] = gnome_config_get_int(setting);
  g_free(setting);

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_blue=%d", name, b);
  settings.color_blue[num] = gnome_config_get_int(setting);
  g_free(setting);
}

/**
 * write_settings - put gzw_settings into the config file
 */
void write_settings(void)
{
  gnome_config_set_string("/" MYNAME "/settings/inferior",
			  settings.inferior);
  gnome_config_set_int("/" MYNAME "/settings/set_wgfile",
		       settings.set_wgfile);
  gnome_config_set_int("/" MYNAME "/settings/exit_on_death",
		       settings.exit_on_death);

  gnome_config_set_int("/" MYNAME "/terminal/cursor_blinks",
		       settings.cursor_blinks);
  gnome_config_set_string("/" MYNAME "/terminal/font_name",
			  settings.font_name);
  write_color("color0", 0);
  write_color("color1", 1);
  write_color("color2", 2);
  write_color("color3", 3);
  write_color("color4", 4);
  write_color("color5", 5);
  write_color("color6", 6);
  write_color("color7", 7);
  write_color("color8", 8);
  write_color("color9", 9);
  write_color("color10", 10);
  write_color("color11", 11);
  write_color("color12", 12);
  write_color("color13", 13);
  write_color("color14", 14);
  write_color("color15", 15);
  write_color("foreground", 16);
  write_color("background", 17);
}

void write_color(const char *name, int num)
{
  char *setting;

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_red", name);
  gnome_config_set_int(setting, settings.color_red[num]);
  g_free(setting);

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_green", name);
  gnome_config_set_int(setting, settings.color_green[num]);
  g_free(setting);

  setting = g_strdup_printf("/" MYNAME "/terminal/%s_blue", name);
  gnome_config_set_int(setting, settings.color_blue[num]);
  g_free(setting);
}


/**
 * subs_to_clist - populate a GtkCList based on subscription info
 */
void subs_to_clist(GtkWidget *clist)
{
  unsigned short port;
  int i;
  Code_t result;
  ZSubscription_t sub;
  gchar *items[4];
  
  i = ZGetWGPort();
  if (i == -1)
    {
      gnome_app_error(GNOME_APP(app),
		      "Couldn't find a port for a Zephyr windowgram client; "
		      "check that zwgc or some other client is running.");
      return;
    }
  port = (unsigned short) i;
  result = ZRetrieveSubscriptions(port, &i);
  if (result != ZERR_NONE)
    {
      gnome_app_error(GNOME_APP(app),
		      "An error occurred when trying to retrieve your "
		      "Zephyr subscriptions.");
      return;
    }

  i = 1;
  while (ZGetSubscriptions(&sub, &i) == ZERR_NONE)
    {
      if (i == 0)
	break;
      items[0] = sub.zsub_class;
      items[1] = sub.zsub_classinst;
      items[2] = sub.zsub_recipient;
      items[3] = NULL;
      gtk_clist_append(GTK_CLIST(clist), items);
    }
  ZFlushSubscriptions();
}

/**
 * file_open - File/Open menu callback
 * @data: unused
 */
void file_open(gpointer data)
{
  GtkWidget *selection;
  selection = gtk_file_selection_new("Open .anyone");
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(selection), ".anyone");
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->ok_button),
     "clicked", GTK_SIGNAL_FUNC(do_file_open), (gpointer)selection);
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->cancel_button),
     "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)selection);
  gtk_window_position(GTK_WINDOW(selection), GTK_WIN_POS_MOUSE);
  gtk_widget_show(selection);
}

/**
 * do_file_open - Open a file based on information from a file selection
 * @data: pointer to GtkFileSelection
 */
void do_file_open(gpointer data)
{
  GtkWidget *selection = data;
  gchar *filename =
    gtk_file_selection_get_filename(GTK_FILE_SELECTION(selection));
  GString *str = g_string_new(filename);
  znol_open(str);
  g_string_free(str, TRUE);
  znol_update();
}

/**
 * file_reopen - File/Revert menu callback
 * @data: unused
 */
void file_reopen(gpointer data)
{
  znol_reopen();
  znol_update();
}

/**
 * file_update - File/Update menu callback
 * @data: unused
 */
void file_update(gpointer data)
{
  znol_update();
}

/**
 * file_exit - File/Exit menu callback
 * @data: unused
 */
void file_exit(gpointer data)
{
  gtk_exit(0);
}

/**
 * settings_zephyr - Settings/Zephyr menu callback
 * @data: unused
 */
void settings_zephyr(gpointer data)
{
  /* These settings are:
   * 0  GnomePropertyBox
   * 1  GtkVScale with the exposure setting
   * 2  GtkCList containing subscriptions
   */
  static GtkWidget *widgets[4];
  GtkObject *adjustment;
  GtkWidget *vbox, *hbox, *widget;

  if (!widgets[0])
    {
      widgets[0] = gnome_property_box_new();
      /* gtk_signal_connect(GTK_OBJECT(widgets[0]), "apply",
	 settings_inferior_apply, widgets); */
      gtk_signal_connect(GTK_OBJECT(widgets[0]), "destroy",
			 settings_any_destroy, widgets);

      hbox = gtk_hbox_new(FALSE, GNOME_PAD);
      adjustment = gtk_adjustment_new(4.0, 1.0, 6.0, 1.0, 1.0, 0.0);
      // Connect the signal *now*; then it will happen first, before
      // signals the vscale connects.
      gtk_signal_connect(GTK_OBJECT(adjustment), "value-changed",
			 settings_adjustment_integral, NULL);
      widgets[1] = gtk_vscale_new(GTK_ADJUSTMENT(adjustment));
      gtk_scale_set_draw_value(GTK_SCALE(widgets[1]), FALSE);
      gtk_widget_show(widgets[1]);
      gtk_box_pack_start(GTK_BOX(hbox), widgets[1], FALSE, FALSE, GNOME_PAD);

      vbox = gtk_vbox_new(TRUE, GNOME_PAD);
      
      widget = gtk_label_new("Completely silent");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      widget = gtk_label_new("Receive personals");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      widget = gtk_label_new("Visible to local realm");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      widget = gtk_label_new("Announced to local realm");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      widget = gtk_label_new("Visible to all realms");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      widget = gtk_label_new("Announced to all realms");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_LEFT);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      gtk_widget_show(vbox);
      gtk_box_pack_start_defaults(GTK_BOX(hbox), vbox);
      gtk_widget_show(hbox);

      gnome_property_box_append_page(GNOME_PROPERTY_BOX(widgets[0]),
				     hbox, gtk_label_new(N_("Exposure")));


      /* Moving along...a page for subscriptions. */
      vbox = gtk_vbox_new(FALSE, GNOME_PAD);

      widgets[2] = gtk_clist_new(3);
      gtk_clist_set_column_title(GTK_CLIST(widgets[2]), 0, "Class");
      gtk_clist_set_column_title(GTK_CLIST(widgets[2]), 1, "Instance");
      gtk_clist_set_column_title(GTK_CLIST(widgets[2]), 2, "Recipient");
      gtk_clist_set_column_auto_resize(GTK_CLIST(widgets[2]), 0, TRUE);
      gtk_clist_set_column_auto_resize(GTK_CLIST(widgets[2]), 1, TRUE);
      gtk_clist_set_column_auto_resize(GTK_CLIST(widgets[2]), 2, TRUE);
      gtk_clist_column_titles_show(GTK_CLIST(widgets[2]));
      gtk_widget_show(widgets[2]);

      widget = gtk_scrolled_window_new(NULL, NULL);
      gtk_container_add(GTK_CONTAINER(widget), widgets[2]);
      gtk_widget_show(widget);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widget);

      hbox = gtk_hbutton_box_new();
      widget = gnome_pixmap_button
	(gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_ADD), "Add");
      gtk_widget_show(widget);
      gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
      widget = gnome_pixmap_button
	(gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_REMOVE), "Remove");
      gtk_widget_show(widget);
      gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
      gtk_widget_show(hbox);
      gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

      gtk_widget_show(vbox);
      gnome_property_box_append_page(GNOME_PROPERTY_BOX(widgets[0]),
				     vbox, gtk_label_new(N_("Subscriptions")));
    }

  gtk_clist_clear(GTK_CLIST(widgets[2]));
  subs_to_clist(widgets[2]);

  gtk_widget_show(widgets[0]);
}

/**
 * settings_term - Settings/Terminal menu callback
 * @data: unused
 */
void settings_term(gpointer data)
{
  /* These settings are:
   * 0  GnomePropertyBox
   * 1  GnomeFontPicker with font name
   * 2..17 GnomeColorPickers with display colors
   * 18..19 GnomeColorPickers with foreground, background
   */
  static GtkWidget* widgets[20];
  GtkWidget *table, *widget;
  int i, n;
  
  /* If we haven't already created the propbox, do it now. */
  if (!widgets[0])
    {
      widgets[0] = gnome_property_box_new();
      gtk_signal_connect(GTK_OBJECT(widgets[0]), "apply",
			 settings_term_apply, widgets);
      gtk_signal_connect(GTK_OBJECT(widgets[0]), "destroy",
			 settings_any_destroy, widgets);
  
      /* Build up the page for Inferior settings. */
      table = gtk_table_new(6, 2, FALSE);
      gnome_property_box_append_page(GNOME_PROPERTY_BOX(widgets[0]),
				     table, gtk_label_new(N_("Terminal")));

      widget = gtk_label_new("Font:");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_RIGHT);
      gtk_table_attach(GTK_TABLE(table), widget, 0, 1, 0, 1,
		       GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);

      widgets[1] = gnome_font_picker_new();
      gnome_font_picker_set_mode(GNOME_FONT_PICKER(widgets[1]),
				 GNOME_FONT_PICKER_MODE_FONT_INFO);
      gtk_table_attach(GTK_TABLE(table), widgets[1], 1, 2, 0, 1,
		       GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL,
		       GNOME_PAD, GNOME_PAD);

      widget = gtk_label_new("Colors:");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_RIGHT);
      gtk_table_attach(GTK_TABLE(table), widget, 0, 1, 1, 2,
		       GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);

      widget = gtk_table_new(2, 8, TRUE);
      gtk_table_attach_defaults(GTK_TABLE(table), widget, 1, 2, 1, 2);

      for (i = 0; i < 16; i++)
	{
	  n = i + 2;
	  widgets[n] = gnome_color_picker_new();
	  gtk_table_attach_defaults(GTK_TABLE(widget), widgets[n],
				    i % 8, i % 8 + 1,
				    i / 8, i / 8 + 1);
	}

      widget = gtk_label_new("Foreground:");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_RIGHT);
      gtk_table_attach(GTK_TABLE(table), widget, 0, 1, 2, 3,
		       GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);

      widgets[18] = gnome_color_picker_new();
      gtk_table_attach_defaults(GTK_TABLE(table), widgets[18], 1, 2, 2, 3);

      widget = gtk_label_new("Background:");
      gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_RIGHT);
      gtk_table_attach(GTK_TABLE(table), widget, 0, 1, 3, 4,
		       GTK_FILL, GTK_FILL, GNOME_PAD, GNOME_PAD);

      widgets[19] = gnome_color_picker_new();
      gtk_table_attach_defaults(GTK_TABLE(table), widgets[19], 1, 2, 3, 4);

      // Set widget values now to prevent spurious update signals.
      settings_term_put(widgets);

      gtk_signal_connect(GTK_OBJECT(widgets[1]),
			 "font-set", settings_any_changed, widgets);
      for (i = 2; i < 20; i++)
	gtk_signal_connect(GTK_OBJECT(widgets[i]),
			   "color-set", settings_any_changed, widgets);
    }
  
  settings_term_put(widgets);
  gtk_widget_show_all(widgets[0]);
}

void settings_term_take(gpointer data)
{
  GtkWidget **widgets = data;
  GdkFont *font;
  int i;
  
  settings.font_name =
    gnome_font_picker_get_font_name(GNOME_FONT_PICKER(widgets[1]));
  font = gdk_font_load(settings.font_name);
  zvt_term_set_fonts(ZVT_TERM(zvt), font, NULL);

  for (i = 0; i < 18; i++)
    {
      gnome_color_picker_get_i16(GNOME_COLOR_PICKER(widgets[i+2]),
				 &settings.color_red[i],
				 &settings.color_green[i],
				 &settings.color_blue[i],
				 NULL);
    }
  zvt_term_set_color_scheme(ZVT_TERM(zvt), settings.color_red,
			    settings.color_green, settings.color_blue);

  write_settings();
}

void settings_term_put(gpointer data)
{
  GtkWidget **widgets = data;
  int i;
  
  gnome_font_picker_set_font_name(GNOME_FONT_PICKER(widgets[1]),
				  settings.font_name);

  for (i = 0; i < 18; i++)
    {
      gnome_color_picker_set_i16(GNOME_COLOR_PICKER(widgets[i+2]),
				 settings.color_red[i],
				 settings.color_green[i],
				 settings.color_blue[i],
				 0);
    }
}

void settings_term_apply(GnomePropertyBox *pbox, gint arg1, gpointer data)
{
  settings_term_take(data);
}

/**
 * settings_inferior - Settings/Inferior menu callback
 * @data: unused
 */
void settings_inferior(gpointer data)
{
  /* These settings are:
   * 0  GnomePropertyBox
   * 1  GnomeEntry containing inferior name
   * 2  GtkCheckButton specifying whether to set WGFILE
   * 3  GtkCheckButton specifying whether to exit on inferior death
   */
  static GtkWidget* widgets[4];
  GtkWidget *vbox, *hbox, *widget;
  
  /* If we haven't already created the propbox, do it now. */
  if (!widgets[0])
    {
      widgets[0] = gnome_property_box_new();
      gtk_signal_connect(GTK_OBJECT(widgets[0]), "apply",
			 settings_inferior_apply, widgets);
      gtk_signal_connect(GTK_OBJECT(widgets[0]), "destroy",
			 settings_any_destroy, widgets);
  
      /* Build up the page for Inferior settings. */
      vbox = gtk_vbox_new(FALSE, GNOME_PAD);
      gnome_property_box_append_page(GNOME_PROPERTY_BOX(widgets[0]),
				     vbox, gtk_label_new(N_("Inferior")));
      gtk_widget_show(vbox);
  
      hbox = gtk_hbox_new(FALSE, GNOME_PAD);
      gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox);
      gtk_widget_show(hbox);
      widget = gtk_label_new(N_("Program to run:"));
      gtk_box_pack_start(GTK_BOX(hbox), widget, FALSE, FALSE, 0);
      gtk_widget_show(widget);
      widgets[1] = gnome_entry_new(MYNAME "_inferior");
      gtk_box_pack_start_defaults(GTK_BOX(hbox), widgets[1]);
      gtk_widget_show(widgets[1]);
  
      widgets[2] = gtk_check_button_new_with_label
	(N_("Set WGFILE environment variable"));
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widgets[2]);
      gtk_widget_show(widgets[2]);
  
      widgets[3] = gtk_check_button_new_with_label
	(N_("Exit when child process dies"));
      gtk_box_pack_start_defaults(GTK_BOX(vbox), widgets[3]);
      gtk_widget_show(widgets[3]);

      // Set widget values now to prevent spurious update signals.
      settings_inferior_put(widgets);

      gtk_signal_connect(GTK_OBJECT
			 (gnome_entry_gtk_entry(GNOME_ENTRY(widgets[1]))),
			 "changed", settings_any_changed, widgets);
      gtk_signal_connect(GTK_OBJECT(widgets[2]),
			 "toggled", settings_any_changed, widgets);
      gtk_signal_connect(GTK_OBJECT(widgets[3]),
			 "toggled", settings_any_changed, widgets);
    }
  
  settings_inferior_put(widgets);
  gtk_widget_show(widgets[0]);
}

void settings_inferior_take(gpointer data)
{
  GtkWidget **widgets = data;
  
  settings.inferior =
    gtk_editable_get_chars(GTK_EDITABLE
			   (gnome_entry_gtk_entry(GNOME_ENTRY(widgets[1]))),
			   0, -1);
  settings.set_wgfile =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets[2]));
  settings.exit_on_death =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets[3]));
  write_settings();
}

void settings_inferior_put(gpointer data)
{
  GtkWidget **widgets = data;
  
  gtk_entry_set_text(GTK_ENTRY(gnome_entry_gtk_entry(GNOME_ENTRY(widgets[1]))),
		     settings.inferior);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widgets[2]),
			       settings.set_wgfile);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widgets[3]),
			       settings.exit_on_death);
}

void settings_inferior_apply(GnomePropertyBox *pbox, gint arg1, gpointer data)
{
  settings_inferior_take(data);
}

void settings_adjustment_integral(GtkAdjustment *adj, gpointer data)
{
  // Force the value of the adjustment to be an integer.  If this happens
  // early enough, we can fake out a slider.
  adj->value = rint(adj->value);
  
}

void settings_any_changed(GtkObject *object, gpointer data)
{
  GtkWidget **widgets = data;
  gnome_property_box_changed(GNOME_PROPERTY_BOX(widgets[0]));
}

void settings_any_destroy(GtkObject *object, gpointer data)
{
  GtkWidget **widgets = data;
  /* The GnomePropertyBox got destroyed; zero its pointer in the struct. */
  widgets[0] = NULL;
}

/**
 * help_about - Help/About menu callback
 * @data: unused
 */
void help_about(gpointer data)
{
  GtkWidget *widget;
  
  static const char *authors[] = { "David Maze <dmaze@mit.edu>", NULL };
  
  widget = gnome_about_new(MYNAME, VERSION,
			   "(C) 1999-2001, David Z. Maze", authors,
			   "Test program to display a gznola widget "
			   "and a zephyr client.\n"
			   "Project home page: "
			   "http://www.mit.edu/~dmaze/xznola/\n",
			   NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(widget), GTK_WINDOW(app));
  gtk_widget_show(widget);
  gnome_dialog_run_and_close(GNOME_DIALOG(widget));
}


/**
 * child_died_event - Called when the inferior process exits
 * @data: unused
 */
void child_died_event(gpointer data)
{
  if (settings.exit_on_death)
    gtk_exit(0);
}
