/*
 * anyone.c: Read special spluftified .anyone files
 * David Maze <dmaze@mit.edu>
 * $Id: anyone.c,v 1.1 1999/04/11 15:16:52 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "anyone.h"
#include "zusertable.h"
#include <stdio.h>

static void parse_group_line(const char *line, GString **parent,
			     GString **group, GString **rest);
/* modifies: *parent, *group, *rest
 *  effects: Scans line of the form #+parent:group:rest, and returns
 *           GStrings containing the text of parent, group, and rest
 *           in the relevant parameters.  These strings may be empty
 *           if the line is malformed. */

zgroup *read_anyone(const GString *filename)
{
  FILE *file;
  char line[256];
  gchar *nline;
  GString *strline;
  zgroup *top, *group;
  GString *parentname, *groupname, *label;
  
  file = fopen(filename->str, "r");
  if (!file)
  {
    g_error("Can't open .anyone file '%s'", filename->str);
    return NULL;
  }

  top = zgroup_new(g_string_new(" "), g_string_new(" "), zusertable_new());
  group = top;

  while (fgets(line, sizeof(line), file))
  {
    nline = g_strstrip(line);
    if (nline[0] == '#')
    {
      /* We have a comment line in a .anyone file.  This could be one of
       * ours; if it is, the second character will be a '+' or a '-'. */
      if (nline[1] == '+' || nline[1] == '-')
      {
	parse_group_line(line, &parentname, &groupname, &label);
	group = zgroup_find(top, parentname);
	if (!group)
	{
	  g_error("Couldn't find group '%s'", parentname->str);
	  group = top;
	}
	group = zgroup_new_child(group, groupname, label);
	zgroup_set_expanded(group, line[1] == '+');
	g_string_free(parentname, TRUE);
	g_string_free(groupname, TRUE);
	g_string_free(label, TRUE);
      }
    }
    else
    {
      strline = g_string_new(nline);
      zgroup_append_username(group, strline);
      g_string_free(strline, TRUE);
    }
  }

  fclose(file);

  return top;
}

zgroup *read_anyone_default(void)
{
  GString *filename;
  gchar *fn;
  zgroup *group;
  
  filename = default_anyone();
  group = read_anyone(filename);
  g_string_free(filename, TRUE);

  return group;
}

GString *default_anyone(void)
{
  GString *filename;
  gchar *fn;
  zgroup *group;
  
  fn = g_getenv("ANYONE");
  if (fn)
    filename = g_string_new(fn);
  else
  {
    filename = g_string_new(g_get_home_dir());
    filename = g_string_append(filename, G_DIR_SEPARATOR_S ".anyone");
  }

  return filename;
}

static void parse_group_line(const char *line, GString **parent,
			     GString **group, GString **rest)
{
  char *word;
  const char *last;
  unsigned count;

  line += 2; /* Ignore # and +/- chars */

  last = line;
  count = 1;
  while (*line && *line != ':')
    {
      count++;
      line++;
    }
  word = g_strndup(last, count - 1);
  *parent = g_string_new(word);
  g_free(word);

  if (*line)
    line++;
  count = 1;
  last = line;
  while (*line && *line != ':')
    {
      count++;
      line++;
    }
  word = g_strndup(last, count - 1);
  *group = g_string_new(word);
  g_free(word);

  if (*line)
    line++;
  *rest = g_string_new(line);

  return;
}
