/*
 * zuser.c: Describe a single user
 * David Maze <dmaze@mit.edu>
 * $Id*
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "com_err.h"
#include "zuser.h"

static void zuser_clone_locs(zuser *user);
static void zuser_free_locs(zuser *user);

struct _zuser
{
  GString *name;
  GString *principal;
  int numlocs;
  ZLocations_t *locs;
};

/* Zephyr...! */
Code_t ZGetLocations(ZLocations_t location[], int *numloc);

zuser *zuser_new(const GString *name)
{
  zuser *user = g_new(zuser, 1);

  /* Split name into at most two bits on @ signs. */
  gchar **nparts = g_strsplit(name->str, "@", 2);
  /* Is there a second part? */
  if (nparts[1] != NULL)
  {
    /* Okay: name is the full name, nparts[0] is the abbrev name. */
    user->name = g_string_new(nparts[0]);
    user->principal = g_string_new(name->str);
  }
  else
  {
    /* No @, use the default realm for the principal. */
    user->name = g_string_new(name->str);
    user->principal = g_string_new("");
    g_string_sprintf(user->principal, "%s@%s", user->name->str, ZGetRealm());
  }
  g_strfreev(nparts);
    
  /* Also set up location stuff. */
  user->numlocs = 0;
  user->locs = NULL;

  return user;
}

void zuser_delete(zuser *user)
{
  g_assert(user != NULL);
  g_assert(user->name != NULL);
  zuser_free_locs(user);
  g_string_free(user->name, TRUE);
  user->name = NULL;
  g_free(user->locs);
  user->locs = NULL;
  user->numlocs = 0;
  g_free(user);
}

GString *zuser_name(const zuser *user)
{
  g_assert(user->name != NULL);
  return g_string_new(user->name->str);
}

GString *zuser_principal(const zuser *user)
{
  g_assert(user != NULL);
  g_assert(user->principal != NULL);
  return g_string_new(user->principal->str);
}

void zuser_update(zuser *user)
{  
  int cresult, numlocs;

  g_assert(user != NULL);
  g_assert(user->principal != NULL);
  
  cresult = ZLocateUser(user->principal->str, &numlocs, ZAUTH);
  if (cresult != ZERR_NONE)
  {
    g_error("Trying to update users: %s", error_message(cresult));
    return;
  }

  /* We know there are valid locations; discard any old state. */
  zuser_free_locs(user);
  g_free(user->locs);
  user->locs = NULL;
  user->numlocs = numlocs;
  
  /* If there aren't any locations, don't bother looking them up. */
  if (user->numlocs == 0)
    return;

  user->locs = g_new(ZLocations_t, user->numlocs);
  cresult = ZGetLocations(user->locs, &user->numlocs);
  switch (cresult)
  {
  case ZERR_NONE:
    zuser_clone_locs(user);
    break; /* All is well */

    /* UTSLing, these aren't so good; we get these if there were never
     * any locations to start with (ZERR_NOLOCATIONS), or if we've
     * already retrieved all of the locations there are to be
     * retrieved (ZERR_NOMORELOCS).  In either case, our locations
     * don't get updated. */
  case ZERR_NOLOCATIONS:
  case ZERR_NOMORELOCS:
  default:
    g_free(user->locs);
    user->locs = NULL;
    user->numlocs = 0;
    g_error("Trying to update %s: %s", user->principal->str,
            error_message(cresult));
  }
}

int zuser_count_locs(const zuser *user)
{
  return user->numlocs;
}

const ZLocations_t *zuser_get_loc(const zuser *user, int n)
{
  g_assert(n >= 0);
  g_assert(n < user->numlocs);
  
  return user->locs + n;
}

static void zuser_clone_locs(zuser *user)
{
  /* So, zephyr hates us.  Specifically, ZGetLocations() copies the
   * zephyr internal location buffers into its parameter.  But this
   * is only a shallow clone; the pointers in the copied structure
   * are freed on the next ZFlushLocations() call (or the next
   * ZLocateUser() call).  strdup() all of those pointers so they
   * stay valid. */
  int i;
  for (i = 0; i < user->numlocs; i++)
  {
    user->locs[i].host = g_strdup(user->locs[i].host);
    user->locs[i].time = g_strdup(user->locs[i].time);
    user->locs[i].tty  = g_strdup(user->locs[i].tty);
  }
}

static void zuser_free_locs(zuser *user)
{
  /* Free the memory we just allocated. */
  int i;
  for (i = 0; i < user->numlocs; i++)
  {
    g_free(user->locs[i].host);
    user->locs[i].host = NULL;
    g_free(user->locs[i].time);
    user->locs[i].time = NULL;
    g_free(user->locs[i].tty);
    user->locs[i].tty = NULL;
  }
}
