/*
 * zgroup.h: Describe groups of users
 * David Maze <dmaze@mit.edu>
 * $Id: zgroup.h,v 1.3 1999/10/05 14:47:31 dmaze Exp $
 */

#ifndef __ZGROUP_H
#define __ZGROUP_H

#include <glib.h>

#include "zuser.h"
#include "zusertable.h"

typedef struct _zgroup zgroup;

/* A zgroup is some collection of users that can appear online.  It
 * contains a list of zusers and a list of zgroups that are children
 * of this one.  No user can be in a given group more than once,
 * though they can appear in multiple groups.  A group also has an
 * internal name, by which other groups can refer to it, and a label,
 * which is displayed to the user.  The zgroup can either be expanded
 * or not expanded, which controls how it will display in a tree. */

zgroup *zgroup_new(const GString *name, const GString *label,
		   zusertable *table);
/*  effects: Returns a new group with internal name name and label label,
 *           with no users and no subgroups.  User names are looked up in
 *           table. */

zgroup *zgroup_new_child(zgroup *group,
			 const GString *name, const GString *label);
/* modifies: group
 *  effects: Creates a new group as in zgroup_new() using the same table
 *           as in group, and adds it as a child of group.  Returns the
 *           new group. */

void zgroup_delete (zgroup *group);
/*  effects: Destroys group. */

GString *zgroup_name(const zgroup *group);
/*  effects: Returns the internal name of group. */

GString *zgroup_label(const zgroup *group);
/*  effects: Returns the label of group. */

zusertable *zgroup_table(const zgroup *group);
/*  effects: Returns the table group uses. */

GList *zgroup_groups(zgroup *group);
/*  effects: Returns a list of the subgroups of group.
 * warnings: You may modify this list; doing so will modify group as
 *           well.  Do not free this list.  Do not free the items to
 *           this list.  Any saved pointers to this list will become
 *           invalid if group is destroyed. */

GList *zgroup_usernames(zgroup *group);
/*  effects: Returns a list of the usernames in group.
 * warnings: As in zgroup_groups(). */

GList *zgroup_users(const zgroup *group);
/*  effects: Returns a list of the zusers in group.
 * warnings: Modifying the returned list has no effect on anything.  You
 *           should probably free it when you're done with it. */

GList *zgroup_find_user(const zgroup *group, zuser *user);
/*  effects: Returns a list of the zusers in group, as in zgroup_users().
 *           The returned glist pointer points to the specified user.
 *           Returns NULL if user is not in group. */

zgroup *zgroup_append_group(zgroup *group, zgroup *added);
/*  effects: Adds 'added' to the subgroup list of group.  Returns added.
 * modifies: group */

const GString *zgroup_append_username(zgroup *group, const GString *name);
/*  effects: Adds 'name' to the user list of group, if name is not already
 *           in group.  Returns name.
 * modifies: group */

zuser *zgroup_append_user(zgroup *group, zuser *user);
/*  effects: Adds 'user' to the user list of group, if user is not already
 *           in group.  Returns user.
 * modifies: group */

zgroup *zgroup_find(zgroup *group, const GString *name);
/*  effects: Returns a zgroup in the heirarchy starting at group with
 *           internal name name, or NULL if there is no such group. */

int zgroup_set_expanded(zgroup *group, int expanded);
/* modifies: group
 *  effects: Sets group to be expanded if expanded is TRUE, or not expanded
 *           if expanded is FALSE.  Returns the old expanded status. */

int zgroup_get_expanded(const zgroup *group);
/*  effects: Returns the expanded status of group. */

#endif /* __ZGROUP_H */
