/*
 * gznola.c: A GNOME-based xznola
 * David Maze <dmaze@mit.edu>
 * $Id: gznola.c,v 1.13 2002/03/07 02:13:53 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "zgroup.h"
#include "anyone.h"
#include "gtkanyone.h"
#include "gtksaneprogressbar.h"

/* Deal with bizarre name conflicts between des.h and GNOME. */
#undef _
#include "gnome.h"

/* Some global constants */
#define MYNAME "gznola"

/* Open a .anyone file, and display a tree for it in the main window. */
void znol_open(GString *file);

/* Close and then re-open the current file. */
void znol_reopen(void);

/* Update all of the users in the current file. */
void znol_update(void);

/* Menu item callbacks */
void file_open(gpointer data);
void file_reopen(gpointer data);
void file_update(gpointer data);
void file_locate(gpointer data);
void file_preferences(gpointer data);
void file_exit(gpointer data);

void help_about(gpointer data);

void do_file_open(gpointer data);
void do_file_locate(gchar *string, gpointer data);
void file_preferences_changed(GtkWidget *widget, gpointer data);
void file_preferences_apply(GnomePropertyBox *propbox,
                            gint page, gpointer data);

void read_settings(void);
void write_settings(void);
void update_setup(void);
gint update_timeout(gpointer data);

/* Definition of the main menu */
static GnomeUIInfo menu_file[] = {
  GNOMEUIINFO_MENU_OPEN_ITEM(file_open, NULL),
  GNOMEUIINFO_MENU_REVERT_ITEM(file_reopen, NULL),
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("_Update"),
    N_("Find out which users are logged in currently"),
    file_update, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
    GDK_U, GDK_CONTROL_MASK, NULL },
  { GNOME_APP_UI_ITEM, N_("_Locate"),
    N_("Locate a particular user"),
    file_locate, NULL, NULL,
    GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH,
    GDK_L, GDK_CONTROL_MASK, NULL },
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_PREFERENCES_ITEM(file_preferences, NULL),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_MENU_EXIT_ITEM(file_exit, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_help[] = {
  GNOMEUIINFO_HELP(MYNAME),
  GNOMEUIINFO_MENU_ABOUT_ITEM(help_about, NULL),
  GNOMEUIINFO_END
};

static GnomeUIInfo menu_main[] = {
  GNOMEUIINFO_MENU_FILE_TREE(menu_file),
  GNOMEUIINFO_MENU_HELP_TREE(menu_help),
  GNOMEUIINFO_END
};

/* Definition of the toolbar */
static GnomeUIInfo toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("Update"),
			 N_("Find out which users are logged in currently"),
			 file_update,
			 GNOME_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_ITEM_STOCK(N_("Locate"),
			 N_("Locate a particular user"),
			 file_locate,
			 GNOME_STOCK_PIXMAP_SEARCH),
  GNOMEUIINFO_ITEM_STOCK(N_("Revert"),
			 N_("Re-open the current file"),
			 file_reopen,
			 GNOME_STOCK_PIXMAP_REVERT),
  GNOMEUIINFO_END
};

struct preferences
{
  int do_zsubs;
  int do_time;
  int time_interval;
  int timer;
} prefs;
struct file_preferences_data
{
  GtkWidget *zsub_check, *time_check, *time_interval;
};
static GtkWidget *app, *progbar;
static GtkAnyone *anyone;

int main(int argc, char **argv)
{
  int cresult;
  GString *file;

  cresult = ZInitialize();
  if (cresult != ZERR_NONE)
  {
    g_error("Trying to initialize zephyr: %s", error_message(cresult));
    return;
  }
  gnome_init(MYNAME, VERSION, argc, argv);

  prefs.timer = 0;
  read_settings();

  app = gnome_app_new(MYNAME, MYNAME);
  gnome_app_create_menus(GNOME_APP(app), menu_main);
  gnome_app_create_toolbar(GNOME_APP(app), toolbar);
  progbar = gtk_sane_progress_bar_new();
  gnome_app_set_statusbar(GNOME_APP(app), progbar);
  /* Let the window shrink itself */
  gtk_window_set_policy(GTK_WINDOW(app), TRUE, TRUE, TRUE);
  gtk_widget_show(app);

  /* Open the default .anyone file. */
  anyone = NULL;
  file = default_anyone();
  znol_open(file);
  g_string_free(file, TRUE);

  znol_update();
  update_setup();
  
  gtk_main();
  return 0;
}

/**
 * znol_open - open a named file
 * @file: GString containing the name of the file to open
 *
 * znol_open() tries to create a &GtkAnyone widget, which reads in
 * @file.  If it fails, nothing happens; otherwise, the existing
 * &GtkAnyone structure is destroyed and replaced with the new one.
 */
void znol_open(GString *file)
{
  GtkWidget *my_anyone = gtk_anyone_open(file);
  
  /* If the open failed, don't do anything; just leave the existing UI up. */
  if (!my_anyone)
    {
      g_message("Couldn't open file %s", file->str);
      return;
    }
  
  /* Otherwise close the old UI and set up the new one. */
  anyone = GTK_ANYONE(my_anyone);
  gnome_app_set_contents(GNOME_APP(app), my_anyone);
  gtk_widget_show(my_anyone);
}

/**
 * znol_reopen - reopen the current file
 *
 * znol_reopen() creates a new &gtk_anyone widget for the same file as
 * the current one, as in znol_open().  This effectively causes the
 * current file to be reread.
 */
void znol_reopen(void)
{
  GString *file;
  
  /* This just shouldn't work without an open file. */
  g_assert(anyone != NULL);
  
  file = gtk_anyone_get_filename(anyone);
  znol_open(file);
  g_string_free(file, TRUE);
}

/**
 * znol_update - update all users
 *
 * znol_update() attempts to locate all of the users listed in the current
 * &GtkAnyone object.  The visible user interface is updated accordingly.
 */
void znol_update(void)
{
  if (anyone)
    gtk_anyone_update(anyone, GTK_PROGRESS(progbar));
}

/**
 * file_open - File/Open menu callback
 * @data: unused
 */
void file_open(gpointer data)
{
  GtkWidget *selection;
  selection = gtk_file_selection_new("Open .anyone");
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(selection), ".anyone");
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->ok_button),
     "clicked", GTK_SIGNAL_FUNC(do_file_open), (gpointer)selection);
  gtk_signal_connect_object
    (GTK_OBJECT(GTK_FILE_SELECTION(selection)->cancel_button),
     "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer)selection);
  gtk_window_position(GTK_WINDOW(selection), GTK_WIN_POS_MOUSE);
  gtk_widget_show(selection);
}

/**
 * do_file_open - Open a file based on information from a file selection
 * @data: pointer to GtkFileSelection
 */
void do_file_open(gpointer data)
{
  GtkWidget *selection = data;
  gchar *filename =
    gtk_file_selection_get_filename(GTK_FILE_SELECTION(selection));
  GString *str = g_string_new(filename);
  znol_open(str);
  g_string_free(str, TRUE);
  znol_update();
}

/**
 * file_reopen - File/Revert menu callback
 * @data: unused
 */
void file_reopen(gpointer data)
{
  znol_reopen();
  znol_update();
}

/**
 * file_update - File/Update menu callback
 * @data: unused
 */
void file_update(gpointer data)
{
  znol_update();
}

/**
 * file_locate - File/Locate menu callback
 * @data: unused
 */
void file_locate(gpointer data)
{
  gnome_request_dialog(FALSE,
		       _("Locate which user?"),
		       NULL,
		       256,
		       do_file_locate,
		       NULL,
		       GTK_WINDOW(app));
}

/**
 * do_file_locate - Locate a user and display their location in a dialog
 * @string: User to locate
 * @data: unused
 */
void do_file_locate(gchar *string, gpointer data)
{
  zuser *user;
  GString *str = g_string_new(string);
  int n;
  const ZLocations_t *loc;

  /* Create and update a user object. */
  user = zuser_new(str);
  g_string_free(str, TRUE);
  zuser_update(user);

  /* Create a message string. */
  str = zuser_name(user);
  g_string_append(str, ":\n");
  
  /* If the user isn't there, indicate this. */
  if (zuser_count_locs(user) == 0)
  {
    g_string_append(str, "Not logged in (or no such user)");
  }
  /* Otherwise, construct the message string. */
  else
  {
    for (n = 0; n < zuser_count_locs(user); n++)
    {
      loc = zuser_get_loc(user, n);
      g_string_sprintfa(str, "%s (%s): %s\n", loc->host, loc->tty, loc->time);
    }
  }

  /* Display the message. */
  gnome_ok_dialog_parented(str->str, GTK_WINDOW(app));
  
  /* Clean up. */
  g_string_free(str, TRUE);
}

/**
 * file_preferences - File/Preferences menu callback
 * @data: unused
 */
void file_preferences(gpointer data)
{
  /* Create the preferences dialog. */
  GtkWidget *dialog, *vbox, *hbox, *label;
  struct file_preferences_data d;
  gchar *interval;

  dialog = gnome_property_box_new();
  gnome_dialog_set_parent(GNOME_DIALOG(dialog), GTK_WINDOW(app));

  vbox = gtk_vbox_new(FALSE, GNOME_PAD);
  
  d.zsub_check =
    gtk_check_button_new_with_label("Update using zephyr subscriptions");
  gtk_object_ref(GTK_OBJECT(d.zsub_check));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(d.zsub_check),
                               prefs.do_zsubs);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), d.zsub_check);
  gtk_signal_connect(GTK_OBJECT(d.zsub_check), "toggled",
                     GTK_SIGNAL_FUNC(file_preferences_changed), dialog);
  /* Until we actually do this. */
  gtk_widget_set_sensitive(d.zsub_check, FALSE);

  hbox = gtk_hbox_new(FALSE, 0);
  label = gtk_label_new("Update every ");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  d.time_interval = gtk_entry_new();
  gtk_object_ref(GTK_OBJECT(d.time_interval));
  interval = g_strdup_printf("%d", prefs.time_interval);
  gtk_entry_set_text(GTK_ENTRY(d.time_interval), interval);
  g_free(interval);
  gtk_box_pack_start(GTK_BOX(hbox), d.time_interval, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(d.time_interval), "changed",
                     GTK_SIGNAL_FUNC(file_preferences_changed), dialog);
  label = gtk_label_new(" seconds");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  d.time_check = gtk_check_button_new();
  gtk_object_ref(GTK_OBJECT(d.time_check));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(d.time_check), prefs.do_time);
  gtk_container_add(GTK_CONTAINER(d.time_check), hbox);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), d.time_check);
  gtk_signal_connect(GTK_OBJECT(d.time_check), "toggled",
                     GTK_SIGNAL_FUNC(file_preferences_changed), dialog);

  label = gtk_label_new("Updating");
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(dialog), vbox, label);

  gtk_signal_connect(GTK_OBJECT(dialog), "apply",
                     GTK_SIGNAL_FUNC(file_preferences_apply), &d);

  gtk_widget_show_all(dialog);
  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));

  gtk_object_unref(GTK_OBJECT(d.zsub_check));
  gtk_object_unref(GTK_OBJECT(d.time_check));
  gtk_object_unref(GTK_OBJECT(d.time_interval));
}

/**
 * file_preferences_changed - file_preferences callback for when a
 * widget state changes
 */
void file_preferences_changed(GtkWidget *widget, gpointer data)
{
  gnome_property_box_changed(GNOME_PROPERTY_BOX(data));
}

/**
 * file_preferences_apply - file_preferences dialog callback
 */
void file_preferences_apply(GnomePropertyBox *propbox,
                            gint page, gpointer data)
{
  int ttime;
  char *text, *text2;
  struct file_preferences_data *d = data;
  prefs.do_zsubs =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(d->zsub_check));
  prefs.do_time =
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(d->time_check));
  text = gtk_entry_get_text(GTK_ENTRY(d->time_interval));
  ttime = strtol(text, &text2, 10);
  if (*text2 == '\0' && ttime > 0)
    prefs.time_interval = ttime;
  update_setup();
  write_settings();
}

/**
 * file_exit - File/Exit menu callback
 * @data: unused
 */
void file_exit(gpointer data)
{
  gtk_exit(0);
}

/**
 * help_about - Help/About menu callback
 * @data: unused
 */
void help_about(gpointer data)
{
  GtkWidget *widget;
  
  static const char *authors[] = { "David Maze <dmaze@mit.edu>", NULL };
  
  widget = gnome_about_new(MYNAME, VERSION,
			   "(C) 1999-2000, David Z. Maze", authors,
			   "Displays a heirarchical tree of users "
			   "visible over Zephyr.\n"
			   "Project home page: "
			   "http://www.mit.edu/~dmaze/xznola/\n",
			   NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(widget), GTK_WINDOW(app));
  gtk_widget_show(widget);
  gnome_dialog_run_and_close(GNOME_DIALOG(widget));
}

void read_settings(void)
{
  prefs.do_zsubs = gnome_config_get_int("/" MYNAME "/settings/do_zsubs=1");
  prefs.do_time = gnome_config_get_int("/" MYNAME "/settings/do_time=1");
  prefs.time_interval =
    gnome_config_get_int("/" MYNAME "/settings/time_interval=60");
  update_setup();
}

void write_settings(void)
{
  gnome_config_set_int("/" MYNAME "/settings/do_zsubs", prefs.do_zsubs);
  gnome_config_set_int("/" MYNAME "/settings/do_time", prefs.do_time);
  gnome_config_set_int("/" MYNAME "/settings/time_interval",
                       prefs.time_interval);
  gnome_config_sync();
}

void update_setup(void)
{
  if (prefs.timer)
  {
    gtk_timeout_remove(prefs.timer);
    prefs.timer = 0;
  }
  
  if (prefs.do_time)
    prefs.timer = gtk_timeout_add(prefs.time_interval * 1000,
                                  update_timeout,
                                  NULL);
}

gint update_timeout(gpointer data)
{
  znol_update();
  return TRUE;
}
