/*
 * gtk_anyone.c: Show the user tree as a Gtk+ widget
 * David Maze <dmaze@mit.edu>
 * $Id: gtkanyone.c,v 1.29 2002/12/22 19:17:21 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "gtkanyone.h"
#include "gtkzgroupitem.h"
#include "gtkzuseritem.h"
#include "anyone.h"

/* Local function prototypes */
static void gtk_anyone_class_init(GtkAnyoneClass *klass);
static void gtk_anyone_init(GtkAnyone *anyone);

static void build_children(GtkAnyone *anyone);
static void rebuild_children(GtkAnyone *anyone);
static void set_tooltip(GtkAnyone *anyone, GtkWidget *child, zuser *user);

/* Get some help with updating */
struct update_iter_data
{
  GtkAnyone *anyone;
  GtkProgress *progress;
  int count;
  int total;
};
static void update_iter(zuser *user, gpointer data);

static void gtk_anyone_class_init(GtkAnyoneClass *klass)
{
  /* Do nothing for now; see the Gtk+ tutorial if we ever add signals. */
}

static void gtk_anyone_init(GtkAnyone *anyone)
{
  anyone->group = NULL;
  anyone->filename = NULL;
  anyone->tips = NULL;
}

GtkType gtk_anyone_get_type(void)
{
  static GtkType anyone_type = 0;
  
  if (!anyone_type)
  {
    static const GtkTypeInfo anyone_info =
    {
      "GtkAnyone",
      sizeof(GtkAnyone),
      sizeof(GtkAnyoneClass),
      (GtkClassInitFunc) gtk_anyone_class_init,
      (GtkObjectInitFunc) gtk_anyone_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL
    };
    
    anyone_type = gtk_type_unique(gtk_tree_get_type(), &anyone_info);  
  }

  return anyone_type;
}

GtkWidget *gtk_anyone_new(zgroup *group)
{
  GtkWidget *anyone;
  
  g_return_val_if_fail(group != NULL, NULL);
  
  anyone = gtk_type_new(gtk_anyone_get_type());
  gtk_anyone_construct(GTK_ANYONE(anyone), group, NULL);
  return anyone;
}

GtkWidget *gtk_anyone_open(GString *filename)
{
  GtkWidget *anyone;
  zgroup *group;

  g_return_val_if_fail(filename != NULL, NULL);

  group = read_anyone(filename);
  if (!group)
    return NULL;
  
  anyone = gtk_type_new(gtk_anyone_get_type());
  gtk_anyone_construct(GTK_ANYONE(anyone), group, filename);
  return anyone;
}

void gtk_anyone_construct(GtkAnyone *anyone, zgroup *group, GString *filename)
{
  g_return_if_fail(anyone != NULL);
  g_return_if_fail(group != NULL);
  /* filename can be NULL */

  anyone->group = group;
  if (filename)
    anyone->filename = g_string_new(filename->str);
  else
    anyone->filename = NULL;
  anyone->tips = gtk_tooltips_new();

  build_children(anyone);
}

static void build_children(GtkAnyone *anyone)
{
  GList *list, *base;
  GString *name;
  GtkWidget *child;
  GtkWidget *subtree;
  zgroup *group;
  zuser *user;

  g_return_if_fail(anyone != NULL);

  /* Recursively add all of the group's child groups. */
  base = zgroup_groups(anyone->group);
  for (list = base; list; list = g_list_next(list))
  {
    /* Build the child group object... */
    group = list->data;

    /* Build a GtkTreeItem for it... */
    child = gtk_zgroup_item_new(group);

    /* Have to attach the child to the parent before we tweak things
     * like subtrees.  Remember, kids, documentation is for wusses, and
     * invariant checking doubly so.  *sigh* */
    gtk_tree_append(GTK_TREE(anyone), child);

    /* Make sure we show the children. */
    gtk_widget_show(GTK_WIDGET(child));

    /* Only create the tree now if the group has children of its own.
    /* rebuild_children() will create groups for users if they don't
     * already exist, to get around gtk+ brokenness. */
    if (zgroup_groups(group))
      gtk_zgroup_item_force_subtree(GTK_ZGROUP_ITEM(child));
  }
}

zgroup *gtk_anyone_group(const GtkAnyone *anyone)
{
  return anyone->group;
}

GString *gtk_anyone_get_filename(const GtkAnyone *anyone)
{
  if (!anyone->filename)
    return NULL;
  return g_string_new(anyone->filename->str);
}

void gtk_anyone_update(GtkAnyone *anyone, GtkProgress *progress)
{
  struct update_iter_data uidata;
  zusertable *table;
  
  /* Assume the entire tree uses the same user table. */
  table = zgroup_table(anyone->group);
  
  /* Ask the table to update all of our users for us. */
  uidata.anyone = anyone;
  uidata.progress = progress;
  uidata.count = 0;
  uidata.total = zusertable_count(table);
  zusertable_foreach(table, update_iter, &uidata);

  rebuild_children(anyone);
}

static void update_iter(zuser *user, gpointer data)
{
  struct update_iter_data *uidata = data;

  /* Figure out where they are. */
  zuser_update(user);

  /* To consider: incremental updates of the user tree? */

  /* Update the progress bar. */
  uidata->count++;
  if (uidata->progress)
  {
    gtk_progress_set_percentage(uidata->progress, 
				((gfloat)uidata->count)/uidata->total);
    while (gtk_events_pending())
      gtk_main_iteration();
  }
}

void gtk_anyone_update_user(GtkAnyone *anyone, zuser *user)
{
  /* If this isn't a root tree, and we have a root tree that is a GtkAnyone,
   * do the update there instead. */
  if (!GTK_IS_ROOT_TREE(anyone) && IS_GTK_ANYONE(GTK_TREE_ROOT_TREE(anyone)))
  {
    gtk_anyone_update_user(GTK_ANYONE(GTK_TREE_ROOT_TREE(anyone)), user);
    return;
  }

  /* Update the user and show them. */
  zuser_update(user);
  rebuild_children(anyone);
}

void gtk_anyone_update_async(GtkAnyone *anyone)
{
  zusertable *table;
  
  /* Assume the entire tree uses the same user table. */
  table = zgroup_table(anyone->group);

  /* Blindly launch asynchronous update requests for the entire table. */
  zusertable_foreach(table, (zuseriter)zuser_async_update, NULL);
}

void gtk_anyone_update_user_async(GtkAnyone *anyone, zuser *user)
{
  zuser_async_update(user);
}

void gtk_anyone_result_async(GtkAnyone *anyone, ZNotice_t *notice)
{
  /* Under normal circumstances, we expect to only see notices on
   * class LOCATE_CLASS, with the instance being the principal of
   * the target.  The table might only have usernames of principals
   * in the default realm, though. */
  GString *name = g_string_new(notice->z_class_inst);
  zuser *user;
  zusertable *table = zgroup_table(anyone->group);
  char *at;
  
  user = zusertable_lookup(table, name);
  if (user != NULL)
    zuser_async_result(user, notice);
  
  /* Also check to see if the reported principal is in the default
   * realm. */
  at = strchr(name->str, '@');
  if (at)
  {
    if (!g_strcasecmp(at + 1, ZGetRealm()))
    {
      name = g_string_truncate(name, at - name->str);
      user = zusertable_lookup(table, name);
      if (user != NULL)
        zuser_async_result(user, notice);
    }
  }

  /* We'd prefer to do this only once, but... */
  rebuild_children(anyone);
}

static void rebuild_children(GtkAnyone *anyone)
{
  /* The plan: iterate simultaneously through the list of Gtk+ widget
   * children and the znola list of users.  When there are discrepancies,
   * patch up the widget tree to match.  There gets to be at most one
   * child widget per user.  While we're at it, make sure that all of
   * the child widgets are visible, and update the tooltips on all of
   * them. */
  GList *luser, *lwidget;
  int pos = 0;
  luser = zgroup_users(anyone->group);
  lwidget = gtk_container_children(GTK_CONTAINER(anyone));

  while (lwidget)
  {
    /* Assume that things aren't totally messed up, and that lwidget's
     * object data points to a valid user.  Walk through the list of
     * users until we find that one, inserting things as we go. */
    GtkWidget *widget = GTK_WIDGET(lwidget->data);
    zuser *target, *user;

    /* Sanity check: the widget should be a GtkZUserItem.  Stop if it
     * isn't (since then we'll have reached the end of our own users). */
    if (!IS_GTK_ZUSER_ITEM(widget))
      break;

    target = gtk_zuser_item_user(GTK_ZUSER_ITEM(widget));
    user = luser->data;

    while (user != target)
    {
      /* If the user is visible, create a widget for them. */
      if (zuser_count_locs(user) > 0)
      {
        GtkWidget *child;

        child = gtk_zuser_item_new(user);
        gtk_tree_insert(GTK_TREE(anyone), child, pos++);
        gtk_widget_show(child);
        set_tooltip(anyone, child, user);
      }
      
      luser = luser->next;
      /* We often lose here; why? */
      if (luser == NULL)
      {
        GString *tname = zuser_principal(target);
        GString *gname = zgroup_name(anyone->group);
        g_warning("BUG: group update failure\n"
                  "We were looking for %s in the %s group, but didn't find it.\n"
                  "Proceeding anyways, in the hopes that recovery might be\n"
                  "possible.  If you have any insight as to why this happens,\n"
                  "please tell dmaze@mit.edu.\n",
                  tname->str, gname->str);
        g_string_free(tname, TRUE);
        g_string_free(gname, TRUE);
        return;
      }
      user = luser->data;
    }
    /* Now we have a guarantee that user == target. */

    /* Advance to the next widget now, before we possibly destroy the
     * current item. */
    lwidget = lwidget->next;
    luser = luser->next;
    pos++;

    /* Check that the user is visible... */
    if (zuser_count_locs(user) > 0)
    {
      set_tooltip(anyone, widget, user);
    }
    else
    {
      /* Nope, not around.  Delete their widget item. */
      gtk_container_remove(GTK_CONTAINER(anyone), widget);
      pos--;
    }
  }

  /* At this point, we've gone through all of the user widgets that
   * exist.  If there are still users left, process and add them. */
  while (luser)
    {
      zuser *user = luser->data;
      
      /* If the user is visible, create a widget for them. */
      if (zuser_count_locs(user) > 0)
      {
        GtkWidget *child;

        child = gtk_zuser_item_new(user);
        gtk_tree_insert(GTK_TREE(anyone), child, pos++);
        gtk_widget_show(child);
        set_tooltip(anyone, child, user);
      }
      
      luser = luser->next;
    }

  /* Recurse into subgroups. */
  for (; lwidget; lwidget = g_list_next(lwidget))
  {
    /* Every item now should be a group item. */
    GtkZGroupItem *item = GTK_ZGROUP_ITEM(lwidget->data);
    GtkWidget *subtree = gtk_zgroup_item_force_subtree(item);
    
    rebuild_children(GTK_ANYONE(subtree));

    /* Possible situations here:
     * (1) subtree has children, and exists, and we want it to exist.
     * (2) subtree started with children, but doesn't have them anymore.
     *     In this case, subtree is invalid since Gtk helpfully destroyed
     *     it for us.
     * (3) subtree started with no children; we created it, and didn't
     *     add anything to it, and it now exists and is empty.  In this
     *     case (only), we want to get rid of it.
     */
    if (GTK_TREE_ITEM_SUBTREE(item))
      if (gtk_container_children(GTK_CONTAINER(subtree)) == NULL)
        gtk_tree_item_remove_subtree(GTK_TREE_ITEM(item));
  }
}

static void set_tooltip(GtkAnyone *anyone, GtkWidget *child, zuser *user)
{
  GString *tip;
  int n;
  const ZLocations_t *loc;
  
  tip = zuser_principal(user);
  g_string_append(tip, "\n");
  
  /* Also pull the locations out of the user object. */
  for (n = 0; n < zuser_count_locs(user); n++)
  {
    loc = zuser_get_loc(user, n);
    g_string_sprintfa(tip, "%s (%s): %s\n", loc->host, loc->tty, loc->time);
  }

  /* Set the tooltip. */
  gtk_tooltips_set_tip(anyone->tips, child, tip->str, tip->str);
  g_string_free(tip, TRUE);
}

