/*
 * gtkxznola.c: The main xznola window
 * David Maze <dmaze@mit.edu>
 * $Id: gtkxznola.c,v 1.4 1999/10/10 23:40:35 dmaze Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gtkxznola.h"

struct gtk_xznola
{
  GtkWidget *window;
  GtkWidget *vbox;
  GtkWidget *menu;

  GtkAnyone *anyone;
};

static GtkWidget *get_main_menu(gtk_xznola *xznola);

gtk_xznola *gtk_xznola_new()
{
  gtk_xznola *xznola = g_new(gtk_xznola, 1);

  /* Start with nothing bound. */
  xznola->anyone = NULL;

  /* Create a top-level window. */
  xznola->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(xznola->window), "xznola");
  gtk_window_set_wmclass(GTK_WINDOW(xznola->window), "xznola", "Xznola");
  /* Let the window shrink itself */
  gtk_window_set_policy(GTK_WINDOW(xznola->window), TRUE, TRUE, TRUE);

  /* Create a vbox to stuff things into. */
  xznola->vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(xznola->window), xznola->vbox);
  gtk_widget_show(xznola->vbox);

  /* Get a menu. */
  xznola->menu = get_main_menu(xznola);
  gtk_container_add(GTK_CONTAINER(xznola->vbox), xznola->menu);
  gtk_widget_show(xznola->menu);

  /* Okay, all done for now. */
  return xznola;
}

void gtk_xznola_delete(gtk_xznola *xznola)
{
  /* In theory, if we free the top-level window, everything gets freed. */
  gtk_widget_destroy(xznola->window);

  g_free(xznola);
}

GtkAnyone *gtk_xznola_get_anyone(const gtk_xznola *xznola)
{
  return xznola->anyone;
}

GtkAnyone *gtk_xznola_bind_anyone(gtk_xznola *xznola, GtkAnyone *anyone)
{
  GtkAnyone *last = gtk_xznola_unbind_anyone(xznola);

  xznola->anyone = anyone;
  gtk_container_add(GTK_CONTAINER(xznola->vbox), GTK_WIDGET(anyone));
  gtk_widget_show(GTK_WIDGET(anyone));
  
  return last;
}

GtkAnyone *gtk_xznola_unbind_anyone(gtk_xznola *xznola)
{
  GtkAnyone *last = xznola->anyone;
  if (xznola->anyone)
    gtk_container_remove(GTK_CONTAINER(xznola->vbox),
			 GTK_WIDGET(xznola->anyone));
  xznola->anyone = NULL;
  return last;
}

GtkAnyone *gtk_xznola_open_anyone(gtk_xznola *xznola, GString *filename)
{
  GtkWidget *anyone = gtk_anyone_open(filename);
  if (!anyone)
    return NULL;
  return gtk_xznola_bind_anyone(xznola, GTK_ANYONE(anyone));
}

void gtk_xznola_close_anyone(gtk_xznola *xznola)
{
  GtkAnyone *anyone = xznola->anyone;
  if (!anyone)
    return;
  
  gtk_xznola_unbind_anyone(xznola);
  gtk_widget_destroy(GTK_WIDGET(anyone));
}

GtkWidget *gtk_xznola_window(const gtk_xznola *xznola)
{
  return xznola->window;
}

void gtk_xznola_update(gtk_xznola *xznola)
{
  /* We need some black magic for this to happen properly.  Punt on that
   * for now.  We won't get any UI response in this version, but that's
   * okay. */
  if (xznola->anyone)
    gtk_anyone_update(xznola->anyone, NULL);
}

/* Top-level menu for the application */

/* Menu callbacks */
static void file_open(gtk_xznola *xznola);
static void file_reopen(gtk_xznola *xznola);
static void file_close(gtk_xznola *xznola);
static void file_update(gtk_xznola *xznola);
static void file_quit(gtk_xznola *xznola);

/* Related callbacks */
struct open_ok_data
{
  GtkFileSelection *sel;
  gtk_xznola *xznola;
};
static void open_ok(GtkWidget *button, struct open_ok_data *data);

/* Structure members: {where on menu, accel key, callback, data, item type} */
static GtkItemFactoryEntry menu_items[] =
{
  {"/_File", NULL, NULL, 0, "<Branch>"},
  {"/File/_Open", "<control>O", file_open, 0, NULL},
  {"/File/_Reopen", "<control>R", file_reopen, 0, NULL},
  {"/File/_Close", "<control>W", file_close, 0, NULL},
  {"/File/sep1", NULL, NULL, 0, "<Separator>"},
  {"/File/_Update", "<control>U", file_update, 0, NULL},
  {"/File/_Locate", "<control>L", /* file_locate */ NULL, 0, "<Title>"},
  {"/File/_ZWGC", NULL, NULL, 0, "<Branch>"},
  {"/File/ZWGC/Get LOGIN _notification", NULL,
   /* zwgc_get_subs */ NULL, 0, "<Title>"},
  {"/File/sep2", NULL, NULL, 0, "<Separator>"},
  {"/File/E_xit", "<control>Q", file_quit, 0, NULL},
  {"/_Help", NULL, NULL, 0, "<LastBranch>"},
  {"/Help/_About", NULL, /* help_about */ NULL, 0, "<Title>"}
};

static GtkWidget *get_main_menu(gtk_xznola *xznola)
{
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  accel_group = gtk_accel_group_new();
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
		  		      accel_group);

  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, xznola);

  gtk_accel_group_attach(accel_group, GTK_OBJECT(xznola->window));

  return gtk_item_factory_get_widget(item_factory, "<main>");
}

static void file_open(gtk_xznola *xznola)
{
  GtkWidget *sel;
  struct open_ok_data data;
  
  sel = gtk_file_selection_new("Open File");
  data.sel = GTK_FILE_SELECTION(sel);
  data.xznola = xznola;
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(sel)->ok_button),
		     "clicked", (GtkSignalFunc)open_ok, &data);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(sel)->cancel_button),
			    "clicked", (GtkSignalFunc)gtk_widget_destroy,
			    GTK_OBJECT(sel));
  gtk_widget_show(sel);
  gtk_main();
}

static void file_reopen(gtk_xznola *xznola)
{
  GtkAnyone *anyone = gtk_xznola_get_anyone(xznola);
  GString *fn;

  if (!anyone)
    return;
  fn = gtk_anyone_get_filename(anyone);
  gtk_xznola_open_anyone(xznola, fn);
  g_string_free(fn, TRUE);
  gtk_xznola_update(xznola);
}

static void file_close(gtk_xznola *xznola)
{
  gtk_xznola_close_anyone(xznola);
}

static void file_update(gtk_xznola *xznola)
{
  gtk_xznola_update(xznola);
}

static void file_quit(gtk_xznola *xznola)
{
  gtk_main_quit();
}

static void open_ok(GtkWidget *button, struct open_ok_data *data)
{
  GString *fn = g_string_new(gtk_file_selection_get_filename(data->sel));
  gtk_widget_destroy(GTK_WIDGET(data->sel));
  gtk_xznola_open_anyone(data->xznola, fn);
  g_string_free(fn, TRUE);
  gtk_xznola_update(data->xznola);
  gtk_main_quit();
}
