/** @file cf_io.c
  * @brief This file contains CF bus driver.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	10/04/05: Add Doxygen format comments
	
********************************************************/

#include  "cfio_io.h"

/********************************************************
		Local Variables
********************************************************/

#define __NO_VERSION__

#define CS_ERROR(h,f,r) { \
        error_info_t  err = {f,r}; \
        pcmcia_report_error(h,&err); \
      }
#define INT_MODULE_PARM(n, v) static int n = v; MODULE_PARM(n, "i")

dev_info_t cfio_dev_info = "mcf25";
struct pcmcia_device *dev_list 	 = NULL;

/* Module Variables */
typedef struct _if_pcmcia_info_t {
  dev_node_t node;
  int stop;
  struct bus_operations *bus;
  struct net_device *eth_dev;
} if_pcmcia_info_t;

/********************************************************
		Global Variables
********************************************************/

struct cf_card_rec cardp;

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief release allocated CF bus driver resource
 *  @param arg       pointer to struct pcmcia_device
 *  @return 	     N/A
 */
static void cf_release(ulong arg)
{
  struct pcmcia_device *link = (struct pcmcia_device *) arg;

  pcmcia_disable_device(link);
} /* cf_release */



/** 
 *  @brief detach CF BUS driver 
 *  @param link      pointer to struct pcmcia_device
 *  @return 	     N/A
 */

static void cf_detach(struct pcmcia_device * link)
{
} /* cf_detach */

/** 
 *  @brief CF BUS driver Configuration
 *  @param link      pointer to struct pcmcia_device
 *  @return 	     N/A
 */
static void cf_config(struct pcmcia_device * link)
{
  client_handle_t handle = link;
  if_pcmcia_info_t *dev = link->priv;
  tuple_t tuple;
  cisparse_t parse;
  cistpl_cftable_entry_t *cfg = &parse.cftable_entry;
#define BUF_LEN 64
  u8 buf[BUF_LEN];
  config_info_t conf;

  tuple.DesiredTuple = CISTPL_CONFIG;
  tuple.Attributes = 0;
  tuple.TupleData = buf;
  tuple.TupleDataMax = sizeof(buf);
  tuple.TupleOffset = 0;

  if (pcmcia_get_first_tuple(handle, &tuple))
    goto onerror;
  if (pcmcia_get_tuple_data(handle, &tuple))
    goto onerror;
  if (pcmcia_parse_tuple(handle, &tuple, &parse))
    goto onerror;

  link->conf.ConfigBase = parse.config.base;
  link->conf.Present = parse.config.rmask[0];

  if (pcmcia_get_configuration_info(handle, &conf))
    goto onerror;

  /*
     The Configuration table consists of a series of configuration table
     entry tuples. Each entry consists of up to seven data structures that
     describe operational characteristsics of the PC card.
   */

  tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;

  if (pcmcia_get_first_tuple(handle, &tuple))
    goto onerror;

  if (pcmcia_get_tuple_data(handle, &tuple) != CS_SUCCESS)
    goto onerror;

  if (pcmcia_parse_tuple(handle, &tuple, &parse) != CS_SUCCESS)
    goto onerror;


  link->conf.ConfigIndex = cfg->index;

  /* Interrupt request description */
  if (cfg->irq.IRQInfo1)
    link->conf.Attributes |= CONF_ENABLE_IRQ;

  /* IO Address space description */
  link->io.NumPorts1 = link->io.NumPorts2 = 0;
  if ((cfg->io.nwin > 0)) {
    cistpl_io_t *io;

    if (cfg->io.nwin)
      io = &cfg->io;

    link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;

    if (!(io->flags & CISTPL_IO_8BIT))
      link->io.Attributes1 = IO_DATA_PATH_WIDTH_16;

    if (!(io->flags & CISTPL_IO_16BIT))
      link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;

    link->io.IOAddrLines = io->flags & CISTPL_IO_LINES_MASK;

    link->io.BasePort1 = io->win[0].base;
    link->io.NumPorts1 = io->win[0].len;

    if (io->nwin > 1) {
      link->io.Attributes2 = link->io.Attributes1;
      link->io.BasePort2 = io->win[1].base;
      link->io.NumPorts2 = io->win[1].len;
    }

    if (pcmcia_request_io(link, &link->io)
        != CS_SUCCESS) {
      pcmcia_disable_device(link);
      printk("Request IO Error !!\n");
      goto onerror;
    }
  }

  if (link->conf.Attributes & CONF_ENABLE_IRQ)
    if (pcmcia_request_irq(link, &link->irq))
      goto onerror;

  if (pcmcia_request_configuration(link, &link->conf))
    goto onerror;

  cardp.irq = link->irq.AssignedIRQ;
  cardp.port = link->io.BasePort1;
  printk("BasePort1=0x%x, AssignedIRQ=%d\n",
         link->io.BasePort1, link->irq.AssignedIRQ);

  if (!(cardp.add(&cardp))) {
    printk("Call to cardp.add failed\n");
    goto onerror;
  }

  printk("After calling wlan_add_card function\n");

  ((if_pcmcia_info_t *) link->priv)->eth_dev = cardp.eth_dev;

  if (!((if_pcmcia_info_t *) link->priv)->eth_dev)
    goto onerror;

  strcpy(dev->node.dev_name, cfio_dev_info);
  strcpy(dev->node.dev_name, cardp.eth_dev->name);
  dev->node.major = dev->node.minor = 0;
  link->dev_node = &dev->node;

  return;

onerror:
  printk("card configuration failed...calling cf_release function\n");
  cf_release((u32) link);
  cardp.flag = 1;

} /* cf_config */


/** 
 *  @brief attach CF BUS driver 
 *  @return 	     pointer to struct pcmcia_device
 */

static int cf_attach(struct pcmcia_device *link)
{
  if_pcmcia_info_t *ifinfo;

  printk("Entering cf_attach()\n");

  /* Allocate space for PCMCIA information */
  if (!(ifinfo = kmalloc(sizeof(if_pcmcia_info_t), GFP_KERNEL))) {
    printk("Error: kmalloc for PCMCIA info failed.\n");
    return 0;
  }

  memset(ifinfo, 0, sizeof(if_pcmcia_info_t));
  link->priv = ifinfo;

  link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
  link->irq.IRQInfo1 = IRQ_INFO2_VALID | IRQ_LEVEL_ID;

  /* Handler will be installed later */
  link->irq.Handler = NULL;

  link->conf.Attributes = 0;
#define VCC_VALUE	50
  link->conf.IntType = INT_MEMORY_AND_IO;
  link->conf.ConfigIndex = 1;
  link->conf.Present = PRESENT_OPTION;

  dev_list = link;

  printk("Before registering the client\n");

  cf_config(link);
  printk("Leaving cf_attach()\n");

  return 0;
}       /* cf_attach */


/********************************************************
		Global Functions
********************************************************/

static struct pcmcia_device_id mrv_ids[] = {
        PCMCIA_DEVICE_MANF_CARD(0x02df, 0x8103),
        PCMCIA_DEVICE_NULL,
};
MODULE_DEVICE_TABLE(pcmcia, mrv_ids);

static struct pcmcia_driver mrv_driver = {
        .owner          = THIS_MODULE,
        .drv            = {
                .name   = "mrv8381",
        },
	.probe = &cf_attach,
	.remove = &cf_detach,
        .id_table = mrv_ids
};

/** 
 *  @brief register CF bus driver with OS and CardService
 *  @param add       add callback function
 *  @param remove    remove callback function
 *  @param arg       pointer to callback argument
 *  @return 	     pointer to card data
 */
u32 *register_cf_driver(cf_notifier_fn_add add, 
			cf_notifier_fn_remove remove, void *arg)
{
  cardp.add = add;
  cardp.remove = remove;
  cardp.host_int_mask = 0;

  printk("Before calling pcmcia_register_driver\n");
  pcmcia_register_driver(&mrv_driver);
  printk("After calling pcmcia_register_driver\n");

  return (u32 *) &cardp;
}

/** 
 *  @brief unregister CF bus Driver
 *  @return 	     N/A
 */
void unregister_cf_driver( void )
{

  pcmcia_unregister_driver(&mrv_driver);

  cf_detach(dev_list);
}


/** 
 *  @brief module initialization procedure
 *  @return 	     N/A
 */
int cfio_init_module(void)
{
  return 0;
}

/** 
 *  @brief module cleanup
 *  @return 	     N/A
 */
void cfio_cleanup_module(void)
{

}

/** 
 *  @brief read CIS information
 *  @param priv      pointer to wlan_private
 *  @return 	     0
 */
s16 cfio_read_cfg_reg(void* priv)
{
  conf_reg_t reg;

  reg.Function  = 0;
  reg.Action    = CS_READ;
  reg.Offset    = 0;
  reg.Value     = 0;

  pcmcia_access_configuration_register(dev_list, &reg);
  return 0;
}

module_init(cfio_init_module);
module_exit(cfio_cleanup_module);

EXPORT_SYMBOL(cardp);
EXPORT_SYMBOL(cfio_dev_info);
EXPORT_SYMBOL(register_cf_driver);
EXPORT_SYMBOL(unregister_cf_driver);
EXPORT_SYMBOL(cfio_read_cfg_reg);

