package maslab.data;

import java.io.*;
import java.util.*;

import maslab.util.*;
import maslab.laser.*;
import maslab.odom.*;

public class DataLogReader
{
    BufferedReader ins;
    Logger log=new Logger(this);

    HashMap<String, Boolean> enabledChannels=new HashMap<String,Boolean>();

    /** A log specification:
	logfile.log:CHANNELS:TO:READ
	
	If no channels are specified, then all channels are enabled.
    **/
    public DataLogReader(String logspecification) throws IOException
    {
	int colonpos=logspecification.indexOf(':');
	String path=logspecification;

	if (colonpos>0)
	    {
		String[] fields=logspecification.split(":");
		path=fields[0];

		for (int i=1;i<fields.length;i++)
		    {
			enabledChannels.put(fields[i].toUpperCase(),true);
			log.verbose("Allowing type "+fields[i]);
		    }

		if (enabledChannels.size()==0)
		    log.warn("No channels enabled!");
	    }

	ins=new BufferedReader(new FileReader(path));
    }

    public DataEvent read() throws IOException
    {
	while (true)
	    {
	    String l=ins.readLine();
	    if (l==null)
		return null;

	    if (l.charAt(0)=='#')
		continue;
	    
	    int colonpos=l.indexOf(':');
	    
	    String header=l.substring(0,colonpos);
	    String[] headerfields=header.split("\\s+");
	    String data=l.substring(colonpos+1).trim();

	    if (headerfields.length<3)
		{
		    log.output("Bad data record: "+l);
		    continue;
		}

	    double timeStamp=Double.parseDouble(headerfields[0]);
	    String parserName=headerfields[1];
	    String channelName=headerfields[2];

	    Boolean b=enabledChannels.get(parserName);
	    if (b==null)
		{
		    log.output("Logger is dropping events with parsername="+parserName);
		    enabledChannels.put(parserName, false);
		    continue;
		}
	    
	    if (!b)
		continue;

	    if (parserName.equals("LASER"))
		{
		    return new LaserScanData(timeStamp, channelName, data);
		}

	    if (parserName.equals("ODOM"))
		{
		    return new OdomData(timeStamp, channelName, data);
		}

	    log.warn("Unknown parser: "+parserName);
	    }
    }

    public void close() throws IOException
    {
	ins.close();
    }
}
