package maslab.examples;

import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.*;

import maslab.camera.*;
import maslab.orc.*;
import maslab.telemetry.*;
import maslab.util.*;

/** A simple robot implementation. **/
public class SimpleRobot
{
    // Keep these instance variables listed first so wrappers
    // below can be initialized.
    Orc orc = new Orc();
    //    Camera cam = new Camera();

    // Create some wrappers....
    Motor leftMotor  = new Motor(orc, 0, true);
    Motor rightMotor = new Motor(orc, 1, false);
    Gyro  gyro       = new Gyro(orc);
    IRRangeSensor forwardIR = IRRangeSensor.makeGP2D12(orc, 8);
    Servo servo = Servo.makeHitecHS300(orc, 0);

    Logger log = new Logger(this);
    JugLoggerPublisher logPublisher = new JugLoggerPublisher("LogMessage");
    
    /** How long should we run (seconds?) **/
    static final long RUNTIME = 25;

    public static void main(String[] args)
    {
	SimpleRobot robot;

	try {
	    robot = new SimpleRobot();
	} catch (Exception ex) {
	    System.out.println("Couldn't start: "+ex);
	    return;
	}

	robot.run();

    }

    /** Create a new robot. This constructor throws IOException
     * because creating the Orc or Camera objects might fail. 
     **/
    public SimpleRobot() throws IOException
    {
	logPublisher.setLevel("SimpleRobot", Logger.VERBOSE);
    }
    
    public void run()
    {
	long starttime = System.currentTimeMillis();

	orc.lcdConsoleHome();

	orc.lcdConsoleWrite("Calibrating gyro...");
	gyro.calibrate(1000);
	orc.lcdConsoleWrite("done.\n");

	orc.lcdConsoleWrite("Starting...\n");

	double lastReportedTime=0;

	boolean driving=false;

	while (true)
	    {
		try{
		    Thread.sleep(50);
		} catch(InterruptedException ex){
		}

		// get the time since we started.
		double time = (System.currentTimeMillis() - starttime)/1000.0;

		if (time > RUNTIME)
		    break;

		if (time > lastReportedTime + 1) {
		    double timeRemaining = RUNTIME - time;
		    orc.lcdDrawText(Orc.LcdFont.TINY_INV, 112, 8, 
				    StringUtil.padLeft(""+((int) timeRemaining),4,' '));
		    lastReportedTime++;
		}

		double range = forwardIR.getRange();
		
		if(driving){
		    log.output("range = "+String.format("%.2f", (double) range)+". driving");
		}
		else
		    log.output("range = "+StringUtil.formatDouble(range, 2)+". stopped");
		    


		if (range > .35) {

		    driving=true;
		    leftMotor.set(200);
		    rightMotor.set(200);
		    servo.seek(0);
		} else {

		    if (driving){
			orc.lcdConsoleWrite("Look out!\n");
			driving=false;

			leftMotor.set(0);
			rightMotor.set(0);
			
			servo.seek(5);
		    }
		}
	    }

	orc.lcdConsoleWrite("Stopping! Press a key...\n");
	leftMotor.set(0);
	rightMotor.set(0);
	orc.padButtonsGet();
    }
}
