package maslab.orc;

import maslab.util.*;

/** Orc wrapper for Gyro angular rate sensor. **/
public class Gyro
{
    Orc orc;
    int port;

    GyroThread gyroThread;

    protected double gyroTheta = 0;
    public double gyroCorrectionPerTick = 625;
    public double mvPerDegreePerSecond = 5.0; // mV per degree per second.

    protected long   gyroLastTime = 0;
    protected long   gyroLastValue = 0;

    Logger log = new Logger(this);

    /** Create a new Orc object, using the default gyro port (15) **/
    public Gyro(Orc orc)
    {
	this(orc, 15);
    }

    public Gyro(Orc orc, int port)
    {
	this.orc=orc;
	this.port=port;

	gyroThread=new GyroThread();
	gyroThread.start();

	orc.pinModeWrite(port, Orc.PinMode.ANALOG_IN);

	calibrate(50); // do a really quick calibration
	gyroTheta=0;

    }

    /** Set the scale factor of the gyro, in millivolts per degree per
     * second.  The nominal value is 5. You can test this by rotating
     * the gyro 360 degrees (physically) and comparing it to the
     * calculated value. If the sensor says N degrees, set the
     * parameter to 5*N/360
     **/
    public void setRateConstant(double mvPerDegreePerSecond)
    {
	this.mvPerDegreePerSecond = mvPerDegreePerSecond;
    }

    /** Automatically perform gyro calibration by waiting a while and
     *  measuring the drift. The robot should be stationary during
     *  this period. The function will return after calibration is
     *  complete (i.e., this method is synchronous).
     *
     * @param ms Number of milliseconds to integrate over. According
     * to Analog Devices, periods greater than 5000ms are of marginal
     * benefit.
     *
     * @return The calibration value (for sanity checking; the value
     * is remembered internally).
     **/
    public synchronized double calibrate(int ms)
    {
	long starttime = orc.clockRead();
	long startvalue = orc.gyroReadRaw();

	long endtime=starttime;

	while (endtime-starttime<=0)
	    {
		try {
		    Thread.sleep(ms);
		} catch (InterruptedException ex) {
		}
		
		endtime = orc.clockRead();
	    }

	long endvalue = orc.gyroReadRaw();

	double deltavalue=orc.diff32(endvalue, startvalue);
	double deltatime=orc.diff16(endtime, starttime);

	gyroCorrectionPerTick=deltavalue/deltatime;

	log.output("dtime: "+deltatime);
	log.output("dval:  "+deltavalue);

	log.output("Gyro Correction: "+gyroCorrectionPerTick);
	log.output("(Nominal voltage: "+(gyroCorrectionPerTick*5/65535+2.5));
	gyroTheta=0;
	gyroLastTime=endtime;
	gyroLastValue=endvalue;

	return gyroCorrectionPerTick;
    }

    /** 
     * Read the (calibrated) gyro orientation.
     *
     * @return The gyro's orientation, in radians, [-PI, PI].
     **/
    public double getRadians()
    {
	update();
	return gyroTheta;
    }

    /** Reset the integrated position to zero. **/
    public void reset()
    {
	gyroTheta=0;
    }

    /** Internally update the integrator. **/
    protected synchronized void update()
    {

	long thistime=orc.clockRead();
	long thisvalue=orc.gyroReadRaw();
	
	// the Gyro value returns integral(v-2.5) with respect to t,
	// e.g., volt*seconds
	// The gyro is rated for 5mV/degree/seconds
	
	double delta = Orc.diff32(thisvalue,gyroLastValue) -
	    gyroCorrectionPerTick*Orc.diff16(thistime,gyroLastTime);
	
	double angledelta = - delta*5.0/512/mvPerDegreePerSecond*1000/65535.0/180.0*Math.PI;
	
	gyroTheta += angledelta;
	
	double degdelta=angledelta*180.0/Math.PI;
	
	gyroTheta = MathUtil.mod2pi(gyroTheta);
	
	gyroLastValue = thisvalue;
	gyroLastTime = thistime;
    }

    /** Make sure we update the gyro adequately often to avoid overflow. **/
    class GyroThread extends Thread
    {
	public GyroThread()
	{
	    setDaemon(true);
	}

	public void run()
	{
	    while(true)
		{
		    try {
			Thread.sleep(1000);
		    } catch (InterruptedException ex) {
		    }

		    update();
		}
	}
    }
}
